/* -----------------------------------------------------------------
                                 PCX.CPP

     This code sits on top of the Bitmap class structure, and
  provides services for encoding and decoding PCX files.

     Sample code for the article "Gearing Up For Games" in EDM/2.

                 Article and code by Michael T. Duffy

   ----------------------------------------------------------------- */

// ***********************************************************************
//  Header Files
// ***********************************************************************

// OS/2 type definitions

#define __IBMC__
#include <os2def.h>


#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "newtypes.hpp"
#include "canvas.hpp"
#include "pcx.hpp"

#include "errcodes.hpp"

typedef struct {
  BYTE             byManufacturer;
  BYTE             byVersion;
  BYTE             byEncoding;
  BYTE             byBitsPerPixel;
  USHORT           usXmin;
  USHORT           usYmin;
  USHORT           usXmax;
  USHORT           usYmax;
  USHORT           usHres;
  USHORT           usVres;
  BYTE             abyPalette [48];
  BYTE             byReserved;
  BYTE             byColorPlanes;
  USHORT           usBytesPerLine;
  USHORT           usPaletteType;
  BYTE             abyFiller [58];
               } PCXHEADER;
typedef PCXHEADER *  PPCXHEADER;


const ULONG        ulPcxType = 0x20584350;  // FOURCC code for "PCX "

// ***********************************************************************
//  Code
// ***********************************************************************


//........................................................................
PcxPainter::PcxPainter
//........................................................................
  (
  VOID
  )
{
SetType (ulPcxType);
};

//........................................................................
PcxPainter::~PcxPainter
//........................................................................
  (
  VOID
  )
{
DissociateBuffer ();
DissociateCanvas ();
};

//........................................................................
PcxPainter::PcxPainter
//........................................................................
  (
  PBYTE            pbyPcxIn,
  ULONG            ulBufferSizeIn
  )
{
SetType (ulPcxType);
AssociateBuffer (pbyPcxIn, ulBufferSizeIn);
};

//........................................................................
PcxPainter::PcxPainter
//........................................................................
  (
  PBYTE            pbyPcxIn,
  ULONG            ulBufferSizeIn,
  PCANVAS          pcnvCanvasIn
  )
{
AssociateBuffer (pbyPcxIn, ulBufferSizeIn);
AssociateCanvas (pcnvCanvasIn);
};

//........................................................................
ERRORSTATUS PcxPainter::AssociateBuffer
//........................................................................
  (
  PBYTE            pbyBufferIn,
  ULONG            ulBufferSizeIn
  )
// Note:  This function replaces the AssociateBuffer routine in the
//   BitmapPainter object.  It does this in order to add the ability to
//   determine whether or not the buffer contains valid information before
//   it accepts it.
{
PPCXHEADER         ppcxhdr;


// Make sure the buffer is of a valid size
if (ulBufferSizeIn <= sizeof (PCXHEADER))
  {
  usLastErrorCode = ERR_PCX_EMPTY_BUFFER;
  return (ES_ERROR);
  };

// Test the header of the buffer, to check if the encoded image is in a
//  format that the decoder understands.
ppcxhdr = (PPCXHEADER)pbyBufferIn;

if (ppcxhdr->byManufacturer == 0x0a)
  {
  // make sure the picture is the right version and color depth
  //  (i.e. 256 color PCX file)
  if ((ppcxhdr->byVersion <= 5) &&
      ((ppcxhdr->byBitsPerPixel == 8) || (ppcxhdr->byBitsPerPixel == 1)))
    {
    // The header info is valid.
    // Extract needed info from the header
    usBytesPerLine  = ppcxhdr->usBytesPerLine;

    // Accept the buffer and store it in needed internal variables.
    pbySourceBuffer = pbyBufferIn;
    ulBufferSize    = ulBufferSizeIn;
    ulBufferOffset  = sizeof (PCXHEADER);
    return (ES_NO_ERROR);
    }
  else
    {
    usLastErrorCode = ERR_PCX_BAD_FORMAT;
    return (ES_ERROR);
    };
  }
else
  {
  usLastErrorCode = ERR_PCX_NOT_PCX;
  return (ES_ERROR);
  };
};


//........................................................................
ERRORSTATUS PcxPainter::PaintCanvas
//........................................................................
  (
  VOID
  )
{


// Make sure that the Canvas exists, and that the buffer pointer is valid
if ((pbySourceBuffer == NULL) || (pcnvDestCanvas == NULL))
  {
  usLastErrorCode = ERR_GEN_MISSING_PARAMS;
  return (ES_ERROR);
  };

if (pcnvDestCanvas->Validate () == FALSE)
  {
  usLastErrorCode = ERR_CNV_BAD_CANVAS;
  return (ES_ERROR);
  };

// Clear the canvas so you also erase the areas of the canvas that the
//   PCX does not overwrite.
pcnvDestCanvas->Clear (0);

// Decode the buffer to the canvas.
UnpackBuffer ();

// If the bits per pixel equals 8, then a 256 color palette should be
//   tacked onto the end of the PCX data

if (((PPCXHEADER)pbySourceBuffer)->byBitsPerPixel == 8)
  {
  // Make sure the palette is where it should be.
  if (pbySourceBuffer [ulBufferSize - 769] == 0x0c)
    {
    // Copy the palette to the internal palette array

    memset (abyPalette, 0, sizeof (abyPalette));
    memcpy (abyPalette,
            pbySourceBuffer + ulBufferSize - 768,
            768);
    };
  };

return(ES_NO_ERROR);
};

//........................................................................
VOID PcxPainter::UnpackBuffer
//........................................................................
  (
  VOID
  )
{
LONG               lRow;
USHORT             usImageHeight;
BYTE *             pbyLineAddr;
PPCXHEADER         ppcxhdr;
USHORT             usCanvasWidth;
LONG               lCanvasRowIncrement;


// Set a pointer to the PCX header.
ppcxhdr = (PPCXHEADER)pbySourceBuffer;

// Find the height of the PCX
usImageHeight = (USHORT)(ppcxhdr->usYmax - ppcxhdr->usYmin + 1);

// Restrict the height of the PCX to the height of the canvas
usImageHeight = min (usImageHeight, pcnvDestCanvas->QueryHeight());

// Move some Canvas variables local for speed and convenience.
usCanvasWidth       = pcnvDestCanvas->QueryWidth ();
pbyLineAddr         = pcnvDestCanvas->QueryRowAddress (0);
lCanvasRowIncrement = pcnvDestCanvas->QueryRowIncrement ();

// Set the buffer index to the start of the data, just past the header.
ulBufferOffset      = sizeof (PCXHEADER);

// Step through each row of the canvas....
for (lRow = 0; lRow < usImageHeight; ++lRow)
  {
  // And decode one line depending on it's format
  if (ppcxhdr->byBitsPerPixel == 8)
    {
    ReadPcxLine8 (pbyLineAddr, usCanvasWidth);
    }
  else if (ppcxhdr->byBitsPerPixel == 1)
    {
    ReadPcxLine1 (pbyLineAddr, usCanvasWidth);
    };

  // Move to the next row of the canvas.
  pbyLineAddr += lCanvasRowIncrement;
  };
};

//........................................................................
VOID PcxPainter::ReadPcxLine8
//........................................................................
  (
  PBYTE             pbyLine,
  USHORT            usCanvasWidth
  )
// Decode a line of an 8-bit (256 color) PCX file
{
LONG                lDestIndex;
BYTE                byCodeIn;
BYTE                byRunCounter;


// Prepare to write to the beginning of the target row.
lDestIndex = 0;

// loop while there are bytes left to write.
// Decode both bytes that are written and bytes that are off the right edge
//   of the canvas so that the offset into the encoded buffer is moved
//   through the entire current line, and thus to the beginning of the next.

do
  {
  // get a key byte
  byCodeIn = pbySourceBuffer [ulBufferOffset++];

  // if it's a run of bytes field
  if ((byCodeIn & 0xc0) == 0xc0)
    {
    // and off the high bits
    byRunCounter = (BYTE)(byCodeIn & (BYTE)0x3f);

    // get the run byte
    byCodeIn = pbySourceBuffer [ulBufferOffset++];

    // run the byte
    while (byRunCounter--)
      {
      // Make sure the byte is in bounds of the canvas
      if (lDestIndex < usCanvasWidth)
        {
        // Write each byte
        pbyLine[lDestIndex++] = byCodeIn;
        };
      };
    }
  else
    {
    // It is a single byte to be written.

    // Make sure the byte is in bounds of the canvas
    if (lDestIndex < usCanvasWidth)
      {
      // Write the byte
      pbyLine[lDestIndex++] = byCodeIn;
      };
    };
  } while (lDestIndex < usBytesPerLine);
};

//........................................................................
VOID PcxPainter::ReadPcxLine1
//........................................................................
  (
  PBYTE             pbyLine,
  USHORT            usCanvasWidth
  )
// Decode a line of a 1-bit (monochrome) PCX file
//
// Warning!  Because black and white PCX files are stored with their width
//  as an increment of 8 (since there are 8 pixels per byte), this routine
//  can only write to canvases that have a width that is either a multiple
//  of 8, or is wider than the PCX width.  The code could be altered to
//  avoid this limitation, but it would require a bounds check for _every_
//  pixel, and the slowdown in unacceptable.
{
LONG                lDestIndex;
BYTE                byCodeIn;
BYTE                byRunCounter;
ULONG               ulBytesProcessed = 0;


// Prepare to write to the beginning of the target row.
lDestIndex = 0;

// loop while there are bytes left to write.
// Decode both bytes that are written and bytes that are off the right edge
//   of the canvas so that the offset into the encoded buffer is moved
//   through the entire current line, and thus to the beginning of the next.

do
  {
  // get a key byte
  byCodeIn = pbySourceBuffer [ulBufferOffset++];

  // if it's a run of bytes field
  if ((byCodeIn & 0xc0) == 0xc0)
    {
    // and off the high bits
    byRunCounter = (BYTE)(byCodeIn & (BYTE)0x3f);

    // get the run byte
    byCodeIn = pbySourceBuffer [ulBufferOffset++];

    // run the byte
    while (byRunCounter--)
      {
      // Make sure the bytes are in bounds of the canvas
      if (lDestIndex < usCanvasWidth)
        {
        // Extract and write each pixel in the byte
        pbyLine [lDestIndex++] = (byCodeIn & 0x80) ? byWhite : byBlack;
        pbyLine [lDestIndex++] = (byCodeIn & 0x40) ? byWhite : byBlack;
        pbyLine [lDestIndex++] = (byCodeIn & 0x20) ? byWhite : byBlack;
        pbyLine [lDestIndex++] = (byCodeIn & 0x10) ? byWhite : byBlack;
        pbyLine [lDestIndex++] = (byCodeIn & 0x08) ? byWhite : byBlack;
        pbyLine [lDestIndex++] = (byCodeIn & 0x04) ? byWhite : byBlack;
        pbyLine [lDestIndex++] = (byCodeIn & 0x02) ? byWhite : byBlack;
        pbyLine [lDestIndex++] = (byCodeIn & 0x01) ? byWhite : byBlack;
        };
      ++ulBytesProcessed;
      };
    }
  else
    {
    // Make sure the bytes are in bounds of the canvas
    if (lDestIndex < usCanvasWidth)
      {
      // Extract and write each pixel in the byte
      pbyLine [lDestIndex++] = (byCodeIn & 0x80) ? byWhite : byBlack;
      pbyLine [lDestIndex++] = (byCodeIn & 0x40) ? byWhite : byBlack;
      pbyLine [lDestIndex++] = (byCodeIn & 0x20) ? byWhite : byBlack;
      pbyLine [lDestIndex++] = (byCodeIn & 0x10) ? byWhite : byBlack;
      pbyLine [lDestIndex++] = (byCodeIn & 0x08) ? byWhite : byBlack;
      pbyLine [lDestIndex++] = (byCodeIn & 0x04) ? byWhite : byBlack;
      pbyLine [lDestIndex++] = (byCodeIn & 0x02) ? byWhite : byBlack;
      pbyLine [lDestIndex++] = (byCodeIn & 0x01) ? byWhite : byBlack;
      };
    ++ulBytesProcessed;
    };
  } while (ulBytesProcessed < usBytesPerLine);
};

//........................................................................
ERRORSTATUS PcxPainter::QueryBitmapStats
//........................................................................
  (
  PBITMAPSTATS     pbmsStatsOut
  )
{
PPCXHEADER         ppcxhdr;


if (pbySourceBuffer == NULL)
  {
  usLastErrorCode = ERR_GEN_MISSING_PARAMS;
  return (ES_ERROR);
  };

// Set a pointer to the bitmap header.
ppcxhdr = (PPCXHEADER) pbySourceBuffer;

// Fill the provided structure with the requested information.
pbmsStatsOut->sWidth     = (SHORT)(ppcxhdr->usXmax - ppcxhdr->usXmin + 1);
pbmsStatsOut->sHeight    = (SHORT)(ppcxhdr->usYmax - ppcxhdr->usYmin + 1);
pbmsStatsOut->byBitDepth = ppcxhdr->byBitsPerPixel;
return (ES_NO_ERROR);
};


//........................................................................
ERRORSTATUS PcxPainter::EncodeCanvas
//........................................................................
  (
  PBYTE            pbyPalette,
  PBYTE *          ppbyBufferOut,
  PULONG           pulBufferSizeOut
  )
// Returns a buffer with the canvas encoded as a PCX file.
{
PPCXHEADER         ppcxhdr;


// Make sure valid variables were passed to the routine
if ((pcnvDestCanvas   == NULL) ||
    (ppbyBufferOut    == NULL) ||
    (pulBufferSizeOut == NULL))
  {
  usLastErrorCode = ERR_GEN_MISSING_PARAMS;
  return (ES_ERROR);
  };

// Make sure canvas is ok
if (pcnvDestCanvas->Validate () == FALSE)
  {
  usLastErrorCode = ERR_CNV_BAD_CANVAS;
  return (ES_ERROR);
  };

// Setup an encode buffer
if (InitializeEncodeBuffer () == ES_ERROR)
  {
  // Error code set in InitializeEncodeBuffer
  return (ES_ERROR);
  };

// Set a pointer to the beginning of the buffer where the header is located
ppcxhdr = (PPCXHEADER) &pbyEncodeBuffer [ulEncBufferOffset];

// Assure the memory before you write to it
if (AssureEncodeMemory (sizeof (PCXHEADER)) == ES_ERROR)
  {
  // Error code set in AssureEncodeMemory
  return (ES_ERROR);
  };

// Create the header for the PCX buffer.  Save as 256 color PCX
memset (ppcxhdr, 0, sizeof (PCXHEADER));
ppcxhdr->byManufacturer   = 0x0a;
ppcxhdr->byVersion        = 0x05;
ppcxhdr->byEncoding       = 0x01;
ppcxhdr->byBitsPerPixel   = 0x08;
ppcxhdr->usXmin           = 0;
ppcxhdr->usYmin           = 0;
ppcxhdr->usXmax           = (USHORT)(pcnvDestCanvas->QueryWidth ()  - 1);
ppcxhdr->usYmax           = (USHORT)(pcnvDestCanvas->QueryHeight () - 1);
ppcxhdr->usHres           = 0;
ppcxhdr->usVres           = 0;
ppcxhdr->byColorPlanes    = 1;
ppcxhdr->usBytesPerLine   = pcnvDestCanvas->QueryWidth ();
ppcxhdr->usPaletteType    = 2;

ulEncBufferOffset += sizeof (PCXHEADER);

// Encode and write the canvas into the buffer
if (PackPcxFile () == ES_ERROR)
  {
  // Error code set in PackPcxFile
  return (ES_ERROR);
  };

// Assure the memory before you write to it
if (AssureEncodeMemory (768 + 1) == ES_ERROR)
  {
  // Error code set in AssureEncodeMemory
  return (ES_ERROR);
  };

// Write out the palette marker.
pbyEncodeBuffer [ulEncBufferOffset] = 0x0c;
++ulEncBufferOffset;

// Write out the palette.
memcpy (&pbyEncodeBuffer [ulEncBufferOffset], pbyPalette, 768);
ulEncBufferOffset += 768;

// Reduce memory useage to minimum possible
if (TrimEncodeBuffer () == ES_ERROR)
  {
  // Error code set in TrimEncodeBuffer
  return (ES_ERROR);
  };

// Fill out the return values
*ppbyBufferOut    = pbyEncodeBuffer;
*pulBufferSizeOut = ulEncBufferSize;

return (ES_NO_ERROR);
};

//........................................................................
ERRORSTATUS PcxPainter::PackPcxFile
//........................................................................
  (
  VOID
  )
{
USHORT             usRowIndex;
USHORT             usEncBytesPerLine;
USHORT             usImageHeight;
BYTE *             pbyLineAddr;
LONG               lRowIncrement;


// Load the local variables with the width and height of the PCX
usImageHeight     = pcnvDestCanvas->QueryHeight ();
usEncBytesPerLine = pcnvDestCanvas->QueryWidth ();

lRowIncrement = pcnvDestCanvas->QueryRowIncrement ();

pbyLineAddr = pcnvDestCanvas->QueryRowAddress (0);

for (usRowIndex = 0; usRowIndex < usImageHeight; ++usRowIndex)
  {
  if (WritePcxLine8 (pbyLineAddr, usEncBytesPerLine) == ES_ERROR)
    {
    // Error code set in WritePcxLine8
    return (ES_ERROR);
    };
  pbyLineAddr += lRowIncrement;
  };
return (ES_NO_ERROR);
};

//........................................................................
ERRORSTATUS PcxPainter::WritePcxLine8
//........................................................................
  (
  PBYTE             pbyLine,
  USHORT            usLineWidth
  )
{
LONG                lSourceIndex;
BYTE                byRunCounter;

// Prepare to read from the beginning of the row.
lSourceIndex = 0;

// Step through the row while there are pixels left.
do
  {
  // Count the number of similar pixels in a row.

  byRunCounter = 0;
  while (((lSourceIndex + byRunCounter) < usLineWidth) &&
         (byRunCounter < 63) &&
         (pbyLine [lSourceIndex + byRunCounter] ==
          pbyLine [lSourceIndex + byRunCounter + 1]))
    {
    ++byRunCounter;
    };

  if (byRunCounter > 0)
    {
    // You have a run of the same byte.

    // Assure the memory before you write to it
    if (AssureEncodeMemory (2) == ES_ERROR)
      {
      // Error code set in AssureEncodeMemory
      return (ES_ERROR);
      };

    // Write the run counter/marker
    pbyEncodeBuffer [ulEncBufferOffset] = (BYTE)(byRunCounter | 0xc0);
    ++ulEncBufferOffset;

    // Write the byte to duplicate
    pbyEncodeBuffer [ulEncBufferOffset] = pbyLine [lSourceIndex];
    ++ulEncBufferOffset;

    lSourceIndex += byRunCounter;
    }
  else
    {
    // You have a single pixel to write

    if ((pbyLine [lSourceIndex] & 0xc0) == 0xc0)
      {
      // data exists in the top two bits, so we must write this byte as a
      //   run length of 1

      // Assure the memory before you write to it
      if (AssureEncodeMemory (2) == ES_ERROR)
        {
        // Error code set in AssureEncodeMemory
        return (ES_ERROR);
        };

      // Write the run counter/marker
      pbyEncodeBuffer [ulEncBufferOffset] = 0xc1;
      ++ulEncBufferOffset;

      // Write the byte to duplicate
      pbyEncodeBuffer [ulEncBufferOffset] = pbyLine [lSourceIndex];
      ++ulEncBufferOffset;

      ++lSourceIndex;
      }
    else
      {
      // Assure the memory before you write to it
      if (AssureEncodeMemory (1) == ES_ERROR)
        {
        // Error code set in AssureEncodeMemory
        return (ES_ERROR);
        };

      // Write the byte
      pbyEncodeBuffer [ulEncBufferOffset] = pbyLine [lSourceIndex];
      ++ulEncBufferOffset;

      ++lSourceIndex;
      };
    };
  } while (lSourceIndex < usLineWidth);

return (ES_NO_ERROR);
};


