/* Poly.cmd
Usage: P=poly(C1,C2,...,Cn)
Creates a polynomial P of degree n-1 with coefficients C1 to Cn.  C1
is
the
coefficient of x~(n-1).
*/
numeric digits 16
p=.poly~new(arg())
do i=1 to arg()
  p[i]=arg(i)
end
return p

::class poly

::method init
  expose coeff count
  parse arg count
  coeff=.array~new(count)

::method '[]='
  expose coeff
  numeric digits 16
  parse arg x,i
  coeff[i]=x

::method '[]'
  expose coeff count
  numeric digits 16
  parse arg i
  return coeff[i]

::method degree
  expose coeff count
  if arg()>0 then do
    parse arg n
    count=n+1
  end
  else return count-1

::method string
  return polyfmt(self)

::method copy
  expose coeff count
  numeric digits 16
  x=.poly~new(count)
  do i=1 to count
    x[i]=coeff[i]
  end
  return x

::method value
  expose coeff count
  numeric digits 16
  use arg x
  n=count
  z=coeff[1]
  do k=2 to n
    z=(x*z)+coeff[k]
  end
  return z

::method '+'
  numeric digits 16
  use arg x
  if datatype(x)='NUM' then x=poly(x)
  cx=x~degree+1
  cs=self~degree+1
  cp=max(cx,cs)
  p=.poly~new(cp)
  do i=1 to cp
    p[i]=0
  end
  do i=cp-cs+1 to cp
    p[i]=p[i]+self[cs-cp+i]
  end
  do i=cp-cx+1 to cp
    p[i]=p[i]+x[cx-cp+i]
  end
  return p~reduce

::method '-'
  numeric digits 16
  use arg x
  if arg(1,'omitted') then return self*-1
  else return (self+x*-1)~reduce

::method '*'
  expose coeff count
  numeric digits 16
  use arg q
  if datatype(q)='NUM' then q=poly(q)
  dp=count
  dq=q~degree+1
  dr=dp+dq-1
  r=.poly~new(dr)
  do i=1 to dr
    r[i]=0
  end
  do i=1 to dp
    do j=1 to dq
      k=i+j-1
      r[k]=r[k]+coeff[i]*q[j]
    end
  end
  return r

::method '/'
  numeric digits 16
  num=self~copy
  use arg den
  if datatype(den)='NUM' then den=poly(den)
  dnum=num~degree+1
  dden=den~degree+1
  dquot=dnum-dden+1
  if dquot<1 then return poly(0)
  quot=.poly~new(dquot)
  do i=1 to dquot
    quot[i]=num[i]/den[1]
    do j=1 to dden
      k=j+i-1
      num[k]=num[k]-quot[i]*den[j]
    end
  end
  return quot

::method '//'   /* remainder */
  numeric digits 16
  num=self
  use arg den
  quot=num/den
  rem=num-den*quot
  return rem~reduce

::method reduce /* remove leading 0 terms */
  numeric digits 16
  if abs(self[1])>1e-9 then return self
  d=self~degree
  if d=0 then return self
  do i=2 to d
    if abs(self[i])>1e-9 then leave
  end
  x=.poly~new(d+2-i)
  do k=i to d+1
    x[k-i+1]=self[k]
  end
  return x

::method '**'
  numeric digits 16
  parse arg n
  if n=0 then return poly(1)
  x=self~copy
  do n-1
    x=x*self
  end
  return x

::method ddx  /* derivative of a polynomial */
  numeric digits 16
  n=self~degree
  r=.poly~new(n)
  do i=1 to n
    r[i]=(n+1-i)*self[i]
  end
  return r

::method gcf         /* greatest common factor */
  numeric digits 16
  u=self
  use arg v
  if u~degree<v~degree then do
    p=v
    q=u
  end
  else do
    p=u
    q=v
  end

  r=p//q
  do while r~degree>0
    p=q
    q=r
    r=p//q
  end

  if abs(r[1])<1e-9 then return q
  else return poly(1)
