/*
 * update_prob.c --
 * Chris Hyde
 * created: 
 *
 * Modified: Dean Collins Sat Feb 12 13:47:00 1994
 * Added unistd.h inclusion for SunOS as requested by
 * magnush@isy.liu.se (Magnus Hammerin).
 *
 * Modified: Dean Collins Wed Feb 23 17:00:42 1994
 * Changed PTSROOT (a constant) to PtsRoot (a variable).
 *
 * Modified: Dean Collins Sat Apr 09 23:49:04 1994
 * Changed order of #includes for SCO.
 *
 * Modified: Dean Collins Wed Nov 16 12:38:50 PST 1994
 * Added code for priority field of index record.
 *
 * Modified: Chris Hyde 1/10/95
 * Removed include <stdio.h>, since it's now included in zdbm.h.
 * Made changes to I/O to reflect zdbm names (zread(), zwrite(), etc.)
 * Replaced direct references to the index records with functions/macros
 *   defined for that purpose.
 *
 * Modified: Dean Collins Thu Jun  8 13:31:27 PDT 1995
 * Replaced #ifdef DEBUG method of debug output with calls to
 * the zdebug macros.
 *
 */

/*
 * Copyright (c) 1992,1993,1994,1995 Christopher G. Hyde
 * Copyright (c) 1992 University of Idaho, Moscow, Idaho.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation free of charge for any purpose is hereby granted without
 * fee, provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear in
 * supporting documentation, and that the name of the University of Idaho
 * not be used in advertising or publicity pertaining to distribution of
 * the software without specific, written prior permission.  The University
 * of Idaho makes no representations about the suitability of this software
 * for any purpose.  It is provided "as is" without express or implied warranty.
 *
 * THE UNIVERSITY OF IDAHO DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 * SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS.
 * IN NO EVENT SHALL THE UNIVERSITY OF IDAHO BE LIABLE FOR ANY SPECIAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
 * LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE
 * OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */

#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/file.h>
#include <sys/stat.h>
#ifdef SUNOS
#include <unistd.h>
#endif /*SUNOS*/

#include "zdbm.h"


int solve_problem(char *db_path, char *prid)
{
 index_record index;
 char tprid[MAXPRID],
      tpath[DBPATHLEN];
 ZFILE *temp;

 zdebug("\nIn solve_problem()\n");
 zdebug2("\tSolving %s/%s\n", db_path, prid);

 if ((temp=zopen(db_path, ".index", "r+", INDEX))==NULL)
    return(FAIL);

 if (find_problem(temp, prid, &index)==NULL)
 {  /* error locating desired problem */
    zclose(temp);
    return(FAIL);
 }

 zdebug("solve_problem() - found problem to solve\n");

 if (is_gone(&index))
 {  /* you can't solve a deleted problem */
    db_errorno=DELETED;
    zclose(temp);
    return(FAIL);
 }
    
 if (is_solved(&index, ""))
 {  /* problem has already been solved */
    zclose(temp);
    return(SUCESS);
 }

 if (is_linked(&index)) /* problem does not reside in this leaf */
 {

    zdebug("solve_problem() - problem given is linked\n");

/* since find_problem overwrites the data in index, we need to store the
 * path and prid of the original problem somewhere else.
 */
    nl_strncpy(tprid, get_ln_prid(&index), MAXPRID);
    nl_strncpy(tpath, get_ln_path(&index), DBPATHLEN);
    zclose(temp); /* we don't need this index anymore */

    if ((temp=zopen(tpath, ".index", "r+", INDEX))==NULL)
    {
       return(FAIL);
    }

    if (find_problem(temp, tprid, &index)==NULL)
    {   /* error locating original problem */
       zclose(temp);
       return(FAIL);
    }

    zdebug("solve_problem() - found the original link\n");

    if (solve_record(temp, &index, tpath, tprid)==FAIL)
    {
       zclose(temp);
       return(FAIL);
    }
    else
    {
       zclose(temp);
       return(SUCESS);
    } /* end else */
 } /* end LINKED problem */
 else
 { /* the problem is not linked */
    if (solve_record(temp, &index, db_path, prid)==FAIL)
    { /* problem solving problem */
       zclose(temp);
       return(FAIL);
    }
    else
    {
       zclose(temp);
       return(SUCESS);
    }
 } /* end else not LINKED */
} /* end solve_problem */

 
int solve_record(ZFILE *fp, index_record *index, char *path, char *prid)
{
 int test;

 zdebug("In solve_record()\n");
 zdebug2("\trecord to solve - %s/%s\n", path, prid);

 if (zlock(fp)==FAIL)
    return(FAIL);

 set_sf(index, SOLVED);

/* since fp is pointing to the end of the record we want to update, we need
 * to move the pointer back one record
 */

 zseek(fp, -(long)sizeof(index_record), SEEK_CUR);
 if (zwrite(index, sizeof(index_record), 1, fp)<1)
 {
    db_errorno=errno;
    zunlock(fp);
    return(FAIL);
 }

 zdebug("solve_record() - wrote out new index entry\n");

 if (exists(PtsRoot, path, prid, ".rep_old"))
    test=append(path, prid, ".rep", prid, ".rep_old");
 else
    test=rename_file(path, prid, ".rep", path, prid, ".rep_old");

 zunlock(fp);
 if (test==SUCESS)
    remove_file(path, prid, ".rep");

 return(test);
} /* end solve_record() */

 
int delete_problem(char *db_path, char *prid, int perm)
{
 index_record index;
 ZFILE *temp;

 zdebug("\nIn delete_problem()\n");
 zdebug1("\tPath to problem - %s\n", db_path);
 zdebug1("\tProblem ID - %s\n", prid);
 zdebug1("\tDelete permission - %d\n", perm);

 db_errorno=0;

 if ((temp=zopen(db_path, ".index", "r+", INDEX))==NULL)
    return(FAIL);

 if (find_problem(temp, prid, &index)==NULL)
 {
    zclose(temp);
    return(FAIL);
 }

 zdebug("delete_problem() - Found problem to delete\n");

/* if the problem to be deleted is an original link or the problem has been
 * requested for reopening, and no permission is given then return an error
 */
 if ((is_org_link(&index) || is_reopen_prob(&index)) && (perm==DELETENOTOK)) 
 {
    db_errorno=LINKERR;
    zclose(temp);
    return(FAIL);
 }

 if (is_in_edit(&index))  /* Typo fixed by DC, 2/13/93 */
 { /* the problem is currently being edited */
    db_errorno=ETXTBSY;
    zclose(temp);
    return(FAIL);
 }

/* reset to the begining of the current index entry, the one to be deleted */
 zseek(temp, -(long)(sizeof(index_record)), SEEK_CUR);
 set_gone(&index);

/* lock the index file before updating the index entry */
 if (zlock(temp)==FAIL)
 {
    zclose(temp);
    return(FAIL);
 }

/* write out the new index entry */
 if (zwrite(&index, sizeof(index_record), 1, temp)<1)
 {  /* the write failed */
    db_errorno=errno;
    zunlock(temp);
    zclose(temp);
    return(FAIL);
 }
 else /* the write succeeded */
 {
    zunlock(temp);
    zclose(temp);
 }

 zdebug("delete_problem() - Marked index entry of problem as deleted\n");
 zdebug("delete_problem() - Now removing associated files\n");

/* after updating the index, remove all the associated files, if they exist */
 remove_file(db_path, prid, ".rep");
 remove_file(db_path, prid, ".rep_old");
 remove_file(db_path, prid, ".log");
 return(SUCESS);
} /* end delete_problem */

 
char *add_problem(reporter_record *rep_rec, char *problem_description,
          char *short_description, char *db_path, char priority)
{
 index_record index;
 ZFILE *temp,
      *temp2;
 long size;
 static char *tptr;

 zdebug("\nIn add_problem()\n");
 zdebug1("\treporter_record.account_name = %s\n", rep_rec->account_name);
 zdebug1("\treporter_record.node_name = %s\n", rep_rec->node_name);
 zdebug1("\treporter_record.date_time = %s\n", rep_rec->date_time);
 zdebug1("\tproblem_description = %s\n", problem_description);
 zdebug1("\tshort_description = %s\n", short_description);
 zdebug1("\tdb_path = %s\n", db_path);
 zdebug1("\tpriority = %d\n", priority);

 db_errorno=0;

/* set up the new index record */
 set_gone(&index);
 set_sf(&index, UNSOLVED);
 set_edit(&index, FALSE);
 set_account_name(&index, "", 1);
 set_node_name(&index, "", 1);
 set_pid(&index, 0);
 set_sd(&index, short_description, MAXSUMLEN);
 set_ln_prid(&index, "", 1);
 set_ln_path(&index, "", 1);
 set_priority(&index, priority);
 set_fr(&index, "", 1);

 zdebug("add_problem() - created new index record\n");

 if ((temp=zopen(db_path, ".index", "w+", INDEX))==NULL)
    if (errno==ENOENT) /* the index does not exist */
       if (make_branch(db_path)==FAIL) /* so create it */
          return(NULL); /* there was a problem creating the new index */
       else /* we  created the index, so open it for update */
          if ((temp=zopen(db_path, ".index", "w+", INDEX))==NULL)
              return(NULL);
          else; /* dummy else to go with if above */
    else /* we couldn't open the file for some other reason, so quit */
       return(NULL);

 if (zlock(temp)==FAIL)
 {  /* the lock failed */
    zclose(temp);
    return(NULL);
 }
 if ((size=file_size(temp, sizeof(index_record)))==ERROR)
 {  /* could not get file size */
    zunlock(temp);
    zclose(temp);
    return(NULL);
 }
 if ((tptr=new_prid(size))==NULL)
 {  /* error getting new id */
    zunlock(temp);
    zclose(temp);
    return(NULL);
 }

 zdebug1("add_problem() - got new problem id - %s\n", tptr);

 set_prid(&index, tptr, MAXPRID);

/* now that the record has been created, move to the end of the index file
 * and write it */
 zseek(temp, 0L, SEEK_END);
 if (zwrite(&index, sizeof(index_record), 1, temp)<1)
 {  /* the write failed */
    db_errorno=errno;
    zunlock(temp);
    zclose(temp);
    return(NULL);
 }

 zdebug("add_problem() - wrote out new index record\n");

/* else the write suceeded, so unlock the index and create the problem files */
 zunlock(temp);

    /* create the reporter file */
 if ((temp2=create_file(db_path, get_prid(&index), ".rep",
            FILE_PERMS))==NULL)
 {  /* the creation failed */
    zclose(temp);
    return(NULL);
 }

 zdebug("add_problem() - created reporter file for new problem\n");

 if (zwrite(rep_rec, sizeof(reporter_record), 1, temp2)<1)
 {  /* the write failed */
    db_errorno=errno;
    zclose(temp);
    zclose(temp2);
    remove_file(db_path, get_prid(&index), ".rep");
    return(NULL);
 }
 zclose(temp2);

 zdebug("add_problem() - wrote out reporter record\n");

/* making the reporter file suceeded, so create the log file */
if (write_log(db_path, tptr, problem_description)==FAIL)
{
   zclose(temp);
   remove_file(db_path, tptr, ".rep");
   return(FAIL);
}

 zdebug("add_problem() - created log file for new problem\n");

/* the two problem files were created, so go back and mark the index entry 
 * as not deleted, so it will be visable to the user.
 */
 zseek(temp, -(long)sizeof(index_record), SEEK_CUR);
 set_mrd(&index, 0);
 if (zwrite(&index, sizeof(index_record), 1, temp)<1)
 {  /* the write failed */
    db_errorno=errno;
    zclose(temp);
    remove_file(db_path, get_prid(&index), ".rep");
    remove_file(db_path, get_prid(&index), ".log");
    return(NULL);
 }

 zdebug("add_problem() - updated index entry\n");

/* else the whole thing suceeded, so return the new prid */
 zclose(temp);
 return(tptr);
} /* end add_problem */

 
char *add_reporter(char *db_path, char *prid, reporter_record *rep_rec)
{
 index_record index;
 char *tprid,
      *path;
 ZFILE *temp;

 zdebug("\nIn add_reporter()\n");
 zdebug2("\tPath = %s.\n\tPrid = %s.\n", db_path, prid);

 db_errorno=0;

 if ((temp=zopen(db_path, ".index", "r+", INDEX))==NULL)
    return(NULL);
 if (find_problem(temp, prid, &index)==NULL)
 {  /* could not find problem requested */
    zclose(temp);
    return(NULL);
 }
 zclose(temp); /* we found what we needed */

 zdebug("add_reporter() - found index record\n");

/* now check to be sure a reporter can be added */
 if (is_gone(&index))
 {  /* problem was deleted, so you can't add a reporter */
    db_errorno=DELETED;
    return(NULL);
 }
 if (is_solved(&index, ""))
 {  /* can't add a reporter to a solved problem */
    db_errorno=ISSOLVED;
    return(NULL);
 }

 if (is_linked(&index)) /* problem does not reside in this leaf, so */
 {  /* update the entry */
    path=get_ln_path(&index); /* path and tprid are used to access files */
    tprid=get_ln_prid(&index);
 } /* end if LINKED problem */
 else /* the problem resides in this leaf, so assign values to path and tprid */
 {
    path=db_path;
    tprid=prid;
 } /* end else problem not LINKED */

/* we know where the reporter file is (path) and its name (tprid), so now 
 * append the reporter record to the end of the reporter file 
 */
 if (append_reporter(path, tprid, rep_rec)==FAIL)
    return(NULL); /* the appending failed */
 else
    return(prid); /* the appending succeed */
} /* end add_reporter */

 
int append_reporter(char *db_path, char *prid, reporter_record *rep_rec)
{
 static char fn[FNAMELEN];
 ZFILE *temp;

 zdebug("\nIn append_reporter()\n");
 zdebug1("\tDB Path = %s\n", db_path);
 zdebug1("\tPRID = %s\n", prid);

 sprintf(fn, "/%s.rep", prid);
 if ((temp=zopen(db_path, fn, "r+", REP))==NULL)
    return(FAIL);
 if (zlock(temp)==FAIL)
 {  /* the lock failed */
    zclose(temp);
    return(FAIL);
 }
 if (double_reporter(temp, rep_rec))
 {  /* reporter is already listed */
    zunlock(temp);
    zclose(temp);
    return(FAIL);
 }

/* no double reporter, so move to end of file and add the new reporter */
 zseek(temp, 0L, SEEK_END);
 if (zwrite(rep_rec, sizeof(reporter_record), 1, temp)<1)
 {  /* the write failed */
    db_errorno=errno;
    zunlock(temp);
    zclose(temp);
    return(FAIL);
 }
/* else the write suceeded, so cleanup and signal sucess */
 zunlock(temp);
 zclose(temp);
 return(SUCESS);
} /* end append_reporter */

 
int double_reporter(ZFILE *db_pointer, reporter_record *rep_rec)
{
 reporter_record in_rec;
 int found=FALSE;

 zdebug("\nIn double_reporter()\n");

 while (found==FALSE)
 {
    if (zread(&in_rec, sizeof(reporter_record), 1, db_pointer)<1)
    {
       if (zerror(db_pointer))
       { /* there was an I/O error */
          db_errorno=errno;
          found=ERROR;

          zdebug1("double_reporter() - I/O error %d\n", errno);

       } /* end I/O error */
#ifdef DEBUG
       else
          zdebug("double_reporter() - no doubles\n");
#endif
       break;
    }

    zdebug1("double_reporter() - account name looking for '%s'\n", 
            rep_rec->account_name);
    zdebug1("double_reporter() - current account name '%s'\n",
            in_rec.account_name);

    if (strcmp(in_rec.account_name, rep_rec->account_name)==0)
    {
       db_errorno=DOUBLEREP;
       found=TRUE;

       zdebug("double_reporter() - found a double\n");
    }
 }
 return(found);
} /* end double_reporter */

/* end update_prob.c */
