/*
 * edit_prob.c --
 * Chris Hyde
 * created: 
 *
 * Modified: Dean Collins Sat Feb 12 18:46:02 1994
 * Fixed several typos.
 * 
 * Modified: Dean Collins Wed Feb 23 17:00:42 1994
 * Changed PTSROOT (a constant) to PtsRoot (a variable).
 *
 * Modified: Dean Collins Wed Nov 23 10:40:27 PST 1994
 * Added priority-handling code to edit_log() and unlock_from_edit().
 *
 * Modified: Chris Hyde 1/10/95
 * Removed include <stdio.h>, since it's now included in zdbm.h.
 * Made changes to I/O to reflect zdbm names (zread(), zwrite(), etc.)
 * Replaced direct references to the index records with functions/macros
 *   defined for that purpose.
 *
 * Modified: Dean Collins Thu Jun  8 13:31:27 PDT 1995
 * Replaced #ifdef DEBUG method of debug output with calls to
 * the zdebug macros.
 */

/*
 * Copyright (c) 1992,1993,1994,1995 Christopher G. Hyde
 * Copyright (c) 1992 University of Idaho, Moscow, Idaho.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation free of charge for any purpose is hereby granted without
 * fee, provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear in
 * supporting documentation, and that the name of the University of Idaho
 * not be used in advertising or publicity pertaining to distribution of
 * the software without specific, written prior permission.  The University
 * of Idaho makes no representations about the suitability of this software
 * for any purpose.  It is provided "as is" without express or implied warranty.
 *
 * THE UNIVERSITY OF IDAHO DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 * SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS.
 * IN NO EVENT SHALL THE UNIVERSITY OF IDAHO BE LIABLE FOR ANY SPECIAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
 * LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE
 * OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */

#include <string.h>
#include <sys/param.h>
#include <errno.h>
#include <pwd.h>
#ifdef SUNOS
#include <unistd.h>
#endif /*SUNOS*/

#include "zdbm.h"

/* the only global variable in zdbm, it is set when an error occurs, it set
 * to either a standard errno (in the case of a an error of a library function
 * or to a zdbm error code (see zdbm.h) in the case of a zdbm error.
 *
 * this is defined as an extern in zdbm.h
 */

 int db_errorno;

 
int edit_log(char *db_path, char *prid, char *mod_log, int mode,
         signed char newpriority, char *short_description)
{
 char tprid[MAXPRID];
 char tpath[DBPATHLEN];
 index_record index;
 ZFILE *temp;

 zdebug("\nIn edit_log()\n");
 zdebug3("\t(Un)Marking problem %s%s/%s\n", PtsRoot, db_path, prid);
 zdebug1("\tMarking mode = %d\n", mode);

 db_errorno=0;

 if ((temp=zopen(db_path, ".index", "r+", INDEX))==NULL)
    return(FAIL);

 if (find_problem(temp, prid, &index)==NULL)
 {  /* could not find problem requested */
    zclose(temp);
    return(FAIL);
 }

 zdebug("edit_log() - found problem to edit, or unedit\n");

 if (is_linked(&index)) /* problem does not reside in leaf given */
 {  /* so get the original problem record */

 zdebug("edit_log() - problem given is a link\n");

/* since the call to find_problem() destroys the data in index while it is 
 * trying to find the problem, the prid and the path of the original link 
 * need to be separate.
 */
    nl_strncpy(tprid, get_ln_prid(&index), MAXPRID); 
    nl_strncpy(tpath, get_ln_path(&index), DBPATHLEN);
    zclose(temp);

/* open the index of the original problem, and get it */
    if ((temp=zopen(tpath, ".index", "r+", INDEX))==NULL)
       return(FAIL);

    if (find_problem(temp, tprid, &index)==NULL)
    {   /* could not find problem requested */
       zclose(temp);
       return(FAIL);
    }

    zdebug("edit_log() - found original problem\n");

 } /* end if LINKED problem */
 else
 {  /* we're using tpath and tprid for the edit log */
    nl_strncpy(tprid, prid, MAXPRID);
    nl_strncpy(tpath, db_path, DBPATHLEN);
 }  /* end else */
 
/* now that we have the real problem, check to be sure it can be edited */

 zdebug("edit_log() - checking validity of call\n");

 if (is_gone(&index))
 {  /* problem was deleted */
    db_errorno=DELETED;
    zclose(temp);
    return(FAIL);
 }

 if (zlock(temp)==FAIL)
 {  /* lock failed */
    zclose(temp);
    return(FAIL);
 }

/* problem can be edited, more or less, now perform action indicated by mode */
switch(mode)
{
    case START:
            if (lock_for_edit(temp, &index)==FAIL)
            {   /* could not lock the record for editing */
               zunlock(temp);
               zclose(temp);
               return(FAIL);
            }
            zdebug("edit_log() - started edit session\n");

            break;
   
    case END:       
            if (unlock_from_edit(temp, &index, tpath, mod_log,
                                        newpriority, short_description) == FAIL)
            {   /* could not unlock the record */
               zunlock(temp);
               zclose(temp);
               return(FAIL);
            }
            zdebug("edit_log() - ended edit session\n");

            break;

    default:
	    db_errorno=BADTYPE;
            zunlock(temp);
            zclose(temp);
            return(FAIL);
 } /* end switch */
/* there were no errors, so clean up and return sucess */
 zunlock(temp);
 zclose(temp);
 return(SUCESS);
} /* end edit_log */

 
int lock_for_edit(ZFILE *db_pointer, index_record *index)
{
 struct passwd *pwd;

 zdebug("\nIn lock_for_edit()\n");

 if (is_in_edit(index))
 {  /* the problem is currently being edited */
    db_errorno=errno=ETXTBSY;
    return(FAIL);
 }

 if (is_solved(index, ""))
 {  /* can't edit a solved problem */
    db_errorno=ISSOLVED;
    return(FAIL);
 }

/* OK to edit problem, so reset to the begining of the index entry and update
 * it.
 */

 zseek(db_pointer, -(long)(sizeof(index_record)), SEEK_CUR);

 zdebug("lock_for_edit() - marking record as being in edit\n");

 set_pid(index, get_pid(index));
 gethostname(get_node_name(index), MAXHOSTNAMELEN);
 pwd=getpwuid(geteuid());
 set_account_name(index, pwd->pw_name, MAXANAMELEN);
 set_edit(index, TRUE);

 if (zwrite(index, sizeof(index_record), 1, db_pointer)<1)
 {  /* error writing new entry */
    db_errorno=errno;
    return(FAIL);
 }
 else
 {
    zdebug("lock_for_edit() - updated record as being in edit\n");

    return(SUCESS);
 } /* end else */
} /* end lock_for_edit */

 
int unlock_from_edit(ZFILE *db_pointer, index_record *index, char *db_path,
            char *new_log, signed char newpriority,
            char *short_description)
{
 struct passwd *pwd;

 zdebug("\nIn unlock_from_edit()\n");

 if (!is_in_edit(index)) /* file is not currently being edited */
 {
    zdebug("unlock_from_edit() - problem is not being edited\n");
    db_errorno=NOTINEDIT;
    return(FAIL);
 }

 pwd=getpwuid(geteuid());
 if (strcmp(get_acc_name(index), pwd->pw_name)!=0)
 {  /* only the person that started the edit can remove it */
    db_errorno=NOTOWNER;
    return(FAIL);
 }

 if (new_log!=NULL) /* there is something to write out */
 {
    zdebug("unlock_from_edit() - writing out new problem log\n");

    if (write_log(db_path, get_prid(index), new_log)==FAIL)
       return(FAIL);

    zdebug("unlock_from_edit() - wrote out new problem log\n");

 } /* end != NULL */
#ifdef DEBUG
 else
    zdebug("unlock_from_edit() - log is NULL\n");
#endif
 
/* we can remove the edit lock now, so do it */
 zseek(db_pointer, -(long)(sizeof(index_record)), SEEK_CUR);

 set_edit(index, FALSE); 
 set_priority(index, newpriority);
 if (short_description != NULL)
    nl_strncpy(get_sd(index), short_description, MAXSUMLEN) ;

 zdebug("unlock_from_edit() - removing edit lock\n");

 if (zwrite(index, sizeof(index_record), 1, db_pointer) == 0)
 {  /* write failed */
    db_errorno=errno;
    return(FAIL);
 }
 else
 {
    zdebug("unlock_from_edit() - removed edit lock\n");

    return(SUCESS);
 } /* end else */
} /* end unlock_from_edit */

 
int remove_edit(char *db_path, char *prid)
{
 index_record index;
 ZFILE *temp;
 char tprid[MAXPRID];

 zdebug("\nIn remove_edit()\n");
 zdebug2("\tProblem to remove the edit lock from - %s/%s.\n", db_path, prid);

 db_errorno=0;

 if ((temp=zopen(db_path, ".index", "r+", INDEX))==NULL)
    return(FAIL);

 if (find_problem(temp, prid, &index)==NULL)
 {  /* could not find problem specified */
    zclose(temp);
    return(FAIL);
 }

 zdebug("remove_edit() - found problem given\n");

 if (is_linked(&index)) /* problem does not reside in leaf given */
 {

 zdebug("remove_edit() - problem given is linked, getting original problem\n");

/* since find_problem() overwrites the data in index while searching, the
 * prid needs to be seaperate
 */

    zclose(temp);
    if ((temp=zopen(get_ln_path(&index), ".index", "r+", INDEX))==NULL)
       return(FAIL);
    nl_strncpy(tprid, get_ln_prid(&index), MAXPRID);
    if (find_problem(temp, tprid, &index)==NULL)
    {   /* could not find problem specified */
    zclose(temp);
    return(FAIL);
    }

    zdebug("remove_edit() - found original problem\n");

 } /* end LINKED problem */

/* index is now the correct record, and temp is the right index file */

 zdebug("remove_edit() - updating index entry\n");

 set_edit(&index, FALSE);
 zseek(temp, -(long)(sizeof(index_record)), SEEK_CUR);
 if (zlock(temp)==FAIL)
 {  /* the lock failed */
    zclose(temp);
    return(FAIL);
 }
 if (zwrite(&index, sizeof(index_record), 1, temp) == 0)
 {  /* the write failed */
    db_errorno=errno;
    zunlock(temp);
    zclose(temp);
    return(FAIL);
 }
 zunlock(temp);
 zclose(temp);

 zdebug("remove_edit() - updated index entry\n");

 return(SUCESS);
} /* end remove_edit */

 
edit_record *edit_info(char *db_path, char *prid)
{
 index_record index;
 static edit_record er;
 ZFILE *temp;

 zdebug("\nIn edit_info()\n");
 zdebug2("\tProblem to get edit info on  - %s/%s\n", db_path, prid);

 if ((temp=zopen(db_path, ".index", "r+", INDEX))==NULL)
    return(NULL);

 if (find_problem(temp, prid, &index)==NULL)
 {  /* could not find problem specified */
    zclose(temp);
    return(NULL);
 }

 zdebug("edit_info() - found problem given\n");
 zdebug("edit_info() - creating edit_record\n");

/* now that we have an index record, copy the appropriate parts into
 * the info record and return it
 */

 er.ef=get_edit(&index); /* the edit flag */ 
 /* who owns the edit */
 nl_strncpy(er.account_name, get_acc_name(&index), MAXANAMELEN);
 /* where is it running */
 nl_strncpy(er.node_name, get_node_name(&index), MAXHOSTNAMELEN);
 er.pid=get_pid(&index); /* what owns the edit */
 zclose(temp);
 return(&er);
} /* end edit_info */

/* end of edit_prob.c */
