/*
 * $Id: timing.c,v 1.2 1994/11/18 08:52:52 lm Exp $
 *
 * Copyright (c) 1994 Larry McVoy.   
 *
 * All output goes to stderr.
 */
#include "bench.h"
#include <stdio.h>
#include <signal.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/time.h>

#ifdef	RUSAGE
#include <sys/resource.h>

#define	secs(tv)	(tv.tv_sec + tv.tv_usec / 1000000.0)
#endif

#define	nz(x)	((x) == 0 ? 1 : (x))
#define	MB	(1024*1024.0)
#define	KB	(1024.0)

#ifdef	RUSAGE

#define	mine(f)		(ru_stop.f - ru_start.f)

static struct rusage ru_start, ru_stop;

void
rusage()
{
	struct rusage r;
	double  sys, user, idle;
	double  per;
	double  timespent();

	sys = secs(ru_stop.ru_stime) - secs(ru_start.ru_stime);
	user = secs(ru_stop.ru_utime) - secs(ru_start.ru_utime);
	idle = timespent() - (sys + user);
	per = idle / timespent() * 100;
	fprintf(stderr, "real=%.2f sys=%.2f user=%.2f idle=%.2f stall=%.0f%% ",
	    timespent(), sys, user, idle, per);
	fprintf(stderr, "rd=%d wr=%d min=%d maj=%d ctx=%d\n",
	    mine(ru_inblock), mine(ru_oublock),
	    mine(ru_minflt), mine(ru_majflt),
	    mine(ru_nvcsw) + mine(ru_nivcsw));
}

#endif

static struct timeval start_tv, stop_tv;

/*
 * Start timing now.
 */
void
start()
{
	(void) gettimeofday(&start_tv, (struct timezone *) 0);
#ifdef	RUSAGE
	getrusage(RUSAGE_SELF, &ru_start);
#endif
}

/*
 * Stop timing and return real time in microseconds.
 */
ulong
stop()
{
	struct timeval tdiff;

	(void) gettimeofday(&stop_tv, (struct timezone *) 0);
#ifdef	RUSAGE
	getrusage(RUSAGE_SELF, &ru_stop);
#endif

	tvsub(&tdiff, &stop_tv, &start_tv);
	return (tdiff.tv_sec * 1000000 + tdiff.tv_usec);
}

ulong
now()
{
	struct timeval t;

	(void) gettimeofday(&t, (struct timezone *) 0);
	return (t.tv_usec / 100);
}

/*
 * Adjust time spent by usec amount.
 *
 * XXX - tv_usec may be too big after this.
 */
void
adjust(usec)
	int	usec;
{
	int     sec = usec / 1000000;

	usec -= sec * 1000000;
	stop_tv.tv_sec += sec;
	stop_tv.tv_usec += usec;
}

void
bandwidth(bytes, verbose)
	int	bytes;
{
	struct timeval td;
	double  s, bs;

	tvsub(&td, &stop_tv, &start_tv);
	s = td.tv_sec + td.tv_usec / 1000000.0;
	bs = bytes / nz(s);
	if (verbose) {
		(void) fprintf(stderr, "%.2f MB in %.2f secs, %.2f KB/sec\n",
		    bytes / MB, s, bs / KB);
	} else {
		(void) fprintf(stderr, "%.2f %.2f\n",
		    bytes / MB, bs / MB);
	}
}

void
kb(bytes)
	int	bytes;
{
	struct timeval td;
	double  s, bs;

	tvsub(&td, &stop_tv, &start_tv);
	s = td.tv_sec + td.tv_usec / 1000000.0;
	bs = bytes / nz(s);
	(void) fprintf(stderr, "%.0f KB/sec\n", bs / KB);
}

void
mb(bytes)
	int	bytes;
{
	struct timeval td;
	double  s, bs;

	tvsub(&td, &stop_tv, &start_tv);
	s = td.tv_sec + td.tv_usec / 1000000.0;
	bs = bytes / nz(s);
	(void) fprintf(stderr, "%.2f MB/sec\n", bs / MB);
}

void
latency(xfers, size)
	int	xfers, size;
{
	struct timeval td;
	double  s;

	tvsub(&td, &stop_tv, &start_tv);
	s = td.tv_sec + td.tv_usec / 1000000.0;
	fprintf(stderr,
	    "%d xfers in %.2f secs, %.4f millisec/xfer, %.2f KB/sec\n",
	    xfers, s, s * 1000 / xfers,
	    (xfers * size) / (1024. * s));
}

void
context(xfers)
	int	xfers;
{
	struct timeval td;
	double  s;

	tvsub(&td, &stop_tv, &start_tv);
	s = td.tv_sec + td.tv_usec / 1000000.0;
	fprintf(stderr,
	    "%d context switches in %.2f secs, %.0f microsec/switch\n",
	    xfers, s, s * 1000000 / xfers);
}

void
micro(s, n)
	char	*s;
	int	n;
{
	struct timeval td;
	int	micro;

	tvsub(&td, &stop_tv, &start_tv);
	micro = td.tv_sec * 1000000 + td.tv_usec;
	fprintf(stderr, "%s: %d microseconds\n", s, micro / n);
}

void
milli(s, n)
	char	*s;
	int	n;
{
	struct timeval td;
	int	milli;

	tvsub(&td, &stop_tv, &start_tv);
	milli = td.tv_sec * 1000 + td.tv_usec / 1000;
	fprintf(stderr, "%s: %d milliseconds\n", s, milli / n);
}

void
ptime(n)
	int	n;
{
	struct timeval td;
	double  s;

	tvsub(&td, &stop_tv, &start_tv);
	s = td.tv_sec + td.tv_usec / 1000000.0;
	fprintf(stderr,
	    "%d in %.2f secs, %.0f microseconds each\n", n, s, s * 1000000 / n);
}

void
tvsub(tdiff, t1, t0)
	struct timeval *tdiff, *t1, *t0;
{
	tdiff->tv_sec = t1->tv_sec - t0->tv_sec;
	tdiff->tv_usec = t1->tv_usec - t0->tv_usec;
	if (tdiff->tv_usec < 0)
		tdiff->tv_sec--, tdiff->tv_usec += 1000000;
}

double
timespent()
{
	struct timeval td;

	tvsub(&td, &stop_tv, &start_tv);
	return (td.tv_sec + td.tv_usec / 1000000);
}
