/*
 * tcp_lib.c - routines for managing TCP connections.
 *
 * Copyright (c) 1994 Larry McVoy.
 */
#include	"bench.h"
#include	<sys/types.h>
#include	<sys/socket.h>
#include	<netinet/in.h>
#include	<netdb.h>
#include	<arpa/inet.h>

u_short	pmap_getport();

/*
 * Get a TCP socket, bind it, figure out the port,
 * and advertise the port as program "prog".
 *
 * XXX - it would be nice if you could advertise ascii strings.
 */
tcp_server(prog)
	u_long	prog;
{
	int	sock, namelen;
	struct	sockaddr_in s;

	if ((sock = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP)) < 0) {
		perror("socket");
		exit(1);
	}
	bzero((char*)&s, sizeof(s));
	s.sin_family = AF_INET;
#ifdef	NO_PORTMAPPER
	s.sin_port = htons(prog);
#endif
	if (bind(sock, (struct sockaddr*)&s, sizeof(s)) < 0) {
		perror("bind");
		exit(2);
	}
	if (listen(sock, 1) < 0) {
		perror("listen");
		exit(4);
	}
#ifndef	NO_PORTMAPPER
	namelen = sizeof(s);
	if (getsockname(sock, (struct sockaddr *)&s, &namelen) < 0) {
		perror("getsockname");
		exit(3);
	}
	(void)pmap_unset(prog, (u_long)1);
	if (!pmap_set(prog, (u_long)1, (u_long)IPPROTO_TCP,
	    (u_long)ntohs(s.sin_port))) {
		perror("pmap_set");
		exit(5);
	}
#endif
	return (sock);
}

/*
 * Unadvertise the socket
 */
tcp_done(prog)
{
	pmap_unset((u_long)prog, (u_long)1);
}

/*
 * Accept a connection and return it
 */
tcp_accept(sock)
{
	struct	sockaddr_in s;
	int	newsock, namelen;

	namelen = sizeof(s);
	bzero((char*)&s, namelen);

	if ((newsock = accept(sock, (struct sockaddr*)&s, &namelen)) < 0) {
		perror("accept");
		exit(6);
	}
	return (newsock);
}

/*
 * Connect to the TCP socket advertised as "prog" on "host" and
 * return the connected socket.
 *
 * Hacked Thu Oct 27 1994 to cache pmap_getport calls.  This saves
 * about 4000 usecs in loopback lat_connect calls.  I suppose we
 * should time gethostbyname() & pmap_getprot(), huh?
 */
tcp_connect(host, prog)
	char	*host;
	u_long	prog;
{
	static	struct hostent *h;
	static	struct sockaddr_in sin;
	static	u_short	port;
	static	u_long save_prog;
	static	char *save_host;
	int	sock;
	int     sockbuf = SOCKBUF;

	if ((sock = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP)) < 0) {
		perror("socket");
		exit(1);
	}
#ifndef	NO_SETSOCKOPT
	while (setsockopt(sock, SOL_SOCKET, SO_SNDBUF, &sockbuf, sizeof(int))) {
		sockbuf -= 128;
	}
#endif
	if (!h || host != save_host || prog != save_prog) {
		save_host = host;	/* XXX - counting on their not
					 * changing it - benchmark only`
					 */
		save_prog = prog;
		if (!(h = gethostbyname(host))) {
			perror(host);
			exit(2);
		}
		bzero((char *) &sin, sizeof(sin));
		sin.sin_family = AF_INET;
		bcopy(h->h_addr, (char *) &sin.sin_addr, h->h_length);
#ifdef	NO_PORTMAPPER
		sin.sin_port = prog;
#else
		port = pmap_getport(&sin, prog, (u_long)1, IPPROTO_TCP);
		if (!port) {
			perror("lib TCP: No port found");
			exit(3);
		}
		sin.sin_port = htons(port);
#endif
	}
	if (connect(sock, (struct sockaddr*)&sin, sizeof(sin)) < 0) {
		perror("connect");
		exit(4);
	}
	return (sock);
}
