/* Copyright (c) 1987, 1988  Stanley T. Shebs, University of Utah. */
/* This program may be used, copied, modified, and redistributed freely */
/* for noncommercial purposes, so long as this notice remains intact. */

/* RCS $Header: /home/users/brossard/X/commands/x11conq/RCS/order.c,v 2.0 90/10/20 12:50:02 brossard Exp Locker: brossard $ */

#include "config.h"
#include "misc.h"
#include "dir.h"
#include "period.h"
#include "side.h"
#include "unit.h"

char *ordernames[] = ORDERNAMES;   /* full names of orders */
char *dirnames[] = DIRNAMES;       /* short names of directions */
char orderbuf[BUFSIZE];            /* buffer for printed form of an order */
char oargbuf[BUFSIZE];             /* buffer for order's arguments */

int orderargs[] = ORDERARGS;       /* types of parameters for each order */

/* General routine to wake a unit up (and maybe all its cargo). */

wake_unit(unit, wakeocc, reason, other)
Unit *unit, *other;
int reason;
bool wakeocc;
{
    Unit *occ;

    if (humanside(unit->side) && unit->movesleft > 0 && !unit->side->loopagain) {
	unit->side->loopagain = TRUE;
    }
    unit->orders.type = AWAKE;
    unit->orders.rept = 0;
    unit->orders.flags = NORMAL;
    unit->wakeup_reason = reason;
#ifdef AUTO
    if (reason == WAKEFULL) {
	unit->orders.morder = FALSE;
    }
#endif
    if (reason == WAKEOWNER)
	unit->area = area_index(unit->x, unit->y);
    if (reason == WAKEENEMY) {
	unit->waking_side = other->side;
	unit->waking_type = other-> type;
	unit->area = area_index(unit->x, unit->y);
    }
    if (wakeocc) {
	for_all_occupants(unit, occ) wake_unit(occ, wakeocc, reason, other);
    }
}

/* Stash a "wakeup call" - will only be for main unit, not occupants. */

cache_awake(side)
Side *side;
{
    side->tmporder->type = AWAKE;
    side->tmporder->rept = 0;
#ifdef AUTO
    side->tmporder->morder = FALSE;
#endif
    finish_teach(side);
}

/* Give a unit sentry orders. */

order_sentry(unit, n)
Unit *unit;
int n;
{
    unit->orders.type = SENTRY;
    unit->orders.rept = n;
}

/* Stash sentry orders. */

cache_sentry(side, n)
Side *side;
int n;
{
    side->tmporder->type = SENTRY;
    side->tmporder->rept = n;
    finish_teach(side);
}

/* Fill in the given unit with direction-moving orders. */

order_movedir(unit, dir, n)
Unit *unit;
int dir, n;
{
    unit->orders.type = MOVEDIR;
    unit->orders.p.dir = dir;
    unit->orders.rept = n;
}

cache_movedir(side, dir, n)
Side *side;
int dir, n;
{
    side->tmporder->type = MOVEDIR;
    side->tmporder->p.dir = dir;
    side->tmporder->rept = n;
    finish_teach(side);
}

/* Give the unit orders to move to a given place - it only needs to do this */
/* once, repetition is nonsensical. */

order_moveto(unit, x, y)
Unit *unit;
int x, y;
{
    unit->orders.type = MOVETO;
    unit->orders.rept = 1;
    unit->orders.p.pt[0].x = x;
    unit->orders.p.pt[0].y = y;
}

cache_moveto(side, x, y)
Side *side;
int x, y;
{
    side->tmporder->type = MOVETO;
    side->tmporder->rept = 1;
    side->tmporder->p.pt[0].x = x;
    side->tmporder->p.pt[0].y = y;
    finish_teach(side);
}

/* Order to follow an edge just needs to remember the direction. */

order_edge(unit, d, n)
Unit *unit;
int d, n;
{
    unit->orders.type = EDGE;
    unit->orders.rept = n;
    unit->orders.p.dir = d;
}

cache_edge(side, d, n)
Side *side;
int d, n;
{
    side->tmporder->type = EDGE;
    side->tmporder->rept = n;
    side->tmporder->p.dir = d;
    finish_teach(side);
}

/* Order to follow a unit just needs the unit to follow. */

order_follow(unit, leader, n)
Unit *unit, *leader;
int n;
{
    unit->orders.type = FOLLOW;
    unit->orders.rept = n;
#ifdef LEADER_ID
    unit->orders.p.leader_id = leader->id;
#else
    unit->orders.p.leader = leader;
#endif
}

cache_follow(side, leader, n)
Side *side;
Unit *leader;
int n;
{
    side->tmporder->type = FOLLOW;
    side->tmporder->rept = n;
#ifdef LEADER_ID
    side->tmporder->p.leader_id = leader->id;
#else
    side->tmporder->p.leader = leader;
#endif
    finish_teach(side);
}

/* A two-waypoint patrol suffices for many purposes. */
/* Should have a more general patrol routine eventually (> 2 waypoints). */

order_patrol(unit, x0, y0, x1, y1, n)
Unit *unit;
int x0, y0, x1, y1, n;
{
    unit->orders.type = PATROL;
    unit->orders.rept = n;
    unit->orders.p.pt[0].x = x0;
    unit->orders.p.pt[0].y = y0;
    unit->orders.p.pt[1].x = x1;
    unit->orders.p.pt[1].y = y1;
}

cache_patrol(side, x0, y0, x1, y1, n)
Side *side;
int x0, y0, x1, y1, n;
{
    side->tmporder->type = PATROL;
    side->tmporder->rept = n;
    side->tmporder->p.pt[0].x = x0;
    side->tmporder->p.pt[0].y = y0;
    side->tmporder->p.pt[1].x = x1;
    side->tmporder->p.pt[1].y = y1;
    finish_teach(side);
}

/* Wait around to embark on some unit when it comes to our location. */

order_embark(unit, n)
Unit *unit;
int n;
{
    unit->orders.type = EMBARK;
    unit->orders.rept = n;
}

cache_embark(side, n)
Side *side;
int n;
{
    side->tmporder->type = EMBARK;
    side->tmporder->rept = n;
    finish_teach(side);
}

#ifdef AUTO
/* Note that this leaves all other orders the same. */

cache_auto(side)
Side *side;
{
    Order *oldorder;

    if ((oldorder = side->sounit->standing->orders[side->soutype]) != NULL)
	copy_orders(side->tmporder, oldorder);
    else {
	side->tmporder->type = AWAKE;
	side->tmporder->rept = 0;
    }
    side->tmporder->morder = TRUE;
    finish_teach(side);
}
#endif

/* Switch from standing order teaching mode back to normal, and confirm */
/* that the standing order has been set. */

finish_teach(side)
Side *side;
{
    Unit *occ;

    side->teach = FALSE;
    if (side->sounit->standing->orders[side->soutype] != NULL)
	free(side->sounit->standing->orders[side->soutype]);
    side->sounit->standing->orders[side->soutype] = side->tmporder;
#ifdef AUTO
    notify(side, "%s has orders for %s to %s%0s.",
	unit_handle(side, side->sounit), utypes[side->soutype].name,
	order_desig(side->tmporder),
	((side->tmporder->morder) ? " and auto" : ""));
#else
    notify(side, "%s has orders for %s to %s.",
	   unit_handle(side, side->sounit), utypes[side->soutype].name,
	   order_desig(side->tmporder));
#endif
    for_all_occupants(side->sounit, occ) {
	if (occ->type == side->soutype) {
	    get_standing_orders(occ, side->sounit);
	}
    }
    show_timemode(side);
}

/* Display orders in some coherent fashion.  Use the information about the */
/* types of order parameters to decide how to display them. */

char *
order_desig(orders)
Order *orders;
{
    switch (orderargs[orders->type]) {
    case NOARG:
	sprintf(oargbuf, "");
	break;
    case DIR:
	sprintf(oargbuf, "%s ", dirnames[orders->p.dir]);
	break;
    case POS:
	sprintf(oargbuf, "%d,%d ", orders->p.pt[0].x, orders->p.pt[0].y);
	break;
    case LEADER:
#ifdef LEADER_ID
	sprintf(oargbuf, "%s ", unit_handle((Side *) NULL,
			find_unit(orders->p.leader_id)));
#else
	sprintf(oargbuf, "%s ", unit_handle((Side *) NULL, orders->p.leader));
#endif
	break;
    case WAYPOINTS:
	sprintf(oargbuf, "%d,%d %d,%d ",
		orders->p.pt[0].x, orders->p.pt[0].y,
		orders->p.pt[1].x, orders->p.pt[1].y);
	break;
    default:
	case_panic("order arg type", orderargs[orders->type]);
    }
    if (orders->rept > 1)
	sprintf(orderbuf, "%s %s(%d)",
		ordernames[orders->type], oargbuf, orders->rept);
    else
	sprintf(orderbuf, "%s %s",
		ordernames[orders->type], oargbuf);
    return orderbuf;
}

/* Yeah yeah, assignment statements supposedly copy structures. */

copy_orders(dst, src)
Order *dst, *src;
{
    dst->type = src->type;
    dst->rept = src->rept;
    dst->flags = src->flags;
    switch (orderargs[src->type]) {
    case NOARG:
	break;
    case DIR:
	dst->p.dir = src->p.dir;
	break;
    case POS:
	dst->p.pt[0].x = src->p.pt[0].x;
	dst->p.pt[0].y = src->p.pt[0].y;
	break;
    case LEADER:
#ifdef LEADER_ID
	dst->p.leader_id = src->p.leader_id;
#else
	dst->p.leader = src->p.leader;
#endif
	break;
    case WAYPOINTS:	
	dst->p.pt[0].x = src->p.pt[0].x;
	dst->p.pt[0].y = src->p.pt[0].y;
	dst->p.pt[1].x = src->p.pt[1].x;
	dst->p.pt[1].y = src->p.pt[1].y;
	break;
    default:
	case_panic("order arg type", orderargs[src->type]);
    }
#ifdef AUTO
    dst->morder = src->morder;
#endif
}
