/* Copyright (c) 1987, 1988  Stanley T. Shebs, University of Utah. */
/* This program may be used, copied, modified, and redistributed freely */
/* for noncommercial purposes, so long as this notice remains intact. */

/* RCS $Header: /home/users/brossard/X/commands/x11conq/RCS/ginit.c,v 2.0 90/10/20 12:48:24 brossard Exp Locker: brossard $ */

/* Initialization and random routines for the display part of xconq. */

#include "config.h"
#include "misc.h"
#include "dir.h"
#include "period.h"
#include "side.h"
#include "unit.h"
#include "map.h"
#include "global.h"

/* Always use four lines for the unit info display, even on small screens. */

#define INFOLINES 4

int maxnamelen = 0;		/* length of longest side name+host */

/* Find a unit and put it at center (or close to it) of first view.  I guess */
/* can't use put_on_screen because not enough stuff is inited yet. */

init_curxy(side)
Side *side;
{
    Unit *unit;

    for_all_units(unit) {
	if (unit->side == side) {
	    side->cx = unit->x;  side->cy = unit->y;
	    side->vcx = unit->x;
	    side->vcy = max(unit->y, side->vw2+1);
	    side->vcy = min(side->vcy, (world.height-1) - side->vh2);
	    return;
	}
    }
    side->vcx = side->vw2;  side->vcy = side->vh2;
    side->cx = side->vcx;  side->cy = side->vcy;
}

/* The very first step in using X is to open up all the desired displays. */
/* In our case, there are many displays each with many windows.  If the */
/* display opens successfully, do an initial "redraw" to set the screen up. */

init_displays()
{
    int len;
    Side *side;

    for_all_sides(side) {
	len = 8 + strlen(side->name) + 1;
	maxnamelen = max(maxnamelen,
			 len + (side->host ? hostlen(side->host) + 3 : 0));
    }
    for_all_sides(side) {
	if (side->host != NULL) {
	    init_display(side);
	    if (active_display(side)) {
		init_curxy(side);
	    }
	}
    }
}

/* The length of the printed host name is the length till the first */
 /* period or the total length */

int hostlen(str)
char str[];
{
    register int len = 0;

    while ((str[len] != '.') && (str[len] != '\0')) len++;
    return len;
}

/* Open display, create all the windows we'll need, do misc setup things, */
/* and initialize some globals to out-of-range values for recognition later. */

init_display(side)
Side *side;
{
    int i;

    if (Debug) printf("Will try to open display \"%s\" ...\n", side->host);

    if (!open_display(side)) {
	fprintf(stderr, "Display \"%s\" could not be opened!\n", side->host);
	exit_xconq();
    }
    active_display(side);   /* done for side effect */
    init_colors(side);
    init_misc(side);
    init_sizes(side);
    create_main_window(side);
    side->msg =
	create_window(side, 0, 0,
		      side->lw, side->nh * side->fh);
    side->info =
	create_window(side, 0, side->nh * side->fh + side->bd,
		      side->lw, INFOLINES * side->fh);
    side->prompt =
	create_window(side, 0, (side->nh + INFOLINES) * side->fh + 2*side->bd,
		      side->lw, side->fh);
    side->map =
	create_window(side, 0, side->th,
		      side->vw * side->hw,
		      side->vh * side->hch + (side->hh - side->hch));
    set_retain(side, side->map);
    side->sides =
	create_window(side, side->lw + side->bd + 1, 0,
		      side->sw * side->fw, numsides * side->fh);
    side->timemode =
	create_window(side, side->lw + side->fw, side->th - 4 * side->fh,
		      2 * side->margin + 8 * side->fw, 2 * side->fh);
    side->clock =
	create_window(side, side->lw + 13 * side->fw, side->th - 2 * side->fh,
		      8 * side->fw, side->fh);
    side->state =
	create_window(side, side->lw + 1, side->th,
		      22 * side->fw, period.numutypes*max(side->hh, side->fh));
    if (world_display(side)) {
	side->world =
	    create_window(side,
			  side->lw+side->bd, side->mh-side->mm*world.height,
			  world.width * side->mm, world.height * side->mm);
	set_retain(side, side->world);
    }
    create_help_window(side);
    fixup_windows(side);
    enable_input(side);
    for (i = 0; i < MAXNOTES; ++i) side->noticebuf[i] = " ";
    for (i = 0; i < MAXUTYPES; ++i) side->bvec[i] = 0;
    /* Flag some values as uninitialized */
    side->vcx = side->vcy = -1;
    side->lastvcx = side->lastvcy = -1;
    side->lastx = side->lasty = -1;
    if (Debug) printf("Successfully opened \"%s\"!\n", side->host);
}

/* Decide/compute all the sizes of things.  Our main problem here is that */
/* the display might be smaller than we really desire, so things have to */
/* be adjusted to fit. */

init_sizes(side)
Side *side;
{
    int alw, abh;

    alw = (3 * display_width(side)) / 4;
    side->vw = min(world.width, alw / side->hw);
    side->nw = min(BUFSIZE, alw / side->fw);
    side->sw = maxnamelen;
    abh = (2 * display_height(side)) / 3;
    side->vh = min(world.height, abh / side->hch);
    side->nh = max(1, min((abh/side->fh)/2 - 5, MAXNOTES));
    side->mm = min(5, (max(world.width, side->fw*side->sw) / world.width));
    set_sizes(side);
    if (side->vw < MINWIDTH || side->vh < MINHEIGHT) {
	fprintf(stderr, "Display \"%s\" is too small!\n", side->host);
	exit_xconq();
    }
}

/* This fn is a "ten-line horror"; effectively a mini tiled window manager */
/* that keeps the subwindows from overlapping no matter what the display and */
/* requested sizes are.  Siemens eat your heart out... */

set_sizes(side)
Side *side;
{
    int ulhgt, llhgt, urhgt, lrhgt;

    /* Make sure map window dimensions are OK */
    side->vw = min(world.width, side->vw);
    side->vh = min(world.height, side->vh);
    side->vw2 = side->vw / 2;  side->vh2 = side->vh / 2;
    /* Compute subregion sizes (left/right upper/lower width/height) */
    side->lw = max(side->nw * side->fw, side->hw * side->vw);
    side->rw = max((world_display(side) ? world.width * side->mm : 0),
		   side->sw * side->fw);
    urhgt = (numsides + 4) * side->fh + side->bd;
    ulhgt = side->fh * (side->nh + INFOLINES + 1) + 3 * side->bd;
    side->th = max(ulhgt, urhgt);
    lrhgt = period.numutypes * max(side->hh, side->fh);
    if (world_display(side)) lrhgt += side->mm * world.height + side->bd;
    llhgt = side->hch * side->vh + (side->hh - side->hch);
    side->bh = max(llhgt, lrhgt);
    side->mw = side->lw + side->bd + side->rw + 1;
    side->mh = side->th + side->bd + side->bh + 1;
    /* Only vcy needs adjustment, since vcx can never be close to an edge */
    side->vcy = min(max(side->vh2, side->vcy), (world.height-1)-side->vh2);
}

/* Acquire a set of colors.  There are too many to make it feasible to */
/* customize them via .Xdefaults, so we don't even try.  If there aren't */
/* enough colors, drop into monochrome mode.  This doesn't take the window */
/* manager into account - it may have grabbed some color space. */

init_colors(side)
Side *side;
{
    if (Debug) printf("%d colors available ...\n", display_colors(side));
    side->monochrome = (display_colors(side) <= 2);
    side->bonw = (side->monochrome ? BLACKONWHITE : FALSE);
    set_colors(side);
}

/* This will set up the correct set of colors at any point in the game. */
/* Note that terrain colors get NO value if in monochrome mode. */
/* If the colors requested are not available, it is up to the graphics */
/* interface to supply a substitute. */

set_colors(side)
Side *side;
{
    int	t;
    long fg, bg;

    fg = (side->bonw ? black_color(side) : white_color(side));
    bg = (side->bonw ? white_color(side) : black_color(side));
    side->bgcolor = side->owncolor = side->altcolor = side->diffcolor = bg;
    side->fgcolor = side->bdcolor = side->graycolor = side->enemycolor = fg;
    side->neutcolor = side->goodcolor = side->badcolor = fg;
    if (!side->monochrome) {
	for_all_terrain_types(t) {
	    side->hexcolor[t] = request_color(side, ttypes[t].color);
	}
	side->owncolor = request_color(side, "black");
	side->altcolor = request_color(side, "blue");
	side->diffcolor = request_color(side, "maroon");
	side->bdcolor = request_color(side, "blue");
	side->graycolor = request_color(side, "light gray");
	side->enemycolor = request_color(side, "red");
	side->neutcolor = request_color(side, "light gray");
	side->goodcolor = request_color(side, "green");
	side->badcolor = request_color(side, "red");
    }
}

/* Move windows and change their sizes to correspond with the new sizes of */
/* viewports, etc. */

reconfigure_display(side)
Side *side;
{
    int sy, sdy;

    if (active_display(side)) {
	set_sizes(side);
	set_colors(side);
	reset_misc(side);
	sy = 0;  sdy = side->nh * side->fh;
	change_window(side, side->msg, 0, sy, side->lw, sdy);
	sy += sdy + side->bd;  sdy = INFOLINES * side->fh;
	change_window(side, side->info, 0, sy, side->lw, sdy);
	sy += sdy + side->bd;  sdy = 1 * side->fh;
	change_window(side, side->prompt, 0, sy, side->lw, sdy);
	change_window(side, side->map, 0, side->th,
		      side->vw * side->hw,
		      side->vh * side->hch + (side->hh - side->hch));
	change_window(side, side->timemode,
		      side->lw + side->fw, side->th - 4 * side->fh, -1, -1);
	change_window(side, side->sides, side->lw+1, 0, -1, -1);
	change_window(side, side->state, side->lw+1, side->th, -1, -1);
	if (world_display(side)) {
	    change_window(side, side->world,
			  side->lw + 1, side->mh - world.height * side->mm,
			  world.width * side->mm, world.height * side->mm);
	}
	change_window(side, side->main, -1, -1, side->mw, side->mh);
    }
}

