/* Copyright (c) 1987, 1988  Stanley T. Shebs, University of Utah. */
/* Copyright 1988 by Chris D. Peterson, MIT. */
/* Many improvements by Tim Moore, University of Utah. */
/* This program may be used, copied, modified, and redistributed freely */
/* for noncommercial purposes, so long as this notice remains intact.  */

/* RCS $Header: /home/users/brossard/X/commands/x11conq/RCS/X11.c,v 2.0 90/10/20 12:47:30 brossard Exp Locker: brossard $
 *
 * $Log:	X11.c,v $
 * Revision 2.0  90/10/20  12:47:30  brossard
 * First version that tries to merge changes from 5.4
 * 
 * Revision 1.7  90/07/30  17:05:18  brossard
 * Fixed the conversion from hex to display (unwrap), always take
 * into account the y when doing the unwrapping.
 * 
 * Revision 1.6  90/07/30  11:41:38  brossard
 * Better check for an invalid DISPLAY
 * 
 * Revision 1.5  90/07/21  15:37:21  brossard
 * Check if XOpenDisplay fails
 * 
 * Revision 1.4  90/07/19  18:12:33  brossard
 * The character representing the side number are not in sequence
 * 
 * Revision 1.3  90/07/17  18:34:47  brossard
 * Removed get_input (now in pinput.c)
 * Removed the wait for the expose event when mapping the help window
 * 
 * Revision 1.2  90/07/16  10:34:33  brossard
 * Expose events are handled on a per window basis:  this means that
 * help.c doesn't write anything to its window until it has received
 * an expose event.
 * Split X11.c the creation of gc's is now in Xgc.c
 * Changed flush_input to only flush events dealing with key or buttons
 * presses.  It also deals with expose events.  It used to flush expose
 * events!
 * 
 * 
 */

/* Interface implementations for the X11 version of xconq. */

#include "config.h"
#include "misc.h"
#include "period.h"
#include "side.h"
#include "unit.h"
#include "map.h"
#include "global.h"

/* careful of the path of the X header files. */

#ifdef UNIX
#include <signal.h>   /* needed for ^C disabling */
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#endif UNIX

#include "X11.h"

/* This is the name of a family of programs, so argv[0] inadequate. */

#define PROGRAMNAME "x11conq"

/* Name of default font - why is this wired in? */

#define STANDARD "x11conq_standard"

/* Magic number meaning that this pixmap intentionally left blank. */

#define NOPIXMAP 17

#define INFOLINES 4

#define BH 32

/* The array of screen objects. */

Screeno screens[MAXSIDES];      /* All the "screen objects" */

/* Values of parameters generally tied to fonts and the like. */

int hw = 20;
int hh = 22;
int hch = 17;
int margin = 2;
int bd = 1;

int helpwinlines = 1;           /* size of help window */

/* Put in a default player, probably the invoker of the program. */
/* An empty host name will confuse everybody. */

add_default_player()
{
#ifdef UNIX
    add_player(TRUE, getenv("DISPLAY"));
#endif UNIX
}

/* Function to print debugging info when a segmentation violation or */
/* bus error occurs. */

void program_crash(sig, code, scp, addr)
int sig, code;
struct sigcontext *scp;
char *addr;
{
  int i;
  static bool already_been_here = FALSE;

  close_displays();
  printf("Fatal error encountered. Signal %d code %d\n", sig, code);
  if (already_been_here) exit(1);
  else {
    already_been_here = TRUE;
    write_savefile("emergency.save.xconq");
    if (sig != 2)
      abort(1);
    else exit(1);
  }
}

/* Handlers for X catastrophes attempt to do a save first. */
/* Could be more friendly and check success of save, but can't do anything */
/* about it anyway... */

static int
handle_x_error (disp, evt)
Display *disp;
XErrorEvent *evt;
{
    static int num_errors = 0;
    char buf[BUFSIZE];

    XGetErrorText(disp, evt->error_code, buf, BUFSIZE);
    printf("\nX error on display %s: %s\n", DisplayString(disp), buf);
    if (++num_errors >= 10) {
        printf("\nX error: trying emergency save!\n");
        write_savefile(SAVEFILE);
      exit(1);
    }
}

static int
handle_xio_error (disp)
Display *disp;
{
    printf("\nX IO error on display %s: trying emergency save!\n",
       DisplayString(disp));
    write_savefile(SAVEFILE);
    exit(1);
}

/* Ignore ^C if humans in the game, do it otherwise, including when the */
/* last human player turns into a machine (this is called by option cmd). */
/* Attempts to be more clever seem to be bad news. */

init_sighandlers()
{
#ifdef UNIX
    signal(SIGINT, SIG_IGN);
    signal(1, program_crash);
    signal(SIGHUP, program_crash);
    signal(SIGBUS, program_crash);
    signal(SIGSEGV, program_crash);
    signal(SIGFPE, program_crash);
    signal(SIGILL, program_crash);
    signal(SIGSYS, program_crash);
    signal(SIGINT, program_crash);
    signal(SIGQUIT, program_crash);
    signal(SIGTERM, program_crash);
#endif UNIX
    XSetIOErrorHandler(handle_xio_error);
    XSetErrorHandler(handle_x_error);
}

/* Note that the open_display function syncronizes the X server when the */
/* Debug flag is set.  Returns TRUE if succeeded.  */

open_display(side)
Side *side;
{
    side->display = (long) &(screens[side_number(side)]);
    if( ! (sdd() = XOpenDisplay(side->host) )) {
	side->display = (long) NULL;
	return FALSE;
    }
    if (Debug) {
      XSynchronize(sdd(), TRUE);
      printf("Synching the X server.\n");
    }
    side->main = XCreateSimpleWindow(sdd(), DefaultRootWindow(sdd()),
				     50, 3,
				     display_width(side), display_height(side),
				     3, white_color(side), black_color(side));
    return TRUE;
}

/* A predicate that tests whether our display can safely be written to. */

active_display(side)
Side *side;
{
    return (side && side->host && !side->lost && side->display);
}

display_width(side)
Side *side;
{
    return ((19 * XDisplayWidth(sdd(), DefaultScreen(sdd()))) / 20);
}

display_height(side)
Side *side;
{
    return ((19 * XDisplayHeight(sdd(), DefaultScreen(sdd()))) / 20);
}

/* Most X displays have enough screen to do a world map. */

world_display(side) Side *side; {  return TRUE;  }

/* Could use handlers for X failures... */

/* Do misc setup thingies. */

/* Do the rigmarole to convert all those short arrays into X-approved */
/* Bitmaps.  Note that this has to be for *each* display separately (!) */
/* Also get the cursor shape.  If the hardware can't hack the desired */
/* cursor size, warn about it and just use the root window's cursor. */
/* 0x0 cursor specs seem to be don't cares - machine can handle any size */
/* X11 also needs gazillions of GCs, since we've got so many fonts and */
/* colors and bitmaps. */

init_misc(side)
Side *side;
{
    char *name;
    side->margin = margin;
    side->bd = bd;
    side->hw = hw;
    side->hh = hh;
    side->hch = hch;

    if (
#ifdef RESTORE
    !midturnrestore &&
#endif
      (name = XGetDefault(sdd(), PROGRAMNAME, "SideName")) != NULL)
      side->name = name;
    sd()->helpfont = open_font(side, HELPFONT, "HelpFont", sd()->textfont);
    get_font_size(sd()->helpfont, &(side->hfw), &(side->hfh) );
    sd()->textfont = open_font(side, TEXTFONT, "TextFont", NULL);
    get_font_size(sd()->textfont, &(side->fw), &(side->fh) );
    sd()->iconfont = open_font(side, ICONFONT, "IconFont", sd()->textfont);
    get_font_size(sd()->iconfont, &(side->hw), &(side->hh) );
    if (period.fontname != NULL && strlen(period.fontname) > 0) {
	sd()->unitfont = 
	    open_font(side, period.fontname, "UnitFont", sd()->textfont);
	get_font_size(sd()->unitfont, &(side->uw), &(side->uh) );
    } else {
      	sd()->unitfont = 
	    open_font(side, STANDARD, "UnitFont", sd()->textfont);
	get_font_size(sd()->unitfont, &(side->uw), &(side->uh) );
    }

    create_gc( side );
}

/* Since font lookup is still not smart among Xs, this is a general routine */
/* that can deal with unopenable, missing, etc, fonts, as well as the use of */
/* .Xdefaults.  One of the inputs is another font that can be substituted if */
/* necessary. */

XFontStruct *
open_font(side, name, xdefault, altfont)
Side *side;
char *name, *xdefault;
XFontStruct * altfont;
{
    char *firstname, *altname;
    XFontStruct * font;

    if ((altname = XGetDefault(sdd(), PROGRAMNAME, xdefault)) != NULL)
	name = altname;
    firstname = name;
    if ((font = XLoadQueryFont(sdd(), name)) == NULL) {
	make_pathname(XCONQLIB, name, "snf", spbuf);
	name = spbuf;
	if ((font = XLoadQueryFont(sdd(), name)) == NULL) {
	    fprintf(stderr, "Can't open fonts \"%s\" or \"%s\" on \"%s\"\n",
		    firstname, name, side->host);
	    if (altfont != NULL) {
		fprintf(stderr, "Substituting another font...\n");
		return altfont;
	    } else {
		fprintf(stderr, "No font to substitute!!\n");
		exit(1);
	    }
	}
    }
    if (Debug) printf("Opened font \"%s\" ...\n", name);
    return font;
}

/* Force X11 font fanciness into semblance of X10 font plainness. */

void
get_font_size(font, width, height)
XFontStruct * font;
short *width, *height;
{
    *width = font->max_bounds.rbearing - font->min_bounds.lbearing;
    *height = font->max_bounds.ascent + font->max_bounds.descent;
    if (Debug) {
	printf("rbearing = %d lbearing = %d\n",
	       (int)font->max_bounds.rbearing, (int)font->min_bounds.lbearing);
	printf("returning font size %d %d\n", (int)*width, (int)*height);
    }
}

/* This routine has to be able to cope with window managers constraining */
/* size.  Actually, the main window was already opened when the display */
/* was opened, so the name is not quite accurate! */

create_main_window(side)
Side *side;
{
    Pixmap dots;
    XSizeHints hints;
		  
    XStoreName(sdd(), side->main, PROGRAMNAME);
    dots = (side->bonw ? sd()->bwdots : sd()->wbdots); 
    XSetWindowBackgroundPixmap(sdd(), side->main, dots);

    hints.width = side->mw;
    hints.height = side->mh;
    hints.min_width = side->mw;
    hints.min_height = side->mh;
    hints.flags = PSize|PMinSize;
    XSetNormalHints(sdd(), side->main, &hints);
}

/* Help window is not necessarily a subwindow, though it might be sometimes. */

create_help_window(side)
Side *side;
{
    helpwinlines =
	max(HELP_WINDOW_SIZE,
		(24 + period.numrtypes + period.numttypes + period.numutypes));

    side->help = XCreateSimpleWindow(sdd(), DefaultRootWindow(sdd()),
				     0, 0, 80*side->fw+1,
				     helpwinlines*side->hfh+1,
				     1, side->fgcolor, side->bgcolor);
    XStoreName(sdd(), side->help, "xconq-help");
}

/* Subwindow creator. */

create_window(side, x, y, w, h)
Side *side;
int x, y, w, h;
{
    return XCreateSimpleWindow(sdd(), side->main, x, y, w, h,
			       1, side->fgcolor, side->bgcolor);
}

/* force x to retain this window. */
set_retain(side, window)
Side *side;
long window;
{
  XSetWindowAttributes foo;
  char *retain;

  if ((retain = XGetDefault(sdd(), PROGRAMNAME, "Retain")) == NULL ||
      (*retain != 'N' && *retain != 'n') ) {
    foo.backing_store = 1;
    XChangeWindowAttributes(sdd(), window, CWBackingStore, &foo);
  }
}

/* Do little things necesary to make it all go, in this case mapping all */
/* the windows (except help win). */

fixup_windows(side)
Side *side;
{
    XMapWindow(sdd(), side->main);
    XMapSubwindows(sdd(), side->main);
    if (!global.giventime) XUnmapWindow(sdd(), side->clock);
    create_cursor( side );
}

/* Specify the sorts of input that will be allowed - main window needs to */
/* see mouse buttons so unit type selection works right. */

enable_input(side)
Side *side;
{
    XSelectInput(sdd(), side->main, KeyPressMask);
    XSelectInput(sdd(), side->help, KeyPressMask|ExposureMask);
    XSelectInput(sdd(), side->msg, ExposureMask|ButtonPressMask);
    XSelectInput(sdd(), side->info, ExposureMask);
    XSelectInput(sdd(), side->prompt, ExposureMask);
    XSelectInput(sdd(), side->map, ButtonPressMask|ExposureMask);
    XSelectInput(sdd(), side->sides, ExposureMask);
    XSelectInput(sdd(), side->timemode, ExposureMask);
    XSelectInput(sdd(), side->clock, ExposureMask);
    XSelectInput(sdd(), side->state, ButtonPressMask|ExposureMask);
    XSelectInput(sdd(), side->world, ExposureMask);
}

/* Move windows and change their sizes to correspond with the new sizes of */
/* viewports, etc */

reset_misc(side)
Side *side;
{
    Pixmap dots;
    XSizeHints hints;
    XGCValues values;
    unsigned int gcmask;

    dots = (side->bonw ? sd()->bwdots : sd()->wbdots); 

    XResizeWindow(sdd(), side->main, side->mw, side->mh);
    hints.width = side->mw;  hints.height = side->mh;
    hints.min_width = side->mw;  hints.min_height = side->mh;
    hints.flags = PSize|PMinSize;
    XSetNormalHints(sdd(), side->main, &hints);

    XSetWindowBackgroundPixmap(sdd(), side->main, dots);
    XSetWindowBackground(sdd(), side->msg, side->bgcolor);
    XSetWindowBackground(sdd(), side->info, side->bgcolor); 
    XSetWindowBackground(sdd(), side->prompt, side->bgcolor); 
    XSetWindowBackground(sdd(), side->map, side->bgcolor);
    XSetWindowBackground(sdd(), side->timemode, side->bgcolor);
    XSetWindowBackground(sdd(), side->clock, side->bgcolor);
    XSetWindowBackground(sdd(), side->state, side->bgcolor);
    XSetWindowBackground(sdd(), side->help, side->bgcolor);
    XSetWindowBackground(sdd(), side->sides, side->bgcolor);
    XSetWindowBackground(sdd(), side->world, side->bgcolor);
    XSetWindowBorder(sdd(), side->msg, side->fgcolor);
    XSetWindowBorder(sdd(), side->info, side->fgcolor); 
    XSetWindowBorder(sdd(), side->prompt, side->fgcolor); 
    XSetWindowBorder(sdd(), side->map, side->fgcolor);
    XSetWindowBorder(sdd(), side->timemode, side->fgcolor);
    XSetWindowBorder(sdd(), side->clock, side->fgcolor);
    XSetWindowBorder(sdd(), side->state, side->fgcolor);
    XSetWindowBorder(sdd(), side->help, side->fgcolor);
    XSetWindowBorder(sdd(), side->sides, side->fgcolor);
    XSetWindowBorder(sdd(), side->world, side->fgcolor);

    gcmask = GCForeground | GCBackground;
    values.foreground = side->fgcolor;
    values.background = side->bgcolor;
    XChangeGC(sdd(), sd()->gc, gcmask, &values);
    XChangeGC(sdd(), sd()->flashgc, gcmask, &values);
    XChangeGC(sdd(), sd()->textgc, gcmask, &values);
    XChangeGC(sdd(), sd()->helpgc, gcmask, &values);
    XChangeGC(sdd(), sd()->icongc, gcmask, &values);
    XChangeGC(sdd(), sd()->unitgc, gcmask, &values);
    values.foreground = side->bgcolor;
    values.background = side->fgcolor;
    XChangeGC(sdd(), sd()->invicongc, gcmask, &values);
}

/* Alter the size and position of a window. */

change_window(side, win, x, y, w, h)
Side *side;
Window win;
int x, y, w, h;
{
    unsigned int mask;
    XWindowChanges changes;

    if (active_display(side)) {
	if (x >= 0) {
	    if (w >= 0) {
		mask = CWX | CWY | CWWidth | CWHeight;
		changes.x = x;  changes.y = y;
		changes.width = w;  changes.height = h;
	    } else {
		mask = CWX | CWY;
		changes.x = x;  changes.y = y;
	    }
	} else {
	    mask = CWWidth | CWHeight;
	    changes.width = w;  changes.height = h;
	}
    }
    XConfigureWindow(sdd(), win, mask, &changes);
}

/* Return the number of colors - this is used to guess about monochromeness. */

display_colors(side)
Side *side;
{
    return XDisplayCells(sdd(), DefaultScreen(sdd()));
}

white_color(side)
Side *side;
{
    return WhitePixel(sdd(), DefaultScreen(sdd()));
}

black_color(side)
Side *side;
{
    return BlackPixel(sdd(), DefaultScreen(sdd()));
}

/* Get a color set up and warn if not getting what was asked for. */

long
request_color(side, name)
Side *side;
char *name;
{
    XColor c, avail;

    if (Debug) printf("Allocating %s\n", name);
    XAllocNamedColor(sdd(), DefaultColormap(sdd(), DefaultScreen(sdd())), name,
		     &avail, &c);
    if (c.red != avail.red || c.green != avail.green || c.blue != avail.blue) {
	fprintf(stderr, "Warning: %s color not exact on \"%s\"!\n",
		name, side->host);
	fprintf(stderr, "Is %d %d %d instead of %d %d %d\n",
		avail.red, avail.green, avail.blue, c.red, c.green, c.blue);
    }
    return avail.pixel;
}

/* Look at a single event and fill the request structure appropriately. */

process_events(side)
Side *side;
{
    XEvent evt;
    char buf[BUFSIZE];
    int nchar, rawx, rawy;
    unsigned int junk;

    side->reqtype = GARBAGE;
    XNextEvent(sdd(), &evt);
    switch (evt.type) {
    case KeyPress:
	if (evt.xkey.window != side->main && evt.xkey.window != side->help) 
	    return;
	nchar = XLookupString( (XKeyEvent *)&evt, buf, BUFSIZE,
				(KeySym*)NULL, (XComposeStatus *)NULL);
	if (nchar > 0) {
	    side->reqtype = KEYBOARD;
	    side->reqch = *buf;
	    if (Debug) printf("Host %s returns key '%c'\n",
			      side->host, side->reqch);
	}
	break;
    case ButtonPress:
	if (evt.xbutton.window == side->map) {
	    rawx = evt.xbutton.x;  rawy = evt.xbutton.y; 
	    side->reqtype = MAPPOS;
	    deform(side, rawx, rawy, &(side->reqx), &(side->reqy));
	    if (Debug) printf("Host %s returns map %d %d\n",
			      side->host, side->reqx, side->reqy);
	} else if (evt.xbutton.window == side->state) {
	    rawx = evt.xbutton.x;  rawy = evt.xbutton.y; 
	    side->reqtype = UNITTYPE;
	    side->requtype = rawy / max(side->hh, side->fh);
	    if (Debug) printf("Host %s returns unit type %d\n",
			      side->host, side->requtype);
	} else if (evt.xbutton.window == side->msg) {
	    if (evt.xbutton.y < side->nh * side->fh / 2)
		side->reqtype = SCROLLUP;
	    else side->reqtype = SCROLLDOWN;
	}
	break;
    case Expose:
	expose( side, &evt );
	break;
    default:
        case_panic("event type", evt.type);
	break;
    }
}

/* Freeze everything until given side supplies input, use sparingly. */

freeze_wait(side)
Side *side;
{
    XEvent evt;
    char buf[BUFSIZE];
    int nchar;

    if (Debug) printf("Waiting for a %s event\n", side->host);
    flush_input(side);
    while(TRUE) {
	XNextEvent(sdd(), &evt);
	if (evt.type == KeyPress) {
	    nchar = XLookupString( (XKeyEvent *)&evt, buf, BUFSIZE,
				(KeySym*)NULL, (XComposeStatus *)NULL);
	    if (nchar > 0) 
		return *buf;
	    else
		return '\0';
	}
    }  
}

#ifdef	__STDC__
void expose( register Side *side, register XEvent *evt )
#else
void expose( side, evt )
    register Side *side;
    register XEvent *evt;
#endif
{
   register Window	w;
   if (Debug) fprintf( stderr, "Host %s exposes itself\n", side->host);
   do {		/* while there are expose events, let treat them all */
/*
 * check if there are other expose events for this window if so,
 * discard this one
 */
	while ( XCheckTypedWindowEvent( sdd(), evt->xexpose.window,
		evt->xexpose.type, evt ) );
/*
 *	Now find which window had an expose event and draw it
 */
	if( (w = evt->xexpose.window) == side->msg )
	    show_note(side);
	else if( w == side->help )
	    expose_help(side);
	else if( w == side->info )
	    show_info(side);
	else if( w == side->prompt )
	    show_prompt(side);
	else if( w == side->map ) {
	    show_map(side);
	    draw_cursor( side );
	} else if( w == side->sides )
	    show_all_sides(side);
	else if( w == side->timemode )
	    show_timemode(side);
	else if( w == side->clock )
	    show_clock(side);
	else if( w == side->state )
	    show_state(side);
	else if( w == side->world )
	    show_world(side);
	else
	    fprintf( stderr, "Unknown window for expose event: %d, side: %s\n",
		w, side->host );
    } while( XCheckTypedEvent( sdd(), evt->xexpose.type, evt ) );
}
/* Get rid of extra key/mouse clicks. And ONLY key/mouse clicks */
/* While we are here, may as well deal with expose events */

flush_input(side)
Side *side;
{
    XEvent evt;
    char buf[BUFSIZE];
    int nchar, rawx, rawy;
    unsigned int junk;

    if (!humanside(side) || !active_display(side) ) return;
    if( XPending( sdd() ) ){	/* There are events to be read */
	do {
	    XNextEvent(sdd(), &evt);
	    switch (evt.type) {
		case KeyPress:		/* Flush those events */
		case ButtonPress:
		    break;
		case Expose:
		    expose( side, &evt );
		    break;
		default:
		    case_panic("event type", evt.type);
		    break;
	    }	/* Don't go back to the server to get more events */
	} while( XEventsQueued( sdd(), QueuedAlready ) );
    }
}

/* Trivial abstraction - sometimes other routines like to ensure all output */
/* actually on the screen. */

flush_output(side) 
Side *side; 
{  
    if (active_display(side)) XFlush(sdd());  
}

/* General window clearing. */

clear_window(side, win)
Side *side;
Window win;
{
    XClearWindow(sdd(), win);
}

/* Draw a single horizontal constant-color bar on the world map.  If part */
/* would not be drawn because of the map's obliqueness, cut it in two and */
/* wrap one of the pieces around. */

#ifdef __STDC__
void draw_bar( register Side *side, int x, int y, int len,
		unsigned long color, GC gc )
#else
void draw_bar( side, x, y, len, color, gc )
Side *side;
int x, y, len;
unsigned long color;
GC gc;
#endif
{
    int sx1, sx2, sy, sww;

    if( !gc ) return;
    if( color ) XSetForeground( sdd(), gc, color );
    w_xform(side, x, y, &sx1, &sy);
    w_xform(side, x + len, y, &sx2, &sy);
    sww = side->mm * world.width;
    if (sx1 < sww && sx2 >= sww) {
	XFillRectangle(sdd(), side->world, gc,
		       sx1, sy, (unint) sww - sx1, (unint) side->mm);
	XFillRectangle(sdd(), side->world, gc,
		       0, sy, (unint) sx2 - sww, (unint) side->mm);
    } else {
	sx1 %= sww;
	sx2 %= sww;
	XFillRectangle(sdd(), side->world, gc,
		       sx1, sy, (unint) sx2 - sx1, (unint) side->mm);
    }
}

/* Invert the outline box on the world map.  This is a little tricky, */
/* because we want the lines to run through the middle of the world's */
/* hexes, and because the line drawn should not overlap (or the overlaps */
/* will be doubly inverted and look strange). */

invert_box(side, vcx, vcy)
Side *side;
int vcx, vcy;
{
    int x1, y1, x2, y2, sx1, sy1, sx2, sy2, mm2 = side->mm/2;

    x1 = vcx - side->vw2 + side->vh2/2;  y1 = vcy - side->vh2;
    x2 = vcx + side->vw2 - side->vh2/2;  y2 = vcy + side->vh2;
    w_xform(side, x1, y1, &sx1, &sy1);
    w_xform(side, x2, y2, &sx2, &sy2);
    sx1 += mm2;  sy1 -= mm2;  sx2 += mm2;  sy2 += mm2;
    XSetFunction(sdd(), sd()->gc, GXinvert);
    /* is this next call really necessary? */
    XSetLineAttributes(sdd(), sd()->gc, 1, LineSolid, CapButt, JoinMiter);
    XDrawLine(sdd(), side->world, sd()->gc, sx1, sy1, sx2, sy1);
    XDrawLine(sdd(), side->world, sd()->gc, sx2, sy1-1, sx2, sy2+1);
    XDrawLine(sdd(), side->world, sd()->gc, sx2, sy2, sx1, sy2);
    XDrawLine(sdd(), side->world, sd()->gc, sx1, sy2+1, sx1, sy1-1);
    XSetFunction(sdd(), sd()->gc, GXcopy);
}

/* This interfaces higher-level drawing decisions to the rendition of */
/* individual pieces of display. */

draw_terrain_row(side, sx, sy, buf, len, color)
Side *side;
int sx, sy, len, color;
char *buf;
{
    sy += sd()->iconfont->max_bounds.ascent;
    XSetForeground(sdd(), sd()->icongc, color);
    XDrawString(sdd(), side->map, sd()->icongc, sx, sy, buf, len);
}

/* Flash a pair of lines up, slow enough to draw the eye, but not so slow */
/* as to get in the way. */

flash_position(side, sx, sy, tm)
Side *side;
int sx, sy, tm;
{
    int sx1, sy1, sx2, sy2;

    if (tm > 0) {
	sx1 = sx - 50 + side->hw/2;  sy1 = sy + 50 + side->hch/2;
	sx2 = sx + 50 + side->hw/2;  sy2 = sy - 50 + side->hch/2;
	XDrawLine(sdd(), side->map, sd()->flashgc, sx1, sy1, sx2, sy2);
	XDrawLine(sdd(), side->map, sd()->flashgc, sx1, sy2, sx2, sy1);
	flush_output(side);
	nap(tm);
	XDrawLine(sdd(), side->map, sd()->flashgc, sx1, sy1, sx2, sy2);
	XDrawLine(sdd(), side->map, sd()->flashgc, sx1, sy2, sx2, sy1);
    }
}

/* The "cursor icon" is just a pair of special chars - nothing to do with */
/* X's notion of cursors. */

draw_cursor_icon(side, sx, sy)
Side *side;
int sx, sy;
{
    sy += sd()->iconfont->max_bounds.ascent;
    XDrawString(sdd(), side->map, sd()->invicongc, sx, sy, "[", 1);
    XSetForeground(sdd(), sd()->icongc, side->fgcolor);
    XDrawString(sdd(), side->map, sd()->icongc, sx, sy, "]", 1);
}

/* Draw the icon for a hex (given as a char). */

draw_hex_icon(side, win, sx, sy, color, ch)
Side *side;
Window win;
int sx, sy, color;
char ch;
{
    XSetForeground(sdd(), sd()->varicongc, color);
    sy += sd()->iconfont->max_bounds.ascent;
    XDrawString(sdd(), win, sd()->varicongc, sx, sy, &ch, 1);
}

/* Draw the number of an unfriendly side (never called for own units). */

draw_side_number(side, win, sx, sy, n, color)
Side *side;
Window win;
int sx, sy, n, color;
{
    char ch = side_num_to_char(n);

    if (n >= 0) {
	XSetForeground(sdd(), sd()->varicongc, color);
	sy += sd()->iconfont->max_bounds.ascent;
	XDrawString(sdd(), win, sd()->varicongc, sx, sy, &ch, 1);
    }
}

draw_blast_icon(side, win, sx, sy, type, color)
Side *side;
Window win;
int sx, sy, color;
char type;
{
    char buf[1];

    XSetForeground(sdd(), sd()->varicongc, color);
    buf[0] = type;
    sy += sd()->iconfont->max_bounds.ascent;
    XDrawString(sdd(), win, sd()->varicongc, sx, sy, buf, 1);
}

/* Flash the player's screen in an unmistakable way. */

invert_whole_map(side)
Side *side;
{
    int sw = side->vw * side->hw, sh = side->vh * side->hh;

    /* GC needs to be set for inverted drawing */
    XDrawRectangle(sdd(), side->map, sd()->gc, 0, 0, sw, sh);
    flush_output(side);
}

/* Draw just one of the mushroom cloud shapes. */

draw_mushroom(side, x, y, i)
Side *side;
int x, y, i;
{
    int sx, sy;
    int color;

    color = ((side->monochrome || i == 3) ? side->fgcolor : side->bgcolor);
    xform(side, unwrap(side, x, y), y, &sx, &sy);
    XSetForeground(sdd(), sd()->unitgc, color);
    XSetClipMask(sdd(), sd()->unitgc, sd()->bombpics[i]);
    XSetClipOrigin(sdd(), sd()->unitgc, sx-BH/4, sy-BH/2);
    XFillRectangle(sdd(), side->map, sd()->unitgc, sx-BH/4, sy-BH/2, BH, BH);
    flush_output(side);
}

/* Confirm that we can indeed do bar graph displays. */

bar_graphs(side) Side *side;  {  return TRUE;  }

/* Do yet another X-toolkit-type function.  This draws a bar graph. */

draw_graph(side, number, amount, total, critical, title)
Side *side;
int number, amount, total, critical;
{
    int boxwidth, boxheight, boxoffset, boxleft, barwidth, barheight;

    if (total > 0) {
	boxwidth = 5*side->fw;
	boxheight = (INFOLINES-1)*side->fh - 2*side->margin;
	boxoffset = side->margin;
	boxleft = 30*side->fw + number * boxwidth;
	barwidth = boxwidth / 3;
	barheight = (boxheight * amount) / total;
	XSetForeground(sdd(), sd()->gc, side->fgcolor);
	XFillRectangle(sdd(), side->info, sd()->gc,
		       boxleft + boxwidth/3 - 1, boxoffset - 1,
		       barwidth + 2, boxheight + 2);
	XSetForeground(sdd(), sd()->gc, side->bgcolor);
	XFillRectangle(sdd(), side->info, sd()->gc,
		       boxleft + boxwidth/3, boxoffset,
		       barwidth, boxheight);
	if ( amount > critical)
	  XSetForeground(sdd(), sd()->gc, side->goodcolor);
	else
	  XSetForeground(sdd(), sd()->gc, side->badcolor);
	XFillRectangle(sdd(), side->info, sd()->gc,
		       boxleft + boxwidth/3, boxoffset + boxheight - barheight,
		       barwidth, barheight);
	draw_text(side, side->info,
		  boxleft+(boxwidth-strlen((char *)title)*side->fw)/2,
		  (INFOLINES-1)*side->fh, title, side->fgcolor);
    }
}

/* Splash a unit image (either bitmap or font char) onto some window. */

draw_unit_icon(side, win, x, y, u, color)
Side *side;
Window win;
int x, y, u, color;
{
    char buf[1];

    y += 3;			/* fudge factor to make x11 look */
    x += 2;			/*  like X10 (ugh). */
    if (utypes[u].bitmapname != NULL ) {
      XSetForeground(sdd(), sd()->unitgc, color);
      XSetClipMask(sdd(), sd()->unitgc, sd()->unitpics[u]);
      XSetClipOrigin(sdd(), sd()->unitgc, x, y);
      XFillRectangle(sdd(), win, sd()->unitgc, x, y, side->uw, side->uh);
    } else {
        XSetForeground(sdd(), sd()->unittextgc, color);
	buf[0] = utypes[u].uchar;
	y += sd()->unitfont->max_bounds.ascent;
	XDrawString(sdd(), win, sd()->unittextgc, x, y, buf, 1);
    }
}

/* General text drawer. */

draw_text(side, win, x, y, str, color)
Side *side;
Window win;
int x, y, color;
char *str;
{
  if (win == side->help) draw_help_text(side, win, x, y, str, color);
  else {
    y += sd()->textfont->max_bounds.ascent;
    if (color != side->bgcolor) {
	XSetForeground(sdd(), sd()->textgc, color);
	XDrawImageString(sdd(), win, sd()->textgc, x, y, str, strlen(str));
    } else {
	XSetForeground(sdd(), sd()->textgc, side->bgcolor);
	XSetBackground(sdd(), sd()->textgc, side->fgcolor);
	XDrawImageString(sdd(), win, sd()->textgc, x, y, str, strlen(str));
	XSetBackground(sdd(), sd()->textgc, side->bgcolor);
    }
  }
}

/* General text drawer. */

draw_help_text(side, win, x, y, str, color)
Side *side;
Window win;
int x, y, color;
char *str;
{
    y += sd()->helpfont->max_bounds.ascent;
    if (color != side->bgcolor) {
      XSetForeground(sdd(), sd()->helpgc, color);
      XDrawImageString(sdd(), win, sd()->helpgc, x, y, str, strlen(str));
    } else {
      XSetForeground(sdd(), sd()->helpgc, side->bgcolor);
      XSetBackground(sdd(), sd()->helpgc, side->fgcolor);
      XDrawImageString(sdd(), win, sd()->helpgc, x, y, str, strlen(str));
      XSetBackground(sdd(), sd()->helpgc, side->bgcolor);
      }
    }

/* Draw a line through some side's title. */

draw_scratchout(side, pos)
Side *side;
int pos;
{
    XSetForeground(sdd(), sd()->textgc, side->fgcolor);
    XDrawLine(sdd(), side->sides, sd()->textgc, 0, pos, 30*side->fw, pos);
}

/* Beep the beeper! */

beep(side)
Side *side;
{
    XBell(sdd(), DefaultScreen(sdd()));
}

/* Little routines to pop up the help window and make it go away again */
/* They only get called when display is in use. */

reveal_help(side)
Side *side;
{
    XEvent evt;

    XMapWindow(sdd(), side->help);
    return TRUE;
}

conceal_help(side)
Side *side;
{
    XUnmapWindow(sdd(), side->help);
}

/* Shut a single display down, but only if there's one to operate on. */
/* Hit the display slot, for safety. */

close_display(side)
Side *side;
{
    XCloseDisplay(sdd());
    side->display = (long) NULL;
}

