/*
 * sonic.c - simple real time audio spatialization example with graphics 
 *
 * This code was originally written in IRIS GL and that version
 * appeared in the March/April 1993 Issue of the Silicon Graphics 
 * support magazine "Pipeline", Volume 4, Number 2 accompanying 
 * the article titled "Adding Audio to an Existing Graphics
 * Application" as a coding example.
 * sonic.c - 11/92 Dan Fink
 * ported by Mason Woo to OpenGL/Motif 11/94
 * For use with the Digital Media Development Option
 * On Indigo family hardware platforms running IRIX 4.0.5F or better 
 * compile with -lGLw -lGLU -lGL -lXm -lXt -lX11 -laudio -laudiofile -lm
 */

#include <limits.h>
#include <sys/types.h>
#include <sys/prctl.h>
#include <sys/schedctl.h>
#include <sys/lock.h>
#include <stdlib.h>
#include <stdio.h>
#include <malloc.h>
#include <Xm/Form.h>
#include <Xm/Frame.h>
#include <X11/GLw/GLwMDrawA.h>
#include <X11/keysym.h>
#include <X11/Xutil.h>
#include <GL/gl.h>
#include <GL/glu.h>
#include <GL/glx.h>
#include <math.h>
#include <audio.h>
#include <audiofile.h>

/* AUDIO DEFINES */
#define BUFFERSIZE 4000            /* Size of output sample buffer. */
#define PI 3.141592653             /* Our old friend from trig.     */
/* the following originated from the dmedia_tools.data.prosonus subsystem:
#define BALLFILENAME "/usr/lib/sounds/prosonus/instr/metal_triad_power.E1.aiff"
#define WALLFILENAME "/usr/lib/sounds/prosonus/musictags/slinky_slap.aiff"
*/
#define BALLFILENAME "metal_triad_power.E1.aiff"
#define WALLFILENAME "slinky_slap.aiff"

/* OTHER DEFINES */
#define SPHERERADIUS 1.0
#define BOXSIZE 20.0
#define NO_HIT 0
#define JUST_HIT 1
#define BEEN_HIT 2

/* GRAPHICS PROTOTYPES */
void movesphere();
void drawScene (Widget);
void sceneinit();
Boolean updateDraw (XtPointer);

/* AUDIO PROTOTYPES */
void audioinit();
long init_sound(float **, char*);
void audioloop(void *);
void process_oneshot_audio( float *, float, float, long);
void process_continuous_audio( float *, float, float, long);
void audioexit(int);

/* AUDIO RELATED GLOBALS */
float *ballbuf, *wallbuf;       /* Sample buffers for ball and wall sounds */
long  ballframes,wallframes;    /* Number of samples in each of these buffers */
ALconfig audioconfig;           /* Audio Library port configuration */
ALport audioport;               /* Audio Library audio port */
AFfilesetup afsetup;            /* Audio File Library file setup */
AFfilehandle affh;              /* Audio File Library file handle */

/* OTHER GLOBALS */
GLuint sphere,box;              /* GL object identifiers for ball and box */
GLfloat rx, ry;                   /* Angles of rotation for observer */
GLfloat sphx,sphy,sphz;           /* Coordinates of sphere */
GLfloat movx,movy,movz;           /* Vector for sphere motion */
GLfloat hitx, hity, hitz;         /* Coordinates of ball hitting wall */
int hit;                        /* Status of ball hitting wall */
int done;                       /* Indicates completion of graphics process */
int audiodone;                  /* Indicates completion of audio process */
int moveball = TRUE;
GLfloat winWidth, winHeight;

static int snglBuf[] = {GLX_RGBA, None};

static int dblBuf[] = {GLX_RGBA, GLX_DOUBLEBUFFER, None};

static String   fallbackResources[] =
{
    "*glxarea*width: 300", "*glxarea*height: 300",
    "*frame*x: 20", "*frame*y: 20",
    "*frame*topOffset: 20", "*frame*bottomOffset: 20",
    "*frame*rightOffset: 20", "*frame*leftOffset: 20",
    "*frame*shadowType: SHADOW_IN",
    NULL};

Display        *dpy;
XtAppContext    app;
XtWorkProcId    workId = 0;
Widget          toplevel, form, frame, glxarea;
XVisualInfo    *visinfo;
GLXContext      glxcontext;
char           *status;
GLUquadricObj  *quadObj;

GLboolean       doubleBuffer = GL_TRUE, animation = GL_FALSE;


void movesphere()        /* Add motion to sphere.  Check for wall collisions */
{
    sphx += movx; sphy += movy; sphz += movz;    
                        /* Move sphere in current direction */
    
    if (sphx > (BOXSIZE - SPHERERADIUS)) {    /* Check for x bounce */
        movx = -movx;
        sphx = 2*(BOXSIZE - SPHERERADIUS) - sphx; 
        hit = JUST_HIT;
    } 
    else if (sphx < (SPHERERADIUS - BOXSIZE)) {
        movx = -movx;
        sphx = 2*(SPHERERADIUS - BOXSIZE) - sphx; 
        hit = JUST_HIT;
    }

    if (sphy > (BOXSIZE - SPHERERADIUS)) {    /* Check for y bounce */
        movy = -movy;
        sphy = 2*(BOXSIZE - SPHERERADIUS) - sphy; 
        hit = JUST_HIT;
    } 
    else if (sphy < (SPHERERADIUS - BOXSIZE)) {
        movy = -movy;
        sphy = 2*(SPHERERADIUS - BOXSIZE) - sphy; 
        hit = JUST_HIT;
    }

    if (sphz > (BOXSIZE - SPHERERADIUS)) {    /* Check for z bounce */
        movz = -movz;
        sphz = 2*(BOXSIZE - SPHERERADIUS) - sphz; 
        hit = JUST_HIT;
    } 
    else if (sphz < (SPHERERADIUS - BOXSIZE)) {
        movz = -movz;
        sphz = 2*(SPHERERADIUS - BOXSIZE) - sphz; 
        hit = JUST_HIT;
    }

    if (hit == JUST_HIT) {                     /* Record where bounced */
            hitx = sphx; hity=sphy; hitz=sphz;
    }
}

void sceneinit() {
    static GLfloat v[8][3] = {
            { -1.0, -1.0, -1.0 }, {  1.0, -1.0, -1.0 },
            {  1.0,  1.0, -1.0 }, { -1.0,  1.0, -1.0 },
            { -1.0, -1.0,  1.0 }, {  1.0, -1.0,  1.0 },
            {  1.0,  1.0,  1.0 }, { -1.0,  1.0,  1.0 } 
    };

    glShadeModel (GL_SMOOTH);

    glMatrixMode (GL_PROJECTION);
    glLoadIdentity ();
    gluPerspective (60.0, 1.0, .25, BOXSIZE*1.718);
    glMatrixMode (GL_MODELVIEW);

    box = glGenLists(2);
    if (box == 0) {
	fprintf (stderr, "No display lists remaining\n");
	exit (0);	
    }
    sphere = box + 1;    

    glNewList (box, GL_COMPILE);
    glBegin (GL_TRIANGLE_STRIP);
	glColor3f (0x60, 0, 0);		glVertex3fv (v[1]);
	glColor3f (0, 0, 0);		glVertex3fv (v[0]);
	glColor3f (0x60, 0x60, 0);	glVertex3fv (v[2]);
	glColor3f (0, 0x60, 0);		glVertex3fv (v[3]);
	glColor3f (0x60, 0x60, 0x60);	glVertex3fv (v[6]);
    glEnd ();
    glBegin (GL_TRIANGLE_STRIP);
	glColor3f (0x60, 0x60, 0x60);	glVertex3fv (v[6]);
	glColor3f (0, 0x60, 0);		glVertex3fv (v[3]);
	glColor3f (0, 0x60, 0x60);	glVertex3fv (v[7]);
	glColor3f (0, 0, 0);		glVertex3fv (v[0]);
	glColor3f (0, 0, 0x60);		glVertex3fv (v[4]);
    glEnd ();
    glBegin (GL_TRIANGLE_STRIP);
	glColor3f (0, 0, 0x60);		glVertex3fv (v[4]);
	glColor3f (0, 0, 0);		glVertex3fv (v[0]);
	glColor3f (0x60, 0, 0x60);	glVertex3fv (v[5]);
	glColor3f (0x60, 0, 0);		glVertex3fv (v[1]);
    glEnd ();
    glBegin (GL_TRIANGLE_STRIP);
	glColor3f (0x60, 0, 0);		glVertex3fv (v[1]);
	glColor3f (0x60, 0, 0x60);	glVertex3fv (v[5]);
	glColor3f (0x60, 0x60, 0);	glVertex3fv (v[2]);
	glColor3f (0x60, 0x60, 0x60);	glVertex3fv (v[6]);
	glColor3f (0, 0x60, 0x60);	glVertex3fv (v[7]);
    glEnd ();
    glBegin (GL_TRIANGLE_STRIP);
	glColor3f (0, 0x60, 0x60);	glVertex3fv (v[7]);
	glColor3f (0x60, 0x60, 0x60);	glVertex3fv (v[6]);
	glColor3f (0, 0, 0x60);		glVertex3fv (v[4]);
	glColor3f (0x60, 0, 0x60);	glVertex3fv (v[5]);
    glEnd();

    glPushMatrix();
	glScalef (0.99, 0.99, 0.99);
	glColor3f (0xA0, 0xA0, 0xA0);
	glBegin (GL_LINE_STRIP);
	    glVertex3fv (v[0]);
	    glVertex3fv (v[1]);
	    glVertex3fv (v[2]);
	    glVertex3fv (v[3]);
	    glVertex3fv (v[0]);
	    glVertex3fv (v[4]);
	    glVertex3fv (v[5]);
	    glVertex3fv (v[6]);
	    glVertex3fv (v[7]);
	    glVertex3fv (v[4]);
	glEnd ();
	glBegin (GL_LINES);
	    glVertex3fv (v[1]);
	    glVertex3fv (v[5]);
	    glVertex3fv (v[2]);
	    glVertex3fv (v[6]);
	    glVertex3fv (v[3]);
	    glVertex3fv (v[7]);
	glEnd ();
    glPopMatrix();
    glEndList ();

    glNewList (sphere, GL_COMPILE);
    quadObj = gluNewQuadric();
    gluQuadricDrawStyle (quadObj, GLU_FILL);
    gluQuadricNormals (quadObj, GLU_SMOOTH);
    gluSphere (quadObj, 1.0, 8, 8);
    glEndList ();

    {
	static float ambient[] = {.1, .1, .1};
	static float matDiffuse[] = {0, .369, .165};
	static float matSpec[] = {.5, .5, .5};
	static float whiteLight[] = {1.0, 1.0, 1.0};
	static float lightPos[] = {0.0, 0.0, 1.0, 0.0};

	glMaterialfv (GL_FRONT, GL_AMBIENT, ambient);
	glMaterialfv (GL_FRONT, GL_DIFFUSE, matDiffuse);
	glMaterialfv (GL_FRONT, GL_SPECULAR, matSpec);
	glMaterialf (GL_FRONT, GL_SHININESS, 10.0);
	glLightModelfv (GL_LIGHT_MODEL_AMBIENT, ambient);
	glLightModeli (GL_LIGHT_MODEL_LOCAL_VIEWER, GL_TRUE);
	glLightfv (GL_LIGHT0, GL_AMBIENT, ambient);
	glLightfv (GL_LIGHT0, GL_DIFFUSE, whiteLight);
	glLightfv (GL_LIGHT0, GL_SPECULAR, whiteLight);
	glLightfv (GL_LIGHT0, GL_POSITION, lightPos);
    }
}

void drawScene (Widget w) {
    glPushMatrix ();
    glRotatef(ry, 0.0, 1.0, 0.0);
    glRotatef(rx, 1.0, 0.0, 0.0);

    glDisable (GL_LIGHTING);
    glDisable (GL_LIGHT0);
    glDisable (GL_CULL_FACE);
    glPushMatrix ();
    glScalef (BOXSIZE, BOXSIZE, BOXSIZE);
    glCallList (box);
    glPopMatrix();

    glEnable (GL_LIGHTING);
    glEnable (GL_LIGHT0);
    glEnable (GL_CULL_FACE);
    glPushMatrix ();
    glTranslatef (sphx, sphy, sphz);
    glScalef (SPHERERADIUS, SPHERERADIUS, SPHERERADIUS);
    glCallList (sphere);    
    glPopMatrix();

    glPopMatrix();
    if (doubleBuffer) 
	glXSwapBuffers (dpy, XtWindow(w));
    glFlush ();
}

/* AUDIO ROUTINES */
void audioinit() /* Configure the audio port */
{
    ballframes = init_sound( &ballbuf, BALLFILENAME); /* Get the ball waveform*/
    wallframes = init_sound( &wallbuf, WALLFILENAME); /* Get the wall waveform*/

    audioconfig = ALnewconfig();             /* Get a new audio configuration */    

    ALsetsampfmt( audioconfig, AL_SAMPFMT_FLOAT); /* Sample format uses floats*/
    ALsetfloatmax(audioconfig, 1.0);      /* Floats will vary from -1.0 to 1.0*/

    ALsetqueuesize(audioconfig,2*BUFFERSIZE); 
            /* Set the sample queue size to be twice that of our sound buffer */

    ALsetchannels(audioconfig,AL_STEREO);             /* Use stereo */

    audioport = ALopenport("sine","w",audioconfig); /* Open audio for writing */
    if (audioport == NULL) {
        printf("couldn't open audio port.\n");
        audioexit(-1);
    }
}


long init_sound(float **floatbuf, char *filename)         /* Read audio file */
{
/*  This routine reads the specified audio file, converts the samples
    to floating point, and return the length of the sample buffer.
    For simplicity sake, we're looking to read in 16 bit sounds in 
    AF_SAMPFMT_TWOSCOMP format.  Since our sound is eminating from a point
    source, we can only use mono information.  For stereo sound files,
    only the left buffer is used.  */

    long sampfmt;       /* Sample format of the audio file */
    long numbits;       /* Number of bits wide the samples are in audio file */
    long numchannels;   /* Number of channels (1=mono,2=stereo) in audio file */
    long numframes;     /* Number of sample frames in audio file */
    short *tempbuf;     /* Temporary buffer to store integer samples */
    int i;              /* Counter */

    if((afsetup = AFnewfilesetup()) == NULL) { /* Create new audio file setup */
        printf("Null file setup.\n");
        audioexit(-2);
    }

    if((affh = AFopenfile(filename,"r",afsetup)) == NULL) {  /*Open audio file*/
        printf("Couldn't open audio file %s.\n",filename);
        audioexit(-3);
    }

    /* Determine validity of sample format */
    AFgetsampfmt(affh, AF_DEFAULT_TRACK, &sampfmt,&numbits); 
    if(sampfmt != AF_SAMPFMT_TWOSCOMP) {
        printf("Strange audio file format in %s.\n",filename);
        audioexit(-4);
    }

    if (numbits != 16) {   /* Check sample width */
        printf("Sample format isn't 16 bits in file %s.\n", filename);
        audioexit(-5);
    }

    numchannels = AFgetchannels(affh, AF_DEFAULT_TRACK);
    numframes = AFgetframecnt(affh, AF_DEFAULT_TRACK);    /* Find buffer size */
    tempbuf = (short *)malloc(numframes*numchannels*sizeof(short));

    /* Read 16 bit samples into temp buffer before float conversion */
    if(AFreadframes(affh,AF_DEFAULT_TRACK,tempbuf,numframes) != numframes) {
        printf("Error reading audio file %s.\n",filename);
        audioexit(-6);
    }
    
    *floatbuf = (float *)malloc(numframes*sizeof(float)); 
                /* Create a buffer of floats - floats are much nicer than */
                /* integers for the amplitude calculations later on.      */
 
    for(i=0; i<numframes; i++)
        (*floatbuf)[i] = (float)tempbuf[i*numchannels]/32767.0;

                    /* Convert two's complement data into floating point data.*/
                    /* Scale all samples to fit between -1.0 and 1.0. If the  */
                    /* file is stereo, then only use data from the left       */
                    /* channel. We are using a point source of sound here.    */

    free(tempbuf);        /* Free structures and close the audio file */
    AFclosefile(affh);
    affh = AF_NULL_FILEHANDLE;
    AFfreefilesetup(afsetup);

    return(numframes);    /* Return number of frames read in */
}


void audioloop(void *dummy)		/* Audio process main loop */
{
    float distance,amplitude,balance;   /* What we need to find out */
    float soundbuf[BUFFERSIZE];         /* Sample buffer for computations */
    float relx,rely,relz;      /* Coordinates of ball relative to orientation */
    float radx,rady;           /* Angle describing orientation in radians */

    if(schedctl(NDPRI,0,NDPHIMAX) == -1)
      printf("Couldn't run at high, non-degrading priority.\n");
                            /* Make this process run at a high, non-        */
                            /* degrading priority. This help significantly  */
                            /* "smooth out" audio clicking caused by losing */
                            /* the CPU to other processes.  Works best when */
                            /* the effective user id is the root user.      */
    if(plock(PROCLOCK)!=0)
      printf("Couldn't lock process into memory.\n");
                               /* Lock this process into memory - make it  */
                               /* immune to page swapping. Again, this is  */
                               /* another aid in preventing clicking which */
                               /* works when the effective user id is the  */
                               /* root user.                               */


    audioinit();               /* Initialize audio */
    while(!done) {             /* Keep going until gfx process says otherwise */

        /* Put sphere's coordinates through viewing transformation */
        /* Rot x (rx) * Rot y (rx) just like the graphics are doing */
        /* If translate() or scale() calls were used, they would need to */
        /* be represented here as well. Remember that the graphics pipe */
        /* pre-multiplies for matrix operations! */
        
        radx = PI*rx/180.0;
        rady = PI*ry/180.0;

        relx =  sphx*cos(rady) + sphy*sin(radx)*sin(rady) + 
                sphz*cos(radx)*sin(rady);
        rely =  sphy*cos(radx) - sphz*sin(radx);
        relz = -sphx*sin(rady) + sphy*sin(radx)*cos(rady) + 
                sphz*cos(radx)*cos(rady);

        distance = fsqrt( relx*relx + rely*rely + relz*relz );

        amplitude = 1.0 / (distance*distance + 1.0);  
            /* Use inverse square proportion for amplitude */

        balance = acos( relx/distance) / PI;
            /* Compute the audio orientation of the sphere by taking the     */
            /* angle of the projected vector from the observer to the object */
            /* the object onto the xy plane of the observer.  Left-right     */
            /* balance is easiest to achieve with a 2 speaker system such as */
            /* headphones, so we'll need to use the distance in the          */
            /* x direction (relative to the observer's orientation) to find  */
            /* left-right balance.  With only 2 speakers top-bottom balance  */
            /* is very difficult to simulate, so we'll ignore it here.       */

        process_continuous_audio(soundbuf,amplitude,balance,BUFFERSIZE);

        if(hit != NO_HIT) {  /* Don't bother with walls if we don't need to */

            /* Pass the coordinates of the hit through the same */
            /* transformations as above.                        */

            relx =  hitx*cos(rady) + hity*sin(radx)*sin(rady) + 
                    hitz*cos(radx)*sin(rady);
            rely =  hity*cos(radx) - hitz*sin(radx);
            relz = -hitx*sin(rady) + hity*sin(radx)*cos(rady) + 
                    hitz*cos(radx)*cos(rady);

            distance = fsqrt( relx*relx + rely*rely + relz*relz );
            balance = acos( relx/distance) / PI;
            amplitude = 1.0 / (distance*distance + 1.0);  

            process_oneshot_audio(soundbuf,amplitude,balance,BUFFERSIZE);
        }
        
        while(ALgetfillable(audioport) < BUFFERSIZE)
            sginap(1);  /* Wait til theres enough room to write the entire */
                        /* copy of our sound buffer to the audio queue.    */

        ALwritesamps(audioport,soundbuf,BUFFERSIZE); 
                        /* Output samples to audio port */
    }
    
    while(ALgetfilled(audioport) > 0);
        /* Wait till all the sound has been played before closing the port.  */

    audioexit(0);
}


/* Process one-shot audio for bouncing off wall sound - time critical */
void process_oneshot_audio(float *soundbuf, float amplitude,    
                           float balance, long buflen)
{
    static long wallphase;            /* Index into wall's sound buffer */
    long i;                           /* Counter */
    float rbalance = 1.0 - balance;   /* Speeds up calculations */

    amplitude *= 5.0;             /* The wall is usually pretty far away */
                                  /* boost up the volume for collisions  */

    if(hit == JUST_HIT)   /* If this is the first time around, the index     */
    {                     /* into the sound buffer needs to be reinitialized */
        hit = BEEN_HIT;   /* ialized, otherwise we may need to pick up       */
        wallphase = 0;    /* processing where we left off, since the entire  */
                          /* duration of sound may be longer than the time   */
    }                     /* slice we need to process.                       */

    for(i=0; (i<buflen) && (wallphase < wallframes); wallphase++ ) {

        /* Since we know that this will be processed _after_ there is valid */
        /* audio data in the sample buffer, we'll perform the operation of  */
        /* mixing the new audio data with the old audio data as the samples */
        /* are copied. In this case the mixing operation is a simple add. A */
        /* weighted average may be needed if the sum is likely to clip.     */

        soundbuf[i++] += amplitude * balance * wallbuf[wallphase]; /* Left  */
        soundbuf[i++] += amplitude * rbalance* wallbuf[wallphase]; /* Right */

            /* Note that a stereo soundbuffer is interleaved LRLRLR... */
    }

    if(wallphase == wallframes)      /* If the sound buffer is exhausted, */
        hit = NO_HIT;                /* we don't need to come back.       */
}


/* Process audio sample loop for moving ball - time critical */
void process_continuous_audio(float *soundbuf, float amplitude, 
                              float balance, long buflen)
{
    static long ballphase = 0;        /* Index into wall's sound buffer */
    long i;                           /* Counter */
    float rbalance = 1.0 - balance;   /* Speeds up calculations */

    for(i=0; i<buflen; ballphase %= ballframes) {

        /* Since we know that this will be processed _before_ there is valid */
        /* audio data in the sample buffer, we'll can just copy the data into*/
        /* the buffer, if data were already in the sample buffer, mixing     */
        /* would be necessary.   Continuous looping is acheived by the modulo*/
        /* function above. */

        soundbuf[i++] = amplitude * balance * ballbuf[ballphase];   /* Left  */
        soundbuf[i++] = amplitude * rbalance* ballbuf[ballphase++]; /* Right */

            /* Note that a stereo soundbuffer is interleaved LRLRLR... */
    }
}


void audioexit(int exitval)        /* Gracefully exit the audio process */
{
    if (wallbuf != NULL)            free(wallbuf);
    if (ballbuf != NULL)            free(ballbuf);
    if (audioconfig != NULL)        ALfreeconfig(audioconfig);
    if (audioport != NULL)          ALcloseport(audioport);
    if (affh != AF_NULL_FILEHANDLE) AFclosefile(affh);
    if (afsetup != NULL)            AFfreefilesetup(afsetup);

    printf("audio closed down OK.\n");
    audiodone = TRUE;
    exit(exitval);
}


void
input_callback(Widget w, XtPointer client_data, XtPointer call)
{
    char            buffer[31];
    KeySym          keysym;
    GLwDrawingAreaCallbackStruct *call_data;

    call_data = (GLwDrawingAreaCallbackStruct *) call;

    switch (call_data->event->type) {
    case ButtonPress:
	if (call_data->event->xbutton.button == Button1) {
	    moveball = !moveball;
	    hit = JUST_HIT;
	    hitx = sphx; hity = sphy; hitz = sphz;
	    if (moveball)
		workId = XtAppAddWorkProc(app, updateDraw, NULL);
	    else
		XtRemoveWorkProc(workId);
	}
	else if (call_data->event->xbutton.button == Button3) {
	    done = TRUE;
	    fprintf (stderr, "Right mouse pressed\n");
	    while (!audiodone)
		sginap(1);
	    exit(0);
	    break;
	}
	break;
    case KeyPress:
	XLookupString(&call_data->event->xkey, buffer,
		      1, &keysym, NULL);
	switch (keysym) {
	case XK_Up:
	    rx = rx + 3.0;
	    break;
	case XK_Down:
	    rx = rx - 3.0;
	    break;
	case XK_Right:
	    ry = ry + 3.0;
	    break;
	case XK_Left:
	    ry = ry - 3.0;
	    break;
	}
        if (!moveball)
	    drawScene(glxarea);
	break;
    }
}

Boolean updateDraw (XtPointer clientData)
{
    if (moveball)
	movesphere();
    drawScene(glxarea);
    return False; /* leave work proc active */
}

void
resize_callback(Widget w, XtPointer client_data, XtPointer call)
{
    GLwDrawingAreaCallbackStruct *call_data;

    call_data = (GLwDrawingAreaCallbackStruct *) call;
    glViewport(0, 0, call_data->width, call_data->height);

    glMatrixMode (GL_PROJECTION);
    glLoadIdentity ();
    gluPerspective (60.0, 
	(GLfloat) call_data->width/ (GLfloat) call_data->height,
	.25, BOXSIZE*1.718);
    glMatrixMode (GL_MODELVIEW);
}

void
expose_callback(Widget w, XtPointer client_data, XtPointer call)
{
    GLwDrawingAreaCallbackStruct *call_data;

    call_data = (GLwDrawingAreaCallbackStruct *) call;
    drawScene(w);
}

void
init_callback(Widget w, XtPointer client_data, XtPointer call)
{
    XVisualInfo    *visinfo;

    XtVaGetValues(w, GLwNvisualInfo, &visinfo, NULL);
    glxcontext = glXCreateContext(XtDisplay(w), visinfo,
		      /* no sharing */ 0, /* direct if possible */ GL_TRUE);
}


main(int argc, char **argv)
{
    int audio_pid;          /* Process id of child audio process */

    /* Initialize globals */
    done = audiodone = FALSE;
    rx = ry = 0.0;
    sphx = 1.0; sphy = 2.0; sphz = 3.0;
    movx = 0.1; movy = 0.2;  movz = 0.1;
    hit = NO_HIT;

    audio_pid = sproc(audioloop,PR_SALL);     /* Fork audio process */
    if (audio_pid == -1) {
        fprintf(stderr,"Couldn't create audio process.\n");
        exit(-1);
    }

    toplevel = XtAppInitialize(&app, "Sonic", NULL, 0, &argc, argv,
			       fallbackResources, NULL, 0);
    dpy = XtDisplay(toplevel);

    form = XmCreateForm(toplevel, "form", NULL, 0);
    XtManageChild(form);

    frame = XmCreateFrame(form, "frame", NULL, 0);
    XtVaSetValues(frame, XmNbottomAttachment, XmATTACH_FORM,
	  XmNtopAttachment, XmATTACH_FORM, XmNleftAttachment, XmATTACH_FORM,
		  XmNrightAttachment, XmATTACH_FORM, NULL);
    XtManageChild(frame);

    /*
     * We find the XVisualInfo* we want for the GLwMDrawA widget _before_ we
     * create the widget.  The alternative to this is specifying the OpenGL
     * visual attributes as resources arguments when creating the widget but
     * unfortunately, if a visual matching the attributes we specify does not
     * exist, we get a fatal error message like:
     * 
     * Error: GLwMDrawingArea: requested visual not supported
     * 
     * By specifying exactly which XVisualInfo* we want, we avoid this problem,
     * allowing us to fall back to another acceptable set of visual
     * attirbutes (single buffered in this case) and print out our own more
     * informative message if even the second visual selection fails.
     */
    visinfo = glXChooseVisual(dpy, DefaultScreen(dpy), dblBuf);
    if (visinfo == NULL) {
	visinfo = glXChooseVisual(dpy, DefaultScreen(dpy), snglBuf);
	if (visinfo == NULL)
	    XtAppError(app, "no RGB visual with deep enough accumulation buffer");
	doubleBuffer = GL_FALSE;
    }
    glxarea = XtVaCreateManagedWidget("glxarea", glwMDrawingAreaWidgetClass,
				      frame, GLwNvisualInfo, visinfo, NULL);
    XtAddCallback(glxarea, GLwNginitCallback, init_callback, NULL);
    XtAddCallback(glxarea, GLwNexposeCallback, expose_callback, NULL);
    XtAddCallback(glxarea, GLwNresizeCallback, resize_callback, NULL);
    XtAddCallback(glxarea, GLwNinputCallback, input_callback, NULL);

    XtRealizeWidget(toplevel);

    GLwDrawingAreaMakeCurrent(glxarea, glxcontext);

    /*
     * Now the GLwNginitCallback has been called so the OpenGL context has
     * been set up.
     */
    sceneinit ();
    workId = XtAppAddWorkProc(app, updateDraw, NULL);

    XtAppMainLoop(app);
}
