#include <iostream.h>
#include <stdlib.h>
#include <GL/glut.h>

#include <timeit.h>

/////////////////////////////////////////////////////////////////////////////
//
// globals
//
/////////////////////////////////////////////////////////////////////////////

typedef enum motiontype
{
  motion_rotate,
  motion_zoom,
  motion_pan

} motiontype;

typedef struct mouseinfo
{
  GLfloat zoom;
  GLfloat angle[ 2 ];
  GLfloat pan[ 2 ];

  bool moving;
  motiontype type;
  int startx;
  int starty;

} mouseinfo;

mouseinfo mm;
timeobj *fps_timer = timeit_new();
bool silent = false;

/////////////////////////////////////////////////////////////////////////////
//
// adjust the base_spheresize to increase or decrease the workload.
//
/////////////////////////////////////////////////////////////////////////////

const int nspheres = 50;
const float base_spheresize = .001;
float spheresize = base_spheresize;
int divisions = 15;

/////////////////////////////////////////////////////////////////////////////
//
// function prototypes
//
/////////////////////////////////////////////////////////////////////////////

void usage();
void reset();
void init();
void initproj();
void draw();
void idle();
void reshape( int ww, int hh );
void key( unsigned char key, int x, int y );
void mouse( int button, int xx, int yy );
void motion( int xx, int yy );
void mouseinfo_init( mouseinfo *mm );

void set_high_fill();
void set_low_fill();

/////////////////////////////////////////////////////////////////////////////
//
// misc initialization
//
/////////////////////////////////////////////////////////////////////////////

void reset()
{
  cerr << "-- reset" << endl;
  set_high_fill();
  mouseinfo_init( &mm );
  initproj();
  divisions = 15;  
}

void mouseinfo_init( mouseinfo *mm )
{
  mm->angle[ 0 ] = 0;
  mm->angle[ 1 ] = 0;
  mm->pan[ 0 ] = 0;
  mm->pan[ 1 ] = 0;
  mm->zoom = 10.0;
  mm->moving = false;
  mm->type = motion_rotate;
  mm->startx = 0;
  mm->starty = 0;
  
}

void set_high_fill()
{
  cerr << "-- enable high fill load (larger spheres)" << endl;
  spheresize = base_spheresize*10;
}

void set_low_fill()
{
  cerr << "-- enable low fill load (smaller spheres)" << endl;
  spheresize = base_spheresize;
}

void init()
{
  /* gl state */
  glClearColor( 1, 0, 1, 0 );

  glEnable( GL_DEPTH_TEST );
  glDepthFunc( GL_LEQUAL );

  glEnable( GL_LIGHTING );
  glEnable( GL_LIGHT0 );
  float pos[ 4 ] = { 0, 0, 10, 0 };
  glLightfv( GL_LIGHT0, GL_POSITION, pos );
  
  glShadeModel( GL_SMOOTH );

  /* set defaults for program */
  reset();
}

void initproj()
{
  glMatrixMode( GL_MODELVIEW );
  glLoadIdentity();

  glMatrixMode( GL_PROJECTION );
  glOrtho( -1.0, 1.0, -1.0, 1.0, -10.0, 10.0 );

  glMatrixMode( GL_MODELVIEW );
  glLoadIdentity();
}

/////////////////////////////////////////////////////////////////////////////
//
// mouse manipulation
//
/////////////////////////////////////////////////////////////////////////////

void mouse( int button, int state, int xx, int yy )
{
  if ( state == GLUT_DOWN )
    {
      mm.moving = true;
      mm.startx = xx;
      mm.starty = yy;
      switch( button )
	{
	case GLUT_LEFT_BUTTON:
	  {
	    mm.type   = motion_rotate;
	  }
	  break;
	case GLUT_MIDDLE_BUTTON:
	  {
	    mm.type   = motion_zoom;
	  }
	  break;
	case GLUT_RIGHT_BUTTON:
	  {
	    mm.type   = motion_pan;
	  }
	  break;
	}
    }
  else
    {
      mm.moving = false;
    }
}

void motion( int xx, int yy )
{
  const float scale = .01;

  if ( mm.moving == true )
    {
      switch( mm.type )
	{
	case motion_rotate:
	  {
	    mm.angle[ 0 ] += xx - mm.startx;
	    mm.angle[ 1 ] += yy - mm.starty;
	  }
	  break;

	case motion_pan:
	  {
	    mm.pan[ 0 ] += scale * ( xx - mm.startx );
	    mm.pan[ 1 ] += scale * ( yy - mm.starty );
	  }
	  break;

	case motion_zoom:
	  {
	    mm.zoom *= 1.0 + ( scale * ( yy - mm.starty ) );

	  }
	  break;
	}
      
      mm.startx = xx;
      mm.starty = yy;

      glutPostRedisplay();
    }
}

/////////////////////////////////////////////////////////////////////////////
//
// drawing / resizing functions
//
/////////////////////////////////////////////////////////////////////////////

void reshape( int ww, int hh )
{
  glViewport( 0, 0, ww, hh );

  initproj();
}

void draw( )
{
  /* clear the screen */
  glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );

  /* set eyepoint */
  glMatrixMode( GL_MODELVIEW );
  glLoadIdentity();
  glTranslatef( 0, 0, -1 );
  glTranslatef( mm.pan[ 0 ], -mm.pan[ 1 ], 0.0 );
  glRotatef( mm.angle[ 1 ], 1.0, 0.0, 0.0 );
  glRotatef( mm.angle[ 0 ], 0.0, 1.0, 0.0 );
  glScalef( mm.zoom, mm.zoom, mm.zoom );
  
  /* set default material */
  GLfloat green[ 4 ] = { 0, 1, 0, 0 };
  glMaterialfv( GL_FRONT, GL_DIFFUSE, green );

  /* set application default values */
  const float scale = .1;
  const float scale2 = scale/2.0;
  const float step = scale/nspheres;

  //  glPolygonMode( GL_FRONT_AND_BACK, GL_LINE );


  /* draw our geometry */
  glTranslatef( scale2, -scale2-step, 0 );
  timeit_start( fps_timer );
  for( int ii=0; ii<nspheres; ii++ )
    {
      glTranslatef( -scale, step, 0 );
      for( int jj=0; jj<nspheres; jj++ )
	{
	  glTranslatef( step, 0, 0 );
	  glutSolidSphere( spheresize, divisions, divisions );
	}
    }
  
  /* dump a fps each frame we draw */
  if ( !silent )
    {
      glFinish();
      timeit_stop( fps_timer );
      
      cerr << "-- fps: " << 1.0 /
	timeit_getf( fps_timer, timeit_seconds ) << endl;
    }

  /* if we used a doublebuffered visual, we'd need a swap here */
}

/////////////////////////////////////////////////////////////////////////////
//
// key-event handler
//
/////////////////////////////////////////////////////////////////////////////

void usage()
{
  cerr << "-- mouse commands" << endl;
  cerr << "l - rotate the scene" << endl;
  cerr << "m - zoom (scale) the scene" << endl;
  cerr << "r - pan the scene" << endl;
  cerr << "-- keyboard commands" << endl;
  cerr << "r - reset all parameters" << endl;
  cerr << "F - set high fill load" << endl;
  cerr << "f - set low fill load" << endl;
  cerr << "h - display this help" << endl;
  cerr << "? - display this help" << endl;
}

void key( unsigned char key, int, int )
{
  bool need_refresh = false;

  switch( key )
    {
    case 'h':
    case '?':
      {
	usage();
      }
      break;

    case 'r':
      {
	reset();
	need_refresh = true;
      }
      break;
    
    case 'F':
      {
	set_high_fill();
	need_refresh = true;
      }
      break;
      
    case 'f':
      {
	set_low_fill();
	need_refresh = true;
      }
      break;

    case 't':
      {
	cerr << "-- running test" << endl;
	float time = -1;
	float tps = 0;
	int ii;

	reset();
	const int niters = 10;
	timeobj *tt = timeit_new();
	silent = true;

	/* run test of high fill load */
	set_high_fill();
	timeit_start( tt );
	for( ii=0; ii<niters; ii++ )
	  {
	    draw();
	  }
	glFinish();
	timeit_stop( tt );
	time = timeit_getf( tt, timeit_seconds );
	tps = nspheres * nspheres * niters *
	  ( ( divisions * (divisions - 2) ) * 2 + ( divisions * 2 ) )
	    / time;
	
	cerr << "-- test results w/high fill load" << endl;
	cerr << "total time (s)       : " << time << endl;
	cerr << "triangles per second : " << tps << endl;

	/* run test of low fill load */
	set_low_fill();
	timeit_start( tt );
	for( ii=0; ii<niters; ii++ )
	  {
	    draw();
	  }
	glFinish();
	timeit_stop( tt );
	time = timeit_getf( tt, timeit_seconds );
	tps = nspheres * nspheres * niters *
	  ( ( divisions * (divisions - 2) ) * 2 + ( divisions * 2 ) )
	    / time;

	cerr << "-- test results w/low fill load" << endl;
	cerr << "total time (s)       : " << time << endl;
	cerr << "triangles per second : " << tps << endl;

	/* cleanup */
	timeit_delete( tt );
	need_refresh = false;
	silent = false;
      }
      break;
      
    case 'q':
    case 27:
      {
	exit(0);
      }
      break;
    }

  if ( need_refresh == true )
    {
      draw();
    }
}

/////////////////////////////////////////////////////////////////////////////
//
// main (setup, visual selection, callback registration)
//
/////////////////////////////////////////////////////////////////////////////

int main(int argc, char **argv)
{
  glutInit( &argc, argv );
  
  /* note: all drawing is done in single-buffer mode to avoid frame
   * quantization issues as described in the course notes.
   */
  glutInitDisplayMode( GLUT_RGB | GLUT_SINGLE | GLUT_DEPTH );
  glutInitWindowSize( 500, 500 );
  glutCreateWindow( "glut test shell" );
  
  init();
  
  glutReshapeFunc( reshape );
  glutKeyboardFunc( key );
  glutMouseFunc( mouse );
  glutMotionFunc( motion );
  glutDisplayFunc( draw );
  glutMainLoop();

  return( 1 );
}
