#ifndef _SCENEGRAPH_H_
#define _SCENEGRAPH_H_

//
// A class to hold the state settings for each node
//
class calState
{
public:
  calState() {}
  virtual ~calState() {}

  enum LineStyles   {NoLineStyle, Solid, Wire,_LastStyle};
  enum RenderTypes  {NoType, Cube, Sphere, _LastType};
  enum Lighting     {LightingOff, LightingOn, _LastLighting};

  // used for printing the state
  static char *LineStyleNames[_LastStyle];
  static char *RenderTypeNames[_LastType];
  static char *LightingNames[_LastLighting];
  
  //
  // set/get methods
  //
  
  // enable or disable, set to the above enum's
  LineStyles   &style()           { return _style; }
  RenderTypes  &type()            { return _type; }
  Lighting     &lighting()        { return _lighting; }

  // data
  float        *lightPosition( )  { return _position; }
  float        *color( )          { return _color; }

private:
  // variables to hold the state
  LineStyles   _style;
  RenderTypes  _type;
  Lighting     _lighting;
  float        _position[3];
  GLfloat      _color[4];

};

//
// Base class for state management. 
//
//  Subclasses implement setState() and restoreState() to do the 
//  right thing 
//
class calStateMgr
{
public:

  calStateMgr() {}
  virtual ~calStateMgr() {}
  virtual void setState( calState &s ) = 0;
  virtual void restoreState() = 0;
};


//
// Mild mannered everyday state manager. 
//
//  Sets all state on setState(), restores all state back to previous
//  on restoreState()
//
class plainStateMgr
  : public calStateMgr
{
public:
  virtual void setState( calState & );
  virtual void restoreState();
};

//
// Ah ha! It's super state manager!
//
//  Only set state if it doesn't match the previous state. A full
//  implementation would have a state stack so multiple states could
//  be pushed. Before setting new state, check if the current state
//  matches what will be set. If it does, there's nothing to do.  If
//  it doesn't, push a new state on the state stack. This implementation
//  uses a state stack of depth 1. Meaning, current, next, and previous
//  state stacks all point to the same calState. push and pop are noops
//  because of this.
//
//  There is no need to restore state like in the plainStateMgr. During
//  each state setup, the state that needs to be set is synced with the
//  current state of the GL. Any state that is out of sync will be set.
//  Popping a state off the state stack can be a lazy operation so state
//  does not have to be restored until it changes. 
//
class smartStateMgr
  : public calStateMgr
{
public:
  virtual void setState( calState & );
  virtual void restoreState();

  calState &cur()  { return _state; }
  calState &next() { return _state; }
  calState &prev() { return _state; }
  
  void push() {}  // add a new state to the state stack
  void  pop() {}  // pop a state

private:

  calState _state;
};

//
// Base class for scenegraph
//
//  Scene graph holds and array of nodes each of which point back to 
//  their scenegraph instance. Subclass this into a SceneManager to
//  instantiate.
//
class calSceneGraph
{
public:

  calSceneGraph() {}
  virtual ~calSceneGraph() {}

  virtual void build( const int num ) = 0;  // Build a random sg w/ num nodes
  virtual void sort() = 0;                  // Sort sg
  virtual void render() = 0;                // Render sg

  // set/get methods
  calStateMgr *&stateMgr() { return _state; }  

  calStateMgr *_state;
};

//
// Base class for a node
//
//  Creating one by giving it the scenegraph it belongs to. Automagically
//  sets an integer id. Must subclass into a concrete Node which
//  supplies render method.
//
//  When render() on the Node is called the following happens:
//
//    preRender()     - setup state
//    setState()      - subclass specific state setup
//    renderNode()    - subclass rendering happens
//    postRender()    - teardown state
//    resetState      - subclass state teardown
//
class calNode 
{
public:
  calNode( calSceneGraph *g )
    : _scene(g)
      { init(); }  

  virtual ~calNode() {};
  
  // public API
  virtual void render();                      // Render this node
  void dump( ostream &ostrm = cout );         // Print

  int      &id()       { return _id; }        // local id of node
  calState &state()    { return _state; }     // state settings for node
  GLfloat  *location() { return _location; }  // node location in world
  calSceneGraph *&scene() { return _scene; }  // the scenegraph

protected:
  static int     _count;
  calSceneGraph *_scene;                      // pointer back to scene

  void init();                                // Create

  // Provided by subclass
  virtual void renderNode() = 0;              // Actual rendering calls
  virtual void setState() {};                 // Called before a render
  virtual void resetState() {}                // Called after a render

private:
  //
  // Called by render() to do state managment. Calls the subclass
  // setState() and restoreState() and calls into the stateMgr to
  // actually set the state for the subclass nodes
  //
  void preRender();                           // Node pre-render setup
  void postRender();                          // Node post-render teardown

  // variables to hold stuff. note that location isn't part of calState
  int      _id;
  calState _state;             
  GLfloat  _location[3];

};

//
// A cube node. Renders cubes
//
class calCube
  : public calNode
{
public:
  calCube( calSceneGraph *g ) 
    : calNode(g)
    { state().type() = calState::Cube; }     // hard-code type to a cube

  void renderNode();                         // rendering code!
};


//
// A Sphere node, renders Spheres
//
class calSphere
  : public calNode
{
public:
  calSphere( calSceneGraph *g) 
    : calNode(g)
    { state().type() = calState::Sphere; }   // hard-code type to a sphere

  void renderNode();                         // rendering code!
};

//
// Factory for creating nodes
// 
//  Creates spheres or cubes
//
class calNodeFactory
{
public:
  calNodeFactory() {};
  virtual ~calNodeFactory() {};

  calNode *create( calState::RenderTypes t, calSceneGraph *g );
};


//
// Scene Graph Manager concrete class
//
//  Creates, sorts, renders, and copies scene graphs.
//
//  XXX
//  A copy of the scene graph will only do a shallow copy of the nodes.
//  The node will still point to the original scene graph. 
//
class calSceneMgr 
  : public calSceneGraph
{
public:
  calSceneMgr() 
    : _num(0), _scene(0), _keyCB(0) {}
  virtual ~calSceneMgr() { destroy(); }

  calSceneMgr( const calSceneMgr &g ) { shallowCopy(g); }
  void shallowCopy( const calSceneMgr &g );

  // typedefs for callbacks from GLUT
  typedef void (*RenderCB)( calSceneMgr *);
  typedef void (*KeyCB)( calSceneMgr *, unsigned char key, int x, int y);

  // basic scenegraph manipulation
  void build( const int num );                 // create a scene
  void destroy();                              // destroy a scene
  void sort();                                 // sort the nodes in a scene
  void render();                               // render nodes

  // setup methods
  void initGfx( int w, int h, char *name );    // init OpenGL, open window
  void keyboard( KeyCB cb ) { keyCB() = cb; }  // set callback for keypresses
  void mainLoop( RenderCB cb );                // set callback, enter mainloop

  // set/get
  int     &visible() { return _visible; }      // visibility status
  calNode **&nodes() { return _scene; }        // list of nodes
  int     &size()    { return _num; }          // size of list
  
  // a static method to get the calSceneMgr that is being used. 
  // this is needed to interface with the GLUT routines. when a GLUT
  // callback is triggered, calSceneMgr::sceneMgr() is called to 
  // get the calSceneMgr object and call back into the class. 
  //
  static void set( calSceneMgr * );            
  static calSceneMgr *&sceneMgr();

protected:
  // sort method
  static int 
#ifdef WIN32
    __cdecl 
#endif
   _nodeStateCmp( const void *n1, const void *n2);

private:
  // set the callbacks that are used for render and keyboard
  RenderCB &renderCB() { return _renderCB; }
  KeyCB    &keyCB()    { return _keyCB; }

  // general variable
  int       _num;
  calNode   **_scene;
  int       _visible;
  RenderCB  _renderCB;
  KeyCB     _keyCB;

  // static functions that are called from GLUT
  static void displayFunc();
  static void idleFunc();
  static void keyboardFunc( unsigned char, int, int );
  static void visibilityFunc( int );
  static calSceneMgr *_sceneMgr;
};


#endif
