#include <iostream.h>
#include <stdlib.h>

#ifdef WIN32
#include <windows.h>
#else
#include <unistd.h>
#endif

#include <GL/glut.h>

#include "scenegraph.h"

//
// plainStateMgr Implementation
//

// setState
//  Set the state for a node. There's no way to know what the previous
//  state was set to so each time we need to set state. We don't want
//  to lose the old state, so we're saving it using glPushAttrib()
//
void plainStateMgr::setState( calState &s )
{
  // Save previous state
  glPushAttrib( GL_CURRENT_BIT );
  glPushAttrib( GL_ENABLE_BIT );
  glPushAttrib( GL_POLYGON_BIT );

  // Set the current color
  glColor3fv( s.color() );

  // Set lighting
  if ( s.lighting() == calState::LightingOn ) {
    glEnable( GL_LIGHTING );
    glEnable( GL_LIGHT0 );
    glLightfv( GL_LIGHT0, GL_POSITION, s.lightPosition() );
    glLightfv( GL_LIGHT0, GL_AMBIENT, s.color() );
  } else {
    glDisable( GL_LIGHTING );
  }
  
  // Set draw style
  if ( s.style() == calState::Solid ) glPolygonMode( GL_FRONT, GL_FILL );
  if ( s.style() == calState::Wire ) glPolygonMode( GL_FRONT, GL_LINE );
}

// restoreState
void plainStateMgr::restoreState()
{
  glPopAttrib( );
}


//
// smartStateMgr Implementation
//

// setState
//
void smartStateMgr::setState( calState &s )
{

  int changed = 0;
  //
  // Set the current color
  //
#define STATE_NEQ(item, index)    (s.item()[index] != cur().item()[index])
#define STATE_ASSIGN(item, index) (next().item()[index] = s.item()[index])

  // Only change the color if the current color is different and
  // lighting is not on
  if ( s.lighting() == calState::LightingOff &&
       (STATE_NEQ(color, 0) || STATE_NEQ(color, 1) || STATE_NEQ(color, 2)) ) {

    STATE_ASSIGN(color, 0);
    STATE_ASSIGN(color, 1);
    STATE_ASSIGN(color, 2);

    glColor3fv( next().color() );
    changed++;
  }

  //
  // Set lighting
  //
  
  // See if the lighting state changed, if it did update internal
  // state and enable/disable
  if ( s.lighting() != cur().lighting() ) {

    next().lighting() = s.lighting();

    // Lighting changed, turn either on or off
    if ( s.lighting() == calState::LightingOn ) {
        glEnable( GL_LIGHTING );
        glEnable( GL_LIGHT0 );
    } else {
        glDisable( GL_LIGHTING );
    }
    
    changed++;
  }

  // Now that the lighting state is consistent, check to make
  // sure the parameters are right
  if ( s.lighting() == calState::LightingOn ) {
    if ( STATE_NEQ(lightPosition, 0) || STATE_NEQ(lightPosition, 1) ||
	 STATE_NEQ(lightPosition, 2) || STATE_NEQ(lightPosition, 3) ) {

      STATE_ASSIGN(lightPosition, 0);
      STATE_ASSIGN(lightPosition, 1);
      STATE_ASSIGN(lightPosition, 2);
      STATE_ASSIGN(lightPosition, 3);

      glLightfv( GL_LIGHT0, GL_POSITION, next().lightPosition() );

      changed++;
    }

    if ( STATE_NEQ(color, 0) || STATE_NEQ(color, 1) || STATE_NEQ(color, 2) ) {

      STATE_ASSIGN(color, 0);
      STATE_ASSIGN(color, 1);
      STATE_ASSIGN(color, 2);

      glLightfv( GL_LIGHT0, GL_AMBIENT, s.color() );
      
      changed++;
    }
  }

  // Set draw style
  if ( s.style() != cur().style() ) {
   
    next().style() = s.style();

    if ( next().style() == calState::Solid ) glPolygonMode( GL_FRONT, GL_FILL );
    if ( next().style() == calState::Wire ) glPolygonMode( GL_FRONT, GL_LINE );

    changed++;
  }

  // if anything changed save this state
  if ( changed ) push();
}

// restoreState
void smartStateMgr::restoreState()
{
  pop();
}



//
// calNode Implementation
//
void calNode::init()
{
  state().style()    = calState::NoLineStyle;
  state().type()     = calState::NoType;
  state().lighting() = calState::LightingOff;

  id()               = _count++;

}

void calNode::dump( ostream &ostrm )
{
  ostrm << "Node: " << id() << endl;
  ostrm << "Type: "  << calState::RenderTypeNames[  state().type() ] << endl;
  ostrm << "Style: " << calState::LineStyleNames[ state().style() ] << endl;
  ostrm << "Lighting: " << calState::LightingNames[ state().lighting() ] << endl;
  ostrm << "Color: " << state().color()[0] << ", " 
	             << state().color()[1] << ", " 
	             << state().color()[2] << endl;

  ostrm << "Location: " << location()[0] << ", " 
	                << location()[1] << ", " 
	                << location()[2] << endl;
}

void calNode::preRender()
{
  setState();
  scene()->stateMgr()->setState( state() );
}

void calNode::postRender()
{
  scene()->stateMgr()->restoreState();
  resetState();
}

void calNode::render()
{
  preRender();
    renderNode();
  postRender();
}


//
// calCube Implementation
//

void calCube::renderNode()
{
  glPushMatrix();
    glTranslatef( location()[0], location()[1], location()[2] );
    glutSolidCube( 1.0 );
  glPopMatrix();
}

//
// calSpehere Implementation
//
void calSphere::renderNode()
{
  glPushMatrix();
    glTranslatef( location()[0], location()[1], location()[2] );
    glutSolidSphere( 1.0, 20, 20 );
  glPopMatrix();
}

//
// calNodeFactory Implementation
//
calNode *calNodeFactory::create( calState::RenderTypes t, calSceneGraph *g )
{
  if ( t == calState::Cube )   return new calCube( g );
  if ( t == calState::Sphere ) return new calSphere( g );

  return NULL;
}




//
// calSceneMgr Implementation
//
void calSceneMgr::build( const int num ) 
{
  //  srand48( getpid() );
  if ( size() > 0 ) destroy();

  calNode        **n = new calNode*[num];
  calNodeFactory f;

#ifdef WIN32
#define drand48 rand
#endif

#define RANDOM_NODE  (drand48() > .5 ? calState::Sphere : calState::Cube)
#define RANDOM_STYLE (drand48() > .5 ? calState::Wire : calState::Solid)
#define RANDOM_LIGHTING (drand48() > .5 ? calState::LightingOff : calState::LightingOn)
// a number between -2 and 2
#define SCALE ((drand48() < .5 ? -1.0 : 1.0) * 2.0)

  int i;

  for ( i = 0; i < num; ++i ) {
    // create a cube or sphere
    n[i] = f.create( RANDOM_NODE, this );

    // set a random location
    n[i]->location()[0] = SCALE * drand48();
    n[i]->location()[1] = SCALE * drand48();
    n[i]->location()[2] = 20 * drand48();

    // set it up as either wire frame or solid
    n[i]->state().style() = RANDOM_STYLE;

    // set a random color
    n[i]->state().color()[0] = drand48();
    n[i]->state().color()[1] = drand48();
    n[i]->state().color()[2] = drand48();
    n[i]->state().color()[3] = drand48();

    // turn lighting on or off
    n[i]->state().lighting() = RANDOM_LIGHTING;
    n[i]->state().lightPosition()[0] = drand48();
    n[i]->state().lightPosition()[1] = drand48();
    n[i]->state().lightPosition()[2] = drand48();

    
  }

  nodes() = n;
  size()  = num;
}

void calSceneMgr::destroy()
{
  if ( size() == 0 ) return;

  for ( int i = 0; i < size(); ++i ) {
    delete nodes()[i];
  }

  delete nodes();

  nodes() = NULL;
  size()  = 0;
}

void calSceneMgr::initGfx( int w, int h, char *name )
{
  int  argc = 0;
  char *argv[1] = {name};

  glutInit( &argc, argv );
  glutInitDisplayMode( GLUT_SINGLE | GLUT_RGB | GLUT_DEPTH );
  glutInitWindowSize( w, h );
  glutCreateWindow( name );

  glEnable( GL_DEPTH_TEST );
  glMatrixMode( GL_PROJECTION );
  glLoadIdentity();
  gluPerspective( 60.0, 1.0, 1.0, 20 );
  
  glMatrixMode( GL_MODELVIEW );
  glLoadIdentity();
  glTranslatef( 0.0, 0.0, -10.0 );

  glutDisplayFunc( displayFunc );
  glutKeyboardFunc( keyboardFunc );
  glutVisibilityFunc( visibilityFunc );
}

void calSceneMgr::displayFunc()
{
  if ( calSceneMgr::sceneMgr()->visible() ) 
    
    (calSceneMgr::sceneMgr()->renderCB())( calSceneMgr::sceneMgr() );
}

void calSceneMgr::idleFunc()
{
}

void calSceneMgr::visibilityFunc( int vis )
{
  if ( vis == GLUT_VISIBLE ) {
    calSceneMgr::sceneMgr()->visible() = 1;
  }
  else {
    calSceneMgr::sceneMgr()->visible() = 0;
  }
}

void calSceneMgr::keyboardFunc( unsigned char key, int x, int y )
{
  if ( calSceneMgr::sceneMgr()->keyCB() ) 
    (calSceneMgr::sceneMgr()->keyCB())( calSceneMgr::sceneMgr(), key, x, y );
  else
    if ( key == 27 || key == 'q' ) exit( 0 );
}

void calSceneMgr::mainLoop( RenderCB cb )
{
  renderCB() = cb;
  glutMainLoop();
}

int 
#ifdef WIN32
__cdecl 
#endif
calSceneMgr::_nodeStateCmp( const void *n1, const void *n2)
{
  calNode *a = *(calNode **)n1;
  calNode *b = *(calNode **)n2;

  enum Keys {Style, Light};

  int keys[2];
  // just cmp enum values of style for primary sort key
  keys[Style] = ( a->state().style() < b->state().style() ) 
                  ? -1 
                  : !( a->state().style() == b->state().style() );
  
  // either on or off, off goes before on by the order of the enums
  keys[Light] = ( a->state().lighting() < b->state().lighting()  ) 
                  ? -1
                  : !( a->state().lighting() == b->state().lighting() );

  // primary key lighting, secondary key style
  return (keys[Light] == 0) ? keys[Style] : keys[Light];
}

void calSceneMgr::sort( )
{
  qsort( nodes(), size(), sizeof(calNode*), _nodeStateCmp );
}

void calSceneMgr::render()
{
  glClear( GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT );

  for ( int i = 0; i < size(); ++i ) {
    
    nodes()[i]->render();
  }
}

void calSceneMgr::shallowCopy( const calSceneMgr &g )
{
  // g._num cuz g.size() breaks const'ness. 

  size()  = g._num;
  nodes() = new calNode*[g._num]; 
  stateMgr() = g._state;

  for ( int i = 0; i < _num; ++i ) {
      nodes()[i] = g._scene[i];
  }
}

void calSceneMgr::set( calSceneMgr *s )
{
  // set the global scene manager
  sceneMgr() = s;

  // make sure all the nodes in the scene point to the correct
  // scene manager
  for ( int i = 0; i < s->size(); ++i ) {
      s->nodes()[i]->scene() = s;
  }
  
}

calSceneMgr *&calSceneMgr::sceneMgr()
{
  return _sceneMgr;
}


//
// Statics
//

calSceneMgr *calSceneMgr::_sceneMgr;
int         calNode::_count;

char *calState::LineStyleNames[]  = {"None", "Solid", "Wire"};
char *calState::RenderTypeNames[] = {"None", "Cube", "Sphere"};
char *calState::LightingNames[]   = {"Off", "On"};

