// _________________________________________________________________________
//
// Win32 module to redirect calls from OPENGL32.DLL to specified provider DLL, 
// with API function trace to OutputDebugString() or text file
//
// Released into public domain 9-Jul-97
//
// Please forward enhancements and bug reports to jmiles@digitalanvil.com
// _________________________________________________________________________
//
// Instructions:
//
//    - Compile this module with NMAKE /f OPENGL32.MAK to yield OPENGL32.DLL
//
//    - If original OPENGL32.DLL resides in application directory, rename or
//      move it to avoid conflicts with this copy of OPENGL32.DLL
//
//    - After renaming original OPENGL32.DLL (if necessary), copy 
//      OPENGL32.DLL and GLTRACE.INI into target application directory 
//      (e.g, C:\QUAKE)
//
//    - Modify GLTRACE.INI's [Implementation] section to point to renamed or
//      moved copy of original OPENGL32.DLL (e.g., the 3Dfx GLQuake library)
//
//    - Modify other GLTRACE.INI sections, if desired, in accordance with
//      instructions in GLTRACE.INI comment header
//
//    - Run the application under test (e.g., GLQUAKE.EXE) and observe
//      output trace
// _________________________________________________________________________

#define VERSION "0.90"           // (Alpha release, largely untested)

#include <windows.h>
#include <mmsystem.h>
#include <stdio.h>
#include <stdarg.h>
#include <time.h>
#include "assert.h"

#pragma warning (disable:4273)   // No complaints about DLL linkage...
#pragma pack(1)                  // Disable structure rearrangement

typedef unsigned int GLenum;
typedef unsigned char GLboolean;
typedef unsigned int GLbitfield;
typedef signed char GLbyte;
typedef short GLshort;
typedef int GLint;
typedef int GLsizei;
typedef unsigned char GLubyte;
typedef unsigned char *GLstring;
typedef unsigned short GLushort;
typedef unsigned int GLuint;
typedef float GLfloat;
typedef float GLclampf;
typedef double GLdouble;
typedef double GLclampd;
typedef void GLvoid;

#define ARY_CNT(x) (sizeof((x)) / sizeof((x)[0]))

//
// Argument/return type names and their format specifiers
//

enum TYPE
{
   HEX,
   SINT,
   UNINT,
   FLT,
   DBL,
   STR,
   GLEN,
   GLBO,
   PFD
};

struct FORMAT
{
   char *name;
   TYPE  type;
};

FORMAT type_formats[] = 
{
   {"GLenum",                GLEN},
   {"GLboolean",             GLBO},
   {"GLbyte",                SINT},
   {"GLshort",               SINT},
   {"GLint",                 SINT},
   {"GLsizei",               SINT},
   {"GLubyte",               UNINT},
   {"GLushort",              UNINT},
   {"GLuint",                UNINT},
   {"GLfloat",               FLT},
   {"GLclampf",              FLT},
   {"GLdouble",              DBL},
   {"GLclampd",              DBL},
   {"int",                   SINT},
   {"BOOL",                  GLBO},
   {"LPCSTR",                STR},
   {"GLstring",              STR},
   {"PIXELFORMATDESCRIPTOR*",PFD}

   //
   // Add any other types you want to recognize here...
   //
};

//
// Names of OpenGL functions
//

static char *GLN[] = 
{
   "glAccum",
   "glAlphaFunc",
   "glAreTexturesResident",
   "glArrayElement",
   "glBegin",
   "glBindTexture",
   "glBitmap",
   "glBlendFunc",
   "glCallList",
   "glCallLists",
   "glClear",
   "glClearAccum",
   "glClearColor",
   "glClearDepth",
   "glClearIndex",
   "glClearStencil",
   "glClipPlane",
   "glColor3b",
   "glColor3bv",
   "glColor3d",
   "glColor3dv",
   "glColor3f",
   "glColor3fv",
   "glColor3i",
   "glColor3iv",
   "glColor3s",
   "glColor3sv",
   "glColor3ub",
   "glColor3ubv",
   "glColor3ui",
   "glColor3uiv",
   "glColor3us",
   "glColor3usv",
   "glColor4b",
   "glColor4bv",
   "glColor4d",
   "glColor4dv",
   "glColor4f",
   "glColor4fv",
   "glColor4i",
   "glColor4iv",
   "glColor4s",
   "glColor4sv",
   "glColor4ub",
   "glColor4ubv",
   "glColor4ui",
   "glColor4uiv",
   "glColor4us",
   "glColor4usv",
   "glColorMask",
   "glColorMaterial",
   "glColorPointer",
   "glCopyPixels",
   "glCopyTexImage1D",
   "glCopyTexImage2D",
   "glCopyTexSubImage1D",
   "glCopyTexSubImage2D",
   "glCullFace",
   "glDeleteLists",
   "glDeleteTextures",
   "glDepthFunc",
   "glDepthMask",
   "glDepthRange",
   "glDisable",
   "glDisableClientState",
   "glDrawArrays",
   "glDrawBuffer",
   "glDrawElements",
   "glDrawPixels",
   "glEdgeFlag",
   "glEdgeFlagPointer",
   "glEdgeFlagv",
   "glEnable",
   "glEnableClientState",
   "glEnd",
   "glEndList",
   "glEvalCoord1d",
   "glEvalCoord1dv",
   "glEvalCoord1f",
   "glEvalCoord1fv",
   "glEvalCoord2d",
   "glEvalCoord2dv",
   "glEvalCoord2f",
   "glEvalCoord2fv",
   "glEvalMesh1",
   "glEvalMesh2",
   "glEvalPoint1",
   "glEvalPoint2",
   "glFeedbackBuffer",
   "glFinish",
   "glFlush",
   "glFogf",
   "glFogfv",
   "glFogi",
   "glFogiv",
   "glFrontFace",
   "glFrustum",
   "glGenLists",
   "glGenTextures",
   "glGetBooleanv",
   "glGetClipPlane",
   "glGetDoublev",
   "glGetError",
   "glGetFloatv",
   "glGetIntegerv",
   "glGetLightfv",
   "glGetLightiv",
   "glGetMapdv",
   "glGetMapfv",
   "glGetMapiv",
   "glGetMaterialfv",
   "glGetMaterialiv",
   "glGetPixelMapfv",
   "glGetPixelMapuiv",
   "glGetPixelMapusv",
   "glGetPointerv",
   "glGetPolygonStipple",
   "glGetString",
   "glGetTexEnvfv",
   "glGetTexEnviv",
   "glGetTexGendv",
   "glGetTexGenfv",
   "glGetTexGeniv",
   "glGetTexImage",
   "glGetTexLevelParameterfv",
   "glGetTexLevelParameteriv",
   "glGetTexParameterfv",
   "glGetTexParameteriv",
   "glHint",
   "glIndexMask",
   "glIndexPointer",
   "glIndexd",
   "glIndexdv",
   "glIndexf",
   "glIndexfv",
   "glIndexi",
   "glIndexiv",
   "glIndexs",
   "glIndexsv",
   "glIndexub",
   "glIndexubv",
   "glInitNames",
   "glInterleavedArrays",
   "glIsEnabled",
   "glIsList",
   "glIsTexture",
   "glLightModelf",
   "glLightModelfv",
   "glLightModeli",
   "glLightModeliv",
   "glLightf",
   "glLightfv",
   "glLighti",
   "glLightiv",
   "glLineStipple",
   "glLineWidth",
   "glListBase",
   "glLoadIdentity",
   "glLoadMatrixd",
   "glLoadMatrixf",
   "glLoadName",
   "glLogicOp",
   "glMap1d",
   "glMap1f",
   "glMap2d",
   "glMap2f",
   "glMapGrid1d",
   "glMapGrid1f",
   "glMapGrid2d",
   "glMapGrid2f",
   "glMaterialf",
   "glMaterialfv",
   "glMateriali",
   "glMaterialiv",
   "glMatrixMode",
   "glMultMatrixd",
   "glMultMatrixf",
   "glNewList",
   "glNormal3b",
   "glNormal3bv",
   "glNormal3d",
   "glNormal3dv",
   "glNormal3f",
   "glNormal3fv",
   "glNormal3i",
   "glNormal3iv",
   "glNormal3s",
   "glNormal3sv",
   "glNormalPointer",
   "glOrtho",
   "glPassThrough",
   "glPixelMapfv",
   "glPixelMapuiv",
   "glPixelMapusv",
   "glPixelStoref",
   "glPixelStorei",
   "glPixelTransferf",
   "glPixelTransferi",
   "glPixelZoom",
   "glPointSize",
   "glPolygonMode",
   "glPolygonOffset",
   "glPolygonStipple",
   "glPopAttrib",
   "glPopClientAttrib",
   "glPopMatrix",
   "glPopName",
   "glPrioritizeTextures",
   "glPushAttrib",
   "glPushClientAttrib",
   "glPushMatrix",
   "glPushName",
   "glRasterPos2d",
   "glRasterPos2dv",
   "glRasterPos2f",
   "glRasterPos2fv",
   "glRasterPos2i",
   "glRasterPos2iv",
   "glRasterPos2s",
   "glRasterPos2sv",
   "glRasterPos3d",
   "glRasterPos3dv",
   "glRasterPos3f",
   "glRasterPos3fv",
   "glRasterPos3i",
   "glRasterPos3iv",
   "glRasterPos3s",
   "glRasterPos3sv",
   "glRasterPos4d",
   "glRasterPos4dv",
   "glRasterPos4f",
   "glRasterPos4fv",
   "glRasterPos4i",
   "glRasterPos4iv",
   "glRasterPos4s",
   "glRasterPos4sv",
   "glReadBuffer",
   "glReadPixels",
   "glRectd",
   "glRectdv",
   "glRectf",
   "glRectfv",
   "glRecti",
   "glRectiv",
   "glRects",
   "glRectsv",
   "glRenderMode",
   "glRotated",
   "glRotatef",
   "glScaled",
   "glScalef",
   "glScissor",
   "glSelectBuffer",
   "glShadeModel",
   "glStencilFunc",
   "glStencilMask",
   "glStencilOp",
   "glTexCoord1d",
   "glTexCoord1dv",
   "glTexCoord1f",
   "glTexCoord1fv",
   "glTexCoord1i",
   "glTexCoord1iv",
   "glTexCoord1s",
   "glTexCoord1sv",
   "glTexCoord2d",
   "glTexCoord2dv",
   "glTexCoord2f",
   "glTexCoord2fv",
   "glTexCoord2i",
   "glTexCoord2iv",
   "glTexCoord2s",
   "glTexCoord2sv",
   "glTexCoord3d",
   "glTexCoord3dv",
   "glTexCoord3f",
   "glTexCoord3fv",
   "glTexCoord3i",
   "glTexCoord3iv",
   "glTexCoord3s",
   "glTexCoord3sv",
   "glTexCoord4d",
   "glTexCoord4dv",
   "glTexCoord4f",
   "glTexCoord4fv",
   "glTexCoord4i",
   "glTexCoord4iv",
   "glTexCoord4s",
   "glTexCoord4sv",
   "glTexCoordPointer",
   "glTexEnvf",
   "glTexEnvfv",
   "glTexEnvi",
   "glTexEnviv",
   "glTexGend",
   "glTexGendv",
   "glTexGenf",
   "glTexGenfv",
   "glTexGeni",
   "glTexGeniv",
   "glTexImage1D",
   "glTexImage2D",
   "glTexParameterf",
   "glTexParameterfv",
   "glTexParameteri",
   "glTexParameteriv",
   "glTexSubImage1D",
   "glTexSubImage2D",
   "glTranslated",
   "glTranslatef",
   "glVertex2d",
   "glVertex2dv",
   "glVertex2f",
   "glVertex2fv",
   "glVertex2i",
   "glVertex2iv",
   "glVertex2s",
   "glVertex2sv",
   "glVertex3d",
   "glVertex3dv",
   "glVertex3f",
   "glVertex3fv",
   "glVertex3i",
   "glVertex3iv",
   "glVertex3s",
   "glVertex3sv",
   "glVertex4d",
   "glVertex4dv",
   "glVertex4f",
   "glVertex4fv",
   "glVertex4i",
   "glVertex4iv",
   "glVertex4s",
   "glVertex4sv",
   "glVertexPointer",
   "glViewport",
   "wglChoosePixelFormat",
   "wglCopyContext",
   "wglCreateContext",
   "wglCreateLayerContext",
   "wglDeleteContext",
   "wglDescribeLayerPlane",
   "wglDescribePixelFormat",
   "wglGetCurrentContext",
   "wglGetCurrentDC",
   "wglGetDefaultProcAddress",
   "wglGetLayerPaletteEntries",
   "wglGetPixelFormat",
   "wglGetProcAddress",
   "wglMakeCurrent",
   "wglRealizeLayerPalette",
   "wglSetLayerPaletteEntries",
   "wglSetPixelFormat",
   "wglShareLists",
   "wglSwapBuffers",
   "wglSwapLayerBuffers",
   "wglUseFontBitmapsA",
   "wglUseFontBitmapsW",
   "wglUseFontOutlinesA",
   "wglUseFontOutlinesW"
};

//
// Numerical identifiers
//

enum GLID
{
   FP_glAccum = 0,
   FP_glAlphaFunc,
   FP_glAreTexturesResident,
   FP_glArrayElement,
   FP_glBegin,
   FP_glBindTexture,
   FP_glBitmap,
   FP_glBlendFunc,
   FP_glCallList,
   FP_glCallLists,
   FP_glClear,
   FP_glClearAccum,
   FP_glClearColor,
   FP_glClearDepth,
   FP_glClearIndex,
   FP_glClearStencil,
   FP_glClipPlane,
   FP_glColor3b,
   FP_glColor3bv,
   FP_glColor3d,
   FP_glColor3dv,
   FP_glColor3f,
   FP_glColor3fv,
   FP_glColor3i,
   FP_glColor3iv,
   FP_glColor3s,
   FP_glColor3sv,
   FP_glColor3ub,
   FP_glColor3ubv,
   FP_glColor3ui,
   FP_glColor3uiv,
   FP_glColor3us,
   FP_glColor3usv,
   FP_glColor4b,
   FP_glColor4bv,
   FP_glColor4d,
   FP_glColor4dv,
   FP_glColor4f,
   FP_glColor4fv,
   FP_glColor4i,
   FP_glColor4iv,
   FP_glColor4s,
   FP_glColor4sv,
   FP_glColor4ub,
   FP_glColor4ubv,
   FP_glColor4ui,
   FP_glColor4uiv,
   FP_glColor4us,
   FP_glColor4usv,
   FP_glColorMask,
   FP_glColorMaterial,
   FP_glColorPointer,
   FP_glCopyPixels,
   FP_glCopyTexImage1D,
   FP_glCopyTexImage2D,
   FP_glCopyTexSubImage1D,
   FP_glCopyTexSubImage2D,
   FP_glCullFace,
   FP_glDeleteLists,
   FP_glDeleteTextures,
   FP_glDepthFunc,
   FP_glDepthMask,
   FP_glDepthRange,
   FP_glDisable,
   FP_glDisableClientState,
   FP_glDrawArrays,
   FP_glDrawBuffer,
   FP_glDrawElements,
   FP_glDrawPixels,
   FP_glEdgeFlag,
   FP_glEdgeFlagPointer,
   FP_glEdgeFlagv,
   FP_glEnable,
   FP_glEnableClientState,
   FP_glEnd,
   FP_glEndList,
   FP_glEvalCoord1d,
   FP_glEvalCoord1dv,
   FP_glEvalCoord1f,
   FP_glEvalCoord1fv,
   FP_glEvalCoord2d,
   FP_glEvalCoord2dv,
   FP_glEvalCoord2f,
   FP_glEvalCoord2fv,
   FP_glEvalMesh1,
   FP_glEvalMesh2,
   FP_glEvalPoint1,
   FP_glEvalPoint2,
   FP_glFeedbackBuffer,
   FP_glFinish,
   FP_glFlush,
   FP_glFogf,
   FP_glFogfv,
   FP_glFogi,
   FP_glFogiv,
   FP_glFrontFace,
   FP_glFrustum,
   FP_glGenLists,
   FP_glGenTextures,
   FP_glGetBooleanv,
   FP_glGetClipPlane,
   FP_glGetDoublev,
   FP_glGetError,
   FP_glGetFloatv,
   FP_glGetIntegerv,
   FP_glGetLightfv,
   FP_glGetLightiv,
   FP_glGetMapdv,
   FP_glGetMapfv,
   FP_glGetMapiv,
   FP_glGetMaterialfv,
   FP_glGetMaterialiv,
   FP_glGetPixelMapfv,
   FP_glGetPixelMapuiv,
   FP_glGetPixelMapusv,
   FP_glGetPointerv,
   FP_glGetPolygonStipple,
   FP_glGetString,
   FP_glGetTexEnvfv,
   FP_glGetTexEnviv,
   FP_glGetTexGendv,
   FP_glGetTexGenfv,
   FP_glGetTexGeniv,
   FP_glGetTexImage,
   FP_glGetTexLevelParameterfv,
   FP_glGetTexLevelParameteriv,
   FP_glGetTexParameterfv,
   FP_glGetTexParameteriv,
   FP_glHint,
   FP_glIndexMask,
   FP_glIndexPointer,
   FP_glIndexd,
   FP_glIndexdv,
   FP_glIndexf,
   FP_glIndexfv,
   FP_glIndexi,
   FP_glIndexiv,
   FP_glIndexs,
   FP_glIndexsv,
   FP_glIndexub,
   FP_glIndexubv,
   FP_glInitNames,
   FP_glInterleavedArrays,
   FP_glIsEnabled,
   FP_glIsList,
   FP_glIsTexture,
   FP_glLightModelf,
   FP_glLightModelfv,
   FP_glLightModeli,
   FP_glLightModeliv,
   FP_glLightf,
   FP_glLightfv,
   FP_glLighti,
   FP_glLightiv,
   FP_glLineStipple,
   FP_glLineWidth,
   FP_glListBase,
   FP_glLoadIdentity,
   FP_glLoadMatrixd,
   FP_glLoadMatrixf,
   FP_glLoadName,
   FP_glLogicOp,
   FP_glMap1d,
   FP_glMap1f,
   FP_glMap2d,
   FP_glMap2f,
   FP_glMapGrid1d,
   FP_glMapGrid1f,
   FP_glMapGrid2d,
   FP_glMapGrid2f,
   FP_glMaterialf,
   FP_glMaterialfv,
   FP_glMateriali,
   FP_glMaterialiv,
   FP_glMatrixMode,
   FP_glMultMatrixd,
   FP_glMultMatrixf,
   FP_glNewList,
   FP_glNormal3b,
   FP_glNormal3bv,
   FP_glNormal3d,
   FP_glNormal3dv,
   FP_glNormal3f,
   FP_glNormal3fv,
   FP_glNormal3i,
   FP_glNormal3iv,
   FP_glNormal3s,
   FP_glNormal3sv,
   FP_glNormalPointer,
   FP_glOrtho,
   FP_glPassThrough,
   FP_glPixelMapfv,
   FP_glPixelMapuiv,
   FP_glPixelMapusv,
   FP_glPixelStoref,
   FP_glPixelStorei,
   FP_glPixelTransferf,
   FP_glPixelTransferi,
   FP_glPixelZoom,
   FP_glPointSize,
   FP_glPolygonMode,
   FP_glPolygonOffset,
   FP_glPolygonStipple,
   FP_glPopAttrib,
   FP_glPopClientAttrib,
   FP_glPopMatrix,
   FP_glPopName,
   FP_glPrioritizeTextures,
   FP_glPushAttrib,
   FP_glPushClientAttrib,
   FP_glPushMatrix,
   FP_glPushName,
   FP_glRasterPos2d,
   FP_glRasterPos2dv,
   FP_glRasterPos2f,
   FP_glRasterPos2fv,
   FP_glRasterPos2i,
   FP_glRasterPos2iv,
   FP_glRasterPos2s,
   FP_glRasterPos2sv,
   FP_glRasterPos3d,
   FP_glRasterPos3dv,
   FP_glRasterPos3f,
   FP_glRasterPos3fv,
   FP_glRasterPos3i,
   FP_glRasterPos3iv,
   FP_glRasterPos3s,
   FP_glRasterPos3sv,
   FP_glRasterPos4d,
   FP_glRasterPos4dv,
   FP_glRasterPos4f,
   FP_glRasterPos4fv,
   FP_glRasterPos4i,
   FP_glRasterPos4iv,
   FP_glRasterPos4s,
   FP_glRasterPos4sv,
   FP_glReadBuffer,
   FP_glReadPixels,
   FP_glRectd,
   FP_glRectdv,
   FP_glRectf,
   FP_glRectfv,
   FP_glRecti,
   FP_glRectiv,
   FP_glRects,
   FP_glRectsv,
   FP_glRenderMode,
   FP_glRotated,
   FP_glRotatef,
   FP_glScaled,
   FP_glScalef,
   FP_glScissor,
   FP_glSelectBuffer,
   FP_glShadeModel,
   FP_glStencilFunc,
   FP_glStencilMask,
   FP_glStencilOp,
   FP_glTexCoord1d,
   FP_glTexCoord1dv,
   FP_glTexCoord1f,
   FP_glTexCoord1fv,
   FP_glTexCoord1i,
   FP_glTexCoord1iv,
   FP_glTexCoord1s,
   FP_glTexCoord1sv,
   FP_glTexCoord2d,
   FP_glTexCoord2dv,
   FP_glTexCoord2f,
   FP_glTexCoord2fv,
   FP_glTexCoord2i,
   FP_glTexCoord2iv,
   FP_glTexCoord2s,
   FP_glTexCoord2sv,
   FP_glTexCoord3d,
   FP_glTexCoord3dv,
   FP_glTexCoord3f,
   FP_glTexCoord3fv,
   FP_glTexCoord3i,
   FP_glTexCoord3iv,
   FP_glTexCoord3s,
   FP_glTexCoord3sv,
   FP_glTexCoord4d,
   FP_glTexCoord4dv,
   FP_glTexCoord4f,
   FP_glTexCoord4fv,
   FP_glTexCoord4i,
   FP_glTexCoord4iv,
   FP_glTexCoord4s,
   FP_glTexCoord4sv,
   FP_glTexCoordPointer,
   FP_glTexEnvf,
   FP_glTexEnvfv,
   FP_glTexEnvi,
   FP_glTexEnviv,
   FP_glTexGend,
   FP_glTexGendv,
   FP_glTexGenf,
   FP_glTexGenfv,
   FP_glTexGeni,
   FP_glTexGeniv,
   FP_glTexImage1D,
   FP_glTexImage2D,
   FP_glTexParameterf,
   FP_glTexParameterfv,
   FP_glTexParameteri,
   FP_glTexParameteriv,
   FP_glTexSubImage1D,
   FP_glTexSubImage2D,
   FP_glTranslated,
   FP_glTranslatef,
   FP_glVertex2d,
   FP_glVertex2dv,
   FP_glVertex2f,
   FP_glVertex2fv,
   FP_glVertex2i,
   FP_glVertex2iv,
   FP_glVertex2s,
   FP_glVertex2sv,
   FP_glVertex3d,
   FP_glVertex3dv,
   FP_glVertex3f,
   FP_glVertex3fv,
   FP_glVertex3i,
   FP_glVertex3iv,
   FP_glVertex3s,
   FP_glVertex3sv,
   FP_glVertex4d,
   FP_glVertex4dv,
   FP_glVertex4f,
   FP_glVertex4fv,
   FP_glVertex4i,
   FP_glVertex4iv,
   FP_glVertex4s,
   FP_glVertex4sv,
   FP_glVertexPointer,
   FP_glViewport,
   FP_wglChoosePixelFormat,
   FP_wglCopyContext,
   FP_wglCreateContext,
   FP_wglCreateLayerContext,
   FP_wglDeleteContext,
   FP_wglDescribeLayerPlane,
   FP_wglDescribePixelFormat,
   FP_wglGetCurrentContext,
   FP_wglGetCurrentDC,
   FP_wglGetDefaultProcAddress,
   FP_wglGetLayerPaletteEntries,
   FP_wglGetPixelFormat,
   FP_wglGetProcAddress,
   FP_wglMakeCurrent,
   FP_wglRealizeLayerPalette,
   FP_wglSetLayerPaletteEntries,
   FP_wglSetPixelFormat,
   FP_wglShareLists,
   FP_wglSwapBuffers,
   FP_wglSwapLayerBuffers,
   FP_wglUseFontBitmapsA,
   FP_wglUseFontBitmapsW,
   FP_wglUseFontOutlinesA,
   FP_wglUseFontOutlinesW
};

//
// Status values
//

#define SUPPRESS           0     // Never trace this function
#define SHOW_ALWAYS        1     // Trace this function
#define SHOW_IF_SHIFT_DOWN 2     // Trace this function only if user holding down
                                 // SHIFT or ALT key

int show_status[ARY_CNT(GLN)];

//
// Jump table vectors to GLTRACE entries (aka "real" OPENGL32.DLL)
//

struct
{
   void (APIENTRY *glAccum) (GLenum op, GLfloat value);
   void (APIENTRY *glAlphaFunc) (GLenum func, GLclampf ref);
   GLboolean (APIENTRY *glAreTexturesResident) (GLsizei n, const GLuint *textures, GLboolean *residences);
   void (APIENTRY *glArrayElement) (GLint i);
   void (APIENTRY *glBegin) (GLenum mode);
   void (APIENTRY *glBindTexture) (GLenum target, GLuint texture);
   void (APIENTRY *glBitmap) (GLsizei width, GLsizei height, GLfloat xorig, GLfloat yorig, GLfloat xmove, GLfloat ymove, const GLubyte *bitmap);
   void (APIENTRY *glBlendFunc) (GLenum sfactor, GLenum dfactor);
   void (APIENTRY *glCallList) (GLuint list);
   void (APIENTRY *glCallLists) (GLsizei n, GLenum type, const GLvoid *lists);
   void (APIENTRY *glClear) (GLbitfield mask);
   void (APIENTRY *glClearAccum) (GLfloat red, GLfloat green, GLfloat blue, GLfloat alpha);
   void (APIENTRY *glClearColor) (GLclampf red, GLclampf green, GLclampf blue, GLclampf alpha);
   void (APIENTRY *glClearDepth) (GLclampd depth);
   void (APIENTRY *glClearIndex) (GLfloat c);
   void (APIENTRY *glClearStencil) (GLint s);
   void (APIENTRY *glClipPlane) (GLenum plane, const GLdouble *equation);
   void (APIENTRY *glColor3b) (GLbyte red, GLbyte green, GLbyte blue);
   void (APIENTRY *glColor3bv) (const GLbyte *v);
   void (APIENTRY *glColor3d) (GLdouble red, GLdouble green, GLdouble blue);
   void (APIENTRY *glColor3dv) (const GLdouble *v);
   void (APIENTRY *glColor3f) (GLfloat red, GLfloat green, GLfloat blue);
   void (APIENTRY *glColor3fv) (const GLfloat *v);
   void (APIENTRY *glColor3i) (GLint red, GLint green, GLint blue);
   void (APIENTRY *glColor3iv) (const GLint *v);
   void (APIENTRY *glColor3s) (GLshort red, GLshort green, GLshort blue);
   void (APIENTRY *glColor3sv) (const GLshort *v);
   void (APIENTRY *glColor3ub) (GLubyte red, GLubyte green, GLubyte blue);
   void (APIENTRY *glColor3ubv) (const GLubyte *v);
   void (APIENTRY *glColor3ui) (GLuint red, GLuint green, GLuint blue);
   void (APIENTRY *glColor3uiv) (const GLuint *v);
   void (APIENTRY *glColor3us) (GLushort red, GLushort green, GLushort blue);
   void (APIENTRY *glColor3usv) (const GLushort *v);
   void (APIENTRY *glColor4b) (GLbyte red, GLbyte green, GLbyte blue, GLbyte alpha);
   void (APIENTRY *glColor4bv) (const GLbyte *v);
   void (APIENTRY *glColor4d) (GLdouble red, GLdouble green, GLdouble blue, GLdouble alpha);
   void (APIENTRY *glColor4dv) (const GLdouble *v);
   void (APIENTRY *glColor4f) (GLfloat red, GLfloat green, GLfloat blue, GLfloat alpha);
   void (APIENTRY *glColor4fv) (const GLfloat *v);
   void (APIENTRY *glColor4i) (GLint red, GLint green, GLint blue, GLint alpha);
   void (APIENTRY *glColor4iv) (const GLint *v);
   void (APIENTRY *glColor4s) (GLshort red, GLshort green, GLshort blue, GLshort alpha);
   void (APIENTRY *glColor4sv) (const GLshort *v);
   void (APIENTRY *glColor4ub) (GLubyte red, GLubyte green, GLubyte blue, GLubyte alpha);
   void (APIENTRY *glColor4ubv) (const GLubyte *v);
   void (APIENTRY *glColor4ui) (GLuint red, GLuint green, GLuint blue, GLuint alpha);
   void (APIENTRY *glColor4uiv) (const GLuint *v);
   void (APIENTRY *glColor4us) (GLushort red, GLushort green, GLushort blue, GLushort alpha);
   void (APIENTRY *glColor4usv) (const GLushort *v);
   void (APIENTRY *glColorMask) (GLboolean red, GLboolean green, GLboolean blue, GLboolean alpha);
   void (APIENTRY *glColorMaterial) (GLenum face, GLenum mode);
   void (APIENTRY *glColorPointer) (GLint size, GLenum type, GLsizei stride, const GLvoid *pointer);
   void (APIENTRY *glCopyPixels) (GLint x, GLint y, GLsizei width, GLsizei height, GLenum type);
   void (APIENTRY *glCopyTexImage1D) (GLenum target, GLint level, GLenum internalFormat, GLint x, GLint y, GLsizei width, GLint border);
   void (APIENTRY *glCopyTexImage2D) (GLenum target, GLint level, GLenum internalFormat, GLint x, GLint y, GLsizei width, GLsizei height, GLint border);
   void (APIENTRY *glCopyTexSubImage1D) (GLenum target, GLint level, GLint xoffset, GLint x, GLint y, GLsizei width);
   void (APIENTRY *glCopyTexSubImage2D) (GLenum target, GLint level, GLint xoffset, GLint yoffset, GLint x, GLint y, GLsizei width, GLsizei height);
   void (APIENTRY *glCullFace) (GLenum mode);
   void (APIENTRY *glDeleteLists) (GLuint list, GLsizei range);
   void (APIENTRY *glDeleteTextures) (GLsizei n, const GLuint *textures);
   void (APIENTRY *glDepthFunc) (GLenum func);
   void (APIENTRY *glDepthMask) (GLboolean flag);
   void (APIENTRY *glDepthRange) (GLclampd zNear, GLclampd zFar);
   void (APIENTRY *glDisable) (GLenum cap);
   void (APIENTRY *glDisableClientState) (GLenum array);
   void (APIENTRY *glDrawArrays) (GLenum mode, GLint first, GLsizei count);
   void (APIENTRY *glDrawBuffer) (GLenum mode);
   void (APIENTRY *glDrawElements) (GLenum mode, GLsizei count, GLenum type, const GLvoid *indices);
   void (APIENTRY *glDrawPixels) (GLsizei width, GLsizei height, GLenum format, GLenum type, const GLvoid *pixels);
   void (APIENTRY *glEdgeFlag) (GLboolean flag);
   void (APIENTRY *glEdgeFlagPointer) (GLsizei stride, const GLvoid *pointer);
   void (APIENTRY *glEdgeFlagv) (const GLboolean *flag);
   void (APIENTRY *glEnable) (GLenum cap);
   void (APIENTRY *glEnableClientState) (GLenum array);
   void (APIENTRY *glEnd) (void);
   void (APIENTRY *glEndList) (void);
   void (APIENTRY *glEvalCoord1d) (GLdouble u);
   void (APIENTRY *glEvalCoord1dv) (const GLdouble *u);
   void (APIENTRY *glEvalCoord1f) (GLfloat u);
   void (APIENTRY *glEvalCoord1fv) (const GLfloat *u);
   void (APIENTRY *glEvalCoord2d) (GLdouble u, GLdouble v);
   void (APIENTRY *glEvalCoord2dv) (const GLdouble *u);
   void (APIENTRY *glEvalCoord2f) (GLfloat u, GLfloat v);
   void (APIENTRY *glEvalCoord2fv) (const GLfloat *u);
   void (APIENTRY *glEvalMesh1) (GLenum mode, GLint i1, GLint i2);
   void (APIENTRY *glEvalMesh2) (GLenum mode, GLint i1, GLint i2, GLint j1, GLint j2);
   void (APIENTRY *glEvalPoint1) (GLint i);
   void (APIENTRY *glEvalPoint2) (GLint i, GLint j);
   void (APIENTRY *glFeedbackBuffer) (GLsizei size, GLenum type, GLfloat *buffer);
   void (APIENTRY *glFinish) (void);
   void (APIENTRY *glFlush) (void);
   void (APIENTRY *glFogf) (GLenum pname, GLfloat param);
   void (APIENTRY *glFogfv) (GLenum pname, const GLfloat *params);
   void (APIENTRY *glFogi) (GLenum pname, GLint param);
   void (APIENTRY *glFogiv) (GLenum pname, const GLint *params);
   void (APIENTRY *glFrontFace) (GLenum mode);
   void (APIENTRY *glFrustum) (GLdouble left, GLdouble right, GLdouble bottom, GLdouble top, GLdouble zNear, GLdouble zFar);
   GLuint (APIENTRY *glGenLists) (GLsizei range);
   void (APIENTRY *glGenTextures) (GLsizei n, GLuint *textures);
   void (APIENTRY *glGetBooleanv) (GLenum pname, GLboolean *params);
   void (APIENTRY *glGetClipPlane) (GLenum plane, GLdouble *equation);
   void (APIENTRY *glGetDoublev) (GLenum pname, GLdouble *params);
   GLenum (APIENTRY *glGetError) (void);
   void (APIENTRY *glGetFloatv) (GLenum pname, GLfloat *params);
   void (APIENTRY *glGetIntegerv) (GLenum pname, GLint *params);
   void (APIENTRY *glGetLightfv) (GLenum light, GLenum pname, GLfloat *params);
   void (APIENTRY *glGetLightiv) (GLenum light, GLenum pname, GLint *params);
   void (APIENTRY *glGetMapdv) (GLenum target, GLenum query, GLdouble *v);
   void (APIENTRY *glGetMapfv) (GLenum target, GLenum query, GLfloat *v);
   void (APIENTRY *glGetMapiv) (GLenum target, GLenum query, GLint *v);
   void (APIENTRY *glGetMaterialfv) (GLenum face, GLenum pname, GLfloat *params);
   void (APIENTRY *glGetMaterialiv) (GLenum face, GLenum pname, GLint *params);
   void (APIENTRY *glGetPixelMapfv) (GLenum map, GLfloat *values);
   void (APIENTRY *glGetPixelMapuiv) (GLenum map, GLuint *values);
   void (APIENTRY *glGetPixelMapusv) (GLenum map, GLushort *values);
   void (APIENTRY *glGetPointerv) (GLenum pname, GLvoid* *params);
   void (APIENTRY *glGetPolygonStipple) (GLubyte *mask);
   const GLstring (APIENTRY *glGetString) (GLenum name);
   void (APIENTRY *glGetTexEnvfv) (GLenum target, GLenum pname, GLfloat *params);
   void (APIENTRY *glGetTexEnviv) (GLenum target, GLenum pname, GLint *params);
   void (APIENTRY *glGetTexGendv) (GLenum coord, GLenum pname, GLdouble *params);
   void (APIENTRY *glGetTexGenfv) (GLenum coord, GLenum pname, GLfloat *params);
   void (APIENTRY *glGetTexGeniv) (GLenum coord, GLenum pname, GLint *params);
   void (APIENTRY *glGetTexImage) (GLenum target, GLint level, GLenum format, GLenum type, GLvoid *pixels);
   void (APIENTRY *glGetTexLevelParameterfv) (GLenum target, GLint level, GLenum pname, GLfloat *params);
   void (APIENTRY *glGetTexLevelParameteriv) (GLenum target, GLint level, GLenum pname, GLint *params);
   void (APIENTRY *glGetTexParameterfv) (GLenum target, GLenum pname, GLfloat *params);
   void (APIENTRY *glGetTexParameteriv) (GLenum target, GLenum pname, GLint *params);
   void (APIENTRY *glHint) (GLenum target, GLenum mode);
   void (APIENTRY *glIndexMask) (GLuint mask);
   void (APIENTRY *glIndexPointer) (GLenum type, GLsizei stride, const GLvoid *pointer);
   void (APIENTRY *glIndexd) (GLdouble c);
   void (APIENTRY *glIndexdv) (const GLdouble *c);
   void (APIENTRY *glIndexf) (GLfloat c);
   void (APIENTRY *glIndexfv) (const GLfloat *c);
   void (APIENTRY *glIndexi) (GLint c);
   void (APIENTRY *glIndexiv) (const GLint *c);
   void (APIENTRY *glIndexs) (GLshort c);
   void (APIENTRY *glIndexsv) (const GLshort *c);
   void (APIENTRY *glIndexub) (GLubyte c);
   void (APIENTRY *glIndexubv) (const GLubyte *c);
   void (APIENTRY *glInitNames) (void);
   void (APIENTRY *glInterleavedArrays) (GLenum format, GLsizei stride, const GLvoid *pointer);
   GLboolean (APIENTRY *glIsEnabled) (GLenum cap);
   GLboolean (APIENTRY *glIsList) (GLuint list);
   GLboolean (APIENTRY *glIsTexture) (GLuint texture);
   void (APIENTRY *glLightModelf) (GLenum pname, GLfloat param);
   void (APIENTRY *glLightModelfv) (GLenum pname, const GLfloat *params);
   void (APIENTRY *glLightModeli) (GLenum pname, GLint param);
   void (APIENTRY *glLightModeliv) (GLenum pname, const GLint *params);
   void (APIENTRY *glLightf) (GLenum light, GLenum pname, GLfloat param);
   void (APIENTRY *glLightfv) (GLenum light, GLenum pname, const GLfloat *params);
   void (APIENTRY *glLighti) (GLenum light, GLenum pname, GLint param);
   void (APIENTRY *glLightiv) (GLenum light, GLenum pname, const GLint *params);
   void (APIENTRY *glLineStipple) (GLint factor, GLushort pattern);
   void (APIENTRY *glLineWidth) (GLfloat width);
   void (APIENTRY *glListBase) (GLuint base);
   void (APIENTRY *glLoadIdentity) (void);
   void (APIENTRY *glLoadMatrixd) (const GLdouble *m);
   void (APIENTRY *glLoadMatrixf) (const GLfloat *m);
   void (APIENTRY *glLoadName) (GLuint name);
   void (APIENTRY *glLogicOp) (GLenum opcode);
   void (APIENTRY *glMap1d) (GLenum target, GLdouble u1, GLdouble u2, GLint stride, GLint order, const GLdouble *points);
   void (APIENTRY *glMap1f) (GLenum target, GLfloat u1, GLfloat u2, GLint stride, GLint order, const GLfloat *points);
   void (APIENTRY *glMap2d) (GLenum target, GLdouble u1, GLdouble u2, GLint ustride, GLint uorder, GLdouble v1, GLdouble v2, GLint vstride, GLint vorder, const GLdouble *points);
   void (APIENTRY *glMap2f) (GLenum target, GLfloat u1, GLfloat u2, GLint ustride, GLint uorder, GLfloat v1, GLfloat v2, GLint vstride, GLint vorder, const GLfloat *points);
   void (APIENTRY *glMapGrid1d) (GLint un, GLdouble u1, GLdouble u2);
   void (APIENTRY *glMapGrid1f) (GLint un, GLfloat u1, GLfloat u2);
   void (APIENTRY *glMapGrid2d) (GLint un, GLdouble u1, GLdouble u2, GLint vn, GLdouble v1, GLdouble v2);
   void (APIENTRY *glMapGrid2f) (GLint un, GLfloat u1, GLfloat u2, GLint vn, GLfloat v1, GLfloat v2);
   void (APIENTRY *glMaterialf) (GLenum face, GLenum pname, GLfloat param);
   void (APIENTRY *glMaterialfv) (GLenum face, GLenum pname, const GLfloat *params);
   void (APIENTRY *glMateriali) (GLenum face, GLenum pname, GLint param);
   void (APIENTRY *glMaterialiv) (GLenum face, GLenum pname, const GLint *params);
   void (APIENTRY *glMatrixMode) (GLenum mode);
   void (APIENTRY *glMultMatrixd) (const GLdouble *m);
   void (APIENTRY *glMultMatrixf) (const GLfloat *m);
   void (APIENTRY *glNewList) (GLuint list, GLenum mode);
   void (APIENTRY *glNormal3b) (GLbyte nx, GLbyte ny, GLbyte nz);
   void (APIENTRY *glNormal3bv) (const GLbyte *v);
   void (APIENTRY *glNormal3d) (GLdouble nx, GLdouble ny, GLdouble nz);
   void (APIENTRY *glNormal3dv) (const GLdouble *v);
   void (APIENTRY *glNormal3f) (GLfloat nx, GLfloat ny, GLfloat nz);
   void (APIENTRY *glNormal3fv) (const GLfloat *v);
   void (APIENTRY *glNormal3i) (GLint nx, GLint ny, GLint nz);
   void (APIENTRY *glNormal3iv) (const GLint *v);
   void (APIENTRY *glNormal3s) (GLshort nx, GLshort ny, GLshort nz);
   void (APIENTRY *glNormal3sv) (const GLshort *v);
   void (APIENTRY *glNormalPointer) (GLenum type, GLsizei stride, const GLvoid *pointer);
   void (APIENTRY *glOrtho) (GLdouble left, GLdouble right, GLdouble bottom, GLdouble top, GLdouble zNear, GLdouble zFar);
   void (APIENTRY *glPassThrough) (GLfloat token);
   void (APIENTRY *glPixelMapfv) (GLenum map, GLsizei mapsize, const GLfloat *values);
   void (APIENTRY *glPixelMapuiv) (GLenum map, GLsizei mapsize, const GLuint *values);
   void (APIENTRY *glPixelMapusv) (GLenum map, GLsizei mapsize, const GLushort *values);
   void (APIENTRY *glPixelStoref) (GLenum pname, GLfloat param);
   void (APIENTRY *glPixelStorei) (GLenum pname, GLint param);
   void (APIENTRY *glPixelTransferf) (GLenum pname, GLfloat param);
   void (APIENTRY *glPixelTransferi) (GLenum pname, GLint param);
   void (APIENTRY *glPixelZoom) (GLfloat xfactor, GLfloat yfactor);
   void (APIENTRY *glPointSize) (GLfloat size);
   void (APIENTRY *glPolygonMode) (GLenum face, GLenum mode);
   void (APIENTRY *glPolygonOffset) (GLfloat factor, GLfloat units);
   void (APIENTRY *glPolygonStipple) (const GLubyte *mask);
   void (APIENTRY *glPopAttrib) (void);
   void (APIENTRY *glPopClientAttrib) (void);
   void (APIENTRY *glPopMatrix) (void);
   void (APIENTRY *glPopName) (void);
   void (APIENTRY *glPrioritizeTextures) (GLsizei n, const GLuint *textures, const GLclampf *priorities);
   void (APIENTRY *glPushAttrib) (GLbitfield mask);
   void (APIENTRY *glPushClientAttrib) (GLbitfield mask);
   void (APIENTRY *glPushMatrix) (void);
   void (APIENTRY *glPushName) (GLuint name);
   void (APIENTRY *glRasterPos2d) (GLdouble x, GLdouble y);
   void (APIENTRY *glRasterPos2dv) (const GLdouble *v);
   void (APIENTRY *glRasterPos2f) (GLfloat x, GLfloat y);
   void (APIENTRY *glRasterPos2fv) (const GLfloat *v);
   void (APIENTRY *glRasterPos2i) (GLint x, GLint y);
   void (APIENTRY *glRasterPos2iv) (const GLint *v);
   void (APIENTRY *glRasterPos2s) (GLshort x, GLshort y);
   void (APIENTRY *glRasterPos2sv) (const GLshort *v);
   void (APIENTRY *glRasterPos3d) (GLdouble x, GLdouble y, GLdouble z);
   void (APIENTRY *glRasterPos3dv) (const GLdouble *v);
   void (APIENTRY *glRasterPos3f) (GLfloat x, GLfloat y, GLfloat z);
   void (APIENTRY *glRasterPos3fv) (const GLfloat *v);
   void (APIENTRY *glRasterPos3i) (GLint x, GLint y, GLint z);
   void (APIENTRY *glRasterPos3iv) (const GLint *v);
   void (APIENTRY *glRasterPos3s) (GLshort x, GLshort y, GLshort z);
   void (APIENTRY *glRasterPos3sv) (const GLshort *v);
   void (APIENTRY *glRasterPos4d) (GLdouble x, GLdouble y, GLdouble z, GLdouble w);
   void (APIENTRY *glRasterPos4dv) (const GLdouble *v);
   void (APIENTRY *glRasterPos4f) (GLfloat x, GLfloat y, GLfloat z, GLfloat w);
   void (APIENTRY *glRasterPos4fv) (const GLfloat *v);
   void (APIENTRY *glRasterPos4i) (GLint x, GLint y, GLint z, GLint w);
   void (APIENTRY *glRasterPos4iv) (const GLint *v);
   void (APIENTRY *glRasterPos4s) (GLshort x, GLshort y, GLshort z, GLshort w);
   void (APIENTRY *glRasterPos4sv) (const GLshort *v);
   void (APIENTRY *glReadBuffer) (GLenum mode);
   void (APIENTRY *glReadPixels) (GLint x, GLint y, GLsizei width, GLsizei height, GLenum format, GLenum type, GLvoid *pixels);
   void (APIENTRY *glRectd) (GLdouble x1, GLdouble y1, GLdouble x2, GLdouble y2);
   void (APIENTRY *glRectdv) (const GLdouble *v1, const GLdouble *v2);
   void (APIENTRY *glRectf) (GLfloat x1, GLfloat y1, GLfloat x2, GLfloat y2);
   void (APIENTRY *glRectfv) (const GLfloat *v1, const GLfloat *v2);
   void (APIENTRY *glRecti) (GLint x1, GLint y1, GLint x2, GLint y2);
   void (APIENTRY *glRectiv) (const GLint *v1, const GLint *v2);
   void (APIENTRY *glRects) (GLshort x1, GLshort y1, GLshort x2, GLshort y2);
   void (APIENTRY *glRectsv) (const GLshort *v1, const GLshort *v2);
   GLint (APIENTRY *glRenderMode) (GLenum mode);
   void (APIENTRY *glRotated) (GLdouble angle, GLdouble x, GLdouble y, GLdouble z);
   void (APIENTRY *glRotatef) (GLfloat angle, GLfloat x, GLfloat y, GLfloat z);
   void (APIENTRY *glScaled) (GLdouble x, GLdouble y, GLdouble z);
   void (APIENTRY *glScalef) (GLfloat x, GLfloat y, GLfloat z);
   void (APIENTRY *glScissor) (GLint x, GLint y, GLsizei width, GLsizei height);
   void (APIENTRY *glSelectBuffer) (GLsizei size, GLuint *buffer);
   void (APIENTRY *glShadeModel) (GLenum mode);
   void (APIENTRY *glStencilFunc) (GLenum func, GLint ref, GLuint mask);
   void (APIENTRY *glStencilMask) (GLuint mask);
   void (APIENTRY *glStencilOp) (GLenum fail, GLenum zfail, GLenum zpass);
   void (APIENTRY *glTexCoord1d) (GLdouble s);
   void (APIENTRY *glTexCoord1dv) (const GLdouble *v);
   void (APIENTRY *glTexCoord1f) (GLfloat s);
   void (APIENTRY *glTexCoord1fv) (const GLfloat *v);
   void (APIENTRY *glTexCoord1i) (GLint s);
   void (APIENTRY *glTexCoord1iv) (const GLint *v);
   void (APIENTRY *glTexCoord1s) (GLshort s);
   void (APIENTRY *glTexCoord1sv) (const GLshort *v);
   void (APIENTRY *glTexCoord2d) (GLdouble s, GLdouble t);
   void (APIENTRY *glTexCoord2dv) (const GLdouble *v);
   void (APIENTRY *glTexCoord2f) (GLfloat s, GLfloat t);
   void (APIENTRY *glTexCoord2fv) (const GLfloat *v);
   void (APIENTRY *glTexCoord2i) (GLint s, GLint t);
   void (APIENTRY *glTexCoord2iv) (const GLint *v);
   void (APIENTRY *glTexCoord2s) (GLshort s, GLshort t);
   void (APIENTRY *glTexCoord2sv) (const GLshort *v);
   void (APIENTRY *glTexCoord3d) (GLdouble s, GLdouble t, GLdouble r);
   void (APIENTRY *glTexCoord3dv) (const GLdouble *v);
   void (APIENTRY *glTexCoord3f) (GLfloat s, GLfloat t, GLfloat r);
   void (APIENTRY *glTexCoord3fv) (const GLfloat *v);
   void (APIENTRY *glTexCoord3i) (GLint s, GLint t, GLint r);
   void (APIENTRY *glTexCoord3iv) (const GLint *v);
   void (APIENTRY *glTexCoord3s) (GLshort s, GLshort t, GLshort r);
   void (APIENTRY *glTexCoord3sv) (const GLshort *v);
   void (APIENTRY *glTexCoord4d) (GLdouble s, GLdouble t, GLdouble r, GLdouble q);
   void (APIENTRY *glTexCoord4dv) (const GLdouble *v);
   void (APIENTRY *glTexCoord4f) (GLfloat s, GLfloat t, GLfloat r, GLfloat q);
   void (APIENTRY *glTexCoord4fv) (const GLfloat *v);
   void (APIENTRY *glTexCoord4i) (GLint s, GLint t, GLint r, GLint q);
   void (APIENTRY *glTexCoord4iv) (const GLint *v);
   void (APIENTRY *glTexCoord4s) (GLshort s, GLshort t, GLshort r, GLshort q);
   void (APIENTRY *glTexCoord4sv) (const GLshort *v);
   void (APIENTRY *glTexCoordPointer) (GLint size, GLenum type, GLsizei stride, const GLvoid *pointer);
   void (APIENTRY *glTexEnvf) (GLenum target, GLenum pname, GLfloat param);
   void (APIENTRY *glTexEnvfv) (GLenum target, GLenum pname, const GLfloat *params);
   void (APIENTRY *glTexEnvi) (GLenum target, GLenum pname, GLint param);
   void (APIENTRY *glTexEnviv) (GLenum target, GLenum pname, const GLint *params);
   void (APIENTRY *glTexGend) (GLenum coord, GLenum pname, GLdouble param);
   void (APIENTRY *glTexGendv) (GLenum coord, GLenum pname, const GLdouble *params);
   void (APIENTRY *glTexGenf) (GLenum coord, GLenum pname, GLfloat param);
   void (APIENTRY *glTexGenfv) (GLenum coord, GLenum pname, const GLfloat *params);
   void (APIENTRY *glTexGeni) (GLenum coord, GLenum pname, GLint param);
   void (APIENTRY *glTexGeniv) (GLenum coord, GLenum pname, const GLint *params);
   void (APIENTRY *glTexImage1D) (GLenum target, GLint level, GLint internalformat, GLsizei width, GLint border, GLenum format, GLenum type, const GLvoid *pixels);
   void (APIENTRY *glTexImage2D) (GLenum target, GLint level, GLint internalformat, GLsizei width, GLsizei height, GLint border, GLenum format, GLenum type, const GLvoid *pixels);
   void (APIENTRY *glTexParameterf) (GLenum target, GLenum pname, GLfloat param);
   void (APIENTRY *glTexParameterfv) (GLenum target, GLenum pname, const GLfloat *params);
   void (APIENTRY *glTexParameteri) (GLenum target, GLenum pname, GLint param);
   void (APIENTRY *glTexParameteriv) (GLenum target, GLenum pname, const GLint *params);
   void (APIENTRY *glTexSubImage1D) (GLenum target, GLint level, GLint xoffset, GLsizei width, GLenum format, GLenum type, const GLvoid *pixels);
   void (APIENTRY *glTexSubImage2D) (GLenum target, GLint level, GLint xoffset, GLint yoffset, GLsizei width, GLsizei height, GLenum format, GLenum type, const GLvoid *pixels);
   void (APIENTRY *glTranslated) (GLdouble x, GLdouble y, GLdouble z);
   void (APIENTRY *glTranslatef) (GLfloat x, GLfloat y, GLfloat z);
   void (APIENTRY *glVertex2d) (GLdouble x, GLdouble y);
   void (APIENTRY *glVertex2dv) (const GLdouble *v);
   void (APIENTRY *glVertex2f) (GLfloat x, GLfloat y);
   void (APIENTRY *glVertex2fv) (const GLfloat *v);
   void (APIENTRY *glVertex2i) (GLint x, GLint y);
   void (APIENTRY *glVertex2iv) (const GLint *v);
   void (APIENTRY *glVertex2s) (GLshort x, GLshort y);
   void (APIENTRY *glVertex2sv) (const GLshort *v);
   void (APIENTRY *glVertex3d) (GLdouble x, GLdouble y, GLdouble z);
   void (APIENTRY *glVertex3dv) (const GLdouble *v);
   void (APIENTRY *glVertex3f) (GLfloat x, GLfloat y, GLfloat z);
   void (APIENTRY *glVertex3fv) (const GLfloat *v);
   void (APIENTRY *glVertex3i) (GLint x, GLint y, GLint z);
   void (APIENTRY *glVertex3iv) (const GLint *v);
   void (APIENTRY *glVertex3s) (GLshort x, GLshort y, GLshort z);
   void (APIENTRY *glVertex3sv) (const GLshort *v);
   void (APIENTRY *glVertex4d) (GLdouble x, GLdouble y, GLdouble z, GLdouble w);
   void (APIENTRY *glVertex4dv) (const GLdouble *v);
   void (APIENTRY *glVertex4f) (GLfloat x, GLfloat y, GLfloat z, GLfloat w);
   void (APIENTRY *glVertex4fv) (const GLfloat *v);
   void (APIENTRY *glVertex4i) (GLint x, GLint y, GLint z, GLint w);
   void (APIENTRY *glVertex4iv) (const GLint *v);
   void (APIENTRY *glVertex4s) (GLshort x, GLshort y, GLshort z, GLshort w);
   void (APIENTRY *glVertex4sv) (const GLshort *v);
   void (APIENTRY *glVertexPointer) (GLint size, GLenum type, GLsizei stride, const GLvoid *pointer);
   void (APIENTRY *glViewport) (GLint x, GLint y, GLsizei width, GLsizei height);
   int   (APIENTRY *wglChoosePixelFormat)       (HDC a, CONST PIXELFORMATDESCRIPTOR *b);
   BOOL  (APIENTRY *wglCopyContext)             (HGLRC a, HGLRC b, UINT c);
   HGLRC (APIENTRY *wglCreateContext)           (HDC a);
   HGLRC (APIENTRY *wglCreateLayerContext)      (HDC a, int b);
   BOOL  (APIENTRY *wglDeleteContext)           (HGLRC a);
   BOOL  (APIENTRY *wglDescribeLayerPlane)      (HDC a, int b, int c, UINT d, LPLAYERPLANEDESCRIPTOR e);
   int   (APIENTRY *wglDescribePixelFormat)     (HDC a, int b, UINT c, LPPIXELFORMATDESCRIPTOR d);
   HGLRC (APIENTRY *wglGetCurrentContext)       (void);
   HDC   (APIENTRY *wglGetCurrentDC)            (void);
   PROC  (APIENTRY *wglGetDefaultProcAddress)   (LPCSTR a);
   int   (APIENTRY *wglGetLayerPaletteEntries)  (HDC a, int b, int c, int d, COLORREF *e);
   int   (APIENTRY *wglGetPixelFormat)          (HDC a);
   PROC  (APIENTRY *wglGetProcAddress)          (LPCSTR a);
   BOOL  (APIENTRY *wglMakeCurrent)             (HDC a, HGLRC b);
   BOOL  (APIENTRY *wglRealizeLayerPalette)     (HDC a, int b, BOOL c);
   int   (APIENTRY *wglSetLayerPaletteEntries)  (HDC a, int b, int c, int d, CONST COLORREF *e);
   BOOL  (APIENTRY *wglSetPixelFormat)          (HDC a, int b, CONST PIXELFORMATDESCRIPTOR *c);
   BOOL  (APIENTRY *wglShareLists)              (HGLRC a, HGLRC b);
   BOOL  (APIENTRY *wglSwapBuffers)             (HDC a);
   BOOL  (APIENTRY *wglSwapLayerBuffers)        (HDC a, UINT b);
   BOOL  (APIENTRY *wglUseFontBitmapsA)         (HDC a, DWORD b, DWORD c, DWORD d);
   BOOL  (APIENTRY *wglUseFontBitmapsW)         (HDC a, DWORD b, DWORD c, DWORD d);
   BOOL  (APIENTRY *wglUseFontOutlinesA)        (HDC a, DWORD b, DWORD c, DWORD d, FLOAT e, FLOAT f, int g, LPGLYPHMETRICSFLOAT h);
   BOOL  (APIENTRY *wglUseFontOutlinesW)        (HDC a, DWORD b, DWORD c, DWORD d, FLOAT e, FLOAT f, int g, LPGLYPHMETRICSFLOAT h);
}
GLV;

//
// OpenGL function handler prototypes
//

extern "C"
{
void APIENTRY glAccum (GLenum op, GLfloat value);
void APIENTRY glAlphaFunc (GLenum func, GLclampf ref);
GLboolean APIENTRY glAreTexturesResident (GLsizei n, const GLuint *textures, GLboolean *residences);
void APIENTRY glArrayElement (GLint i);
void APIENTRY glBegin (GLenum mode);
void APIENTRY glBindTexture (GLenum target, GLuint texture);
void APIENTRY glBitmap (GLsizei width, GLsizei height, GLfloat xorig, GLfloat yorig, GLfloat xmove, GLfloat ymove, const GLubyte *bitmap);
void APIENTRY glBlendFunc (GLenum sfactor, GLenum dfactor);
void APIENTRY glCallList (GLuint list);
void APIENTRY glCallLists (GLsizei n, GLenum type, const GLvoid *lists);
void APIENTRY glClear (GLbitfield mask);
void APIENTRY glClearAccum (GLfloat red, GLfloat green, GLfloat blue, GLfloat alpha);
void APIENTRY glClearColor (GLclampf red, GLclampf green, GLclampf blue, GLclampf alpha);
void APIENTRY glClearDepth (GLclampd depth);
void APIENTRY glClearIndex (GLfloat c);
void APIENTRY glClearStencil (GLint s);
void APIENTRY glClipPlane (GLenum plane, const GLdouble *equation);
void APIENTRY glColor3b (GLbyte red, GLbyte green, GLbyte blue);
void APIENTRY glColor3bv (const GLbyte *v);
void APIENTRY glColor3d (GLdouble red, GLdouble green, GLdouble blue);
void APIENTRY glColor3dv (const GLdouble *v);
void APIENTRY glColor3f (GLfloat red, GLfloat green, GLfloat blue);
void APIENTRY glColor3fv (const GLfloat *v);
void APIENTRY glColor3i (GLint red, GLint green, GLint blue);
void APIENTRY glColor3iv (const GLint *v);
void APIENTRY glColor3s (GLshort red, GLshort green, GLshort blue);
void APIENTRY glColor3sv (const GLshort *v);
void APIENTRY glColor3ub (GLubyte red, GLubyte green, GLubyte blue);
void APIENTRY glColor3ubv (const GLubyte *v);
void APIENTRY glColor3ui (GLuint red, GLuint green, GLuint blue);
void APIENTRY glColor3uiv (const GLuint *v);
void APIENTRY glColor3us (GLushort red, GLushort green, GLushort blue);
void APIENTRY glColor3usv (const GLushort *v);
void APIENTRY glColor4b (GLbyte red, GLbyte green, GLbyte blue, GLbyte alpha);
void APIENTRY glColor4bv (const GLbyte *v);
void APIENTRY glColor4d (GLdouble red, GLdouble green, GLdouble blue, GLdouble alpha);
void APIENTRY glColor4dv (const GLdouble *v);
void APIENTRY glColor4f (GLfloat red, GLfloat green, GLfloat blue, GLfloat alpha);
void APIENTRY glColor4fv (const GLfloat *v);
void APIENTRY glColor4i (GLint red, GLint green, GLint blue, GLint alpha);
void APIENTRY glColor4iv (const GLint *v);
void APIENTRY glColor4s (GLshort red, GLshort green, GLshort blue, GLshort alpha);
void APIENTRY glColor4sv (const GLshort *v);
void APIENTRY glColor4ub (GLubyte red, GLubyte green, GLubyte blue, GLubyte alpha);
void APIENTRY glColor4ubv (const GLubyte *v);
void APIENTRY glColor4ui (GLuint red, GLuint green, GLuint blue, GLuint alpha);
void APIENTRY glColor4uiv (const GLuint *v);
void APIENTRY glColor4us (GLushort red, GLushort green, GLushort blue, GLushort alpha);
void APIENTRY glColor4usv (const GLushort *v);
void APIENTRY glColorMask (GLboolean red, GLboolean green, GLboolean blue, GLboolean alpha);
void APIENTRY glColorMaterial (GLenum face, GLenum mode);
void APIENTRY glColorPointer (GLint size, GLenum type, GLsizei stride, const GLvoid *pointer);
void APIENTRY glCopyPixels (GLint x, GLint y, GLsizei width, GLsizei height, GLenum type);
void APIENTRY glCopyTexImage1D (GLenum target, GLint level, GLenum internalFormat, GLint x, GLint y, GLsizei width, GLint border);
void APIENTRY glCopyTexImage2D (GLenum target, GLint level, GLenum internalFormat, GLint x, GLint y, GLsizei width, GLsizei height, GLint border);
void APIENTRY glCopyTexSubImage1D (GLenum target, GLint level, GLint xoffset, GLint x, GLint y, GLsizei width);
void APIENTRY glCopyTexSubImage2D (GLenum target, GLint level, GLint xoffset, GLint yoffset, GLint x, GLint y, GLsizei width, GLsizei height);
void APIENTRY glCullFace (GLenum mode);
void APIENTRY glDeleteLists (GLuint list, GLsizei range);
void APIENTRY glDeleteTextures (GLsizei n, const GLuint *textures);
void APIENTRY glDepthFunc (GLenum func);
void APIENTRY glDepthMask (GLboolean flag);
void APIENTRY glDepthRange (GLclampd zNear, GLclampd zFar);
void APIENTRY glDisable (GLenum cap);
void APIENTRY glDisableClientState (GLenum array);
void APIENTRY glDrawArrays (GLenum mode, GLint first, GLsizei count);
void APIENTRY glDrawBuffer (GLenum mode);
void APIENTRY glDrawElements (GLenum mode, GLsizei count, GLenum type, const GLvoid *indices);
void APIENTRY glDrawPixels (GLsizei width, GLsizei height, GLenum format, GLenum type, const GLvoid *pixels);
void APIENTRY glEdgeFlag (GLboolean flag);
void APIENTRY glEdgeFlagPointer (GLsizei stride, const GLvoid *pointer);
void APIENTRY glEdgeFlagv (const GLboolean *flag);
void APIENTRY glEnable (GLenum cap);
void APIENTRY glEnableClientState (GLenum array);
void APIENTRY glEnd (void);
void APIENTRY glEndList (void);
void APIENTRY glEvalCoord1d (GLdouble u);
void APIENTRY glEvalCoord1dv (const GLdouble *u);
void APIENTRY glEvalCoord1f (GLfloat u);
void APIENTRY glEvalCoord1fv (const GLfloat *u);
void APIENTRY glEvalCoord2d (GLdouble u, GLdouble v);
void APIENTRY glEvalCoord2dv (const GLdouble *u);
void APIENTRY glEvalCoord2f (GLfloat u, GLfloat v);
void APIENTRY glEvalCoord2fv (const GLfloat *u);
void APIENTRY glEvalMesh1 (GLenum mode, GLint i1, GLint i2);
void APIENTRY glEvalMesh2 (GLenum mode, GLint i1, GLint i2, GLint j1, GLint j2);
void APIENTRY glEvalPoint1 (GLint i);
void APIENTRY glEvalPoint2 (GLint i, GLint j);
void APIENTRY glFeedbackBuffer (GLsizei size, GLenum type, GLfloat *buffer);
void APIENTRY glFinish (void);
void APIENTRY glFlush (void);
void APIENTRY glFogf (GLenum pname, GLfloat param);
void APIENTRY glFogfv (GLenum pname, const GLfloat *params);
void APIENTRY glFogi (GLenum pname, GLint param);
void APIENTRY glFogiv (GLenum pname, const GLint *params);
void APIENTRY glFrontFace (GLenum mode);
void APIENTRY glFrustum (GLdouble left, GLdouble right, GLdouble bottom, GLdouble top, GLdouble zNear, GLdouble zFar);
GLuint APIENTRY glGenLists (GLsizei range);
void APIENTRY glGenTextures (GLsizei n, GLuint *textures);
void APIENTRY glGetBooleanv (GLenum pname, GLboolean *params);
void APIENTRY glGetClipPlane (GLenum plane, GLdouble *equation);
void APIENTRY glGetDoublev (GLenum pname, GLdouble *params);
GLenum APIENTRY glGetError (void);
void APIENTRY glGetFloatv (GLenum pname, GLfloat *params);
void APIENTRY glGetIntegerv (GLenum pname, GLint *params);
void APIENTRY glGetLightfv (GLenum light, GLenum pname, GLfloat *params);
void APIENTRY glGetLightiv (GLenum light, GLenum pname, GLint *params);
void APIENTRY glGetMapdv (GLenum target, GLenum query, GLdouble *v);
void APIENTRY glGetMapfv (GLenum target, GLenum query, GLfloat *v);
void APIENTRY glGetMapiv (GLenum target, GLenum query, GLint *v);
void APIENTRY glGetMaterialfv (GLenum face, GLenum pname, GLfloat *params);
void APIENTRY glGetMaterialiv (GLenum face, GLenum pname, GLint *params);
void APIENTRY glGetPixelMapfv (GLenum map, GLfloat *values);
void APIENTRY glGetPixelMapuiv (GLenum map, GLuint *values);
void APIENTRY glGetPixelMapusv (GLenum map, GLushort *values);
void APIENTRY glGetPointerv (GLenum pname, GLvoid* *params);
void APIENTRY glGetPolygonStipple (GLubyte *mask);
const GLstring APIENTRY glGetString (GLenum name);
void APIENTRY glGetTexEnvfv (GLenum target, GLenum pname, GLfloat *params);
void APIENTRY glGetTexEnviv (GLenum target, GLenum pname, GLint *params);
void APIENTRY glGetTexGendv (GLenum coord, GLenum pname, GLdouble *params);
void APIENTRY glGetTexGenfv (GLenum coord, GLenum pname, GLfloat *params);
void APIENTRY glGetTexGeniv (GLenum coord, GLenum pname, GLint *params);
void APIENTRY glGetTexImage (GLenum target, GLint level, GLenum format, GLenum type, GLvoid *pixels);
void APIENTRY glGetTexLevelParameterfv (GLenum target, GLint level, GLenum pname, GLfloat *params);
void APIENTRY glGetTexLevelParameteriv (GLenum target, GLint level, GLenum pname, GLint *params);
void APIENTRY glGetTexParameterfv (GLenum target, GLenum pname, GLfloat *params);
void APIENTRY glGetTexParameteriv (GLenum target, GLenum pname, GLint *params);
void APIENTRY glHint (GLenum target, GLenum mode);
void APIENTRY glIndexMask (GLuint mask);
void APIENTRY glIndexPointer (GLenum type, GLsizei stride, const GLvoid *pointer);
void APIENTRY glIndexd (GLdouble c);
void APIENTRY glIndexdv (const GLdouble *c);
void APIENTRY glIndexf (GLfloat c);
void APIENTRY glIndexfv (const GLfloat *c);
void APIENTRY glIndexi (GLint c);
void APIENTRY glIndexiv (const GLint *c);
void APIENTRY glIndexs (GLshort c);
void APIENTRY glIndexsv (const GLshort *c);
void APIENTRY glIndexub (GLubyte c);
void APIENTRY glIndexubv (const GLubyte *c);
void APIENTRY glInitNames (void);
void APIENTRY glInterleavedArrays (GLenum format, GLsizei stride, const GLvoid *pointer);
GLboolean APIENTRY glIsEnabled (GLenum cap);
GLboolean APIENTRY glIsList (GLuint list);
GLboolean APIENTRY glIsTexture (GLuint texture);
void APIENTRY glLightModelf (GLenum pname, GLfloat param);
void APIENTRY glLightModelfv (GLenum pname, const GLfloat *params);
void APIENTRY glLightModeli (GLenum pname, GLint param);
void APIENTRY glLightModeliv (GLenum pname, const GLint *params);
void APIENTRY glLightf (GLenum light, GLenum pname, GLfloat param);
void APIENTRY glLightfv (GLenum light, GLenum pname, const GLfloat *params);
void APIENTRY glLighti (GLenum light, GLenum pname, GLint param);
void APIENTRY glLightiv (GLenum light, GLenum pname, const GLint *params);
void APIENTRY glLineStipple (GLint factor, GLushort pattern);
void APIENTRY glLineWidth (GLfloat width);
void APIENTRY glListBase (GLuint base);
void APIENTRY glLoadIdentity (void);
void APIENTRY glLoadMatrixd (const GLdouble *m);
void APIENTRY glLoadMatrixf (const GLfloat *m);
void APIENTRY glLoadName (GLuint name);
void APIENTRY glLogicOp (GLenum opcode);
void APIENTRY glMap1d (GLenum target, GLdouble u1, GLdouble u2, GLint stride, GLint order, const GLdouble *points);
void APIENTRY glMap1f (GLenum target, GLfloat u1, GLfloat u2, GLint stride, GLint order, const GLfloat *points);
void APIENTRY glMap2d (GLenum target, GLdouble u1, GLdouble u2, GLint ustride, GLint uorder, GLdouble v1, GLdouble v2, GLint vstride, GLint vorder, const GLdouble *points);
void APIENTRY glMap2f (GLenum target, GLfloat u1, GLfloat u2, GLint ustride, GLint uorder, GLfloat v1, GLfloat v2, GLint vstride, GLint vorder, const GLfloat *points);
void APIENTRY glMapGrid1d (GLint un, GLdouble u1, GLdouble u2);
void APIENTRY glMapGrid1f (GLint un, GLfloat u1, GLfloat u2);
void APIENTRY glMapGrid2d (GLint un, GLdouble u1, GLdouble u2, GLint vn, GLdouble v1, GLdouble v2);
void APIENTRY glMapGrid2f (GLint un, GLfloat u1, GLfloat u2, GLint vn, GLfloat v1, GLfloat v2);
void APIENTRY glMaterialf (GLenum face, GLenum pname, GLfloat param);
void APIENTRY glMaterialfv (GLenum face, GLenum pname, const GLfloat *params);
void APIENTRY glMateriali (GLenum face, GLenum pname, GLint param);
void APIENTRY glMaterialiv (GLenum face, GLenum pname, const GLint *params);
void APIENTRY glMatrixMode (GLenum mode);
void APIENTRY glMultMatrixd (const GLdouble *m);
void APIENTRY glMultMatrixf (const GLfloat *m);
void APIENTRY glNewList (GLuint list, GLenum mode);
void APIENTRY glNormal3b (GLbyte nx, GLbyte ny, GLbyte nz);
void APIENTRY glNormal3bv (const GLbyte *v);
void APIENTRY glNormal3d (GLdouble nx, GLdouble ny, GLdouble nz);
void APIENTRY glNormal3dv (const GLdouble *v);
void APIENTRY glNormal3f (GLfloat nx, GLfloat ny, GLfloat nz);
void APIENTRY glNormal3fv (const GLfloat *v);
void APIENTRY glNormal3i (GLint nx, GLint ny, GLint nz);
void APIENTRY glNormal3iv (const GLint *v);
void APIENTRY glNormal3s (GLshort nx, GLshort ny, GLshort nz);
void APIENTRY glNormal3sv (const GLshort *v);
void APIENTRY glNormalPointer (GLenum type, GLsizei stride, const GLvoid *pointer);
void APIENTRY glOrtho (GLdouble left, GLdouble right, GLdouble bottom, GLdouble top, GLdouble zNear, GLdouble zFar);
void APIENTRY glPassThrough (GLfloat token);
void APIENTRY glPixelMapfv (GLenum map, GLsizei mapsize, const GLfloat *values);
void APIENTRY glPixelMapuiv (GLenum map, GLsizei mapsize, const GLuint *values);
void APIENTRY glPixelMapusv (GLenum map, GLsizei mapsize, const GLushort *values);
void APIENTRY glPixelStoref (GLenum pname, GLfloat param);
void APIENTRY glPixelStorei (GLenum pname, GLint param);
void APIENTRY glPixelTransferf (GLenum pname, GLfloat param);
void APIENTRY glPixelTransferi (GLenum pname, GLint param);
void APIENTRY glPixelZoom (GLfloat xfactor, GLfloat yfactor);
void APIENTRY glPointSize (GLfloat size);
void APIENTRY glPolygonMode (GLenum face, GLenum mode);
void APIENTRY glPolygonOffset (GLfloat factor, GLfloat units);
void APIENTRY glPolygonStipple (const GLubyte *mask);
void APIENTRY glPopAttrib (void);
void APIENTRY glPopClientAttrib (void);
void APIENTRY glPopMatrix (void);
void APIENTRY glPopName (void);
void APIENTRY glPrioritizeTextures (GLsizei n, const GLuint *textures, const GLclampf *priorities);
void APIENTRY glPushAttrib (GLbitfield mask);
void APIENTRY glPushClientAttrib (GLbitfield mask);
void APIENTRY glPushMatrix (void);
void APIENTRY glPushName (GLuint name);
void APIENTRY glRasterPos2d (GLdouble x, GLdouble y);
void APIENTRY glRasterPos2dv (const GLdouble *v);
void APIENTRY glRasterPos2f (GLfloat x, GLfloat y);
void APIENTRY glRasterPos2fv (const GLfloat *v);
void APIENTRY glRasterPos2i (GLint x, GLint y);
void APIENTRY glRasterPos2iv (const GLint *v);
void APIENTRY glRasterPos2s (GLshort x, GLshort y);
void APIENTRY glRasterPos2sv (const GLshort *v);
void APIENTRY glRasterPos3d (GLdouble x, GLdouble y, GLdouble z);
void APIENTRY glRasterPos3dv (const GLdouble *v);
void APIENTRY glRasterPos3f (GLfloat x, GLfloat y, GLfloat z);
void APIENTRY glRasterPos3fv (const GLfloat *v);
void APIENTRY glRasterPos3i (GLint x, GLint y, GLint z);
void APIENTRY glRasterPos3iv (const GLint *v);
void APIENTRY glRasterPos3s (GLshort x, GLshort y, GLshort z);
void APIENTRY glRasterPos3sv (const GLshort *v);
void APIENTRY glRasterPos4d (GLdouble x, GLdouble y, GLdouble z, GLdouble w);
void APIENTRY glRasterPos4dv (const GLdouble *v);
void APIENTRY glRasterPos4f (GLfloat x, GLfloat y, GLfloat z, GLfloat w);
void APIENTRY glRasterPos4fv (const GLfloat *v);
void APIENTRY glRasterPos4i (GLint x, GLint y, GLint z, GLint w);
void APIENTRY glRasterPos4iv (const GLint *v);
void APIENTRY glRasterPos4s (GLshort x, GLshort y, GLshort z, GLshort w);
void APIENTRY glRasterPos4sv (const GLshort *v);
void APIENTRY glReadBuffer (GLenum mode);
void APIENTRY glReadPixels (GLint x, GLint y, GLsizei width, GLsizei height, GLenum format, GLenum type, GLvoid *pixels);
void APIENTRY glRectd (GLdouble x1, GLdouble y1, GLdouble x2, GLdouble y2);
void APIENTRY glRectdv (const GLdouble *v1, const GLdouble *v2);
void APIENTRY glRectf (GLfloat x1, GLfloat y1, GLfloat x2, GLfloat y2);
void APIENTRY glRectfv (const GLfloat *v1, const GLfloat *v2);
void APIENTRY glRecti (GLint x1, GLint y1, GLint x2, GLint y2);
void APIENTRY glRectiv (const GLint *v1, const GLint *v2);
void APIENTRY glRects (GLshort x1, GLshort y1, GLshort x2, GLshort y2);
void APIENTRY glRectsv (const GLshort *v1, const GLshort *v2);
GLint APIENTRY glRenderMode (GLenum mode);
void APIENTRY glRotated (GLdouble angle, GLdouble x, GLdouble y, GLdouble z);
void APIENTRY glRotatef (GLfloat angle, GLfloat x, GLfloat y, GLfloat z);
void APIENTRY glScaled (GLdouble x, GLdouble y, GLdouble z);
void APIENTRY glScalef (GLfloat x, GLfloat y, GLfloat z);
void APIENTRY glScissor (GLint x, GLint y, GLsizei width, GLsizei height);
void APIENTRY glSelectBuffer (GLsizei size, GLuint *buffer);
void APIENTRY glShadeModel (GLenum mode);
void APIENTRY glStencilFunc (GLenum func, GLint ref, GLuint mask);
void APIENTRY glStencilMask (GLuint mask);
void APIENTRY glStencilOp (GLenum fail, GLenum zfail, GLenum zpass);
void APIENTRY glTexCoord1d (GLdouble s);
void APIENTRY glTexCoord1dv (const GLdouble *v);
void APIENTRY glTexCoord1f (GLfloat s);
void APIENTRY glTexCoord1fv (const GLfloat *v);
void APIENTRY glTexCoord1i (GLint s);
void APIENTRY glTexCoord1iv (const GLint *v);
void APIENTRY glTexCoord1s (GLshort s);
void APIENTRY glTexCoord1sv (const GLshort *v);
void APIENTRY glTexCoord2d (GLdouble s, GLdouble t);
void APIENTRY glTexCoord2dv (const GLdouble *v);
void APIENTRY glTexCoord2f (GLfloat s, GLfloat t);
void APIENTRY glTexCoord2fv (const GLfloat *v);
void APIENTRY glTexCoord2i (GLint s, GLint t);
void APIENTRY glTexCoord2iv (const GLint *v);
void APIENTRY glTexCoord2s (GLshort s, GLshort t);
void APIENTRY glTexCoord2sv (const GLshort *v);
void APIENTRY glTexCoord3d (GLdouble s, GLdouble t, GLdouble r);
void APIENTRY glTexCoord3dv (const GLdouble *v);
void APIENTRY glTexCoord3f (GLfloat s, GLfloat t, GLfloat r);
void APIENTRY glTexCoord3fv (const GLfloat *v);
void APIENTRY glTexCoord3i (GLint s, GLint t, GLint r);
void APIENTRY glTexCoord3iv (const GLint *v);
void APIENTRY glTexCoord3s (GLshort s, GLshort t, GLshort r);
void APIENTRY glTexCoord3sv (const GLshort *v);
void APIENTRY glTexCoord4d (GLdouble s, GLdouble t, GLdouble r, GLdouble q);
void APIENTRY glTexCoord4dv (const GLdouble *v);
void APIENTRY glTexCoord4f (GLfloat s, GLfloat t, GLfloat r, GLfloat q);
void APIENTRY glTexCoord4fv (const GLfloat *v);
void APIENTRY glTexCoord4i (GLint s, GLint t, GLint r, GLint q);
void APIENTRY glTexCoord4iv (const GLint *v);
void APIENTRY glTexCoord4s (GLshort s, GLshort t, GLshort r, GLshort q);
void APIENTRY glTexCoord4sv (const GLshort *v);
void APIENTRY glTexCoordPointer (GLint size, GLenum type, GLsizei stride, const GLvoid *pointer);
void APIENTRY glTexEnvf (GLenum target, GLenum pname, GLfloat param);
void APIENTRY glTexEnvfv (GLenum target, GLenum pname, const GLfloat *params);
void APIENTRY glTexEnvi (GLenum target, GLenum pname, GLint param);
void APIENTRY glTexEnviv (GLenum target, GLenum pname, const GLint *params);
void APIENTRY glTexGend (GLenum coord, GLenum pname, GLdouble param);
void APIENTRY glTexGendv (GLenum coord, GLenum pname, const GLdouble *params);
void APIENTRY glTexGenf (GLenum coord, GLenum pname, GLfloat param);
void APIENTRY glTexGenfv (GLenum coord, GLenum pname, const GLfloat *params);
void APIENTRY glTexGeni (GLenum coord, GLenum pname, GLint param);
void APIENTRY glTexGeniv (GLenum coord, GLenum pname, const GLint *params);
void APIENTRY glTexImage1D (GLenum target, GLint level, GLint internalformat, GLsizei width, GLint border, GLenum format, GLenum type, const GLvoid *pixels);
void APIENTRY glTexImage2D (GLenum target, GLint level, GLint internalformat, GLsizei width, GLsizei height, GLint border, GLenum format, GLenum type, const GLvoid *pixels);
void APIENTRY glTexParameterf (GLenum target, GLenum pname, GLfloat param);
void APIENTRY glTexParameterfv (GLenum target, GLenum pname, const GLfloat *params);
void APIENTRY glTexParameteri (GLenum target, GLenum pname, GLint param);
void APIENTRY glTexParameteriv (GLenum target, GLenum pname, const GLint *params);
void APIENTRY glTexSubImage1D (GLenum target, GLint level, GLint xoffset, GLsizei width, GLenum format, GLenum type, const GLvoid *pixels);
void APIENTRY glTexSubImage2D (GLenum target, GLint level, GLint xoffset, GLint yoffset, GLsizei width, GLsizei height, GLenum format, GLenum type, const GLvoid *pixels);
void APIENTRY glTranslated (GLdouble x, GLdouble y, GLdouble z);
void APIENTRY glTranslatef (GLfloat x, GLfloat y, GLfloat z);
void APIENTRY glVertex2d (GLdouble x, GLdouble y);
void APIENTRY glVertex2dv (const GLdouble *v);
void APIENTRY glVertex2f (GLfloat x, GLfloat y);
void APIENTRY glVertex2fv (const GLfloat *v);
void APIENTRY glVertex2i (GLint x, GLint y);
void APIENTRY glVertex2iv (const GLint *v);
void APIENTRY glVertex2s (GLshort x, GLshort y);
void APIENTRY glVertex2sv (const GLshort *v);
void APIENTRY glVertex3d (GLdouble x, GLdouble y, GLdouble z);
void APIENTRY glVertex3dv (const GLdouble *v);
void APIENTRY glVertex3f (GLfloat x, GLfloat y, GLfloat z);
void APIENTRY glVertex3fv (const GLfloat *v);
void APIENTRY glVertex3i (GLint x, GLint y, GLint z);
void APIENTRY glVertex3iv (const GLint *v);
void APIENTRY glVertex3s (GLshort x, GLshort y, GLshort z);
void APIENTRY glVertex3sv (const GLshort *v);
void APIENTRY glVertex4d (GLdouble x, GLdouble y, GLdouble z, GLdouble w);
void APIENTRY glVertex4dv (const GLdouble *v);
void APIENTRY glVertex4f (GLfloat x, GLfloat y, GLfloat z, GLfloat w);
void APIENTRY glVertex4fv (const GLfloat *v);
void APIENTRY glVertex4i (GLint x, GLint y, GLint z, GLint w);
void APIENTRY glVertex4iv (const GLint *v);
void APIENTRY glVertex4s (GLshort x, GLshort y, GLshort z, GLshort w);
void APIENTRY glVertex4sv (const GLshort *v);
void APIENTRY glVertexPointer (GLint size, GLenum type, GLsizei stride, const GLvoid *pointer);
void APIENTRY glViewport (GLint x, GLint y, GLsizei width, GLsizei height);
int   APIENTRY wglChoosePixelFormat       (HDC a, CONST PIXELFORMATDESCRIPTOR *b);
BOOL  APIENTRY wglCopyContext             (HGLRC a, HGLRC b, UINT c);
HGLRC APIENTRY wglCreateContext           (HDC a);
HGLRC APIENTRY wglCreateLayerContext      (HDC a, int b);
BOOL  APIENTRY wglDeleteContext           (HGLRC a);
BOOL  APIENTRY wglDescribeLayerPlane      (HDC a, int b, int c, UINT d, LPLAYERPLANEDESCRIPTOR e);
int   APIENTRY wglDescribePixelFormat     (HDC a, int b, UINT c, LPPIXELFORMATDESCRIPTOR d);
HGLRC APIENTRY wglGetCurrentContext       (void);
HDC   APIENTRY wglGetCurrentDC            (void);
PROC  APIENTRY wglGetDefaultProcAddress   (LPCSTR a);
int   APIENTRY wglGetLayerPaletteEntries  (HDC a, int b, int c, int d, COLORREF *e);
int   APIENTRY wglGetPixelFormat          (HDC a);
PROC  APIENTRY wglGetProcAddress          (LPCSTR a);
BOOL  APIENTRY wglMakeCurrent             (HDC a, HGLRC b);
BOOL  APIENTRY wglRealizeLayerPalette     (HDC a, int b, BOOL c);
int   APIENTRY wglSetLayerPaletteEntries  (HDC a, int b, int c, int d, CONST COLORREF *e);
BOOL  APIENTRY wglSetPixelFormat          (HDC a, int b, CONST PIXELFORMATDESCRIPTOR *c);
BOOL  APIENTRY wglShareLists              (HGLRC a, HGLRC b);
BOOL  APIENTRY wglSwapBuffers             (HDC a);
BOOL  APIENTRY wglSwapLayerBuffers        (HDC a, UINT b);
BOOL  APIENTRY wglUseFontBitmapsA         (HDC a, DWORD b, DWORD c, DWORD d);
BOOL  APIENTRY wglUseFontBitmapsW         (HDC a, DWORD b, DWORD c, DWORD d);
BOOL  APIENTRY wglUseFontOutlinesA        (HDC a, DWORD b, DWORD c, DWORD d, FLOAT e, FLOAT f, int g, LPGLYPHMETRICSFLOAT h);
BOOL  APIENTRY wglUseFontOutlinesW        (HDC a, DWORD b, DWORD c, DWORD d, FLOAT e, FLOAT f, int g, LPGLYPHMETRICSFLOAT h);
}

//
// GLenum values from Win32 GL.H
//

struct GLenum_pair
{
   char *name;
   int   tag;
};

GLenum_pair GLenum_list[] =
{
{ "GL_POINTS",                         0x0000},
{ "GL_LINES",                          0x0001},
{ "GL_LINE_LOOP",                      0x0002},
{ "GL_LINE_STRIP",                     0x0003},
{ "GL_TRIANGLES",                      0x0004},
{ "GL_TRIANGLE_STRIP",                 0x0005},
{ "GL_TRIANGLE_FAN",                   0x0006},
{ "GL_QUADS",                          0x0007},
{ "GL_QUAD_STRIP",                     0x0008},
{ "GL_POLYGON",                        0x0009},
{ "GL_ACCUM",                          0x0100},
{ "GL_LOAD",                           0x0101},
{ "GL_RETURN",                         0x0102},
{ "GL_MULT",                           0x0103},
{ "GL_ADD",                            0x0104},
{ "GL_NEVER",                          0x0200},
{ "GL_LESS",                           0x0201},
{ "GL_EQUAL",                          0x0202},
{ "GL_LEQUAL",                         0x0203},
{ "GL_GREATER",                        0x0204},
{ "GL_NOTEQUAL",                       0x0205},
{ "GL_GEQUAL",                         0x0206},
{ "GL_ALWAYS",                         0x0207},
{ "GL_SRC_COLOR",                      0x0300},
{ "GL_ONE_MINUS_SRC_COLOR",            0x0301},
{ "GL_SRC_ALPHA",                      0x0302},
{ "GL_ONE_MINUS_SRC_ALPHA",            0x0303},
{ "GL_DST_ALPHA",                      0x0304},
{ "GL_ONE_MINUS_DST_ALPHA",            0x0305},
{ "GL_DST_COLOR",                      0x0306},
{ "GL_ONE_MINUS_DST_COLOR",            0x0307},
{ "GL_SRC_ALPHA_SATURATE",             0x0308},
{ "GL_FRONT_LEFT",                     0x0400},
{ "GL_FRONT_RIGHT",                    0x0401},
{ "GL_BACK_LEFT",                      0x0402},
{ "GL_BACK_RIGHT",                     0x0403},
{ "GL_FRONT",                          0x0404},
{ "GL_BACK",                           0x0405},
{ "GL_LEFT",                           0x0406},
{ "GL_RIGHT",                          0x0407},
{ "GL_FRONT_AND_BACK",                 0x0408},
{ "GL_AUX0",                           0x0409},
{ "GL_AUX1",                           0x040A},
{ "GL_AUX2",                           0x040B},
{ "GL_AUX3",                           0x040C},
{ "GL_INVALID_ENUM",                   0x0500},
{ "GL_INVALID_VALUE",                  0x0501},
{ "GL_INVALID_OPERATION",              0x0502},
{ "GL_STACK_OVERFLOW",                 0x0503},
{ "GL_STACK_UNDERFLOW",                0x0504},
{ "GL_OUT_OF_MEMORY",                  0x0505},
{ "GL_2D",                             0x0600},
{ "GL_3D",                             0x0601},
{ "GL_3D_COLOR",                       0x0602},
{ "GL_3D_COLOR_TEXTURE",               0x0603},
{ "GL_4D_COLOR_TEXTURE",               0x0604},
{ "GL_PASS_THROUGH_TOKEN",             0x0700},
{ "GL_POINT_TOKEN",                    0x0701},
{ "GL_LINE_TOKEN",                     0x0702},
{ "GL_POLYGON_TOKEN",                  0x0703},
{ "GL_BITMAP_TOKEN",                   0x0704},
{ "GL_DRAW_PIXEL_TOKEN",               0x0705},
{ "GL_COPY_PIXEL_TOKEN",               0x0706},
{ "GL_LINE_RESET_TOKEN",               0x0707},
{ "GL_EXP",                            0x0800},
{ "GL_EXP2",                           0x0801},
{ "GL_CW",                             0x0900},
{ "GL_CCW",                            0x0901},
{ "GL_COEFF",                          0x0A00},
{ "GL_ORDER",                          0x0A01},
{ "GL_DOMAIN",                         0x0A02},
{ "GL_CURRENT_COLOR",                  0x0B00},
{ "GL_CURRENT_INDEX",                  0x0B01},
{ "GL_CURRENT_NORMAL",                 0x0B02},
{ "GL_CURRENT_TEXTURE_COORDS",         0x0B03},
{ "GL_CURRENT_RASTER_COLOR",           0x0B04},
{ "GL_CURRENT_RASTER_INDEX",           0x0B05},
{ "GL_CURRENT_RASTER_TEXTURE_COORDS",  0x0B06},
{ "GL_CURRENT_RASTER_POSITION",        0x0B07},
{ "GL_CURRENT_RASTER_POSITION_VALID",  0x0B08},
{ "GL_CURRENT_RASTER_DISTANCE",        0x0B09},
{ "GL_POINT_SMOOTH",                   0x0B10},
{ "GL_POINT_SIZE",                     0x0B11},
{ "GL_POINT_SIZE_RANGE",               0x0B12},
{ "GL_POINT_SIZE_GRANULARITY",         0x0B13},
{ "GL_LINE_SMOOTH",                    0x0B20},
{ "GL_LINE_WIDTH",                     0x0B21},
{ "GL_LINE_WIDTH_RANGE",               0x0B22},
{ "GL_LINE_WIDTH_GRANULARITY",         0x0B23},
{ "GL_LINE_STIPPLE",                   0x0B24},
{ "GL_LINE_STIPPLE_PATTERN",           0x0B25},
{ "GL_LINE_STIPPLE_REPEAT",            0x0B26},
{ "GL_LIST_MODE",                      0x0B30},
{ "GL_MAX_LIST_NESTING",               0x0B31},
{ "GL_LIST_BASE",                      0x0B32},
{ "GL_LIST_INDEX",                     0x0B33},
{ "GL_POLYGON_MODE",                   0x0B40},
{ "GL_POLYGON_SMOOTH",                 0x0B41},
{ "GL_POLYGON_STIPPLE",                0x0B42},
{ "GL_EDGE_FLAG",                      0x0B43},
{ "GL_CULL_FACE",                      0x0B44},
{ "GL_CULL_FACE_MODE",                 0x0B45},
{ "GL_FRONT_FACE",                     0x0B46},
{ "GL_LIGHTING",                       0x0B50},
{ "GL_LIGHT_MODEL_LOCAL_VIEWER",       0x0B51},
{ "GL_LIGHT_MODEL_TWO_SIDE",           0x0B52},
{ "GL_LIGHT_MODEL_AMBIENT",            0x0B53},
{ "GL_SHADE_MODEL",                    0x0B54},
{ "GL_COLOR_MATERIAL_FACE",            0x0B55},
{ "GL_COLOR_MATERIAL_PARAMETER",       0x0B56},
{ "GL_COLOR_MATERIAL",                 0x0B57},
{ "GL_FOG",                            0x0B60},
{ "GL_FOG_INDEX",                      0x0B61},
{ "GL_FOG_DENSITY",                    0x0B62},
{ "GL_FOG_START",                      0x0B63},
{ "GL_FOG_END",                        0x0B64},
{ "GL_FOG_MODE",                       0x0B65},
{ "GL_FOG_COLOR",                      0x0B66},
{ "GL_DEPTH_RANGE",                    0x0B70},
{ "GL_DEPTH_TEST",                     0x0B71},
{ "GL_DEPTH_WRITEMASK",                0x0B72},
{ "GL_DEPTH_CLEAR_VALUE",              0x0B73},
{ "GL_DEPTH_FUNC",                     0x0B74},
{ "GL_ACCUM_CLEAR_VALUE",              0x0B80},
{ "GL_STENCIL_TEST",                   0x0B90},
{ "GL_STENCIL_CLEAR_VALUE",            0x0B91},
{ "GL_STENCIL_FUNC",                   0x0B92},
{ "GL_STENCIL_VALUE_MASK",             0x0B93},
{ "GL_STENCIL_FAIL",                   0x0B94},
{ "GL_STENCIL_PASS_DEPTH_FAIL",        0x0B95},
{ "GL_STENCIL_PASS_DEPTH_PASS",        0x0B96},
{ "GL_STENCIL_REF",                    0x0B97},
{ "GL_STENCIL_WRITEMASK",              0x0B98},
{ "GL_MATRIX_MODE",                    0x0BA0},
{ "GL_NORMALIZE",                      0x0BA1},
{ "GL_VIEWPORT",                       0x0BA2},
{ "GL_MODELVIEW_STACK_DEPTH",          0x0BA3},
{ "GL_PROJECTION_STACK_DEPTH",         0x0BA4},
{ "GL_TEXTURE_STACK_DEPTH",            0x0BA5},
{ "GL_MODELVIEW_MATRIX",               0x0BA6},
{ "GL_PROJECTION_MATRIX",              0x0BA7},
{ "GL_TEXTURE_MATRIX",                 0x0BA8},
{ "GL_ATTRIB_STACK_DEPTH",             0x0BB0},
{ "GL_CLIENT_ATTRIB_STACK_DEPTH",      0x0BB1},
{ "GL_ALPHA_TEST",                     0x0BC0},
{ "GL_ALPHA_TEST_FUNC",                0x0BC1},
{ "GL_ALPHA_TEST_REF",                 0x0BC2},
{ "GL_DITHER",                         0x0BD0},
{ "GL_BLEND_DST",                      0x0BE0},
{ "GL_BLEND_SRC",                      0x0BE1},
{ "GL_BLEND",                          0x0BE2},
{ "GL_LOGIC_OP_MODE",                  0x0BF0},
{ "GL_INDEX_LOGIC_OP",                 0x0BF1},
{ "GL_COLOR_LOGIC_OP",                 0x0BF2},
{ "GL_AUX_BUFFERS",                    0x0C00},
{ "GL_DRAW_BUFFER",                    0x0C01},
{ "GL_READ_BUFFER",                    0x0C02},
{ "GL_SCISSOR_BOX",                    0x0C10},
{ "GL_SCISSOR_TEST",                   0x0C11},
{ "GL_INDEX_CLEAR_VALUE",              0x0C20},
{ "GL_INDEX_WRITEMASK",                0x0C21},
{ "GL_COLOR_CLEAR_VALUE",              0x0C22},
{ "GL_COLOR_WRITEMASK",                0x0C23},
{ "GL_INDEX_MODE",                     0x0C30},
{ "GL_RGBA_MODE",                      0x0C31},
{ "GL_DOUBLEBUFFER",                   0x0C32},
{ "GL_STEREO",                         0x0C33},
{ "GL_RENDER_MODE",                    0x0C40},
{ "GL_PERSPECTIVE_CORRECTION_HINT",    0x0C50},
{ "GL_POINT_SMOOTH_HINT",              0x0C51},
{ "GL_LINE_SMOOTH_HINT",               0x0C52},
{ "GL_POLYGON_SMOOTH_HINT",            0x0C53},
{ "GL_FOG_HINT",                       0x0C54},
{ "GL_TEXTURE_GEN_S",                  0x0C60},
{ "GL_TEXTURE_GEN_T",                  0x0C61},
{ "GL_TEXTURE_GEN_R",                  0x0C62},
{ "GL_TEXTURE_GEN_Q",                  0x0C63},
{ "GL_PIXEL_MAP_I_TO_I",               0x0C70},
{ "GL_PIXEL_MAP_S_TO_S",               0x0C71},
{ "GL_PIXEL_MAP_I_TO_R",               0x0C72},
{ "GL_PIXEL_MAP_I_TO_G",               0x0C73},
{ "GL_PIXEL_MAP_I_TO_B",               0x0C74},
{ "GL_PIXEL_MAP_I_TO_A",               0x0C75},
{ "GL_PIXEL_MAP_R_TO_R",               0x0C76},
{ "GL_PIXEL_MAP_G_TO_G",               0x0C77},
{ "GL_PIXEL_MAP_B_TO_B",               0x0C78},
{ "GL_PIXEL_MAP_A_TO_A",               0x0C79},
{ "GL_PIXEL_MAP_I_TO_I_SIZE",          0x0CB0},
{ "GL_PIXEL_MAP_S_TO_S_SIZE",          0x0CB1},
{ "GL_PIXEL_MAP_I_TO_R_SIZE",          0x0CB2},
{ "GL_PIXEL_MAP_I_TO_G_SIZE",          0x0CB3},
{ "GL_PIXEL_MAP_I_TO_B_SIZE",          0x0CB4},
{ "GL_PIXEL_MAP_I_TO_A_SIZE",          0x0CB5},
{ "GL_PIXEL_MAP_R_TO_R_SIZE",          0x0CB6},
{ "GL_PIXEL_MAP_G_TO_G_SIZE",          0x0CB7},
{ "GL_PIXEL_MAP_B_TO_B_SIZE",          0x0CB8},
{ "GL_PIXEL_MAP_A_TO_A_SIZE",          0x0CB9},
{ "GL_UNPACK_SWAP_BYTES",              0x0CF0},
{ "GL_UNPACK_LSB_FIRST",               0x0CF1},
{ "GL_UNPACK_ROW_LENGTH",              0x0CF2},
{ "GL_UNPACK_SKIP_ROWS",               0x0CF3},
{ "GL_UNPACK_SKIP_PIXELS",             0x0CF4},
{ "GL_UNPACK_ALIGNMENT",               0x0CF5},
{ "GL_PACK_SWAP_BYTES",                0x0D00},
{ "GL_PACK_LSB_FIRST",                 0x0D01},
{ "GL_PACK_ROW_LENGTH",                0x0D02},
{ "GL_PACK_SKIP_ROWS",                 0x0D03},
{ "GL_PACK_SKIP_PIXELS",               0x0D04},
{ "GL_PACK_ALIGNMENT",                 0x0D05},
{ "GL_MAP_COLOR",                      0x0D10},
{ "GL_MAP_STENCIL",                    0x0D11},
{ "GL_INDEX_SHIFT",                    0x0D12},
{ "GL_INDEX_OFFSET",                   0x0D13},
{ "GL_RED_SCALE",                      0x0D14},
{ "GL_RED_BIAS",                       0x0D15},
{ "GL_ZOOM_X",                         0x0D16},
{ "GL_ZOOM_Y",                         0x0D17},
{ "GL_GREEN_SCALE",                    0x0D18},
{ "GL_GREEN_BIAS",                     0x0D19},
{ "GL_BLUE_SCALE",                     0x0D1A},
{ "GL_BLUE_BIAS",                      0x0D1B},
{ "GL_ALPHA_SCALE",                    0x0D1C},
{ "GL_ALPHA_BIAS",                     0x0D1D},
{ "GL_DEPTH_SCALE",                    0x0D1E},
{ "GL_DEPTH_BIAS",                     0x0D1F},
{ "GL_MAX_EVAL_ORDER",                 0x0D30},
{ "GL_MAX_LIGHTS",                     0x0D31},
{ "GL_MAX_CLIP_PLANES",                0x0D32},
{ "GL_MAX_TEXTURE_SIZE",               0x0D33},
{ "GL_MAX_PIXEL_MAP_TABLE",            0x0D34},
{ "GL_MAX_ATTRIB_STACK_DEPTH",         0x0D35},
{ "GL_MAX_MODELVIEW_STACK_DEPTH",      0x0D36},
{ "GL_MAX_NAME_STACK_DEPTH",           0x0D37},
{ "GL_MAX_PROJECTION_STACK_DEPTH",     0x0D38},
{ "GL_MAX_TEXTURE_STACK_DEPTH",        0x0D39},
{ "GL_MAX_VIEWPORT_DIMS",              0x0D3A},
{ "GL_MAX_CLIENT_ATTRIB_STACK_DEPTH",  0x0D3B},
{ "GL_SUBPIXEL_BITS",                  0x0D50},
{ "GL_INDEX_BITS",                     0x0D51},
{ "GL_RED_BITS",                       0x0D52},
{ "GL_GREEN_BITS",                     0x0D53},
{ "GL_BLUE_BITS",                      0x0D54},
{ "GL_ALPHA_BITS",                     0x0D55},
{ "GL_DEPTH_BITS",                     0x0D56},
{ "GL_STENCIL_BITS",                   0x0D57},
{ "GL_ACCUM_RED_BITS",                 0x0D58},
{ "GL_ACCUM_GREEN_BITS",               0x0D59},
{ "GL_ACCUM_BLUE_BITS",                0x0D5A},
{ "GL_ACCUM_ALPHA_BITS",               0x0D5B},
{ "GL_NAME_STACK_DEPTH",               0x0D70},
{ "GL_AUTO_NORMAL",                    0x0D80},
{ "GL_MAP1_COLOR_4",                   0x0D90},
{ "GL_MAP1_INDEX",                     0x0D91},
{ "GL_MAP1_NORMAL",                    0x0D92},
{ "GL_MAP1_TEXTURE_COORD_1",           0x0D93},
{ "GL_MAP1_TEXTURE_COORD_2",           0x0D94},
{ "GL_MAP1_TEXTURE_COORD_3",           0x0D95},
{ "GL_MAP1_TEXTURE_COORD_4",           0x0D96},
{ "GL_MAP1_VERTEX_3",                  0x0D97},
{ "GL_MAP1_VERTEX_4",                  0x0D98},
{ "GL_MAP2_COLOR_4",                   0x0DB0},
{ "GL_MAP2_INDEX",                     0x0DB1},
{ "GL_MAP2_NORMAL",                    0x0DB2},
{ "GL_MAP2_TEXTURE_COORD_1",           0x0DB3},
{ "GL_MAP2_TEXTURE_COORD_2",           0x0DB4},
{ "GL_MAP2_TEXTURE_COORD_3",           0x0DB5},
{ "GL_MAP2_TEXTURE_COORD_4",           0x0DB6},
{ "GL_MAP2_VERTEX_3",                  0x0DB7},
{ "GL_MAP2_VERTEX_4",                  0x0DB8},
{ "GL_MAP1_GRID_DOMAIN",               0x0DD0},
{ "GL_MAP1_GRID_SEGMENTS",             0x0DD1},
{ "GL_MAP2_GRID_DOMAIN",               0x0DD2},
{ "GL_MAP2_GRID_SEGMENTS",             0x0DD3},
{ "GL_TEXTURE_1D",                     0x0DE0},
{ "GL_TEXTURE_2D",                     0x0DE1},
{ "GL_FEEDBACK_BUFFER_POINTER",        0x0DF0},
{ "GL_FEEDBACK_BUFFER_SIZE",           0x0DF1},
{ "GL_FEEDBACK_BUFFER_TYPE",           0x0DF2},
{ "GL_SELECTION_BUFFER_POINTER",       0x0DF3},
{ "GL_SELECTION_BUFFER_SIZE",          0x0DF4},
{ "GL_TEXTURE_WIDTH",                  0x1000},
{ "GL_TEXTURE_HEIGHT",                 0x1001},
{ "GL_TEXTURE_INTERNAL_FORMAT",        0x1003},
{ "GL_TEXTURE_BORDER_COLOR",           0x1004},
{ "GL_TEXTURE_BORDER",                 0x1005},
{ "GL_DONT_CARE",                      0x1100},
{ "GL_FASTEST",                        0x1101},
{ "GL_NICEST",                         0x1102},
{ "GL_AMBIENT",                        0x1200},
{ "GL_DIFFUSE",                        0x1201},
{ "GL_SPECULAR",                       0x1202},
{ "GL_POSITION",                       0x1203},
{ "GL_SPOT_DIRECTION",                 0x1204},
{ "GL_SPOT_EXPONENT",                  0x1205},
{ "GL_SPOT_CUTOFF",                    0x1206},
{ "GL_CONSTANT_ATTENUATION",           0x1207},
{ "GL_LINEAR_ATTENUATION",             0x1208},
{ "GL_QUADRATIC_ATTENUATION",          0x1209},
{ "GL_COMPILE",                        0x1300},
{ "GL_COMPILE_AND_EXECUTE",            0x1301},
{ "GL_BYTE",                           0x1400},
{ "GL_UNSIGNED_BYTE",                  0x1401},
{ "GL_SHORT",                          0x1402},
{ "GL_UNSIGNED_SHORT",                 0x1403},
{ "GL_INT",                            0x1404},
{ "GL_UNSIGNED_INT",                   0x1405},
{ "GL_FLOAT",                          0x1406},
{ "GL_2_BYTES",                        0x1407},
{ "GL_3_BYTES",                        0x1408},
{ "GL_4_BYTES",                        0x1409},
{ "GL_DOUBLE",                         0x140A},
{ "GL_CLEAR",                          0x1500},
{ "GL_AND",                            0x1501},
{ "GL_AND_REVERSE",                    0x1502},
{ "GL_COPY",                           0x1503},
{ "GL_AND_INVERTED",                   0x1504},
{ "GL_NOOP",                           0x1505},
{ "GL_XOR",                            0x1506},
{ "GL_OR",                             0x1507},
{ "GL_NOR",                            0x1508},
{ "GL_EQUIV",                          0x1509},
{ "GL_INVERT",                         0x150A},
{ "GL_OR_REVERSE",                     0x150B},
{ "GL_COPY_INVERTED",                  0x150C},
{ "GL_OR_INVERTED",                    0x150D},
{ "GL_NAND",                           0x150E},
{ "GL_SET",                            0x150F},
{ "GL_EMISSION",                       0x1600},
{ "GL_SHININESS",                      0x1601},
{ "GL_AMBIENT_AND_DIFFUSE",            0x1602},
{ "GL_COLOR_INDEXES",                  0x1603},
{ "GL_MODELVIEW",                      0x1700},
{ "GL_PROJECTION",                     0x1701},
{ "GL_TEXTURE",                        0x1702},
{ "GL_COLOR",                          0x1800},
{ "GL_DEPTH",                          0x1801},
{ "GL_STENCIL",                        0x1802},
{ "GL_COLOR_INDEX",                    0x1900},
{ "GL_STENCIL_INDEX",                  0x1901},
{ "GL_DEPTH_COMPONENT",                0x1902},
{ "GL_RED",                            0x1903},
{ "GL_GREEN",                          0x1904},
{ "GL_BLUE",                           0x1905},
{ "GL_ALPHA",                          0x1906},
{ "GL_RGB",                            0x1907},
{ "GL_RGBA",                           0x1908},
{ "GL_LUMINANCE",                      0x1909},
{ "GL_LUMINANCE_ALPHA",                0x190A},
{ "GL_BITMAP",                         0x1A00},
{ "GL_POINT",                          0x1B00},
{ "GL_LINE",                           0x1B01},
{ "GL_FILL",                           0x1B02},
{ "GL_RENDER",                         0x1C00},
{ "GL_FEEDBACK",                       0x1C01},
{ "GL_SELECT",                         0x1C02},
{ "GL_FLAT",                           0x1D00},
{ "GL_SMOOTH",                         0x1D01},
{ "GL_KEEP",                           0x1E00},
{ "GL_REPLACE",                        0x1E01},
{ "GL_INCR",                           0x1E02},
{ "GL_DECR",                           0x1E03},
{ "GL_VENDOR",                         0x1F00},
{ "GL_RENDERER",                       0x1F01},
{ "GL_VERSION",                        0x1F02},
{ "GL_EXTENSIONS",                     0x1F03},
{ "GL_S",                              0x2000},
{ "GL_T",                              0x2001},
{ "GL_R",                              0x2002},
{ "GL_Q",                              0x2003},
{ "GL_MODULATE",                       0x2100},
{ "GL_DECAL",                          0x2101},
{ "GL_TEXTURE_ENV_MODE",               0x2200},
{ "GL_TEXTURE_ENV_COLOR",              0x2201},
{ "GL_TEXTURE_ENV",                    0x2300},
{ "GL_EYE_LINEAR",                     0x2400},
{ "GL_OBJECT_LINEAR",                  0x2401},
{ "GL_SPHERE_MAP",                     0x2402},
{ "GL_TEXTURE_GEN_MODE",               0x2500},
{ "GL_OBJECT_PLANE",                   0x2501},
{ "GL_EYE_PLANE",                      0x2502},
{ "GL_NEAREST",                        0x2600},
{ "GL_LINEAR",                         0x2601},
{ "GL_NEAREST_MIPMAP_NEAREST",         0x2700},
{ "GL_LINEAR_MIPMAP_NEAREST",          0x2701},
{ "GL_NEAREST_MIPMAP_LINEAR",          0x2702},
{ "GL_LINEAR_MIPMAP_LINEAR",           0x2703},
{ "GL_TEXTURE_MAG_FILTER",             0x2800},
{ "GL_TEXTURE_MIN_FILTER",             0x2801},
{ "GL_TEXTURE_WRAP_S",                 0x2802},
{ "GL_TEXTURE_WRAP_T",                 0x2803},
{ "GL_CLAMP",                          0x2900},
{ "GL_REPEAT",                         0x2901},
{ "GL_POLYGON_OFFSET_UNITS",           0x2A00},
{ "GL_POLYGON_OFFSET_POINT",           0x2A01},
{ "GL_POLYGON_OFFSET_LINE",            0x2A02},
{ "GL_R3_G3_B2",                       0x2A10},
{ "GL_V2F",                            0x2A20},
{ "GL_V3F",                            0x2A21},
{ "GL_C4UB_V2F",                       0x2A22},
{ "GL_C4UB_V3F",                       0x2A23},
{ "GL_C3F_V3F",                        0x2A24},
{ "GL_N3F_V3F",                        0x2A25},
{ "GL_C4F_N3F_V3F",                    0x2A26},
{ "GL_T2F_V3F",                        0x2A27},
{ "GL_T4F_V4F",                        0x2A28},
{ "GL_T2F_C4UB_V3F",                   0x2A29},
{ "GL_T2F_C3F_V3F",                    0x2A2A},
{ "GL_T2F_N3F_V3F",                    0x2A2B},
{ "GL_T2F_C4F_N3F_V3F",                0x2A2C},
{ "GL_T4F_C4F_N3F_V4F",                0x2A2D},
{ "GL_CLIP_PLANE0",                    0x3000},
{ "GL_CLIP_PLANE1",                    0x3001},
{ "GL_CLIP_PLANE2",                    0x3002},
{ "GL_CLIP_PLANE3",                    0x3003},
{ "GL_CLIP_PLANE4",                    0x3004},
{ "GL_CLIP_PLANE5",                    0x3005},
{ "GL_LIGHT0",                         0x4000},
{ "GL_LIGHT1",                         0x4001},
{ "GL_LIGHT2",                         0x4002},
{ "GL_LIGHT3",                         0x4003},
{ "GL_LIGHT4",                         0x4004},
{ "GL_LIGHT5",                         0x4005},
{ "GL_LIGHT6",                         0x4006},
{ "GL_LIGHT7",                         0x4007},
{ "GL_POLYGON_OFFSET_FILL",            0x8037},
{ "GL_POLYGON_OFFSET_FACTOR",          0x8038},
{ "GL_ALPHA4",                         0x803B},
{ "GL_ALPHA8",                         0x803C},
{ "GL_ALPHA12",                        0x803D},
{ "GL_ALPHA16",                        0x803E},
{ "GL_LUMINANCE4",                     0x803F},
{ "GL_LUMINANCE8",                     0x8040},
{ "GL_LUMINANCE12",                    0x8041},
{ "GL_LUMINANCE16",                    0x8042},
{ "GL_LUMINANCE4_ALPHA4",              0x8043},
{ "GL_LUMINANCE6_ALPHA2",              0x8044},
{ "GL_LUMINANCE8_ALPHA8",              0x8045},
{ "GL_LUMINANCE12_ALPHA4",             0x8046},
{ "GL_LUMINANCE12_ALPHA12",            0x8047},
{ "GL_LUMINANCE16_ALPHA16",            0x8048},
{ "GL_INTENSITY",                      0x8049},
{ "GL_INTENSITY4",                     0x804A},
{ "GL_INTENSITY8",                     0x804B},
{ "GL_INTENSITY12",                    0x804C},
{ "GL_INTENSITY16",                    0x804D},
{ "GL_RGB4",                           0x804F},
{ "GL_RGB5",                           0x8050},
{ "GL_RGB8",                           0x8051},
{ "GL_RGB10",                          0x8052},
{ "GL_RGB12",                          0x8053},
{ "GL_RGB16",                          0x8054},
{ "GL_RGBA2",                          0x8055},
{ "GL_RGBA4",                          0x8056},
{ "GL_RGB5_A1",                        0x8057},
{ "GL_RGBA8",                          0x8058},
{ "GL_RGB10_A2",                       0x8059},
{ "GL_RGBA12",                         0x805A},
{ "GL_RGBA16",                         0x805B},
{ "GL_TEXTURE_RED_SIZE",               0x805C},
{ "GL_TEXTURE_GREEN_SIZE",             0x805D},
{ "GL_TEXTURE_BLUE_SIZE",              0x805E},
{ "GL_TEXTURE_ALPHA_SIZE",             0x805F},
{ "GL_TEXTURE_LUMINANCE_SIZE",         0x8060},
{ "GL_TEXTURE_INTENSITY_SIZE",         0x8061},
{ "GL_PROXY_TEXTURE_1D",               0x8063},
{ "GL_PROXY_TEXTURE_2D",               0x8064},
{ "GL_TEXTURE_PRIORITY",               0x8066},
{ "GL_TEXTURE_RESIDENT",               0x8067},
{ "GL_TEXTURE_BINDING_1D",             0x8068},
{ "GL_TEXTURE_BINDING_2D",             0x8069},
{ "GL_VERTEX_ARRAY",                   0x8074},
{ "GL_NORMAL_ARRAY",                   0x8075},
{ "GL_COLOR_ARRAY",                    0x8076},
{ "GL_INDEX_ARRAY",                    0x8077},
{ "GL_TEXTURE_COORD_ARRAY",            0x8078},
{ "GL_EDGE_FLAG_ARRAY",                0x8079},
{ "GL_VERTEX_ARRAY_SIZE",              0x807A},
{ "GL_VERTEX_ARRAY_TYPE",              0x807B},
{ "GL_VERTEX_ARRAY_STRIDE",            0x807C},
{ "GL_NORMAL_ARRAY_TYPE",              0x807E},
{ "GL_NORMAL_ARRAY_STRIDE",            0x807F},
{ "GL_COLOR_ARRAY_SIZE",               0x8081},
{ "GL_COLOR_ARRAY_TYPE",               0x8082},
{ "GL_COLOR_ARRAY_STRIDE",             0x8083},
{ "GL_INDEX_ARRAY_TYPE",               0x8085},
{ "GL_INDEX_ARRAY_STRIDE",             0x8086},
{ "GL_TEXTURE_COORD_ARRAY_SIZE",       0x8088},
{ "GL_TEXTURE_COORD_ARRAY_TYPE",       0x8089},
{ "GL_TEXTURE_COORD_ARRAY_STRIDE",     0x808A},
{ "GL_EDGE_FLAG_ARRAY_STRIDE",         0x808C},
{ "GL_VERTEX_ARRAY_POINTER",           0x808E},
{ "GL_NORMAL_ARRAY_POINTER",           0x808F},
{ "GL_COLOR_ARRAY_POINTER",            0x8090},
{ "GL_INDEX_ARRAY_POINTER",            0x8091},
{ "GL_TEXTURE_COORD_ARRAY_POINTER",    0x8092},
{ "GL_EDGE_FLAG_ARRAY_POINTER",        0x8093},
{ "GL_COLOR_TABLE_FORMAT_EXT",         0x80D8},
{ "GL_COLOR_TABLE_WIDTH_EXT",          0x80D9},
{ "GL_COLOR_TABLE_RED_SIZE_EXT",       0x80DA},
{ "GL_COLOR_TABLE_GREEN_SIZE_EXT",     0x80DB},
{ "GL_COLOR_TABLE_BLUE_SIZE_EXT",      0x80DC},
{ "GL_COLOR_TABLE_ALPHA_SIZE_EXT",     0x80DD},
{ "GL_COLOR_TABLE_LUMINANCE_SIZE_EXT", 0x80DE},
{ "GL_COLOR_TABLE_INTENSITY_SIZE_EXT", 0x80DF},
{ "GL_BGR_EXT",                        0x80E0},
{ "GL_BGRA_EXT",                       0x80E1},
{ "GL_COLOR_INDEX1_EXT",               0x80E2},
{ "GL_COLOR_INDEX2_EXT",               0x80E3},
{ "GL_COLOR_INDEX4_EXT",               0x80E4},
{ "GL_COLOR_INDEX8_EXT",               0x80E5},
{ "GL_COLOR_INDEX12_EXT",              0x80E6},
{ "GL_COLOR_INDEX16_EXT",              0x80E7}
};

//
// Misc. globals
//

HANDLE hDLLInstance;       // DLL instance handle
unsigned starttime;        // Time when trace started
FILE *text_file = NULL;    // Output text file (NULL for OutputDebugString)

char *GLenum_names[65536]; // Pointer to text string for each GLenum value

//
// Send output to file or debug console
//

void __cdecl print(char *fmt, ...)
{
   static char work_string[4096];

   if ((fmt == NULL) || (strlen(fmt) > sizeof(work_string)))
      {
      strcpy(work_string, "(String missing or too large)");
      }
   else
      {
      va_list ap;

      va_start(ap, 
               fmt);

      vsprintf(work_string, 
               fmt, 
               ap);

      va_end  (ap);
      }

   if (text_file == NULL)
      {
      OutputDebugString(work_string);
      }
   else
      {
      fputs(work_string,text_file);
      }
}

//
// Print typed value by reference
//

void *print_value(TYPE type, const void *value)
{
   int i;
   unsigned int *v = (unsigned int *) value;

   PIXELFORMATDESCRIPTOR *desc;

   switch (type)
      {
      default:    print("0x%X",*(unsigned *) v); ++v;      break;

      case HEX:   print("0x%X",*(unsigned *) v); ++v;      break;
      case SINT:  print("%d",  *(int *) v);      ++v;      break;
      case UNINT: print("%u",  *(unsigned *) v); ++v;      break;
      case FLT:   print("%f",  *(float *) v);    ++v;      break;
      case DBL:   print("%f",  *(float *) v);    ++v; ++v; break;
      case STR:   print("%s",  *(char **) v);    ++v;      break;

      case GLBO:  print("%s",  *(int *) v ? "TRUE" : "FALSE"); 
                  ++v;
                  break;
      
      //
      // Print GLenum type from master name list
      //

      case GLEN:  
      
         i = *(int *) v++;      

         if (GLenum_names[i] == NULL)
            {
            print("[%X]",i);
            }
         else
            {
            print("%s",GLenum_names[i]);
            }
         break;

      //
      // (Example: How to dump contents of OpenGL-specific structure)
      //

      case PFD:   
         desc = *(PIXELFORMATDESCRIPTOR **) v;

         print("\n");
         print("     WORD  nSize = 0x%X\n",desc->nSize); 
         print("     WORD  nVersion = 0x%X\n",desc->nVersion); 
         print("     DWORD dwFlags = 0x%X\n",desc->dwFlags); 
         print("     BYTE  iPixelType = 0x%X\n",desc->iPixelType); 
         print("     BYTE  cColorBits = 0x%X\n",desc->cColorBits); 
         print("     BYTE  cRedBits = 0x%X\n",desc->cRedBits); 
         print("     BYTE  cRedShift = 0x%X\n",desc->cRedShift); 
         print("     BYTE  cGreenBits = 0x%X\n",desc->cGreenBits); 
         print("     BYTE  cGreenShift = 0x%X\n",desc->cGreenShift); 
         print("     BYTE  cBlueBits = 0x%X\n",desc->cBlueBits); 
         print("     BYTE  cBlueShift = 0x%X\n",desc->cBlueShift); 
         print("     BYTE  cAlphaBits = 0x%X\n",desc->cAlphaBits); 
         print("     BYTE  cAlphaShift = 0x%X\n",desc->cAlphaShift); 
         print("     BYTE  cAccumBits = 0x%X\n",desc->cAccumBits); 
         print("     BYTE  cAccumRedBits = 0x%X\n",desc->cAccumRedBits); 
         print("     BYTE  cAccumGreenBits = 0x%X\n",desc->cAccumGreenBits); 
         print("     BYTE  cAccumBlueBits = 0x%X\n",desc->cAccumBlueBits); 
         print("     BYTE  cAccumAlphaBits = 0x%X\n",desc->cAccumAlphaBits); 
         print("     BYTE  cDepthBits = 0x%X\n",desc->cDepthBits); 
         print("     BYTE  cStencilBits = 0x%X\n",desc->cStencilBits); 
         print("     BYTE  cAuxBuffers = 0x%X\n",desc->cAuxBuffers); 
         print("     BYTE  iLayerType = 0x%X\n",desc->iLayerType); 
         print("     BYTE  bReserved = 0x%X\n",desc->bReserved); 
         print("     DWORD dwLayerMask = 0x%X\n",desc->dwLayerMask); 
         print("     DWORD dwVisibleMask = 0x%X\n",desc->dwVisibleMask); 
         print("     DWORD dwDamageMask = 0x%X\n",desc->dwDamageMask); 

         ++v;
         break;
      }

   return v;
}

//
// Print timestamp
//

void write_time()
{
   unsigned t;
   int      h,m,s,c;

   t = timeGetTime() - starttime;

   h = (unsigned short)(t/3600000L);
   m = (unsigned short)((t/60000L)%60L);
   s = (unsigned short)((t/1000L)%60L);
   c = (unsigned short)(t%1000L);

   print("[%.02d:%.02d:%.02d.%.03d] ",h,m,s,c);
}

//
// Diagnostic routines
//

void Function(int index, char *parms, void *arglist)   
{
   if (! ((show_status[index] == SHOW_ALWAYS)
          ||
         ((show_status[index] == SHOW_IF_SHIFT_DOWN) && 
          ((GetAsyncKeyState(VK_SHIFT) & 0x8000) || (GetAsyncKeyState(VK_MENU) & 0x8000)))))
      {
      return;
      }

   //
   // Print timestamp and function name
   // 

   write_time();
   print(GLN[index]);

   //
   // Parse and print arguments
   // 

   print("(");

   char workspace[256];
   char *args = workspace;

   strcpy(args, parms);

   char *type_name;
   char *var_name;

   char *end = &args[strlen(args)];

   int first = 1;
   int state = 0;

   while ((unsigned) args < (unsigned) end)
      {
      switch (state)
         {
         // --------------------------
         // State 0: Acquire type name
         // --------------------------

         case 0:

            // 
            // Skip any leading spaces or commas
            //

            while ((*args == ' ') || (*args == ','))
               {
               ++args;
               }

            //
            // Skip any 'const ' qualifier
            //

            if ((strlen(args) > 6) && (!strnicmp(args,"const ",6)))
               {
               args += 6;
               }

            //
            // Mark start of type name, and find its end
            // Include any trailing '*' in type name; e.g., "GLfloat *foo"
            // becomes "GLfloat*"
            //

            type_name = args;

            while (*args != ' ')
               {
               assert(*args++);
               }

            assert(args[1]);

            if (args[1] == '*')
               {
               *args++ = '*';
               }

            *args++ = 0;

            //
            // Advance to next state to get variable name
            //

            state = 1;
            break;

         // --------------------------
         // State 1: Acquire variable name
         // --------------------------

         case 1:
            
            //
            // Skip any leading spaces or '*'s
            //

            while ((*args == ' ') || (*args == '*'))
               {
               ++args;
               }

            //
            // Mark start of variable name, and find its end
            //

            var_name = args;

            while ((*args) && (*args != ' ') && (*args != ','))
               {
               ++args;
               }

            *args++ = 0;

            //
            // Print variable name and value (with leading comma and 
            // space if this is not the first)
            //

            if (first)
               {
               first = 0;
               }
            else
               {
               print(", ");
               }

            print("%s=",var_name);

            for (int i=0; i < ARY_CNT(type_formats); i++)
               {
               FORMAT *f = &type_formats[i];

               if (!strcmp(type_name, f->name))
                  {
                  arglist = print_value(f->type, arglist);
                  break;
                  }
               }

            if (i == ARY_CNT(type_formats))
               {
               arglist = print_value(HEX, arglist);
               }

            //
            // Continue with next type/name pair...
            //

            state = 0;
            break;
         }

      }

   print(")");
}

void Crout(int index)
{
   if (! ((show_status[index] == SHOW_ALWAYS)
          ||
         ((show_status[index] == SHOW_IF_SHIFT_DOWN) && 
          ((GetAsyncKeyState(VK_SHIFT) & 0x8000) || (GetAsyncKeyState(VK_MENU) & 0x8000)))))
      {
      return;
      }

   print("\n");
}

void Result(int index, char *type, const void *value)
{
   if (! ((show_status[index] == SHOW_ALWAYS)
          ||
         ((show_status[index] == SHOW_IF_SHIFT_DOWN) && 
          ((GetAsyncKeyState(VK_SHIFT) & 0x8000) || (GetAsyncKeyState(VK_MENU) & 0x8000)))))
      {
      return;
      }

   print(" = ");

   for (int i=0; i < ARY_CNT(type_formats); i++)
      {
      FORMAT *f = &type_formats[i];

      if (!strcmp(type, f->name))
         {
         print_value(f->type, value);
         return;
         }
      }

   print_value(HEX, value);
}

//
// OpenGL function handlers
//

void APIENTRY glAccum (GLenum op, GLfloat value)
{
   Function(FP_glAccum,"GLenum op, GLfloat value",&op);
   GLV.glAccum (op,value);
   Crout(FP_glAccum);
}

void APIENTRY glAlphaFunc (GLenum func, GLclampf ref)
{
   Function(FP_glAlphaFunc,"GLenum func, GLclampf ref",&func);
   GLV.glAlphaFunc (func, ref);
   Crout(FP_glAlphaFunc);
}

GLboolean APIENTRY glAreTexturesResident (GLsizei n, const GLuint *textures, GLboolean *residences)
{
   Function(FP_glAreTexturesResident,"GLsizei n, const GLuint *textures, GLboolean *residences",&n);
   GLboolean result = GLV.glAreTexturesResident(n,textures,residences);
   Result(FP_glAreTexturesResident,"GLboolean",&result);
   Crout(FP_glAreTexturesResident);
   return result;
}

void APIENTRY glArrayElement (GLint i)
{
   Function(FP_glArrayElement,"GLint i",&i);
   GLV.glArrayElement (i);
   Crout(FP_glArrayElement);
}

void APIENTRY glBegin (GLenum mode)
{
   Function(FP_glBegin,"GLenum mode",&mode);
   GLV.glBegin (mode);
   Crout(FP_glBegin);
}

void APIENTRY glBindTexture (GLenum target, GLuint texture)
{
   Function(FP_glBindTexture,"GLenum target, GLuint texture",&target);
   GLV.glBindTexture (target, texture);
   Crout(FP_glBindTexture);
}

void APIENTRY glBitmap (GLsizei width, GLsizei height, GLfloat xorig, GLfloat yorig, GLfloat xmove, GLfloat ymove, const GLubyte *bitmap)
{
   Function(FP_glBitmap,"GLsizei width, GLsizei height, GLfloat xorig, GLfloat yorig, GLfloat xmove, GLfloat ymove, const GLubyte *bitmap",&width);
   GLV.glBitmap (width, height, xorig, yorig, xmove, ymove, bitmap);
   Crout(FP_glBitmap);
}

void APIENTRY glBlendFunc (GLenum sfactor, GLenum dfactor)
{
   Function(FP_glBlendFunc,"GLenum sfactor, GLenum dfactor",&sfactor);
   GLV.glBlendFunc (sfactor, dfactor);
   Crout(FP_glBlendFunc);
}

void APIENTRY glCallList (GLuint list)
{
   Function(FP_glCallList,"GLuint list",&list);
   GLV.glCallList (list);
   Crout(FP_glCallList);
}

void APIENTRY glCallLists (GLsizei n, GLenum type, const GLvoid *lists)
{
   Function(FP_glCallLists,"GLsizei n, GLenum type, const GLvoid *lists",&n);
   GLV.glCallLists (n, type, lists);
   Crout(FP_glCallLists);
}

void APIENTRY glClear (GLbitfield mask)
{
   Function(FP_glClear,"GLbitfield mask",&mask);
   GLV.glClear (mask);
   Crout(FP_glClear);
}

void APIENTRY glClearAccum (GLfloat red, GLfloat green, GLfloat blue, GLfloat alpha)
{
   Function(FP_glClearAccum,"GLfloat red, GLfloat green, GLfloat blue, GLfloat alpha",&red);
   GLV.glClearAccum (red, green, blue, alpha);
   Crout(FP_glClearAccum);
}

void APIENTRY glClearColor (GLclampf red, GLclampf green, GLclampf blue, GLclampf alpha)
{
   Function(FP_glClearColor,"GLclampf red, GLclampf green, GLclampf blue, GLclampf alpha",&red);
   GLV.glClearColor (red, green, blue, alpha);
   Crout(FP_glClearColor);
}

void APIENTRY glClearDepth (GLclampd depth)
{
   Function(FP_glClearDepth,"GLclampd depth",&depth);
   GLV.glClearDepth (depth);
   Crout(FP_glClearDepth);
}

void APIENTRY glClearIndex (GLfloat c)
{
   Function(FP_glClearIndex,"GLfloat c",&c);
   GLV.glClearIndex (c);
   Crout(FP_glClearIndex);
}

void APIENTRY glClearStencil (GLint s)
{
   Function(FP_glClearStencil,"GLint s",&s);
   GLV.glClearStencil (s);
   Crout(FP_glClearStencil);
}

void APIENTRY glClipPlane (GLenum plane, const GLdouble *equation)
{
   Function(FP_glClipPlane,"GLenum plane, const GLdouble *equation",&plane);
   GLV.glClipPlane (plane, equation);
   Crout(FP_glClipPlane);
}

void APIENTRY glColor3b (GLbyte red, GLbyte green, GLbyte blue)
{
   Function(FP_glColor3b,"GLbyte red, GLbyte green, GLbyte blue",&red);
   GLV.glColor3b (red, green, blue);
   Crout(FP_glColor3b);
}

void APIENTRY glColor3bv (const GLbyte *v)
{
   Function(FP_glColor3bv,"const GLbyte *v",&v);
   GLV.glColor3bv (v);
   Crout(FP_glColor3bv);
}

void APIENTRY glColor3d (GLdouble red, GLdouble green, GLdouble blue)
{
   Function(FP_glColor3d,"GLdouble red, GLdouble green, GLdouble blue",&red);
   GLV.glColor3d (red, green, blue);
   Crout(FP_glColor3d);
}

void APIENTRY glColor3dv (const GLdouble *v)
{
   Function(FP_glColor3dv,"const GLdouble *v",&v);
   GLV.glColor3dv (v);
   Crout(FP_glColor3dv);
}

void APIENTRY glColor3f (GLfloat red, GLfloat green, GLfloat blue)
{
   Function(FP_glColor3f,"GLfloat red, GLfloat green, GLfloat blue",&red);
   GLV.glColor3f (red, green, blue);
   Crout(FP_glColor3f);
}

void APIENTRY glColor3fv (const GLfloat *v)
{
   Function(FP_glColor3fv,"const GLfloat *v",&v);
   GLV.glColor3fv (v);
   Crout(FP_glColor3fv);
}

void APIENTRY glColor3i (GLint red, GLint green, GLint blue)
{
   Function(FP_glColor3i,"GLint red, GLint green, GLint blue",&red);
   GLV.glColor3i (red, green, blue);
   Crout(FP_glColor3i);
}

void APIENTRY glColor3iv (const GLint *v)
{
   Function(FP_glColor3iv,"const GLint *v",&v);
   GLV.glColor3iv (v);
   Crout(FP_glColor3iv);
}

void APIENTRY glColor3s (GLshort red, GLshort green, GLshort blue)
{
   Function(FP_glColor3s,"GLshort red, GLshort green, GLshort blue",&red);
   GLV.glColor3s (red, green, blue);
   Crout(FP_glColor3s);
}

void APIENTRY glColor3sv (const GLshort *v)
{
   Function(FP_glColor3sv,"const GLshort *v",&v);
   GLV.glColor3sv (v);
   Crout(FP_glColor3sv);
}

void APIENTRY glColor3ub (GLubyte red, GLubyte green, GLubyte blue)
{
   Function(FP_glColor3ub,"GLubyte red, GLubyte green, GLubyte blue",&red);
   GLV.glColor3ub (red, green, blue);
   Crout(FP_glColor3ub);
}

void APIENTRY glColor3ubv (const GLubyte *v)
{
   Function(FP_glColor3ubv,"const GLubyte *v",&v);
   GLV.glColor3ubv (v);
   Crout(FP_glColor3ubv);
}

void APIENTRY glColor3ui (GLuint red, GLuint green, GLuint blue)
{
   Function(FP_glColor3ui,"GLuint red, GLuint green, GLuint blue",&red);
   GLV.glColor3ui (red, green, blue);
   Crout(FP_glColor3ui);
}

void APIENTRY glColor3uiv (const GLuint *v)
{
   Function(FP_glColor3uiv,"const GLuint *v",&v);
   GLV.glColor3uiv (v);
   Crout(FP_glColor3uiv);
}

void APIENTRY glColor3us (GLushort red, GLushort green, GLushort blue)
{
   Function(FP_glColor3us,"GLushort red, GLushort green, GLushort blue",&red);
   GLV.glColor3us (red, green, blue);
   Crout(FP_glColor3us);
}

void APIENTRY glColor3usv (const GLushort *v)
{
   Function(FP_glColor3usv,"const GLushort *v",&v);
   GLV.glColor3usv (v);
   Crout(FP_glColor3usv);
}

void APIENTRY glColor4b (GLbyte red, GLbyte green, GLbyte blue, GLbyte alpha)
{
   Function(FP_glColor4b,"GLbyte red, GLbyte green, GLbyte blue, GLbyte alpha",&red);
   GLV.glColor4b (red, green, blue, alpha);
   Crout(FP_glColor4b);
}

void APIENTRY glColor4bv (const GLbyte *v)
{
   Function(FP_glColor4bv,"const GLbyte *v",&v);
   GLV.glColor4bv (v);
   Crout(FP_glColor4bv);
}

void APIENTRY glColor4d (GLdouble red, GLdouble green, GLdouble blue, GLdouble alpha)
{
   Function(FP_glColor4d,"GLdouble red, GLdouble green, GLdouble blue, GLdouble alpha",&red);
   GLV.glColor4d (red, green, blue, alpha);
   Crout(FP_glColor4d);
}

void APIENTRY glColor4dv (const GLdouble *v)
{
   Function(FP_glColor4dv,"const GLdouble *v",&v);
   GLV.glColor4dv (v);
   Crout(FP_glColor4dv);
}

void APIENTRY glColor4f (GLfloat red, GLfloat green, GLfloat blue, GLfloat alpha)
{
   Function(FP_glColor4f,"GLfloat red, GLfloat green, GLfloat blue, GLfloat alpha",&red);
   GLV.glColor4f (red, green, blue, alpha);
   Crout(FP_glColor4f);
}
                                    
void APIENTRY glColor4fv (const GLfloat *v)
{
   Function(FP_glColor4fv,"const GLfloat *v",&v);
   GLV.glColor4fv (v);
   Crout(FP_glColor4fv);
}

void APIENTRY glColor4i (GLint red, GLint green, GLint blue, GLint alpha)
{
   Function(FP_glColor4i,"GLint red, GLint green, GLint blue, GLint alpha",&red);
   GLV.glColor4i (red, green, blue, alpha);
   Crout(FP_glColor4i);
}

void APIENTRY glColor4iv (const GLint *v)
{                  
   Function(FP_glColor4iv,"const GLint *v",&v);
   GLV.glColor4iv (v);
   Crout(FP_glColor4iv);
}

void APIENTRY glColor4s (GLshort red, GLshort green, GLshort blue, GLshort alpha)
{
   Function(FP_glColor4s,"GLshort red, GLshort green, GLshort blue, GLshort alpha",&red);
   GLV.glColor4s (red, green, blue, alpha);
   Crout(FP_glColor4s);
}

void APIENTRY glColor4sv (const GLshort *v)
{
   Function(FP_glColor4sv,"const GLshort *v",&v);
   GLV.glColor4sv (v);
   Crout(FP_glColor4sv);
}

void APIENTRY glColor4ub (GLubyte red, GLubyte green, GLubyte blue, GLubyte alpha)
{
   Function(FP_glColor4ub,"GLubyte red, GLubyte green, GLubyte blue, GLubyte alpha",&red);
   GLV.glColor4ub (red, green, blue, alpha);
   Crout(FP_glColor4ub);
}

void APIENTRY glColor4ubv (const GLubyte *v)
{
   Function(FP_glColor4ubv,"const GLubyte *v",&v);
   GLV.glColor4ubv (v);
   Crout(FP_glColor4ubv);
}

void APIENTRY glColor4ui (GLuint red, GLuint green, GLuint blue, GLuint alpha)
{
   Function(FP_glColor4ui,"GLuint red, GLuint green, GLuint blue, GLuint alpha",&red);
   GLV.glColor4ui (red, green, blue, alpha);
   Crout(FP_glColor4ui);
}

void APIENTRY glColor4uiv (const GLuint *v)
{
   Function(FP_glColor4uiv,"const GLuint *v",&v);
   GLV.glColor4uiv (v);
   Crout(FP_glColor4uiv);
}

void APIENTRY glColor4us (GLushort red, GLushort green, GLushort blue, GLushort alpha)
{
   Function(FP_glColor4us,"GLushort red, GLushort green, GLushort blue, GLushort alpha",&red);
   GLV.glColor4us (red, green, blue, alpha);
   Crout(FP_glColor4us);
}

void APIENTRY glColor4usv (const GLushort *v)
{
   Function(FP_glColor4usv,"const GLushort *v",&v);
   GLV.glColor4usv (v);
   Crout(FP_glColor4usv);
}

void APIENTRY glColorMask (GLboolean red, GLboolean green, GLboolean blue, GLboolean alpha)
{
   Function(FP_glColorMask,"GLboolean red, GLboolean green, GLboolean blue, GLboolean alpha",&red);
   GLV.glColorMask (red, green, blue, alpha);
   Crout(FP_glColorMask);
}

void APIENTRY glColorMaterial (GLenum face, GLenum mode)
{
   Function(FP_glColorMaterial,"GLenum face, GLenum mode",&face);
   GLV.glColorMaterial (face, mode);
   Crout(FP_glColorMaterial);
}

void APIENTRY glColorPointer (GLint size, GLenum type, GLsizei stride, const GLvoid *pointer)
{
   Function(FP_glColorPointer,"GLint size, GLenum type, GLsizei stride, const GLvoid *pointer",&size);
   GLV.glColorPointer (size, type, stride, pointer);
   Crout(FP_glColorPointer);
}

void APIENTRY glCopyPixels (GLint x, GLint y, GLsizei width, GLsizei height, GLenum type)
{
   Function(FP_glCopyPixels,"GLint x, GLint y, GLsizei width, GLsizei height, GLenum type",&x);
   GLV.glCopyPixels (x, y, width, height, type);
   Crout(FP_glCopyPixels);
}

void APIENTRY glCopyTexImage1D (GLenum target, GLint level, GLenum internalFormat, GLint x, GLint y, GLsizei width, GLint border)
{
   Function(FP_glCopyTexImage1D,"GLenum target, GLint level, GLenum internalFormat, GLint x, GLint y, GLsizei width, GLint border",&target);
   GLV.glCopyTexImage1D (target, level, internalFormat, x, y, width, border);
   Crout(FP_glCopyTexImage1D);
}

void APIENTRY glCopyTexImage2D (GLenum target, GLint level, GLenum internalFormat, GLint x, GLint y, GLsizei width, GLsizei height, GLint border)
{
   Function(FP_glCopyTexImage2D,"GLenum target, GLint level, GLenum internalFormat, GLint x, GLint y, GLsizei width, GLsizei height, GLint border",&target);
   GLV.glCopyTexImage2D (target, level, internalFormat, x, y, width, height, border);
   Crout(FP_glCopyTexImage2D);
}

void APIENTRY glCopyTexSubImage1D (GLenum target, GLint level, GLint xoffset, GLint x, GLint y, GLsizei width)
{
   Function(FP_glCopyTexSubImage1D,"GLenum target, GLint level, GLint xoffset, GLint x, GLint y, GLsizei width",&target);
   GLV.glCopyTexSubImage1D (target, level, xoffset, x, y, width);
   Crout(FP_glCopyTexSubImage1D);
}

void APIENTRY glCopyTexSubImage2D (GLenum target, GLint level, GLint xoffset, GLint yoffset, GLint x, GLint y, GLsizei width, GLsizei height)
{
   Function(FP_glCopyTexSubImage2D,"GLenum target, GLint level, GLint xoffset, GLint yoffset, GLint x, GLint y, GLsizei width, GLsizei height",&target);
   GLV.glCopyTexSubImage2D (target, level, xoffset, yoffset, x, y, width, height);
   Crout(FP_glCopyTexSubImage2D);
}

void APIENTRY glCullFace (GLenum mode)
{
   Function(FP_glCullFace,"GLenum mode",&mode);
   GLV.glCullFace (mode);
   Crout(FP_glCullFace);
}

void APIENTRY glDeleteLists (GLuint list, GLsizei range)
{
   Function(FP_glDeleteLists,"GLuint list, GLsizei range",&list);
   GLV.glDeleteLists (list, range);
   Crout(FP_glDeleteLists);
}

void APIENTRY glDeleteTextures (GLsizei n, const GLuint *textures)
{
   Function(FP_glDeleteTextures,"GLsizei n, const GLuint *textures",&n);
   GLV.glDeleteTextures (n, textures);
   Crout(FP_glDeleteTextures);
}

void APIENTRY glDepthFunc (GLenum func)
{
   Function(FP_glDepthFunc,"GLenum func",&func);
   GLV.glDepthFunc (func);
   Crout(FP_glDepthFunc);
}

void APIENTRY glDepthMask (GLboolean flag)
{
   Function(FP_glDepthMask,"GLboolean flag",&flag);
   GLV.glDepthMask (flag);
   Crout(FP_glDepthMask);
}

void APIENTRY glDepthRange (GLclampd zNear, GLclampd zFar)
{
   Function(FP_glDepthRange,"GLclampd zNear, GLclampd zFar",&zNear);
   GLV.glDepthRange (zNear, zFar);
   Crout(FP_glDepthRange);
}

void APIENTRY glDisable (GLenum cap)
{
   Function(FP_glDisable,"GLenum cap",&cap);
   GLV.glDisable (cap);
   Crout(FP_glDisable);
}

void APIENTRY glDisableClientState (GLenum array)
{
   Function(FP_glDisableClientState,"GLenum array",&array);
   GLV.glDisableClientState (array);
   Crout(FP_glDisableClientState);
}

void APIENTRY glDrawArrays (GLenum mode, GLint first, GLsizei count)
{
   Function(FP_glDrawArrays,"GLenum mode, GLint first, GLsizei count",&mode);
   GLV.glDrawArrays (mode, first, count);
   Crout(FP_glDrawArrays);
}

void APIENTRY glDrawBuffer (GLenum mode)
{
   Function(FP_glDrawBuffer,"GLenum mode",&mode);
   GLV.glDrawBuffer (mode);
   Crout(FP_glDrawBuffer);
}

void APIENTRY glDrawElements (GLenum mode, GLsizei count, GLenum type, const GLvoid *indices)
{
   Function(FP_glDrawElements,"GLenum mode, GLsizei count, GLenum type, const GLvoid *indices",&mode);
   GLV.glDrawElements (mode, count, type, indices);
   Crout(FP_glDrawElements);
}

void APIENTRY glDrawPixels (GLsizei width, GLsizei height, GLenum format, GLenum type, const GLvoid *pixels)
{
   Function(FP_glDrawPixels,"GLsizei width, GLsizei height, GLenum format, GLenum type, const GLvoid *pixels",&width);
   GLV.glDrawPixels (width, height, format, type, pixels);
   Crout(FP_glDrawPixels);
}

void APIENTRY glEdgeFlag (GLboolean flag)
{
   Function(FP_glEdgeFlag,"GLboolean flag",&flag);
   GLV.glEdgeFlag (flag);
   Crout(FP_glEdgeFlag);
}

void APIENTRY glEdgeFlagPointer (GLsizei stride, const GLvoid *pointer)
{
   Function(FP_glEdgeFlagPointer,"GLsizei stride, const GLvoid *pointer",&stride);
   GLV.glEdgeFlagPointer (stride, pointer);
   Crout(FP_glEdgeFlagPointer);
}

void APIENTRY glEdgeFlagv (const GLboolean *flag)
{
   Function(FP_glEdgeFlagv,"const GLboolean *flag",&flag);
   GLV.glEdgeFlagv (flag);
   Crout(FP_glEdgeFlagv);
}

void APIENTRY glEnable (GLenum cap)
{
   Function(FP_glEnable,"GLenum cap",&cap);
   GLV.glEnable (cap);
   Crout(FP_glEnable);
}

void APIENTRY glEnableClientState (GLenum array)
{
   Function(FP_glEnableClientState,"GLenum array",&array);
   GLV.glEnableClientState (array);
   Crout(FP_glEnableClientState);
}

void APIENTRY glEnd ()
{
   Function(FP_glEnd,"",NULL);
   GLV.glEnd();;
   Crout(FP_glEnd);
}

void APIENTRY glEndList ()
{
   Function(FP_glEndList,"",NULL);
   GLV.glEndList();;
   Crout(FP_glEndList);
}

void APIENTRY glEvalCoord1d (GLdouble u)
{
   Function(FP_glEvalCoord1d,"GLdouble u",&u);
   GLV.glEvalCoord1d (u);
   Crout(FP_glEvalCoord1d);
}

void APIENTRY glEvalCoord1dv (const GLdouble *u)
{
   Function(FP_glEvalCoord1dv,"const GLdouble *u",&u);
   GLV.glEvalCoord1dv (u);
   Crout(FP_glEvalCoord1dv);
}

void APIENTRY glEvalCoord1f (GLfloat u)
{
   Function(FP_glEvalCoord1f,"GLfloat u",&u);
   GLV.glEvalCoord1f (u);
   Crout(FP_glEvalCoord1f);
}

void APIENTRY glEvalCoord1fv (const GLfloat *u)
{
   Function(FP_glEvalCoord1fv,"const GLfloat *u",&u);
   GLV.glEvalCoord1fv (u);
   Crout(FP_glEvalCoord1fv);
}

void APIENTRY glEvalCoord2d (GLdouble u, GLdouble v)
{
   Function(FP_glEvalCoord2d,"GLdouble u, GLdouble v",&u);
   GLV.glEvalCoord2d (u,v);
   Crout(FP_glEvalCoord2d);
}

void APIENTRY glEvalCoord2dv (const GLdouble *u)
{
   Function(FP_glEvalCoord2dv,"const GLdouble *u",&u);
   GLV.glEvalCoord2dv (u);
   Crout(FP_glEvalCoord2dv);
}

void APIENTRY glEvalCoord2f (GLfloat u, GLfloat v)
{
   Function(FP_glEvalCoord2f,"GLfloat u, GLfloat v",&u);
   GLV.glEvalCoord2f (u,v);
   Crout(FP_glEvalCoord2f);
}

void APIENTRY glEvalCoord2fv (const GLfloat *u)
{
   Function(FP_glEvalCoord2fv,"const GLfloat *u",&u);
   GLV.glEvalCoord2fv (u);
   Crout(FP_glEvalCoord2fv);
}

void APIENTRY glEvalMesh1 (GLenum mode, GLint i1, GLint i2)
{
   Function(FP_glEvalMesh1,"GLenum mode, GLint i1, GLint i2",&mode);
   GLV.glEvalMesh1 (mode, i1, i2);
   Crout(FP_glEvalMesh1);
}

void APIENTRY glEvalMesh2 (GLenum mode, GLint i1, GLint i2, GLint j1, GLint j2)
{
   Function(FP_glEvalMesh2,"GLenum mode, GLint i1, GLint i2, GLint j1, GLint j2",&mode);
   GLV.glEvalMesh2 (mode, i1, i2, j1, j2);
   Crout(FP_glEvalMesh2);
}

void APIENTRY glEvalPoint1 (GLint i)
{
   Function(FP_glEvalPoint1,"GLint i",&i);
   GLV.glEvalPoint1 (i);
   Crout(FP_glEvalPoint1);
}

void APIENTRY glEvalPoint2 (GLint i, GLint j)
{
   Function(FP_glEvalPoint2,"GLint i, GLint j",&i);
   GLV.glEvalPoint2 (i, j);
   Crout(FP_glEvalPoint2);
}

void APIENTRY glFeedbackBuffer (GLsizei size, GLenum type, GLfloat *buffer)
{
   Function(FP_glFeedbackBuffer,"GLsizei size, GLenum type, GLfloat *buffer",&size);
   GLV.glFeedbackBuffer (size, type, buffer);
   Crout(FP_glFeedbackBuffer);
}

void APIENTRY glFinish ()
{
   Function(FP_glFinish,"",NULL);
   GLV.glFinish();
   Crout(FP_glFinish);
}

void APIENTRY glFlush ()
{
   Function(FP_glFlush,"",NULL);
   GLV.glFlush();
   Crout(FP_glFlush);
}

void APIENTRY glFogf (GLenum pname, GLfloat param)
{
   Function(FP_glFogf,"GLenum pname, GLfloat param",&pname);
   GLV.glFogf (pname, param);
   Crout(FP_glFogf);
}

void APIENTRY glFogfv (GLenum pname, const GLfloat *params)
{
   Function(FP_glFogfv,"GLenum pname, const GLfloat *params",&pname);
   GLV.glFogfv (pname, params);
   Crout(FP_glFogfv);
}

void APIENTRY glFogi (GLenum pname, GLint param)
{
   Function(FP_glFogi,"GLenum pname, GLint param",&pname);
   GLV.glFogi (pname, param);
   Crout(FP_glFogi);
}

void APIENTRY glFogiv (GLenum pname, const GLint *params)
{
   Function(FP_glFogiv,"GLenum pname, const GLint *params",&pname);
   GLV.glFogiv (pname, params);
   Crout(FP_glFogiv);
}

void APIENTRY glFrontFace (GLenum mode)
{
   Function(FP_glFrontFace,"GLenum mode",&mode);
   GLV.glFrontFace (mode);
   Crout(FP_glFrontFace);
}

void APIENTRY glFrustum (GLdouble left, GLdouble right, GLdouble bottom, GLdouble top, GLdouble zNear, GLdouble zFar)
{
   Function(FP_glFrustum,"GLdouble left, GLdouble right, GLdouble bottom, GLdouble top, GLdouble zNear, GLdouble zFar",&left);
   GLV.glFrustum (left, right, bottom, top, zNear, zFar);
   Crout(FP_glFrustum);
}

GLuint APIENTRY glGenLists (GLsizei range)
{
   Function(FP_glGenLists,"GLsizei range",&range);
   GLuint result = GLV.glGenLists(range);
   Result(FP_glGenLists,"GLuint",&result);
   Crout(FP_glGenLists);
   return result;
}

void APIENTRY glGenTextures (GLsizei n, GLuint *textures)
{
   Function(FP_glGenTextures,"GLsizei n, GLuint *textures",&n);
   GLV.glGenTextures (n, textures);
   Crout(FP_glGenTextures);
}

void APIENTRY glGetBooleanv (GLenum pname, GLboolean *params)
{
   Function(FP_glGetBooleanv,"GLenum pname, GLboolean *params",&pname);
   GLV.glGetBooleanv (pname, params);
   Crout(FP_glGetBooleanv);
}

void APIENTRY glGetClipPlane (GLenum plane, GLdouble *equation)
{
   Function(FP_glGetClipPlane,"GLenum plane, GLdouble *equation",&plane);
   GLV.glGetClipPlane (plane, equation);
   Crout(FP_glGetClipPlane);
}

void APIENTRY glGetDoublev (GLenum pname, GLdouble *params)
{
   Function(FP_glGetDoublev,"GLenum pname, GLdouble *params",&pname);
   GLV.glGetDoublev (pname, params);
   Crout(FP_glGetDoublev);
}

GLenum APIENTRY glGetError ()
{
   Function(FP_glGetError,"",NULL);
   GLenum result = GLV.glGetError();
   Result(FP_glGetError,"GLenum",&result);
   Crout(FP_glGetError);
   return result;
}

void APIENTRY glGetFloatv (GLenum pname, GLfloat *params)
{
   Function(FP_glGetFloatv,"GLenum pname, GLfloat *params",&pname);
   GLV.glGetFloatv (pname, params);
   Crout(FP_glGetFloatv);
}

void APIENTRY glGetIntegerv (GLenum pname, GLint *params)
{
   Function(FP_glGetIntegerv,"GLenum pname, GLint *params",&pname);
   GLV.glGetIntegerv (pname, params);
   Crout(FP_glGetIntegerv);
}

void APIENTRY glGetLightfv (GLenum light, GLenum pname, GLfloat *params)
{
   Function(FP_glGetLightfv,"GLenum light, GLenum pname, GLfloat *params",&light);
   GLV.glGetLightfv (light, pname, params);
   Crout(FP_glGetLightfv);
}

void APIENTRY glGetLightiv (GLenum light, GLenum pname, GLint *params)
{
   Function(FP_glGetLightiv,"GLenum light, GLenum pname, GLint *params",&light);
   GLV.glGetLightiv (light, pname, params);
   Crout(FP_glGetLightiv);
}

void APIENTRY glGetMapdv (GLenum target, GLenum query, GLdouble *v)
{
   Function(FP_glGetMapdv,"GLenum target, GLenum query, GLdouble *v",&target);
   GLV.glGetMapdv (target, query, v);
   Crout(FP_glGetMapdv);
}

void APIENTRY glGetMapfv (GLenum target, GLenum query, GLfloat *v)
{
   Function(FP_glGetMapfv,"GLenum target, GLenum query, GLfloat *v",&target);
   GLV.glGetMapfv (target, query, v);
   Crout(FP_glGetMapfv);
}

void APIENTRY glGetMapiv (GLenum target, GLenum query, GLint *v)
{
   Function(FP_glGetMapiv,"GLenum target, GLenum query, GLint *v",&target);
   GLV.glGetMapiv (target, query, v);
   Crout(FP_glGetMapiv);
}

void APIENTRY glGetMaterialfv (GLenum face, GLenum pname, GLfloat *params)
{
   Function(FP_glGetMaterialfv,"GLenum face, GLenum pname, GLfloat *params",&face);
   GLV.glGetMaterialfv (face, pname, params);
   Crout(FP_glGetMaterialfv);
}

void APIENTRY glGetMaterialiv (GLenum face, GLenum pname, GLint *params)
{
   Function(FP_glGetMaterialiv,"GLenum face, GLenum pname, GLint *params",&face);
   GLV.glGetMaterialiv (face, pname, params);
   Crout(FP_glGetMaterialiv);
}

void APIENTRY glGetPixelMapfv (GLenum map, GLfloat *values)
{
   Function(FP_glGetPixelMapfv,"GLenum map, GLfloat *values",&map);
   GLV.glGetPixelMapfv (map, values);
   Crout(FP_glGetPixelMapfv);
}

void APIENTRY glGetPixelMapuiv (GLenum map, GLuint *values)
{
   Function(FP_glGetPixelMapuiv,"GLenum map, GLuint *values",&map);
   GLV.glGetPixelMapuiv (map, values);
   Crout(FP_glGetPixelMapuiv);
}

void APIENTRY glGetPixelMapusv (GLenum map, GLushort *values)
{
   Function(FP_glGetPixelMapusv,"GLenum map, GLushort *values",&map);
   GLV.glGetPixelMapusv (map, values);
   Crout(FP_glGetPixelMapusv);
}

void APIENTRY glGetPointerv (GLenum pname, GLvoid* *params)
{
   Function(FP_glGetPointerv,"GLenum pname, GLvoid* *params",&pname);
   GLV.glGetPointerv (pname, params);
   Crout(FP_glGetPointerv);
}

void APIENTRY glGetPolygonStipple (GLubyte *mask)
{
   Function(FP_glGetPolygonStipple,"GLubyte *mask",&mask);
   GLV.glGetPolygonStipple (mask);
   Crout(FP_glGetPolygonStipple);
}

const GLstring APIENTRY glGetString (GLenum name)
{
   Function(FP_glGetString,"GLenum name",&name);
   const GLstring result = GLV.glGetString(name);
   Result(FP_glGetString,"GLstring",&result);
   Crout(FP_glGetString);
   return result;
}

void APIENTRY glGetTexEnvfv (GLenum target, GLenum pname, GLfloat *params)
{
   Function(FP_glGetTexEnvfv,"GLenum target, GLenum pname, GLfloat *params",&target);
   GLV.glGetTexEnvfv (target, pname, params);
   Crout(FP_glGetTexEnvfv);
}

void APIENTRY glGetTexEnviv (GLenum target, GLenum pname, GLint *params)
{
   Function(FP_glGetTexEnviv,"GLenum target, GLenum pname, GLint *params",&target);
   GLV.glGetTexEnviv (target, pname, params);
   Crout(FP_glGetTexEnviv);
}

void APIENTRY glGetTexGendv (GLenum coord, GLenum pname, GLdouble *params)
{
   Function(FP_glGetTexGendv,"GLenum coord, GLenum pname, GLdouble *params",&coord);
   GLV.glGetTexGendv (coord, pname, params);
   Crout(FP_glGetTexGendv);
}

void APIENTRY glGetTexGenfv (GLenum coord, GLenum pname, GLfloat *params)
{
   Function(FP_glGetTexGenfv,"GLenum coord, GLenum pname, GLfloat *params",&coord);
   GLV.glGetTexGenfv (coord, pname, params);
   Crout(FP_glGetTexGenfv);
}

void APIENTRY glGetTexGeniv (GLenum coord, GLenum pname, GLint *params)
{
   Function(FP_glGetTexGeniv,"GLenum coord, GLenum pname, GLint *params",&coord);
   GLV.glGetTexGeniv (coord, pname, params);
   Crout(FP_glGetTexGeniv);
}

void APIENTRY glGetTexImage (GLenum target, GLint level, GLenum format, GLenum type, GLvoid *pixels)
{
   Function(FP_glGetTexImage,"GLenum target, GLint level, GLenum format, GLenum type, GLvoid *pixels",&target);
   GLV.glGetTexImage (target, level, format, type, pixels);
   Crout(FP_glGetTexImage);
}

void APIENTRY glGetTexLevelParameterfv (GLenum target, GLint level, GLenum pname, GLfloat *params)
{
   Function(FP_glGetTexLevelParameterfv,"GLenum target, GLint level, GLenum pname, GLfloat *params",&target);
   GLV.glGetTexLevelParameterfv (target, level, pname, params);
   Crout(FP_glGetTexLevelParameterfv);
}

void APIENTRY glGetTexLevelParameteriv (GLenum target, GLint level, GLenum pname, GLint *params)
{
   Function(FP_glGetTexLevelParameteriv,"GLenum target, GLint level, GLenum pname, GLint *params",&target);
   GLV.glGetTexLevelParameteriv (target, level, pname, params);
   Crout(FP_glGetTexLevelParameteriv);
}

void APIENTRY glGetTexParameterfv (GLenum target, GLenum pname, GLfloat *params)
{
   Function(FP_glGetTexParameterfv,"GLenum target, GLenum pname, GLfloat *params",&target);
   GLV.glGetTexParameterfv (target, pname, params);
   Crout(FP_glGetTexParameterfv);
}

void APIENTRY glGetTexParameteriv (GLenum target, GLenum pname, GLint *params)
{
   Function(FP_glGetTexParameteriv,"GLenum target, GLenum pname, GLint *params",&target);
   GLV.glGetTexParameteriv (target, pname, params);
   Crout(FP_glGetTexParameteriv);
}

void APIENTRY glHint (GLenum target, GLenum mode)
{
   Function(FP_glHint,"GLenum target, GLenum mode",&target);
   GLV.glHint (target, mode);
   Crout(FP_glHint);
}

void APIENTRY glIndexMask (GLuint mask)
{
   Function(FP_glIndexMask,"GLuint mask",&mask);
   GLV.glIndexMask (mask);
   Crout(FP_glIndexMask);
}

void APIENTRY glIndexPointer (GLenum type, GLsizei stride, const GLvoid *pointer)
{
   Function(FP_glIndexPointer,"GLenum type, GLsizei stride, const GLvoid *pointer",&type);
   GLV.glIndexPointer (type, stride, pointer);
   Crout(FP_glIndexPointer);
}

void APIENTRY glIndexd (GLdouble c)
{
   Function(FP_glIndexd,"GLdouble c",&c);
   GLV.glIndexd (c);
   Crout(FP_glIndexd);
}

void APIENTRY glIndexdv (const GLdouble *c)
{
   Function(FP_glIndexdv,"const GLdouble *c",&c);
   GLV.glIndexdv (c);
   Crout(FP_glIndexdv);
}

void APIENTRY glIndexf (GLfloat c)
{
   Function(FP_glIndexf,"GLfloat c",&c);
   GLV.glIndexf (c);
   Crout(FP_glIndexf);
}

void APIENTRY glIndexfv (const GLfloat *c)
{
   Function(FP_glIndexfv,"const GLfloat *c",&c);
   GLV.glIndexfv (c);
   Crout(FP_glIndexfv);
}

void APIENTRY glIndexi (GLint c)
{
   Function(FP_glIndexi,"GLint c",&c);
   GLV.glIndexi (c);
   Crout(FP_glIndexi);
}

void APIENTRY glIndexiv (const GLint *c)
{
   Function(FP_glIndexiv,"const GLint *c",&c);
   GLV.glIndexiv (c);
   Crout(FP_glIndexiv);
}

void APIENTRY glIndexs (GLshort c)
{
   Function(FP_glIndexs,"GLshort c",&c);
   GLV.glIndexs (c);
   Crout(FP_glIndexs);
}

void APIENTRY glIndexsv (const GLshort *c)
{
   Function(FP_glIndexsv,"const GLshort *c",&c);
   GLV.glIndexsv (c);
   Crout(FP_glIndexsv);
}

void APIENTRY glIndexub (GLubyte c)
{
   Function(FP_glIndexub,"GLubyte c",&c);
   GLV.glIndexub (c);
   Crout(FP_glIndexub);
}

void APIENTRY glIndexubv (const GLubyte *c)
{
   Function(FP_glIndexubv,"const GLubyte *c",&c);
   GLV.glIndexubv (c);
   Crout(FP_glIndexubv);
}

void APIENTRY glInitNames ()
{
   Function(FP_glInitNames,"",NULL);
   GLV.glInitNames();
   Crout(FP_glInitNames);
}

void APIENTRY glInterleavedArrays (GLenum format, GLsizei stride, const GLvoid *pointer)
{
   Function(FP_glInterleavedArrays,"GLenum format, GLsizei stride, const GLvoid *pointer",&format);
   GLV.glInterleavedArrays (format, stride, pointer);
   Crout(FP_glInterleavedArrays);
}

GLboolean APIENTRY glIsEnabled (GLenum cap)
{
   Function(FP_glIsEnabled,"GLenum cap",&cap);
   GLboolean result = GLV.glIsEnabled(cap);
   Result(FP_glIsEnabled,"GLboolean",&result);
   Crout(FP_glIsEnabled);
   return result;
}

GLboolean APIENTRY glIsList (GLuint list)
{
   Function(FP_glIsList,"GLuint list",&list);
   GLboolean result = GLV.glIsList(list);
   Result(FP_glIsList,"GLboolean",&result);
   Crout(FP_glIsList);
   return result;
}

GLboolean APIENTRY glIsTexture (GLuint texture)
{
   Function(FP_glIsTexture,"GLuint texture",&texture);
   GLboolean result = GLV.glIsTexture(texture);
   Result(FP_glIsTexture,"GLboolean",&result);
   Crout(FP_glIsTexture);
   return result;
}

void APIENTRY glLightModelf (GLenum pname, GLfloat param)
{
   Function(FP_glLightModelf,"GLenum pname, GLfloat param",&pname);
   GLV.glLightModelf (pname, param);
   Crout(FP_glLightModelf);
}

void APIENTRY glLightModelfv (GLenum pname, const GLfloat *params)
{
   Function(FP_glLightModelfv,"GLenum pname, const GLfloat *params",&pname);
   GLV.glLightModelfv (pname,params);
   Crout(FP_glLightModelfv);
}

void APIENTRY glLightModeli (GLenum pname, GLint param)
{
   Function(FP_glLightModeli,"GLenum pname, GLint param",&pname);
   GLV.glLightModeli (pname, param);
   Crout(FP_glLightModeli);
}

void APIENTRY glLightModeliv (GLenum pname, const GLint *params)
{
   Function(FP_glLightModeliv,"GLenum pname, const GLint *params",&pname);
   GLV.glLightModeliv (pname,params);
   Crout(FP_glLightModeliv);
}

void APIENTRY glLightf (GLenum light, GLenum pname, GLfloat param)
{
   Function(FP_glLightf,"GLenum light, GLenum pname, GLfloat param",&light);
   GLV.glLightf (light, pname, param);
   Crout(FP_glLightf);
}

void APIENTRY glLightfv (GLenum light, GLenum pname, const GLfloat *params)
{
   Function(FP_glLightfv,"GLenum light, GLenum pname, const GLfloat *params",&light);
   GLV.glLightfv (light, pname, params);
   Crout(FP_glLightfv);
}

void APIENTRY glLighti (GLenum light, GLenum pname, GLint param)
{
   Function(FP_glLighti,"GLenum light, GLenum pname, GLint param",&light);
   GLV.glLighti (light, pname, param);
   Crout(FP_glLighti);
}

void APIENTRY glLightiv (GLenum light, GLenum pname, const GLint *params)
{
   Function(FP_glLightiv,"GLenum light, GLenum pname, const GLint *params",&light);
   GLV.glLightiv (light, pname, params);
   Crout(FP_glLightiv);
}

void APIENTRY glLineStipple (GLint factor, GLushort pattern)
{
   Function(FP_glLineStipple,"GLint factor, GLushort pattern",&factor);
   GLV.glLineStipple (factor, pattern);
   Crout(FP_glLineStipple);
}

void APIENTRY glLineWidth (GLfloat width)
{
   Function(FP_glLineWidth,"GLfloat width",&width);
   GLV.glLineWidth (width);
   Crout(FP_glLineWidth);
}

void APIENTRY glListBase (GLuint base)
{
   Function(FP_glListBase,"GLuint base",&base);
   GLV.glListBase (base);
   Crout(FP_glListBase);
}

void APIENTRY glLoadIdentity ()
{
   Function(FP_glLoadIdentity,"",NULL);
   GLV.glLoadIdentity();
   Crout(FP_glLoadIdentity);
}

void APIENTRY glLoadMatrixd (const GLdouble *m)
{
   Function(FP_glLoadMatrixd,"const GLdouble *m",&m);
   GLV.glLoadMatrixd(m);
   Crout(FP_glLoadMatrixd);
}

void APIENTRY glLoadMatrixf (const GLfloat *m)
{
   Function(FP_glLoadMatrixf,"const GLfloat *m",&m);
   GLV.glLoadMatrixf (m);
   Crout(FP_glLoadMatrixf);
}

void APIENTRY glLoadName (GLuint name)
{
   Function(FP_glLoadName,"GLuint name",&name);
   GLV.glLoadName (name);
   Crout(FP_glLoadName);
}

void APIENTRY glLogicOp (GLenum opcode)
{
   Function(FP_glLogicOp,"GLenum opcode",&opcode);
   GLV.glLogicOp (opcode);
   Crout(FP_glLogicOp);
}

void APIENTRY glMap1d (GLenum target, GLdouble u1, GLdouble u2, GLint stride, GLint order, const GLdouble *points)
{
   Function(FP_glMap1d,"GLenum target, GLdouble u1, GLdouble u2, GLint stride, GLint order, const GLdouble *points",&target);
   GLV.glMap1d (target, u1, u2, stride, order, points);
   Crout(FP_glMap1d);
}

void APIENTRY glMap1f (GLenum target, GLfloat u1, GLfloat u2, GLint stride, GLint order, const GLfloat *points)
{
   Function(FP_glMap1f,"GLenum target, GLfloat u1, GLfloat u2, GLint stride, GLint order, const GLfloat *points",&target);
   GLV.glMap1f (target, u1, u2, stride, order, points);
   Crout(FP_glMap1f);
}

void APIENTRY glMap2d (GLenum target, GLdouble u1, GLdouble u2, GLint ustride, GLint uorder, GLdouble v1, GLdouble v2, GLint vstride, GLint vorder, const GLdouble *points)
{
   Function(FP_glMap2d,"GLenum target, GLdouble u1, GLdouble u2, GLint ustride, GLint uorder, GLdouble v1, GLdouble v2, GLint vstride, GLint vorder, const GLdouble *points",&target);
   GLV.glMap2d (target, u1, u2, ustride, uorder, v1, v2, vstride, vorder, points);
   Crout(FP_glMap2d);
}

void APIENTRY glMap2f (GLenum target, GLfloat u1, GLfloat u2, GLint ustride, GLint uorder, GLfloat v1, GLfloat v2, GLint vstride, GLint vorder, const GLfloat *points)
{
   Function(FP_glMap2f,"GLenum target, GLfloat u1, GLfloat u2, GLint ustride, GLint uorder, GLfloat v1, GLfloat v2, GLint vstride, GLint vorder, const GLfloat *points",&target);
   GLV.glMap2f (target, u1, u2, ustride, uorder, v1, v2, vstride, vorder, points);
   Crout(FP_glMap2f);
}

void APIENTRY glMapGrid1d (GLint un, GLdouble u1, GLdouble u2)
{
   Function(FP_glMapGrid1d,"GLint un, GLdouble u1, GLdouble u2",&un);
   GLV.glMapGrid1d (un, u1, u2);
   Crout(FP_glMapGrid1d);
}

void APIENTRY glMapGrid1f (GLint un, GLfloat u1, GLfloat u2)
{
   Function(FP_glMapGrid1f,"GLint un, GLfloat u1, GLfloat u2",&un);
   GLV.glMapGrid1f (un, u1, u2);
   Crout(FP_glMapGrid1f);
}

void APIENTRY glMapGrid2d (GLint un, GLdouble u1, GLdouble u2, GLint vn, GLdouble v1, GLdouble v2)
{
   Function(FP_glMapGrid2d,"GLint un, GLdouble u1, GLdouble u2, GLint vn, GLdouble v1, GLdouble v2",&un);
   GLV.glMapGrid2d (un, u1, u2, vn, v1, v2);
   Crout(FP_glMapGrid2d);
}

void APIENTRY glMapGrid2f (GLint un, GLfloat u1, GLfloat u2, GLint vn, GLfloat v1, GLfloat v2)
{
   Function(FP_glMapGrid2f,"GLint un, GLfloat u1, GLfloat u2, GLint vn, GLfloat v1, GLfloat v2",&un);
   GLV.glMapGrid2f (un, u1, u2, vn, v1, v2);
   Crout(FP_glMapGrid2f);
}

void APIENTRY glMaterialf (GLenum face, GLenum pname, GLfloat param)
{
   Function(FP_glMaterialf,"GLenum face, GLenum pname, GLfloat param",&face);
   GLV.glMaterialf (face, pname, param);
   Crout(FP_glMaterialf);
}

void APIENTRY glMaterialfv (GLenum face, GLenum pname, const GLfloat *params)
{
   Function(FP_glMaterialfv,"GLenum face, GLenum pname, const GLfloat *params",&face);
   GLV.glMaterialfv (face, pname, params);
   Crout(FP_glMaterialfv);
}

void APIENTRY glMateriali (GLenum face, GLenum pname, GLint param)
{
   Function(FP_glMateriali,"GLenum face, GLenum pname, GLint param",&face);
   GLV.glMateriali (face, pname, param);
   Crout(FP_glMateriali);
}

void APIENTRY glMaterialiv (GLenum face, GLenum pname, const GLint *params)
{
   Function(FP_glMaterialiv,"GLenum face, GLenum pname, const GLint *params",&face);
   GLV.glMaterialiv (face, pname, params);
   Crout(FP_glMaterialiv);
}

void APIENTRY glMatrixMode (GLenum mode)
{
   Function(FP_glMatrixMode,"GLenum mode",&mode);
   GLV.glMatrixMode (mode);
   Crout(FP_glMatrixMode);
}

void APIENTRY glMultMatrixd (const GLdouble *m)
{
   Function(FP_glMultMatrixd,"const GLdouble *m",&m);
   GLV.glMultMatrixd (m);
   Crout(FP_glMultMatrixd);
}

void APIENTRY glMultMatrixf (const GLfloat *m)
{
   Function(FP_glMultMatrixf,"const GLfloat *m",&m);
   GLV.glMultMatrixf (m);
   Crout(FP_glMultMatrixf);
}

void APIENTRY glNewList (GLuint list, GLenum mode)
{
   Function(FP_glNewList,"GLuint list, GLenum mode",&mode);
   GLV.glNewList (list,mode);
   Crout(FP_glNewList);
}

void APIENTRY glNormal3b (GLbyte nx, GLbyte ny, GLbyte nz)
{
   Function(FP_glNormal3b,"GLbyte nx, GLbyte ny, GLbyte nz",&nx);
   GLV.glNormal3b (nx, ny, nz);
   Crout(FP_glNormal3b);
}

void APIENTRY glNormal3bv (const GLbyte *v)
{
   Function(FP_glNormal3bv,"const GLbyte *v",&v);
   GLV.glNormal3bv (v);
   Crout(FP_glNormal3bv);
}

void APIENTRY glNormal3d (GLdouble nx, GLdouble ny, GLdouble nz)
{
   Function(FP_glNormal3d,"GLdouble nx, GLdouble ny, GLdouble nz",&nx);
   GLV.glNormal3d (nx, ny, nz);
   Crout(FP_glNormal3d);
}

void APIENTRY glNormal3dv (const GLdouble *v)
{
   Function(FP_glNormal3dv,"const GLdouble *v",&v);
   GLV.glNormal3dv (v);
   Crout(FP_glNormal3dv);
}

void APIENTRY glNormal3f (GLfloat nx, GLfloat ny, GLfloat nz)
{
   Function(FP_glNormal3f,"GLfloat nx, GLfloat ny, GLfloat nz",&nx);
   GLV.glNormal3f (nx, ny, nz);
   Crout(FP_glNormal3f);
}

void APIENTRY glNormal3fv (const GLfloat *v)
{
   Function(FP_glNormal3fv,"const GLfloat *v",&v);
   GLV.glNormal3fv (v);
   Crout(FP_glNormal3fv);
}

void APIENTRY glNormal3i (GLint nx, GLint ny, GLint nz)
{
   Function(FP_glNormal3i,"GLint nx, GLint ny, GLint nz",&nx);
   GLV.glNormal3i (nx, ny, nz);
   Crout(FP_glNormal3i);
}

void APIENTRY glNormal3iv (const GLint *v)
{
   Function(FP_glNormal3iv,"const GLint *v",&v);
   GLV.glNormal3iv (v);
   Crout(FP_glNormal3iv);
}

void APIENTRY glNormal3s (GLshort nx, GLshort ny, GLshort nz)
{
   Function(FP_glNormal3s,"GLshort nx, GLshort ny, GLshort nz",&nx);
   GLV.glNormal3s (nx, ny, nz);
   Crout(FP_glNormal3s);
}

void APIENTRY glNormal3sv (const GLshort *v)
{
   Function(FP_glNormal3sv,"const GLshort *v",&v);
   GLV.glNormal3sv (v);
   Crout(FP_glNormal3sv);
}

void APIENTRY glNormalPointer (GLenum type, GLsizei stride, const GLvoid *pointer)
{
   Function(FP_glNormalPointer,"GLenum type, GLsizei stride, const GLvoid *pointer",&type);
   GLV.glNormalPointer (type, stride, pointer);
   Crout(FP_glNormalPointer);
}

void APIENTRY glOrtho (GLdouble left, GLdouble right, GLdouble bottom, GLdouble top, GLdouble zNear, GLdouble zFar)
{
   Function(FP_glOrtho,"GLdouble left, GLdouble right, GLdouble bottom, GLdouble top, GLdouble zNear, GLdouble zFar",&left);
   GLV.glOrtho (left, right, bottom, top, zNear, zFar);
   Crout(FP_glOrtho);
}

void APIENTRY glPassThrough (GLfloat token)
{
   Function(FP_glPassThrough,"GLfloat token",&token);
   GLV.glPassThrough (token);
   Crout(FP_glPassThrough);
}

void APIENTRY glPixelMapfv (GLenum map, GLsizei mapsize, const GLfloat *values)
{
   Function(FP_glPixelMapfv,"GLenum map, GLsizei mapsize, const GLfloat *values",&map);
   GLV.glPixelMapfv (map, mapsize, values);
   Crout(FP_glPixelMapfv);
}

void APIENTRY glPixelMapuiv (GLenum map, GLsizei mapsize, const GLuint *values)
{
   Function(FP_glPixelMapuiv,"GLenum map, GLsizei mapsize, const GLuint *values",&map);
   GLV.glPixelMapuiv (map, mapsize, values);
   Crout(FP_glPixelMapuiv);
}

void APIENTRY glPixelMapusv (GLenum map, GLsizei mapsize, const GLushort *values)
{
   Function(FP_glPixelMapusv,"GLenum map, GLsizei mapsize, const GLushort *values",&map);
   GLV.glPixelMapusv (map, mapsize, values);
   Crout(FP_glPixelMapusv);
}

void APIENTRY glPixelStoref (GLenum pname, GLfloat param)
{
   Function(FP_glPixelStoref,"GLenum pname, GLfloat param",&pname);
   GLV.glPixelStoref (pname, param);
   Crout(FP_glPixelStoref);
}

void APIENTRY glPixelStorei (GLenum pname, GLint param)
{
   Function(FP_glPixelStorei,"GLenum pname, GLint param",&pname);
   GLV.glPixelStorei (pname, param);
   Crout(FP_glPixelStorei);
}

void APIENTRY glPixelTransferf (GLenum pname, GLfloat param)
{
   Function(FP_glPixelTransferf,"GLenum pname, GLfloat param",&pname);
   GLV.glPixelTransferf (pname, param);
   Crout(FP_glPixelTransferf);
}

void APIENTRY glPixelTransferi (GLenum pname, GLint param)
{
   Function(FP_glPixelTransferi,"GLenum pname, GLint param",&pname);
   GLV.glPixelTransferi (pname, param);
   Crout(FP_glPixelTransferi);
}

void APIENTRY glPixelZoom (GLfloat xfactor, GLfloat yfactor)
{
   Function(FP_glPixelZoom,"GLfloat xfactor, GLfloat yfactor",&xfactor);
   GLV.glPixelZoom (xfactor, yfactor);
   Crout(FP_glPixelZoom);
}

void APIENTRY glPointSize (GLfloat size)
{
   Function(FP_glPointSize,"GLfloat size",&size);
   GLV.glPointSize (size);
   Crout(FP_glPointSize);
}

void APIENTRY glPolygonMode (GLenum face, GLenum mode)
{
   Function(FP_glPolygonMode,"GLenum face, GLenum mode",&face);
   GLV.glPolygonMode (face, mode);
   Crout(FP_glPolygonMode);
}

void APIENTRY glPolygonOffset (GLfloat factor, GLfloat units)
{
   Function(FP_glPolygonOffset,"GLfloat factor, GLfloat units",&factor);
   GLV.glPolygonOffset (factor, units);
   Crout(FP_glPolygonOffset);
}

void APIENTRY glPolygonStipple (const GLubyte *mask)
{
   Function(FP_glPolygonStipple,"const GLubyte *mask",&mask);
   GLV.glPolygonStipple (mask);
   Crout(FP_glPolygonStipple);
}

void APIENTRY glPopAttrib ()
{
   Function(FP_glPopAttrib,"",NULL);
   GLV.glPopAttrib();
   Crout(FP_glPopAttrib);
}

void APIENTRY glPopClientAttrib ()
{
   Function(FP_glPopClientAttrib,"",NULL);
   GLV.glPopClientAttrib();
   Crout(FP_glPopClientAttrib);
}

void APIENTRY glPopMatrix ()
{
   Function(FP_glPopMatrix,"",NULL);
   GLV.glPopMatrix();
   Crout(FP_glPopMatrix);
}

void APIENTRY glPopName ()
{
   Function(FP_glPopName,"",NULL);
   GLV.glPopName();
   Crout(FP_glPopName);
}

void APIENTRY glPrioritizeTextures (GLsizei n, const GLuint *textures, const GLclampf *priorities)
{
   Function(FP_glPrioritizeTextures,"GLsizei n, const GLuint *textures, const GLclampf *priorities",&n);
   GLV.glPrioritizeTextures (n, textures, priorities);
   Crout(FP_glPrioritizeTextures);
}

void APIENTRY glPushAttrib (GLbitfield mask)
{
   Function(FP_glPushAttrib,"GLbitfield mask",&mask);
   GLV.glPushAttrib (mask);
   Crout(FP_glPushAttrib);
}

void APIENTRY glPushClientAttrib (GLbitfield mask)
{
   Function(FP_glPushClientAttrib,"GLbitfield mask",&mask);
   GLV.glPushClientAttrib (mask);
   Crout(FP_glPushClientAttrib);
}

void APIENTRY glPushMatrix ()
{
   Function(FP_glPushMatrix,"",NULL);
   GLV.glPushMatrix();
   Crout(FP_glPushMatrix);
}

void APIENTRY glPushName (GLuint name)
{
   Function(FP_glPushName,"GLuint name",&name);
   GLV.glPushName (name);
   Crout(FP_glPushName);
}

void APIENTRY glRasterPos2d (GLdouble x, GLdouble y)
{
   Function(FP_glRasterPos2d,"GLdouble x, GLdouble y",&x);
   GLV.glRasterPos2d (x, y);
   Crout(FP_glRasterPos2d);
}

void APIENTRY glRasterPos2dv (const GLdouble *v)
{
   Function(FP_glRasterPos2dv,"const GLdouble *v",&v);
   GLV.glRasterPos2dv (v);
   Crout(FP_glRasterPos2dv);
}

void APIENTRY glRasterPos2f (GLfloat x, GLfloat y)
{
   Function(FP_glRasterPos2f,"GLfloat x, GLfloat y",&x);
   GLV.glRasterPos2f (x, y);
   Crout(FP_glRasterPos2f);
}

void APIENTRY glRasterPos2fv (const GLfloat *v)
{
   Function(FP_glRasterPos2fv,"const GLfloat *v",&v);
   GLV.glRasterPos2fv (v);
   Crout(FP_glRasterPos2fv);
}

void APIENTRY glRasterPos2i (GLint x, GLint y)
{
   Function(FP_glRasterPos2i,"GLint x, GLint y",&x);
   GLV.glRasterPos2i (x, y);
   Crout(FP_glRasterPos2i);
}

void APIENTRY glRasterPos2iv (const GLint *v)
{
   Function(FP_glRasterPos2iv,"const GLint *v",&v);
   GLV.glRasterPos2iv (v);
   Crout(FP_glRasterPos2iv);
}

void APIENTRY glRasterPos2s (GLshort x, GLshort y)
{
   Function(FP_glRasterPos2s,"GLshort x, GLshort y",&x);
   GLV.glRasterPos2s (x, y);
   Crout(FP_glRasterPos2s);
}

void APIENTRY glRasterPos2sv (const GLshort *v)
{
   Function(FP_glRasterPos2sv,"const GLshort *v",&v);
   GLV.glRasterPos2sv (v);
   Crout(FP_glRasterPos2sv);
}

void APIENTRY glRasterPos3d (GLdouble x, GLdouble y, GLdouble z)
{
   Function(FP_glRasterPos3d,"GLdouble x, GLdouble y, GLdouble z",&x);
   GLV.glRasterPos3d (x, y, z);
   Crout(FP_glRasterPos3d);
}

void APIENTRY glRasterPos3dv (const GLdouble *v)
{
   Function(FP_glRasterPos3dv,"const GLdouble *v",&v);
   GLV.glRasterPos3dv (v);
   Crout(FP_glRasterPos3dv);
}

void APIENTRY glRasterPos3f (GLfloat x, GLfloat y, GLfloat z)
{
   Function(FP_glRasterPos3f,"GLfloat x, GLfloat y, GLfloat z",&x);
   GLV.glRasterPos3f (x, y, z);
   Crout(FP_glRasterPos3f);
}

void APIENTRY glRasterPos3fv (const GLfloat *v)
{
   Function(FP_glRasterPos3fv,"const GLfloat *v",&v);
   GLV.glRasterPos3fv (v);
   Crout(FP_glRasterPos3fv);
}

void APIENTRY glRasterPos3i (GLint x, GLint y, GLint z)
{
   Function(FP_glRasterPos3i,"GLint x, GLint y, GLint z",&x);
   GLV.glRasterPos3i (x, y, z);
   Crout(FP_glRasterPos3i);
}

void APIENTRY glRasterPos3iv (const GLint *v)
{
   Function(FP_glRasterPos3iv,"const GLint *v",&v);
   GLV.glRasterPos3iv (v);
   Crout(FP_glRasterPos3iv);
}

void APIENTRY glRasterPos3s (GLshort x, GLshort y, GLshort z)
{
   Function(FP_glRasterPos3s,"GLshort x, GLshort y, GLshort z",&x);
   GLV.glRasterPos3s (x, y, z);
   Crout(FP_glRasterPos3s);
}

void APIENTRY glRasterPos3sv (const GLshort *v)
{
   Function(FP_glRasterPos3sv,"const GLshort *v",&v);
   GLV.glRasterPos3sv (v);
   Crout(FP_glRasterPos3sv);
}

void APIENTRY glRasterPos4d (GLdouble x, GLdouble y, GLdouble z, GLdouble w)
{
   Function(FP_glRasterPos4d,"GLdouble x, GLdouble y, GLdouble z, GLdouble w",&x);
   GLV.glRasterPos4d (x, y, z, w);
   Crout(FP_glRasterPos4d);
}

void APIENTRY glRasterPos4dv (const GLdouble *v)
{
   Function(FP_glRasterPos4dv,"const GLdouble *v",&v);
   GLV.glRasterPos4dv (v);
   Crout(FP_glRasterPos4dv);
}

void APIENTRY glRasterPos4f (GLfloat x, GLfloat y, GLfloat z, GLfloat w)
{
   Function(FP_glRasterPos4f,"GLfloat x, GLfloat y, GLfloat z, GLfloat w",&x);
   GLV.glRasterPos4f (x, y, z, w);
   Crout(FP_glRasterPos4f);
}

void APIENTRY glRasterPos4fv (const GLfloat *v)
{
   Function(FP_glRasterPos4fv,"const GLfloat *v",&v);
   GLV.glRasterPos4fv (v);
   Crout(FP_glRasterPos4fv);
}

void APIENTRY glRasterPos4i (GLint x, GLint y, GLint z, GLint w)
{
   Function(FP_glRasterPos4i,"GLint x, GLint y, GLint z, GLint w",&x);
   GLV.glRasterPos4i (x, y, z, w);
   Crout(FP_glRasterPos4i);
}

void APIENTRY glRasterPos4iv (const GLint *v)
{
   Function(FP_glRasterPos4iv,"const GLint *v",&v);
   GLV.glRasterPos4iv (v);
   Crout(FP_glRasterPos4iv);
}

void APIENTRY glRasterPos4s (GLshort x, GLshort y, GLshort z, GLshort w)
{
   Function(FP_glRasterPos4s,"GLshort x, GLshort y, GLshort z, GLshort w",&x);
   GLV.glRasterPos4s (x, y, z, w);
   Crout(FP_glRasterPos4s);
}

void APIENTRY glRasterPos4sv (const GLshort *v)
{
   Function(FP_glRasterPos4sv,"const GLshort *v",&v);
   GLV.glRasterPos4sv (v);
   Crout(FP_glRasterPos4sv);
}

void APIENTRY glReadBuffer (GLenum mode)
{
   Function(FP_glReadBuffer,"GLenum mode",&mode);
   GLV.glReadBuffer (mode);
   Crout(FP_glReadBuffer);
}

void APIENTRY glReadPixels (GLint x, GLint y, GLsizei width, GLsizei height, GLenum format, GLenum type, GLvoid *pixels)
{
   Function(FP_glReadPixels,"GLint x, GLint y, GLsizei width, GLsizei height, GLenum format, GLenum type, GLvoid *pixels",&x);
   GLV.glReadPixels (x, y, width, height, format, type, pixels);
   Crout(FP_glReadPixels);
}

void APIENTRY glRectd (GLdouble x1, GLdouble y1, GLdouble x2, GLdouble y2)
{
   Function(FP_glRectd,"GLdouble x1, GLdouble y1, GLdouble x2, GLdouble y2",&x1);
   GLV.glRectd (x1, y1, x2, y2);
   Crout(FP_glRectd);
}

void APIENTRY glRectdv (const GLdouble *v1, const GLdouble *v2)
{
   Function(FP_glRectdv,"const GLdouble *v1, const GLdouble *v2",&v1);
   GLV.glRectdv (v1, v2);
   Crout(FP_glRectdv);
}

void APIENTRY glRectf (GLfloat x1, GLfloat y1, GLfloat x2, GLfloat y2)
{
   Function(FP_glRectf,"GLfloat x1, GLfloat y1, GLfloat x2, GLfloat y2",&x1);
   GLV.glRectf (x1, y1, x2, y2);
   Crout(FP_glRectf);
}

void APIENTRY glRectfv (const GLfloat *v1, const GLfloat *v2)
{
   Function(FP_glRectfv,"const GLfloat *v1, const GLfloat *v2",&v1);
   GLV.glRectfv (v1, v2);
   Crout(FP_glRectfv);
}

void APIENTRY glRecti (GLint x1, GLint y1, GLint x2, GLint y2)
{
   Function(FP_glRecti,"GLint x1, GLint y1, GLint x2, GLint y2",&x1);
   GLV.glRecti (x1, y1, x2, y2);
   Crout(FP_glRecti);
}

void APIENTRY glRectiv (const GLint *v1, const GLint *v2)
{
   Function(FP_glRectiv,"const GLint *v1, const GLint *v2",&v1);
   GLV.glRectiv (v1, v2);
   Crout(FP_glRectiv);
}

void APIENTRY glRects (GLshort x1, GLshort y1, GLshort x2, GLshort y2)
{
   Function(FP_glRects,"GLshort x1, GLshort y1, GLshort x2, GLshort y2",&x1);
   GLV.glRects (x1, y1, x2, y2);
   Crout(FP_glRects);
}

void APIENTRY glRectsv (const GLshort *v1, const GLshort *v2)
{
   Function(FP_glRectsv,"const GLshort *v1, const GLshort *v2",&v1);
   GLV.glRectsv (v1, v2);
   Crout(FP_glRectsv);
}

GLint APIENTRY glRenderMode (GLenum mode)
{
   Function(FP_glRenderMode,"GLenum mode",&mode);
   GLint result = GLV.glRenderMode(mode);
   Result(FP_glRenderMode,"GLint",&result);
   Crout(FP_glRenderMode);
   return result;
}

void APIENTRY glRotated (GLdouble angle, GLdouble x, GLdouble y, GLdouble z)
{
   Function(FP_glRotated,"GLdouble angle, GLdouble x, GLdouble y, GLdouble z",&angle);
   GLV.glRotated (angle, x, y, z);
   Crout(FP_glRotated);
}

void APIENTRY glRotatef (GLfloat angle, GLfloat x, GLfloat y, GLfloat z)
{
   Function(FP_glRotatef,"GLfloat angle, GLfloat x, GLfloat y, GLfloat z",&angle);
   GLV.glRotatef (angle, x, y, z);
   Crout(FP_glRotatef);
}

void APIENTRY glScaled (GLdouble x, GLdouble y, GLdouble z)
{
   Function(FP_glScaled,"GLdouble x, GLdouble y, GLdouble z",&x);
   GLV.glScaled (x, y, z);
   Crout(FP_glScaled);
}

void APIENTRY glScalef (GLfloat x, GLfloat y, GLfloat z)
{
   Function(FP_glScalef,"GLfloat x, GLfloat y, GLfloat z",&x);
   GLV.glScalef (x, y, z);
   Crout(FP_glScalef);
}

void APIENTRY glScissor (GLint x, GLint y, GLsizei width, GLsizei height)
{
   Function(FP_glScissor,"GLint x, GLint y, GLsizei width, GLsizei height",&x);
   GLV.glScissor (x, y, width, height);
   Crout(FP_glScissor);
}

void APIENTRY glSelectBuffer (GLsizei size, GLuint *buffer)
{
   Function(FP_glSelectBuffer,"GLsizei size, GLuint *buffer",&size);
   GLV.glSelectBuffer (size, buffer);
   Crout(FP_glSelectBuffer);
}

void APIENTRY glShadeModel (GLenum mode)
{
   Function(FP_glShadeModel,"GLenum mode",&mode);
   GLV.glShadeModel (mode);
   Crout(FP_glShadeModel);
}

void APIENTRY glStencilFunc (GLenum func, GLint ref, GLuint mask)
{
   Function(FP_glStencilFunc,"GLenum func, GLint ref, GLuint mask",&func);
   GLV.glStencilFunc (func, ref, mask);
   Crout(FP_glStencilFunc);
}

void APIENTRY glStencilMask (GLuint mask)
{
   Function(FP_glStencilMask,"GLuint mask",&mask);
   GLV.glStencilMask (mask);
   Crout(FP_glStencilMask);
}

void APIENTRY glStencilOp (GLenum fail, GLenum zfail, GLenum zpass)
{
   Function(FP_glStencilOp,"GLenum fail, GLenum zfail, GLenum zpass",&fail);
   GLV.glStencilOp (fail, zfail, zpass);
   Crout(FP_glStencilOp);
}

void APIENTRY glTexCoord1d (GLdouble s)
{
   Function(FP_glTexCoord1d,"GLdouble s",&s);
   GLV.glTexCoord1d (s);
   Crout(FP_glTexCoord1d);
}

void APIENTRY glTexCoord1dv (const GLdouble *v)
{
   Function(FP_glTexCoord1dv,"const GLdouble *v",&v);
   GLV.glTexCoord1dv (v);
   Crout(FP_glTexCoord1dv);
}

void APIENTRY glTexCoord1f (GLfloat s)
{
   Function(FP_glTexCoord1f,"GLfloat s",&s);
   GLV.glTexCoord1f (s);
   Crout(FP_glTexCoord1f);
}

void APIENTRY glTexCoord1fv (const GLfloat *v)
{
   Function(FP_glTexCoord1fv,"const GLfloat *v",&v);
   GLV.glTexCoord1fv (v);
   Crout(FP_glTexCoord1fv);
}

void APIENTRY glTexCoord1i (GLint s)
{
   Function(FP_glTexCoord1i,"GLint s",&s);
   GLV.glTexCoord1i (s);
   Crout(FP_glTexCoord1i);
}

void APIENTRY glTexCoord1iv (const GLint *v)
{
   Function(FP_glTexCoord1iv,"const GLint *v",&v);
   GLV.glTexCoord1iv (v);
   Crout(FP_glTexCoord1iv);
}

void APIENTRY glTexCoord1s (GLshort s)
{
   Function(FP_glTexCoord1s,"GLshort s",&s);
   GLV.glTexCoord1s (s);
   Crout(FP_glTexCoord1s);
}

void APIENTRY glTexCoord1sv (const GLshort *v)
{
   Function(FP_glTexCoord1sv,"const GLshort *v",&v);
   GLV.glTexCoord1sv (v);
   Crout(FP_glTexCoord1sv);
}

void APIENTRY glTexCoord2d (GLdouble s, GLdouble t)
{
   Function(FP_glTexCoord2d,"GLdouble s, GLdouble t",&s);
   GLV.glTexCoord2d (s, t);
   Crout(FP_glTexCoord2d);
}

void APIENTRY glTexCoord2dv (const GLdouble *v)
{
   Function(FP_glTexCoord2dv,"const GLdouble *v",&v);
   GLV.glTexCoord2dv (v);
   Crout(FP_glTexCoord2dv);
}

void APIENTRY glTexCoord2f (GLfloat s, GLfloat t)
{
   Function(FP_glTexCoord2f,"GLfloat s, GLfloat t",&s);
   GLV.glTexCoord2f (s, t);
   Crout(FP_glTexCoord2f);
}

void APIENTRY glTexCoord2fv (const GLfloat *v)
{
   Function(FP_glTexCoord2fv,"const GLfloat *v",&v);
   GLV.glTexCoord2fv (v);
   Crout(FP_glTexCoord2fv);
}

void APIENTRY glTexCoord2i (GLint s, GLint t)
{
   Function(FP_glTexCoord2i,"GLint s, GLint t",&s);
   GLV.glTexCoord2i (s, t);
   Crout(FP_glTexCoord2i);
}

void APIENTRY glTexCoord2iv (const GLint *v)
{
   Function(FP_glTexCoord2iv,"const GLint *v",&v);
   GLV.glTexCoord2iv (v);
   Crout(FP_glTexCoord2iv);
}

void APIENTRY glTexCoord2s (GLshort s, GLshort t)
{
   Function(FP_glTexCoord2s,"GLshort s, GLshort t",&s);
   GLV.glTexCoord2s (s, t);
   Crout(FP_glTexCoord2s);
}

void APIENTRY glTexCoord2sv (const GLshort *v)
{
   Function(FP_glTexCoord2sv,"const GLshort *v",&v);
   GLV.glTexCoord2sv (v);
   Crout(FP_glTexCoord2sv);
}

void APIENTRY glTexCoord3d (GLdouble s, GLdouble t, GLdouble r)
{
   Function(FP_glTexCoord3d,"GLdouble s, GLdouble t, GLdouble r",&s);
   GLV.glTexCoord3d (s, t, r);
   Crout(FP_glTexCoord3d);
}

void APIENTRY glTexCoord3dv (const GLdouble *v)
{
   Function(FP_glTexCoord3dv,"const GLdouble *v",&v);
   GLV.glTexCoord3dv (v);
   Crout(FP_glTexCoord3dv);
}

void APIENTRY glTexCoord3f (GLfloat s, GLfloat t, GLfloat r)
{
   Function(FP_glTexCoord3f,"GLfloat s, GLfloat t, GLfloat r",&s);
   GLV.glTexCoord3f (s, t, r);
   Crout(FP_glTexCoord3f);
}

void APIENTRY glTexCoord3fv (const GLfloat *v)
{
   Function(FP_glTexCoord3fv,"const GLfloat *v",&v);
   GLV.glTexCoord3fv (v);
   Crout(FP_glTexCoord3fv);
}

void APIENTRY glTexCoord3i (GLint s, GLint t, GLint r)
{
   Function(FP_glTexCoord3i,"GLint s, GLint t, GLint r",&s);
   GLV.glTexCoord3i (s, t, r);
   Crout(FP_glTexCoord3i);
}

void APIENTRY glTexCoord3iv (const GLint *v)
{
   Function(FP_glTexCoord3iv,"const GLint *v",&v);
   GLV.glTexCoord3iv (v);
   Crout(FP_glTexCoord3iv);
}

void APIENTRY glTexCoord3s (GLshort s, GLshort t, GLshort r)
{
   Function(FP_glTexCoord3s,"GLshort s, GLshort t, GLshort r",&s);
   GLV.glTexCoord3s (s, t, r);
   Crout(FP_glTexCoord3s);
}

void APIENTRY glTexCoord3sv (const GLshort *v)
{
   Function(FP_glTexCoord3sv,"const GLshort *v",&v);
   GLV.glTexCoord3sv (v);
   Crout(FP_glTexCoord3sv);
}

void APIENTRY glTexCoord4d (GLdouble s, GLdouble t, GLdouble r, GLdouble q)
{
   Function(FP_glTexCoord4d,"GLdouble s, GLdouble t, GLdouble r, GLdouble q",&s);
   GLV.glTexCoord4d (s, t, r, q);
   Crout(FP_glTexCoord4d);
}

void APIENTRY glTexCoord4dv (const GLdouble *v)
{
   Function(FP_glTexCoord4dv,"const GLdouble *v",&v);
   GLV.glTexCoord4dv (v);
   Crout(FP_glTexCoord4dv);
}

void APIENTRY glTexCoord4f (GLfloat s, GLfloat t, GLfloat r, GLfloat q)
{
   Function(FP_glTexCoord4f,"GLfloat s, GLfloat t, GLfloat r, GLfloat q",&s);
   GLV.glTexCoord4f (s, t, r, q);
   Crout(FP_glTexCoord4f);
}

void APIENTRY glTexCoord4fv (const GLfloat *v)
{
   Function(FP_glTexCoord4fv,"const GLfloat *v",&v);
   GLV.glTexCoord4fv (v);
   Crout(FP_glTexCoord4fv);
}

void APIENTRY glTexCoord4i (GLint s, GLint t, GLint r, GLint q)
{
   Function(FP_glTexCoord4i,"GLint s, GLint t, GLint r, GLint q",&s);
   GLV.glTexCoord4i (s, t, r, q);
   Crout(FP_glTexCoord4i);
}

void APIENTRY glTexCoord4iv (const GLint *v)
{
   Function(FP_glTexCoord4iv,"const GLint *v",&v);
   GLV.glTexCoord4iv (v);
   Crout(FP_glTexCoord4iv);
}

void APIENTRY glTexCoord4s (GLshort s, GLshort t, GLshort r, GLshort q)
{
   Function(FP_glTexCoord4s,"GLshort s, GLshort t, GLshort r, GLshort q",&s);
   GLV.glTexCoord4s (s, t, r, q);
   Crout(FP_glTexCoord4s);
}

void APIENTRY glTexCoord4sv (const GLshort *v)
{
   Function(FP_glTexCoord4sv,"const GLshort *v",&v);
   GLV.glTexCoord4sv (v);
   Crout(FP_glTexCoord4sv);
}

void APIENTRY glTexCoordPointer (GLint size, GLenum type, GLsizei stride, const GLvoid *pointer)
{
   Function(FP_glTexCoordPointer,"GLint size, GLenum type, GLsizei stride, const GLvoid *pointer",&size);
   GLV.glTexCoordPointer (size, type, stride, pointer);
   Crout(FP_glTexCoordPointer);
}

void APIENTRY glTexEnvf (GLenum target, GLenum pname, GLfloat param)
{
   Function(FP_glTexEnvf,"GLenum target, GLenum pname, GLfloat param",&target);
   GLV.glTexEnvf (target, pname, param);
   Crout(FP_glTexEnvf);
}

void APIENTRY glTexEnvfv (GLenum target, GLenum pname, const GLfloat *params)
{
   Function(FP_glTexEnvfv,"GLenum target, GLenum pname, const GLfloat *params",&target);
   GLV.glTexEnvfv (target, pname, params);
   Crout(FP_glTexEnvfv);
}

void APIENTRY glTexEnvi (GLenum target, GLenum pname, GLint param)
{
   Function(FP_glTexEnvi,"GLenum target, GLenum pname, GLint param",&target);
   GLV.glTexEnvi (target, pname, param);
   Crout(FP_glTexEnvi);
}

void APIENTRY glTexEnviv (GLenum target, GLenum pname, const GLint *params)
{
   Function(FP_glTexEnviv,"GLenum target, GLenum pname, const GLint *params",&target);
   GLV.glTexEnviv (target, pname, params);
   Crout(FP_glTexEnviv);
}

void APIENTRY glTexGend (GLenum coord, GLenum pname, GLdouble param)
{
   Function(FP_glTexGend,"GLenum coord, GLenum pname, GLdouble param",&coord);
   GLV.glTexGend (coord, pname, param);
   Crout(FP_glTexGend);
}

void APIENTRY glTexGendv (GLenum coord, GLenum pname, const GLdouble *params)
{
   Function(FP_glTexGendv,"GLenum coord, GLenum pname, const GLdouble *params",&coord);
   GLV.glTexGendv (coord, pname, params);
   Crout(FP_glTexGendv);
}

void APIENTRY glTexGenf (GLenum coord, GLenum pname, GLfloat param)
{
   Function(FP_glTexGenf,"GLenum coord, GLenum pname, GLfloat param",&coord);
   GLV.glTexGenf (coord, pname, param);
   Crout(FP_glTexGenf);
}

void APIENTRY glTexGenfv (GLenum coord, GLenum pname, const GLfloat *params)
{
   Function(FP_glTexGenfv,"GLenum coord, GLenum pname, const GLfloat *params",&coord);
   GLV.glTexGenfv (coord, pname, params);
   Crout(FP_glTexGenfv);
}

void APIENTRY glTexGeni (GLenum coord, GLenum pname, GLint param)
{
   Function(FP_glTexGeni,"GLenum coord, GLenum pname, GLint param",&coord);
   GLV.glTexGeni (coord, pname, param);
   Crout(FP_glTexGeni);
}

void APIENTRY glTexGeniv (GLenum coord, GLenum pname, const GLint *params)
{
   Function(FP_glTexGeniv,"GLenum coord, GLenum pname, const GLint *params",&coord);
   GLV.glTexGeniv (coord, pname, params);
   Crout(FP_glTexGeniv);
}

void APIENTRY glTexImage1D (GLenum target, GLint level, GLint internalformat, GLsizei width, GLint border, GLenum format, GLenum type, const GLvoid *pixels)
{
   Function(FP_glTexImage1D,"GLenum target, GLint level, GLint internalformat, GLsizei width, GLint border, GLenum format, GLenum type, const GLvoid *pixels",&target);
   GLV.glTexImage1D (target, level, internalformat, width, border, format, type, pixels);
   Crout(FP_glTexImage1D);
}

void APIENTRY glTexImage2D (GLenum target, GLint level, GLint internalformat, GLsizei width, GLsizei height, GLint border, GLenum format, GLenum type, const GLvoid *pixels)
{
   Function(FP_glTexImage2D,"GLenum target, GLint level, GLint internalformat, GLsizei width, GLsizei height, GLint border, GLenum format, GLenum type, const GLvoid *pixels",&target);
   GLV.glTexImage2D (target, level, internalformat, width, height, border, format, type, pixels);
   Crout(FP_glTexImage2D);
}

void APIENTRY glTexParameterf (GLenum target, GLenum pname, GLfloat param)
{
   Function(FP_glTexParameterf,"GLenum target, GLenum pname, GLfloat param",&target);
   GLV.glTexParameterf (target, pname, param);
   Crout(FP_glTexParameterf);
}

void APIENTRY glTexParameterfv (GLenum target, GLenum pname, const GLfloat *params)
{
   Function(FP_glTexParameterfv,"GLenum target, GLenum pname, const GLfloat *params",&target);
   GLV.glTexParameterfv (target, pname, params);
   Crout(FP_glTexParameterfv);
}

void APIENTRY glTexParameteri (GLenum target, GLenum pname, GLint param)
{
   Function(FP_glTexParameteri,"GLenum target, GLenum pname, GLint param",&target);
   GLV.glTexParameteri (target, pname, param);
   Crout(FP_glTexParameteri);
}

void APIENTRY glTexParameteriv (GLenum target, GLenum pname, const GLint *params)
{
   Function(FP_glTexParameteriv,"GLenum target, GLenum pname, const GLint *params",&target);
   GLV.glTexParameteriv (target, pname, params);
   Crout(FP_glTexParameteriv);
}

void APIENTRY glTexSubImage1D (GLenum target, GLint level, GLint xoffset, GLsizei width, GLenum format, GLenum type, const GLvoid *pixels)
{
   Function(FP_glTexSubImage1D,"GLenum target, GLint level, GLint xoffset, GLsizei width, GLenum format, GLenum type, const GLvoid *pixels",&target);
   GLV.glTexSubImage1D (target, level, xoffset, width, format, type, pixels);
   Crout(FP_glTexSubImage1D);
}

void APIENTRY glTexSubImage2D (GLenum target, GLint level, GLint xoffset, GLint yoffset, GLsizei width, GLsizei height, GLenum format, GLenum type, const GLvoid *pixels)
{
   Function(FP_glTexSubImage2D,"GLenum target, GLint level, GLint xoffset, GLint yoffset, GLsizei width, GLsizei height, GLenum format, GLenum type, const GLvoid *pixels",&target);
   GLV.glTexSubImage2D (target, level, xoffset, yoffset, width, height, format, type, pixels);
   Crout(FP_glTexSubImage2D);
}

void APIENTRY glTranslated (GLdouble x, GLdouble y, GLdouble z)
{
   Function(FP_glTranslated,"GLdouble x, GLdouble y, GLdouble z",&x);
   GLV.glTranslated (x, y, z);
   Crout(FP_glTranslated);
}

void APIENTRY glTranslatef (GLfloat x, GLfloat y, GLfloat z)
{
   Function(FP_glTranslatef,"GLfloat x, GLfloat y, GLfloat z",&x);
   GLV.glTranslatef (x, y, z);
   Crout(FP_glTranslatef);
}

void APIENTRY glVertex2d (GLdouble x, GLdouble y)
{
   Function(FP_glVertex2d,"GLdouble x, GLdouble y",&x);
   GLV.glVertex2d (x, y);
   Crout(FP_glVertex2d);
}

void APIENTRY glVertex2dv (const GLdouble *v)
{
   Function(FP_glVertex2dv,"const GLdouble *v",&v);
   GLV.glVertex2dv (v);
   Crout(FP_glVertex2dv);
}

void APIENTRY glVertex2f (GLfloat x, GLfloat y)
{
   Function(FP_glVertex2f,"GLfloat x, GLfloat y",&x);
   GLV.glVertex2f (x, y);
   Crout(FP_glVertex2f);
}

void APIENTRY glVertex2fv (const GLfloat *v)
{
   Function(FP_glVertex2fv,"const GLfloat *v",&v);
   GLV.glVertex2fv (v);
   Crout(FP_glVertex2fv);
}

void APIENTRY glVertex2i (GLint x, GLint y)
{
   Function(FP_glVertex2i,"GLint x, GLint y",&x);
   GLV.glVertex2i (x, y);
   Crout(FP_glVertex2i);
}

void APIENTRY glVertex2iv (const GLint *v)
{
   Function(FP_glVertex2iv,"const GLint *v",&v);
   GLV.glVertex2iv (v);
   Crout(FP_glVertex2iv);
}

void APIENTRY glVertex2s (GLshort x, GLshort y)
{
   Function(FP_glVertex2s,"GLshort x, GLshort y",&x);
   GLV.glVertex2s (x, y);
   Crout(FP_glVertex2s);
}

void APIENTRY glVertex2sv (const GLshort *v)
{
   Function(FP_glVertex2sv,"const GLshort *v",&v);
   GLV.glVertex2sv (v);
   Crout(FP_glVertex2sv);
}

void APIENTRY glVertex3d (GLdouble x, GLdouble y, GLdouble z)
{
   Function(FP_glVertex3d,"GLdouble x, GLdouble y, GLdouble z",&x);
   GLV.glVertex3d (x, y, z);
   Crout(FP_glVertex3d);
}

void APIENTRY glVertex3dv (const GLdouble *v)
{
   Function(FP_glVertex3dv,"const GLdouble *v",&v);
   GLV.glVertex3dv (v);
   Crout(FP_glVertex3dv);
}

void APIENTRY glVertex3f (GLfloat x, GLfloat y, GLfloat z)
{
   Function(FP_glVertex3f,"GLfloat x, GLfloat y, GLfloat z",&x);
   GLV.glVertex3f (x, y, z);
   Crout(FP_glVertex3f);
}

void APIENTRY glVertex3fv (const GLfloat *v)
{
   Function(FP_glVertex3fv,"const GLfloat *v",&v);
   GLV.glVertex3fv (v);
   Crout(FP_glVertex3fv);
}

void APIENTRY glVertex3i (GLint x, GLint y, GLint z)
{
   Function(FP_glVertex3i,"GLint x, GLint y, GLint z",&x);
   GLV.glVertex3i (x, y, z);
   Crout(FP_glVertex3i);
}

void APIENTRY glVertex3iv (const GLint *v)
{
   Function(FP_glVertex3iv,"const GLint *v",&v);
   GLV.glVertex3iv (v);
   Crout(FP_glVertex3iv);
}

void APIENTRY glVertex3s (GLshort x, GLshort y, GLshort z)
{
   Function(FP_glVertex3s,"GLshort x, GLshort y, GLshort z",&x);
   GLV.glVertex3s (x, y, z);
   Crout(FP_glVertex3s);
}

void APIENTRY glVertex3sv (const GLshort *v)
{
   Function(FP_glVertex3sv,"const GLshort *v",&v);
   GLV.glVertex3sv (v);
   Crout(FP_glVertex3sv);
}

void APIENTRY glVertex4d (GLdouble x, GLdouble y, GLdouble z, GLdouble w)
{
   Function(FP_glVertex4d,"GLdouble x, GLdouble y, GLdouble z, GLdouble w",&x);
   GLV.glVertex4d (x, y, z, w);
   Crout(FP_glVertex4d);
}

void APIENTRY glVertex4dv (const GLdouble *v)
{
   Function(FP_glVertex4dv,"const GLdouble *v",&v);
   GLV.glVertex4dv (v);
   Crout(FP_glVertex4dv);
}

void APIENTRY glVertex4f (GLfloat x, GLfloat y, GLfloat z, GLfloat w)
{
   Function(FP_glVertex4f,"GLfloat x, GLfloat y, GLfloat z, GLfloat w",&x);
   GLV.glVertex4f (x, y, z, w);
   Crout(FP_glVertex4f);
}

void APIENTRY glVertex4fv (const GLfloat *v)
{
   Function(FP_glVertex4fv,"const GLfloat *v",&v);
   GLV.glVertex4fv (v);
   Crout(FP_glVertex4fv);
}

void APIENTRY glVertex4i (GLint x, GLint y, GLint z, GLint w)
{
   Function(FP_glVertex4i,"GLint x, GLint y, GLint z, GLint w",&x);
   GLV.glVertex4i (x, y, z, w);
   Crout(FP_glVertex4i);
}

void APIENTRY glVertex4iv (const GLint *v)
{
   Function(FP_glVertex4iv,"const GLint *v",&v);
   GLV.glVertex4iv (v);
   Crout(FP_glVertex4iv);
}

void APIENTRY glVertex4s (GLshort x, GLshort y, GLshort z, GLshort w)
{
   Function(FP_glVertex4s,"GLshort x, GLshort y, GLshort z, GLshort w",&x);
   GLV.glVertex4s (x, y, z, w);
   Crout(FP_glVertex4s);
}

void APIENTRY glVertex4sv (const GLshort *v)
{
   Function(FP_glVertex4sv,"const GLshort *v",&v);
   GLV.glVertex4sv (v);
   Crout(FP_glVertex4sv);
}

void APIENTRY glVertexPointer (GLint size, GLenum type, GLsizei stride, const GLvoid *pointer)
{
   Function(FP_glVertexPointer,"GLint size, GLenum type, GLsizei stride, const GLvoid *pointer",&size);
   GLV.glVertexPointer (size, type, stride, pointer);
   Crout(FP_glVertexPointer);
}

void APIENTRY glViewport (GLint x, GLint y, GLsizei width, GLsizei height)
{
   Function(FP_glViewport,"GLint x, GLint y, GLsizei width, GLsizei height",&x);
   GLV.glViewport (x, y, width, height);
   Crout(FP_glViewport);
}

int   APIENTRY wglChoosePixelFormat       (HDC a, CONST PIXELFORMATDESCRIPTOR *b)
{
   Function(FP_wglChoosePixelFormat,"HDC a, CONST PIXELFORMATDESCRIPTOR *b",&a);
   int result = GLV.wglChoosePixelFormat(a,b);
   Result(FP_wglChoosePixelFormat,"int",&result);
   Crout(FP_wglChoosePixelFormat);
   return result;
}

BOOL  APIENTRY wglCopyContext             (HGLRC a, HGLRC b, UINT c)
{
   Function(FP_wglCopyContext,"HGLRC a, HGLRC b, UINT c",&a);
   BOOL result = GLV.wglCopyContext(a,b,c);
   Result(FP_wglCopyContext,"BOOL",&result);
   Crout(FP_wglCopyContext);
   return result;
}

HGLRC APIENTRY wglCreateContext           (HDC a)
{
   Function(FP_wglCreateContext,"HDC a",&a);
   HGLRC result = GLV.wglCreateContext(a);
   Result(FP_wglCreateContext,"HGLRC",&result);
   Crout(FP_wglCreateContext);
   return result;
}

HGLRC APIENTRY wglCreateLayerContext      (HDC a, int b)
{
   Function(FP_wglCreateLayerContext,"HDC a, int b",&a);
   HGLRC result = GLV.wglCreateLayerContext(a,b);
   Result(FP_wglCreateLayerContext,"HGLRC",&result);
   Crout(FP_wglCreateLayerContext);
   return result;
}

BOOL  APIENTRY wglDeleteContext           (HGLRC a)
{
   Function(FP_wglDeleteContext,"HGLRC a",&a);
   BOOL result = GLV.wglDeleteContext(a);
   Result(FP_wglDeleteContext,"BOOL",&result);
   Crout(FP_wglDeleteContext);
   return result;
}

BOOL  APIENTRY wglDescribeLayerPlane      (HDC a, int b, int c, UINT d, LPLAYERPLANEDESCRIPTOR e)
{
   Function(FP_wglDescribeLayerPlane,"HDC a, int b, int c, UINT d, LPLAYERPLANEDESCRIPTOR e",&a);
   BOOL result = GLV.wglDescribeLayerPlane(a,b,c,d,e);
   Result(FP_wglDescribeLayerPlane,"BOOL",&result);
   Crout(FP_wglDescribeLayerPlane);
   return result;
}

int   APIENTRY wglDescribePixelFormat     (HDC a, int b, UINT c, LPPIXELFORMATDESCRIPTOR d)
{
   Function(FP_wglDescribePixelFormat,"HDC a, int b, UINT c, LPPIXELFORMATDESCRIPTOR d",&a);
   int result = GLV.wglDescribePixelFormat(a,b,c,d);
   Result(FP_wglDescribePixelFormat,"int",&result);
   Crout(FP_wglDescribePixelFormat);
   return result;
}

HGLRC APIENTRY wglGetCurrentContext       (void)
{
   Function(FP_wglGetCurrentContext,"",NULL);
   HGLRC result = GLV.wglGetCurrentContext();
   Result(FP_wglGetCurrentContext,"HGLRC",&result);
   Crout(FP_wglGetCurrentContext);
   return result;
}

HDC   APIENTRY wglGetCurrentDC            (void)
{
   Function(FP_wglGetCurrentDC,"",NULL);
   HDC result = GLV.wglGetCurrentDC();
   Result(FP_wglGetCurrentDC,"HDC",&result);
   Crout(FP_wglGetCurrentDC);
   return result;
}

PROC  APIENTRY wglGetDefaultProcAddress   (LPCSTR a)
{
   Function(FP_wglGetDefaultProcAddress,"LPCSTR a",&a);
   PROC result = GLV.wglGetDefaultProcAddress(a);
   Result(FP_wglGetDefaultProcAddress,"PROC",&result);
   Crout(FP_wglGetDefaultProcAddress);
   return result;
}

int   APIENTRY wglGetLayerPaletteEntries  (HDC a, int b, int c, int d, COLORREF *e)
{
   Function(FP_wglGetLayerPaletteEntries,"HDC a, int b, int c, int d, COLORREF *e",&a);
   int result = GLV.wglGetLayerPaletteEntries(a,b,c,d,e);
   Result(FP_wglGetLayerPaletteEntries,"int",&result);
   Crout(FP_wglGetLayerPaletteEntries);
   return result;
}

int   APIENTRY wglGetPixelFormat          (HDC a)
{
   Function(FP_wglGetPixelFormat,"HDC a",&a);
   int result = GLV.wglGetPixelFormat(a);
   Result(FP_wglGetPixelFormat,"int",&result);
   Crout(FP_wglGetPixelFormat);
   return result;
}

PROC  APIENTRY wglGetProcAddress          (LPCSTR a)
{
   Function(FP_wglGetProcAddress,"LPCSTR a",&a);
   PROC result = GLV.wglGetProcAddress(a);
   Result(FP_wglGetProcAddress,"PROC",&result);
   Crout(FP_wglGetProcAddress);
   return result;
}

BOOL  APIENTRY wglMakeCurrent             (HDC a, HGLRC b)
{
   Function(FP_wglMakeCurrent,"HDC a, HGLRC b",&a);
   BOOL result = GLV.wglMakeCurrent(a,b);
   Result(FP_wglMakeCurrent,"BOOL",&result);
   Crout(FP_wglMakeCurrent);
   return result;
}

BOOL  APIENTRY wglRealizeLayerPalette     (HDC a, int b, BOOL c)
{
   Function(FP_wglRealizeLayerPalette,"HDC a, int b, BOOL c",&a);
   BOOL result = GLV.wglRealizeLayerPalette(a,b,c);
   Result(FP_wglRealizeLayerPalette,"BOOL",&result);
   Crout(FP_wglRealizeLayerPalette);
   return result;
}

int   APIENTRY wglSetLayerPaletteEntries  (HDC a, int b, int c, int d, CONST COLORREF *e)
{
   Function(FP_wglSetLayerPaletteEntries,"HDC a, int b, int c, int d, CONST COLORREF *e",&e);
   int result = GLV.wglSetLayerPaletteEntries(a,b,c,d,e);
   Result(FP_wglSetLayerPaletteEntries,"int",&result);
   Crout(FP_wglSetLayerPaletteEntries);
   return result;
}

BOOL  APIENTRY wglSetPixelFormat          (HDC a, int b, CONST PIXELFORMATDESCRIPTOR *c)
{
   Function(FP_wglSetPixelFormat,"HDC a, int b, CONST PIXELFORMATDESCRIPTOR *c",&a);
   BOOL result = GLV.wglSetPixelFormat(a,b,c);
   Result(FP_wglSetPixelFormat,"BOOL",&result);
   Crout(FP_wglSetPixelFormat);
   return result;
}

BOOL  APIENTRY wglShareLists              (HGLRC a, HGLRC b)
{
   Function(FP_wglShareLists,"HGLRC a, HGLRC b",&a);
   BOOL result = GLV.wglShareLists(a,b);
   Result(FP_wglShareLists,"BOOL",&result);
   Crout(FP_wglShareLists);
   return result;
}

BOOL  APIENTRY wglSwapBuffers             (HDC a)
{
   Function(FP_wglSwapBuffers,"HDC a",&a);
   BOOL result = GLV.wglSwapBuffers(a);
   Result(FP_wglSwapBuffers,"BOOL",&result);
   Crout(FP_wglSwapBuffers);
   return result;
}

BOOL  APIENTRY wglSwapLayerBuffers        (HDC a, UINT b)
{
   Function(FP_wglSwapLayerBuffers,"HDC a, UINT b",&a);
   BOOL result = GLV.wglSwapLayerBuffers(a,b);
   Result(FP_wglSwapLayerBuffers,"BOOL",&result);
   Crout(FP_wglSwapLayerBuffers);
   return result;
}

BOOL  APIENTRY wglUseFontBitmapsA         (HDC a, DWORD b, DWORD c, DWORD d)
{
   Function(FP_wglUseFontBitmapsA,"HDC a, DWORD b, DWORD c, DWORD d",&a);
   BOOL result = GLV.wglUseFontBitmapsA(a,b,c,d);
   Result(FP_wglUseFontBitmapsA,"BOOL",&result);
   Crout(FP_wglUseFontBitmapsA);
   return result;
}

BOOL  APIENTRY wglUseFontBitmapsW         (HDC a, DWORD b, DWORD c, DWORD d)
{
   Function(FP_wglUseFontBitmapsW,"HDC a, DWORD b, DWORD c, DWORD d",&a);
   BOOL result = GLV.wglUseFontBitmapsW(a,b,c,d);
   Result(FP_wglUseFontBitmapsW,"BOOL",&result);
   Crout(FP_wglUseFontBitmapsW);
   return result;
}

BOOL  APIENTRY wglUseFontOutlinesA        (HDC a, DWORD b, DWORD c, DWORD d, FLOAT e, FLOAT f, int g, LPGLYPHMETRICSFLOAT h)
{
   Function(FP_wglUseFontOutlinesA,"HDC a, DWORD b, DWORD c, DWORD d, FLOAT e, FLOAT f, int g, LPGLYPHMETRICSFLOAT h",&a);
   BOOL result = GLV.wglUseFontOutlinesA(a,b,c,d,e,f,g,h);
   Result(FP_wglUseFontOutlinesA,"BOOL",&result);
   Crout(FP_wglUseFontOutlinesA);
   return result;
}

BOOL  APIENTRY wglUseFontOutlinesW        (HDC a, DWORD b, DWORD c, DWORD d, FLOAT e, FLOAT f, int g, LPGLYPHMETRICSFLOAT h)
{
   Function(FP_wglUseFontOutlinesW,"HDC a, DWORD b, DWORD c, DWORD d, FLOAT e, FLOAT f, int g, LPGLYPHMETRICSFLOAT h",&a);
   BOOL result = GLV.wglUseFontOutlinesW(a,b,c,d,e,f,g,h);
   Result(FP_wglUseFontOutlinesW,"BOOL",&result);
   Crout(FP_wglUseFontOutlinesW);
   return result;
}

//****************************************************************************
//*                                                                          *
//*  DLLMain() function to acquire DLL instance handle, etc.                 *
//*                                                                          *
//****************************************************************************

BOOL WINAPI DllMain(HINSTANCE hinstDLL,//)
                    DWORD     fdwReason,
                    LPVOID    lpvReserved)
{
   int i;

   if (fdwReason == DLL_PROCESS_DETACH)
      {
      if (text_file != NULL)
         {
         print("Closing output file\n");
         fclose(text_file);
         text_file = NULL;
         }

      return TRUE;
      }

   if (fdwReason != DLL_PROCESS_ATTACH)
      {
      return TRUE;
      }

   //
   // Get DLL path
   //
      
   hDLLInstance = hinstDLL;

   TCHAR DLL_pathname[_MAX_PATH];
   GetModuleFileName(hinstDLL, DLL_pathname, _MAX_PATH);

   //
   // Initialize all entries in show_status table
   //
   // By default, all OpenGL functions are traced
   //

   for (i=0; i < ARY_CNT(show_status); i++)
      {
      show_status[i] = SHOW_ALWAYS;
      }

   //
   // Build index for GLenum name lookup
   //

   for (i=0; i < ARY_CNT(GLenum_names); i++)
      {
      GLenum_names[i] = NULL;
      }

   for (i=0; i < ARY_CNT(GLenum_list); i++)
      {
      GLenum_names[GLenum_list[i].tag] = GLenum_list[i].name;
      }

   //
   // Parse .INI file
   //

   TCHAR INI_pathname[_MAX_PATH];
   strcpy(INI_pathname, DLL_pathname);

   TCHAR OGL_pathname[_MAX_PATH];
   strcpy(OGL_pathname,"GLTRACE.DLL");

   char *c = strrchr(INI_pathname,'\\');

   if (c == NULL)
      {
      c = INI_pathname;
      }
   else
      {
      c++;
      }

   strcpy(c,"GLTRACE.INI");

   print("GLTRACE: Parsing %s...\n",INI_pathname);

   FILE *INI = fopen(INI_pathname,"rt");

   enum section
      {
      NONE,
      IMP,
      EXC,
      OPT,
      TXT
      };

   if (INI != NULL)
      {
      char line[256];

      section s = NONE;

      while (fgets(line, sizeof(line)-1, INI) != NULL)
         {
         if (line[0] == ';')
            {
            continue;
            }

         for (i=strlen(line)-1; i >= 0; i--)
            {
            if (isspace(line[i])) 
               {
               line[i] = 0;
               }
            else
               {
               break;
               }
            }

         if (i == -1)
            {
            continue;
            }

         if (!stricmp(line, "[Implementation]")) { s = IMP; continue; }
         if (!stricmp(line, "[Exclude]"))        { s = EXC; continue; }
         if (!stricmp(line, "[Optional]"))       { s = OPT; continue; }
         if (!stricmp(line, "[Output]"))         { s = TXT; continue; }

         switch (s)
            {
            case IMP:

               strcpy(OGL_pathname,line);
               break;

            case TXT:

               text_file = fopen(line, "w+t");
               break;

            case EXC:

               for (i=0; i < ARY_CNT(GLN); i++)
                  {
                  if (!stricmp(GLN[i], line))
                     {
                     show_status[i] = SUPPRESS;
                     break;
                     }
                  }
               break;

            case OPT:
               
               for (i=0; i < ARY_CNT(GLN); i++)
                  {
                  if (!stricmp(GLN[i], line))
                     {
                     show_status[i] = SHOW_IF_SHIFT_DOWN;
                     break;
                     }
                  }
               break;
            }
         }

      fclose(INI);
      }

   //
   // Load actual OpenGL provider under "GLTRACE.DLL" alias
   //

   HINSTANCE OpenGL_provider = LoadLibrary(OGL_pathname);

   assert(OpenGL_provider != NULL);

   //
   // Map Win32 OpenGL calls to our jump table
   //

   for (i=0; i < ARY_CNT(GLN); i++)
      {
      FARPROC OpenGL_proc = GetProcAddress(OpenGL_provider, GLN[i]);

      assert(OpenGL_proc != NULL);

      ((unsigned long *) (&GLV))[i] = (unsigned long) OpenGL_proc;
      }

   //
   // Indicate start of debug trace
   //

   long elapstime;
   struct tm *adjtime;
   char loctime[32];

   time(&elapstime);
   adjtime = localtime(&elapstime);
   lstrcpy(loctime,asctime(adjtime));
   loctime[24] = 0;

   print("\n");
   print("-------------------------------------------------------------------------------\n");
   print("Win32 OpenGL usage script generated by %s V%s\n",DLL_pathname,VERSION);
   print("OpenGL provider: %s\n",OGL_pathname);
   print("Start time: %s\n",loctime);
   print("-------------------------------------------------------------------------------\n\n");

   starttime = timeGetTime();

   return TRUE;
}

