#include "ReadWriteMLA.h"
#include "Str.h"
#include "Sys.h"
#include "StrSet.h"
#include "config.h"		// XXX
#include <errno.h>
#include <ctype.h>

#define	MLA_DICTFILE	FAX_LIBDATA "/dict"
#define	MLA_DEFSTOP	FAX_LIBDATA "/stop"

static	void usage(void);
static	const char* progName;
static	void readStopList(const char* file, fxStrSet& list);

static fxBool
isDirectory(const char* path)
{
    struct stat sb;
    return (Sys::stat(path, sb) == -1 ?
	FALSE : (sb.st_mode & S_IFMT) == S_IFDIR);
}

int
main(int argc, char* argv[])
{
    fxStr tocFile(MLA_TOCNAME);
    fxStr dictFile(MLA_DICTFILE);
    fxStr ixFile(MLA_IXNAME);
    fxStr dir(".");
    fxStr defAddr;
    u_int maxmsgs = 2500;
    u_int avgmsgsize = 128;
    u_int maxmsgsize = 3072;		// NB: chosen empirically
    u_int trace = 0;
    u_int minword = (u_int) -1;
    u_int maxword = (u_int) -1;
    fxBool update = TRUE;
    fxBool skip = TRUE;
    fxBool overwrite = FALSE;
    fxBool compact = FALSE;
    fxBool process = TRUE;
    fxBool index = FALSE;
    fxStrSet stopList;
    const char* sep = NULL;
    extern char *optarg;
    extern int optind;
    int c;

    progName = argv[0];
    while ((c = getopt(argc, argv, "a:b:cd:D:fil:m:n:ps:S:tT:w:yz")) != -1)
	switch (c) {
	case 'a':	defAddr = optarg; break;
	case 'b':	minword = strtoul(optarg, NULL, 0); break;
	case 'B':	maxword = strtoul(optarg, NULL, 0); break;
	case 'c':	update = FALSE; break;
	case 'd':	dir = optarg; break;
	case 'D':	dictFile = optarg; break;
	case 'f':	overwrite = TRUE; break;
	case 'i':	index = TRUE; break;
	case 'I':	ixFile = optarg; break;
	case 'l':	maxmsgsize = strtoul(optarg, NULL, 0); break;
	case 'm':	avgmsgsize = strtoul(optarg, NULL, 0); break;
	case 'n':	maxmsgs = strtoul(optarg, NULL, 0); break;
	case 'p':	process = FALSE; break;
	case 's':	sep = optarg; break;
	case 't':	trace++; break;
	case 'S':	readStopList(optarg, stopList); break;
	case 'T':	tocFile = optarg; break;
	case 'w':	stopList.add(optarg); break;
	case 'y':	skip = FALSE; break;
	case 'z':	compact = TRUE; break;
	case '?':	usage();
	}

    (void) umask(0);
    if (dir != ".") {
	if (!isDirectory(dir)) {
	    if (mkdir(dir, 0755) == -1) {
		fprintf(stderr, "%s: Could not create archive directory \"%s\"",
		    progName, (const char*) dir);
		return (-1);
	    }
	    if (trace)
		printf("Creating directory \"%s\", mode %o.\n",
		    (const char*) dir, 0755);
	}
	if (Sys::chdir(dir) < 0) {
	    fprintf(stderr, "%s: ", progName);
	    perror("chdir");
	    return (-1);
	}
    }
    ReadWriteMLA* mla;
    if (update && Sys::isRegularFile(tocFile)) {
	mla = ReadWriteMLA::readMLA(tocFile);
	if (mla == NULL)
	    return (-1);
    } else {
	mla = new ReadWriteMLA(tocFile, maxmsgs, avgmsgsize, maxmsgsize);
	/*
	 * If messages are to be indexed and no stop
	 * list is specified, then supply a default list.
	 */
	if (index && stopList.size() == 0)
	    readStopList(MLA_DEFSTOP, stopList);
    }
    mla->setOverwrite(overwrite);
    mla->setTrace(trace);
    mla->setCompactPool(compact);
    if (sep != NULL)
	mla->setMsgSeparator(sep);
    if (defAddr != "")
	mla->setDefaultAddr(defAddr);
    if (index) {
	mla->setIXFile(ixFile);
	if (dictFile != "")
	    mla->setDictionary(dictFile);
	if (stopList.size() > 0) {
	    for (fxStrSetIter iter(stopList); iter.notDone(); iter++) {
		const fxStr& s = iter;
		mla->addStopList(s);
	    }
	}
	if (minword != (u_int) -1)
	    mla->setMinWordLength(minword);
	if (maxword != (u_int) -1)
	    mla->setMaxWordLength(maxword);
    } else
	mla->setIXFile("");
    fxBool status = TRUE;
    if (optind < argc) {
	mla->setupParsing();
	do {
	    if (strcmp(argv[optind], "-")) {
		mla->setInputFile(argv[optind]);
		FILE* fp = fopen(argv[optind], "r");
		if (fp == NULL)
		    mla->fatal("Cannot open file, errno %s", strerror(errno));
		status = mla->load(fp, skip, process);
		fclose(fp);
	    } else {
		mla->setInputFile("<stdin>");
		status = mla->load(stdin, skip, process);
	    }
	} while (status && ++optind < argc);
    }
    return (mla->update() && status ? 0 : 1);
}

static void
readStopList(const char* file, fxStrSet& list)
{
    FILE* fd = fopen(file, "r");
    if (fd != NULL) {
	char line[256];

	while (fgets(line, sizeof (line), fd)) {
	    const char* cp;
	    for (cp = line; isspace(*cp); cp++)
		;
	    if (*cp) {
		const char* w;
		for (w = cp++; *cp && !isspace(*cp); cp++)
		    ;
		list.add(fxStr(w, cp-w));
	    }
	}
	fclose(fd);
    } else
	fprintf(stderr, "%s: %s: Can not open stop list file.\n",
	    progName, file);
}

static void
usage(void)
{
    printf("usage: %s [options] [files]\n", progName);
    printf("where options are:\n");
    printf("-c		create the archive, even if it exists\n");
    printf("-d dir		directory to write archive files\n");
    printf("-f		force overwriting of existing data\n");
    printf("-l #		set the maximum message size (default 3072)\n");
    printf("-m #		set the average message size (default 128)\n");
    printf("-n #		set the maximum number of messages (default 2500)\n");
    printf("-s sep		set the message separator string (default \"From \")\n");
    printf("-t		trace work (multiple times for more info)\n");
    printf("-T file		MLA table-of-contents database file\n");
    printf("-y		do not skip to the first message separator (default FALSE)\n");
    printf("-z		compact the database (default FALSE)\n");
    printf("Giving \"-\" for a file forces messages to be read from standard input\n");
    exit(1);
}
