#include "ReadOnlyMLA.h"
#include "Sys.h"
#include <errno.h>

static	const char* progName;
static	void usage(void);
static	void fatal(const char* fmt ...);
    
#define	dostr(s, ns, ss) {			\
    if (s == 0)					\
	ns++;					\
    else					\
	ss += strlen(mla->checkstr(s))+1;	\
}

int
main(int argc, char* argv[])
{
    fxStr tocFile(MLA_TOCNAME);
    fxStr dir(".");

    extern char *optarg;
    extern int optind;
    int c;
    progName = argv[0];
    while ((c = getopt(argc, argv, "d:T:")) != -1)
	switch (c) {
	case 'd':	dir = optarg; break;
	case 'T':	tocFile = optarg; break;
	case '?':	usage();
	}

    if (dir != "." && Sys::chdir(dir) < 0)
	fatal("%s: %s: %s", progName, (const char*) dir, strerror(errno));
    ReadOnlyMLA* mla = ReadOnlyMLA::readMLA(tocFile);
    if (mla == NULL)
	usage();
    u_int nmsgs = mla->getMsgCount();
    const MailMsg* table = mla->getMsgTable();
    const mnum_t* replies = mla->getReplyTable();
    u_int maxreplies = 0;
    u_int totreplies = 0;
    u_int nullstrs = 0;
    size_t strspace = 0;
    u_int replyoverhead = 0;
    for (u_int i = 0; i < nmsgs; i++) {
	const MailMsg& msg = table[i];
	u_int n = msg.nreplies;
	if (n > maxreplies)
	    maxreplies = n;
	totreplies += n;
	if (n > 1) {
	    u_int blocks = 1;
	    const mnum_t* ix = &replies[msg.replynum];
	    for (; n > 4; n -= 3) {
		blocks++;
		ix = &replies[ix[3]];
	    }
	    replyoverhead += 1+4*blocks - msg.nreplies;
	}
	dostr(msg.name, nullstrs, strspace);
	dostr(msg.emailaddr, nullstrs, strspace);
	dostr(msg.replytoaddr, nullstrs, strspace);
	dostr(msg.to, nullstrs, strspace);
	dostr(msg.date, nullstrs, strspace);
	dostr(msg.subject, nullstrs, strspace);
	dostr(msg.msgid, nullstrs, strspace);
    }
    printf("%u null strings (avg %.1f/msg)\n"
	, nullstrs
	, float(nullstrs) / nmsgs
	);
    printf("%u%% string compaction\n"
	, 100*strspace/mla->getStrSpace()
	);
    printf("%u%% reply overhead (%u of %u)\n"
	, 100*replyoverhead/(totreplies+replyoverhead)
	, replyoverhead
	, totreplies
	);
    return (0);
}

static void
usage(void)
{
    printf("usage: %s [options]\n", progName);
    printf("where options are:\n");
    printf("-d dir		directory to find archive files\n");
    printf("-T file		MLA table-of-contents database file\n");
    printf("\n");
    exit(1);
}

#include <stdarg.h>

static void
fatal(const char* fmt ...)
{
    fprintf(stderr, "%s: ", progName);
    va_list ap;
    va_start(ap, fmt);
    vfprintf(stderr, fmt, ap);
    va_end(ap);
    fprintf(stderr, ".\n");
    exit(-1);
}
