#include "CGIForm.h"
#include "SortedMsgArray.h"
#include "ReadOnlyMLA.h"
#include "RegEx.h"
#include <time.h>

fxDECLARE_SortedMsgArray(ThreadSortedMailMsgArray);
fxDECLARE_SortedMsgArray(DateSortedMailMsgArray);
fxDECLARE_SortedMsgArray(SubjectSortedMailMsgArray);
fxDECLARE_SortedMsgArray(AuthorSortedMailMsgArray);

static	CGIForm form("mlaquery", MLA_QUERYFORM);
static	RegEx* fromPat = NULL;
static	RegEx* subjPat = NULL;
static	u_int nhits;
static	u_int nthreads = (u_int) -1;
static	time_t firstHit = 0;
static	time_t lastHit = 0;
static	fxBool more;

static	void scan(FILE*, const MLA&, const SortedMailMsgArray&,
	    const char*, const char*);

int
main(int argc, char* argv[])
{
    form.setupArgs(argc, argv);

    ReadOnlyMLA* mla = form.readMLA();
    if (mla == NULL)
	return (-1);

    const char* data;
    size_t size;
    form.readForm(*mla, data, size);

    SortedMailMsgArray* hits;
    if (form.getScheme() == "thread")
	hits = new ThreadSortedMailMsgArray;
    else if (form.getScheme() == "date")
	hits = new DateSortedMailMsgArray;
    else if (form.getScheme() == "author")
	hits = new AuthorSortedMailMsgArray;
    else if (form.getScheme() == "subject")
	hits = new SubjectSortedMailMsgArray;
    else 
	mla->fatal("Unknown collation scheme \"%s\"",
	    (const char*) form.getScheme());

    int flags = REG_EXTENDED;
    if (form.getIgnoreCase())
	flags |= REG_ICASE;
    if (form.getFromPat() != "")
	fromPat = new RegEx(form.getFromPat(), flags);
    if (form.getSubjPat() != "")
	subjPat = new RegEx(form.getSubjPat(), flags);
    form.query(*mla, *hits, fromPat, subjPat, more);

    nhits = hits->length();
    if (nhits > 0) {
	firstHit = (*hits)[0]->datetime;
	lastHit = (*hits)[nhits-1]->datetime;
    }
    hits->mla = mla;
    hits->reverse = form.getReverse();
    hits->maxlevels = form.getMaxLevels();
    hits->qsort();
    /*
     * Craft a compact representation of the hit set for
     * passing through the environment to each mlafetch
     * that's invoked to fetch&display a document.  We
     * do this so that mlafetch can present proper navigation
     * guides; e.g. next message/thread in the hit set.
     */
    hits->encode(form.getQueryString());

    scan(stdout, *mla, *hits, data, data+size);
    return (0);
}

static u_int
threadCount(const MLA& mla, const SortedMailMsgArray& hits)
{
    if (nthreads == (u_int) -1) {
	u_int n = howmany(mla.getThreadCount(), 32);
	u_int* x = new u_int[n];
	memset(x, 0, n*sizeof (u_int));
	u_int i;
	for (i = 0; i < nhits; i++) {
	    u_int t = hits[i]->thread;
	    x[t>>5] |= 1<<(t&31);
	}
	nthreads = 0;
	for (i = 0; i < n; i++)
	    if (x[i] != 0) {
		for (u_int mask = 0x80000000; mask; mask >>= 1)
		    if (x[i] & mask)
			nthreads++;
	    }
    }
    return (nthreads);
}

static const char*
nextchr(const char* cp, char c, const char* ep)
{
    for (; cp < ep; cp++)
	if (*cp == c)
	    return (cp);
    return (NULL);
}

static const char*
doif(fxBool b, FILE* fp, const MLA& mla, const SortedMailMsgArray& hits,
    const char* cp, const char* ep)
{
    const char* tp = nextchr(cp+1, cp[0], ep);
    if (b)
	scan(fp, mla, hits, cp+1, tp ? tp : ep);
    return (tp ? tp+1 : ep);
}

static void
scan(FILE* fp, const MLA& mla, const SortedMailMsgArray& hits,
    const char* cp, const char* ep)
{
    const char* tp = nextchr(cp, '%', ep);
    if (tp) {
	do {
	    fwrite(cp, tp-cp, 1, fp);
	    cp = tp+1;
	    switch (*cp++) {
	    case 'h':		// hit count
		fprintf(fp, "%u", nhits);
		break;
	    case 'q':		// query argument string
		fprintf(fp, "%s", (const char*) form.getQueryString());
		break;
	    case 't':		// thread count
		fprintf(fp, "%u", threadCount(mla, hits));
		break;
	    case '[':		// date of first hit
		if (nhits > 0)
		    CGIForm::printDate(fp, *gmtime(&firstHit));
		else
		    fprintf(fp, "<I>Never</I>");
		break;
	    case ']':		// date of last hit
		if (nhits > 0)
		    CGIForm::printDate(fp, *gmtime(&lastHit));
		else
		    fprintf(fp, "<I>Never</I>");
		break;
	    case '*':		// print query results
		if (hits.length() > 0) {
		    hits.print(fp, form.getQueryString());
		    if (more)
			fprintf(fp, "<P><I>...more...</I>\n");
		}
		break;
	    case '?':		// conditional escapes
		switch (*cp++) {
		case 'f':		// if fromPat
		    cp = doif(fromPat != NULL, fp, mla, hits, cp, ep);
		    break;
		case 's':		// if subjPat
		    cp = doif(subjPat != NULL, fp, mla, hits, cp, ep);
		    break;
		case 'h':		// if hits != 0
		    cp = doif(nhits != 0, fp, mla, hits, cp, ep);
		    break;
		case 'H':		// if hits == 0
		    cp = doif(nhits == 0, fp, mla, hits, cp, ep);
		    break;
		default:
		    fprintf(fp, "%c?%c", '%', cp[-1]);
		    break;
		}
		break;
	    case '+':		// forms-related escapes
		form.printEscapes(fp, mla, *cp++);
		break;
	    default:
		fprintf(fp, "%c%c", '%', cp[-1]);
		break;
	    }
	} while (cp < ep && (tp = nextchr(cp, '%', ep)));
    }
    fwrite(cp, ep-cp, 1, fp);
}

int
ThreadSortedMailMsgArray::compareElements(void const* a1, void const* a2) const
{
    const MailMsg* m1 = *(const MailMsg**) a1;
    const MailMsg* m2 = *(const MailMsg**) a2;

    int c = mla->getMsgTable()[mla->getThreadTable()[m1->thread]].datetime
	  - mla->getMsgTable()[mla->getThreadTable()[m2->thread]].datetime;
    if (c == 0)
	c = m1->datetime - m2->datetime;
    return (reverse ? -c : c);
}

void
ThreadSortedMailMsgArray::print(FILE* fp, const char* query) const
{
    fxBool* isPrinted = new fxBool[mla->getThreadCount()];
    memset(isPrinted, FALSE, mla->getThreadCount());
    const mnum_t* threads = mla->getThreadTable();
    const MailMsg* table = mla->getMsgTable();
    for (u_int i = 0, n = length(); i < n; i++) {
	const MailMsg& top = table[threads[(*this)[i]->thread]];
	if (!isPrinted[top.thread]) {
	    fputs("<LI> ", fp);
	    if (find(&top) != fx_invalidArrayIndex)
		printMsg(fp, top, query);
	    else
		printMsg(fp, top);
	    if (top.hasReplies())
		printReplies(fp, top, 1, query);
	    isPrinted[top.thread] = TRUE;
	}
    }
    delete isPrinted;
}
fxIMPLEMENT_SortedMsgArray(ThreadSortedMailMsgArray);

int
DateSortedMailMsgArray::compareElements(void const* a1, void const* a2) const
{
    const MailMsg* m1 = *(const MailMsg**) a1;
    const MailMsg* m2 = *(const MailMsg**) a2;

    return (reverse
	? m2->datetime - m1->datetime
	: m1->datetime - m2->datetime);
}

void
DateSortedMailMsgArray::print(FILE* fp, const char* query) const
{
    for (u_int i = 0, n = length(); i < n; i++) {
	fputs("<LI> ", fp);
	printMsg(fp, *(*this)[i], query);
    }
}
fxIMPLEMENT_SortedMsgArray(DateSortedMailMsgArray);

#include <ctype.h>

static fxBool
isRe(const char* cp)
{
    return ((cp[0] == 'r' || cp[0] == 'R')
	 && (cp[1] == 'e' || cp[1] == 'E')
	 && cp[2] == ':'
    );
}

static const char*
stripre(const char* s)
{
    if (isRe(s)) {
	for (s += 3; isspace(*s); s++)
	    ;
    }
    return (s);
}

int
SubjectSortedMailMsgArray::compareElements(void const* a1, void const* a2) const
{
    const MailMsg* m1 = *(const MailMsg**) a1;
    const MailMsg* m2 = *(const MailMsg**) a2;

    int c = strcmp(
	stripre(mla->checkstr(m1->subject)),
	stripre(mla->checkstr(m2->subject)));
    if (c == 0)
	c = m1->datetime - m2->datetime;
    return (reverse ? -c : c);
}

void
SubjectSortedMailMsgArray::print(FILE* fp, const char* query) const
{
    tnum_t curthread = (u_int) -1;
    for (u_int i = 0, n = length(); i < n; i++) {
	const MailMsg& msg = *(*this)[i];
	if (curthread != msg.thread) {
	    if (curthread != (u_int) -1)
		fputs("</UL>", fp);
	    fputs("<LI> <B>", fp);
	    CGIForm::printString(fp, mla->checkstr(msg.subject));
	    fputs("</B>\n<UL>\n", fp);
	    curthread = msg.thread;
	}
	fputs("<LI> <A HREF=\"?", fp);
	CGIForm::printFetchArgs(fp, msg.msgnum, query);
	fputs("\">", fp);
	CGIForm::printString(fp, mla->checkstr(msg.name));
	fputs("</A> <I>", fp);
	CGIForm::printString(fp, mla->checkstr(msg.date));
	fputs("</I>\n", fp);
    }
    fputs("</UL>", fp);
}
fxIMPLEMENT_SortedMsgArray(SubjectSortedMailMsgArray);

int
AuthorSortedMailMsgArray::compareElements(void const* a1, void const* a2) const
{
    const MailMsg* m1 = *(const MailMsg**) a1;
    const MailMsg* m2 = *(const MailMsg**) a2;

    int c = strcasecmp(mla->checkstr(m1->name), mla->checkstr(m2->name));
    if (c == 0)
	c = m1->datetime - m2->datetime;
    return (reverse ? -c : c);
}

void
AuthorSortedMailMsgArray::print(FILE* fp, const char* query) const
{
    const char* curname = mla->nullstr;
    for (u_int i = 0, n = length(); i < n; i++) {
	const MailMsg& msg = *(*this)[i];
	const char* name = mla->checkstr(msg.name);
	if (strcmp(name, curname)) {
	    if (curname != mla->nullstr)
		fputs("</UL>", fp);
	    const char* replytoaddr = mla->checkstr(msg.replytoaddr);
	    if (strcmp(name, replytoaddr)) {
		fputs("<LI><B>", fp);
		CGIForm::printString(fp, name);
		fprintf(fp, "</B> &lt;<A HREF=\"mailto:%s\"><I>", replytoaddr);
		CGIForm::printString(fp, replytoaddr);
		fputs("</I></A>&gt;", fp);
	    } else {
		fprintf(fp, "<LI><B><A HREF=\"mailto:%s\">", replytoaddr);
		CGIForm::printString(fp, replytoaddr);
		fputs("</A></B>", fp);
	    }
	    fputs("<UL>\n", fp);
	    curname = name;
	}
	fputs("<LI> <A HREF=\"?", fp);
	CGIForm::printFetchArgs(fp, msg.msgnum, query);
	fputs("\">", fp);
	CGIForm::printString(fp, mla->checkstr(msg.subject));
	fputs("</A> <I>", fp);
	CGIForm::printString(fp, mla->checkstr(msg.date));
	fputs("</I>\n", fp);
    }
    if (curname != mla->nullstr)
	fputs("</UL>", fp);
}
fxIMPLEMENT_SortedMsgArray(AuthorSortedMailMsgArray);
