#include "ReadOnlyMLA.h"
#include "Sys.h"
#include <errno.h>

fxBool	showbody = FALSE;

static	const char* progName;
static	void usage(void);
static	void fatal(const char* fmt ...);

static	void printThread(FILE*, const MLA&, const MailMsg&);

int
main(int argc, char* argv[])
{
    fxStr tocFile(MLA_TOCNAME);
    fxStr dir(".");

    extern char *optarg;
    extern int optind;
    int c;
    progName = argv[0];
    while ((c = getopt(argc, argv, "bd:T:")) != -1)
	switch (c) {
	case 'b':	showbody = TRUE; break;
	case 'd':	dir = optarg; break;
	case 'T':	tocFile = optarg; break;
	case '?':	usage();
	}

    if (dir != "." && Sys::chdir(dir) < 0)
	fatal("%s: %s: %s", progName, (const char*) dir, strerror(errno));
    ReadOnlyMLA* mla = ReadOnlyMLA::readMLA(tocFile);
    if (mla == NULL)
	usage();
    u_int nmsgs = mla->getMsgCount();
    u_int nthreads = mla->getThreadCount();
    if (optind == argc) {
	mla->printHeader(stdout);
	if (nmsgs > 0) {
	    char date[80];
	    struct tm* tm = gmtime(&mla->getSortedMsg(0).datetime);
	    strftime(date, sizeof (date), "%a, %d %b %Y %T", tm);
	    printf("First message: %s\n", date);
	    tm = gmtime(&mla->getSortedMsg(nmsgs-1).datetime);
	    strftime(date, sizeof (date), "%a, %d %b %Y %T", tm);
	    printf("Last message:  %s\n", date);
	}
    } else {
	const MailMsg* table = mla->getMsgTable();
	const mnum_t* threads = mla->getThreadTable();
	for (; optind < argc; optind++) {
	    fxBool isThread = FALSE;
	    char* cp = argv[optind];
	    if (*cp == '#')
		isThread = TRUE, cp++;
	    char* tp;
	    u_int a;
	    if (strncmp(cp, "first", 5) == 0)
		a = 0, tp = cp+5;
	    else if (strncmp(cp, "last", 4) == 0)
		a = (isThread ? nthreads : nmsgs) - 1, tp = cp+4;
	    else
		a = (u_int) strtoul(cp, &tp, 0);
	    u_int b;
	    if (tp && *tp == '-') {
		tp++;
		if (strcmp(tp, "first") == 0)
		    b = 0;
		else if (strcmp(tp, "last") == 0)
		    b = (isThread ? nthreads : nmsgs) - 1;
		else
		    b = (u_int) strtoul(tp, NULL, 0);
	    } else
		b = a;
	    if (isThread) {
		if (a >= nthreads || b >= nthreads)
		    mla->fatal("%s: Thread number out of range", argv[optind]);
		for (; a <= b; a++)
		    printThread(stdout, *mla, table[threads[a]]);
	    } else {
		if (a >= nmsgs || b >= nmsgs)
		    mla->fatal("%s: Message number out of range", argv[optind]);
		for (; a <= b; a++) {
		    const MailMsg& msg = table[a];
		    msg.print(stdout, *mla);
		    if (showbody)
			mla->printMsgBody(stdout, msg);
		}
	    }
	}
    }
    return (0);
}

static void
printReplies(FILE* fp, const MLA& mla, const MailMsg& msg)
{
    const MailMsg* table = mla.getMsgTable();
    u_int n = msg.nreplies;
    if (n > 1) {
	const mnum_t* replies = mla.getReplyTable();
	const mnum_t* ix = &replies[msg.replynum];
	for (u_int i = 0; n > 4; i += 3, n -= 3) {
	    printThread(fp, mla, table[ix[0]]);
	    printThread(fp, mla, table[ix[1]]);
	    printThread(fp, mla, table[ix[2]]);
	    ix = &replies[ix[3]];
	}
	switch (n) {
	case 4: printThread(fp, mla, table[*ix++]);
	case 3: printThread(fp, mla, table[*ix++]);
	case 2: printThread(fp, mla, table[*ix++]);
	case 1: printThread(fp, mla, table[*ix]);
	}
    } else
	printThread(fp, mla, table[msg.replynum]);
}

static void
printThread(FILE* fp, const MLA& mla, const MailMsg& msg)
{
    msg.print(fp, mla);
    if (showbody)
	mla.printMsgBody(fp, msg);
    if (msg.hasReplies())
	printReplies(fp, mla, msg);
}

static void
usage(void)
{
    printf("usage: %s [options] [items]\n", progName);
    printf("where options are:\n");
    printf("-b		display message bodies in addition to headers\n");
    printf("-d dir		directory to find archive files\n");
    printf("-T file		MLA table-of-contents database file\n");
    printf("\n");
    printf("items are either message numbers, thread numbers, or message or\n");
    printf("thread sequences.  threads are indicated by a # prefix (#2-4).\n");
    printf("\"first\" and \"last\" refer to the first/last message/thread\n");
    exit(1);
}

#include <stdarg.h>

static void
fatal(const char* fmt ...)
{
    fprintf(stderr, "%s: ", progName);
    va_list ap;
    va_start(ap, fmt);
    vfprintf(stderr, fmt, ap);
    va_end(ap);
    fprintf(stderr, ".\n");
    exit(-1);
}
