#include "ReadOnlyMLA.h"
#include "Sys.h"
#include <errno.h>

static	const char* from = "somebody@somewhere.com";
static	const char* msgsep = "\n\n\n\n";

static	const char* progName;
static	void usage(void);
static	void fatal(const char* fmt ...);

static	void printMsg(FILE* fp, const MLA& mla, const MailMsg& msg);

int
main(int argc, char* argv[])
{
    fxStr tocFile(MLA_TOCNAME);
    fxStr dir(".");

    extern char *optarg;
    extern int optind;
    int c;
    progName = argv[0];
    while ((c = getopt(argc, argv, "d:f:s:T:")) != -1)
	switch (c) {
	case 'd':	dir = optarg; break;
	case 'f':	from = optarg; break;
	case 's':	msgsep = optarg; break;
	case 'T':	tocFile = optarg; break;
	case '?':	usage();
	}

    if (dir != "." && Sys::chdir(dir) < 0)
	fatal("%s: %s: %s", progName, (const char*) dir, strerror(errno));
    ReadOnlyMLA* mla = ReadOnlyMLA::readMLA(tocFile);
    if (mla == NULL)
	usage();
    u_int nmsgs = mla->getMsgCount();
    const MailMsg* table = mla->getMsgTable();
    const char* sep = "";
    for (; optind < argc; optind++) {
	char* cp = argv[optind];
	char* tp;
	mnum_t a;
	if (strncmp(cp, "first", 5) == 0)
	    a = 0, tp = cp+5;
	else if (strncmp(cp, "last", 4) == 0)
	    a = nmsgs-1, tp = cp+4;
	else
	    a = (mnum_t) strtoul(cp, &tp, 0);
	mnum_t b;
	if (tp && *tp == '-') {
	    tp++;
	    if (strcmp(tp, "first") == 0)
		b = 0;
	    else if (strcmp(tp, "last") == 0)
		b = nmsgs-1;
	    else
		b = (mnum_t) strtoul(tp, NULL, 0);
	} else
	    b = a;
	if (a >= nmsgs || b >= nmsgs)
	    mla->fatal("%s: Message number out of range", argv[optind]);
	for (; a <= b; a++) {
	    fputs(sep, stdout);
	    printMsg(stdout, *mla, table[a]), sep = msgsep;
	}
    }
    return (0);
}

static void
printMsg(FILE* fp, const MLA& mla, const MailMsg& msg)
{
    char buf[80];
    strftime(buf, sizeof (buf), "%a %b %d %T %Y", gmtime(&msg.datetime));
    fprintf(fp, "From %s  %s\n", from, buf);
    fprintf(fp, "To: %s\n", mla.checkstr(msg.to));
    fprintf(fp, "Subject: %s\n", mla.checkstr(msg.subject));
    fprintf(fp, "Date: %s\n", mla.checkstr(msg.date));
    const char* email = mla.checkstr(msg.emailaddr);
    if (msg.name)
	fprintf(fp, "From: %s (%s)\n", email, mla.checkstr(msg.name));
    else
	fprintf(fp, "From: %s\n", email);
    const char* replyto = mla.checkstr(msg.replytoaddr);
    if (strcmp(replyto, email))
	fprintf(fp, "Reply-To: %s\n", replyto);
    if (msg.parent)
	fprintf(fp, "In-reply-to: <%s>\n",
	    mla.checkstr(mla.getMsgTable()[msg.parent-1].msgid));
    fprintf(fp, "Message-Id: <%s>\n", mla.checkstr(msg.msgid));
    fputs("\n", fp);
    mla.printMsgBody(fp, msg);
}

static void
usage(void)
{
    printf("usage: %s [options] [msgs]\n", progName);
    printf("where options are:\n");
    printf("-d dir		directory to find archive files\n");
    printf("-f from		specify from mail address\n");
    printf("-s sep		specify inter-message separator string\n");
    printf("-T file		MLA table-of-contents database file\n");
    printf("\n");
    printf("msgs are either message numbers, or message sequences.\n");
    printf("\"first\" and \"last\" refer to the first/last message.\n");
    exit(1);
}

#include <stdarg.h>

static void
fatal(const char* fmt ...)
{
    fprintf(stderr, "%s: ", progName);
    va_list ap;
    va_start(ap, fmt);
    vfprintf(stderr, fmt, ap);
    va_end(ap);
    fprintf(stderr, ".\n");
    exit(-1);
}
