#include "ReadWriteMLA.h"
#include "Str.h"
#include "Sys.h"
#include <sys/file.h>
#include <errno.h>

extern int parseAtSyntax(const char* spec, const struct tm& ref,
	    struct tm& at0, fxStr& emsg);

static	void fatal(const char* fmt ...);
static	void usage(void);
static	const char* progName;
static	void copy(int, int);

static fxBool
isDirectory(const char* path)
{
    struct stat sb;
    return (Sys::stat(path, sb) == -1 ?
	FALSE : (sb.st_mode & S_IFMT) == S_IFDIR);
}

int
main(int argc, char* argv[])
{
    fxStr tocFile(MLA_TOCNAME);
    fxStr dir(".");
    fxStr batchFile("mla.batch");
    fxStr holdspec("now + 5 minutes");
    fxStr maxwaitspec("now + 30 minutes");
    u_int maxmsgs = 2500;
    u_int avgmsgsize = 128;
    u_int maxmsgsize = 3072;		// NB: chosen empirically
    u_int trace = 0;
    fxBool update = TRUE;
    fxBool skip = TRUE;
    fxBool overwrite = FALSE;
    fxBool compact = FALSE;
    fxBool detach = TRUE;
    const char* sep = NULL;
    extern char *optarg;
    extern int optind;
    int c;

    progName = argv[0];
    while ((c = getopt(argc, argv, "b:cd:Dfh:l:m:n:s:tT:w:yz")) != -1)
	switch (c) {
	case 'b':	batchFile = optarg; break;
	case 'c':	update = FALSE; break;
	case 'd':	dir = optarg; break;
	case 'D':	detach = FALSE; break;
	case 'f':	overwrite = TRUE; break;
	case 'h':	holdspec = optarg; break;
	case 'l':	maxmsgsize = strtoul(optarg, NULL, 0); break;
	case 'm':	avgmsgsize = strtoul(optarg, NULL, 0); break;
	case 'n':	maxmsgs = strtoul(optarg, NULL, 0); break;
	case 's':	sep = optarg; break;
	case 't':	trace++; break;
	case 'T':	tocFile = optarg; break;
	case 'w':	maxwaitspec = optarg; break;
	case 'y':	skip = FALSE; break;
	case 'z':	compact = TRUE; break;
	case '?':	usage();
	}

    if (trace)
	setlinebuf(stdout);

    time_t start = Sys::now();
    struct tm now = *localtime(&start);
    struct tm at0;
    fxStr emsg;
    if (!parseAtSyntax(holdspec, now, at0, emsg))
	fatal("%s", (const char*) emsg);
    time_t holdTime = ::mktime(&at0) - start;
    if (!parseAtSyntax(maxwaitspec, now, at0, emsg))
	fatal("%s", (const char*) emsg);
    time_t waitTime = ::mktime(&at0) - start;

    (void) umask(0);
    if (dir != ".") {
	if (!isDirectory(dir)) {
	    if (mkdir(dir, 0755) == -1)
		fatal("Could not create archive directory \"%s\": %s",
		    (const char*) dir, strerror(errno));
	    if (trace)
		printf("Creating directory \"%s\", mode %o.\n",
		    (const char*) dir, 0755);
	}
	if (Sys::chdir(dir) < 0)
	    fatal("chdir: %s", strerror(errno));
    }

    /*
     * If we're the first batch job, then create the batch
     * file and write the message.  Otherwise, there's already
     * another instance of mlabatch waiting to do an update
     * and we can just write the message and terminate.  Must
     * do this carefully to insure that data are not lost.
     */
    fxBool waitForUpdate;
    int batchFd;
    int ntries = 0;
    struct stat sb;
retry:
    waitForUpdate = TRUE;
    batchFd = Sys::open(batchFile, O_RDWR|O_APPEND|O_CREAT|O_EXCL, 0644);
    if (batchFd < 0 && errno == EEXIST) {
	batchFd = Sys::open(batchFile, O_WRONLY|O_APPEND|O_CREAT, 0644);
	waitForUpdate = FALSE;
    }
    if (batchFd < 0)
	fatal("%s: open: %s", (const char*) batchFile, strerror(errno));
    if (trace) {
	if (waitForUpdate)
	    printf("We are the master %s, wait to do update.\n", progName);
	else
	    printf("There appears to be another %s around, "
		"let it do the update.\n", progName);
    }
    /*
     * Only one writer is permitted at a time to avoid
     * intermixing message data.  Also when an update
     * takes place the batch file is locked so that
     * only complete messages are inserted.  If we fail
     * here it may be because an update has completed
     * and there is no longer another batch job waiting
     * to the update--in case we start over again in
     * case now need to act as the job that handles the
     * update procedure.
     */
    if (flock(batchFd, LOCK_EX) < 0)
	fatal("%s: flock: %s", (const char*) batchFile, strerror(errno));
    if (trace)
	printf("Locked the batch file.\n");
    if (Sys::fstat(batchFd, sb) < 0)
	fatal("%s: fstat: %s", (const char*) batchFile, strerror(errno));
    if (!waitForUpdate && sb.st_size == 0) {
	/*
	 * If we thought there was another batch job around
	 * to do the update but when we finally got control
	 * of the batch file it's truncated to zero length,
	 * then the original batch process has terminated and
	 * we need to retry in order to properly decide if
	 * we should stick around and the update ourselves.
	 */
	close(batchFd);
	if (trace)
	    printf("The batch file was truncated, "
		"let's try this again (try %d).\n", ntries);
	if (++ntries > 3)
	    fatal("%s: Confusion over who does the batch update",
		(const char*) batchFile);
	goto retry;
    }

    /*
     * The batch file should now be open and locked;
     * copy the messages and either terminate or wait
     * to do the update.
     */
    if (optind < argc) {
	do {
	    if (strcmp(argv[optind], "-")) {
		int fd = open(argv[optind], O_RDONLY);
		if (fd < 0)
		    fatal("%s: Cannot open, errno %s",
			argv[optind], strerror(errno));
		copy(batchFd, fd);
		close(fd);
	    } else
		copy(batchFd, fileno(stdin));
	} while (++optind < argc);
    }
    if (!waitForUpdate) {			// job done, go away
	if (trace)
	    printf("Work done, someone else should do the update.\n");
	return (0);
    }

    /*
     * Release the lock on the batch file so that
     * other batch jobs may write to the batch file.
     * We are committed to do the update at the
     * appropriate time.
     */
    Sys::fstat(batchFd, sb);
    time_t lastmod = sb.st_mtime;
    (void) flock(batchFd, LOCK_UN);

    /*
     * Fork so the parent can continue.  This is important
     * for tools like MH's slocal that guard against delivery
     * problems by killing delivery agents that take too
     * long to do their job.
     */
    if (detach) {

	ntries = 0;
again:
	switch (::fork()) {
	case -1:				// error
	    if (errno == EAGAIN) {
		if (++ntries <= 3) {
		    sleep(1);
		    goto again;
		}
		fprintf(stderr, "%s: Warning, can not fork,"
		    " doing work in the foreground.\n", progName);
		break;
	    }
	    return (-1);
	case 0:	break;				// child, continue
	default:return (0);			// parent, terminate
	}
    }

    /*
     * Now wait around so that other messages can be
     * appended to the batch file before we do the
     * update.  We wait at least ``hold'' time and at
     * most ``max wait'' time.  If after the hold time
     * we find new messages have arrived, we delay another
     * hold period in the hope more messages will be
     * delivered.  This continues up to the max wait
     * time after which we do the update whether or
     * not more messages have recently arrived.
     */
    start = Sys::now();
    do {
	if (trace)
	    printf("Sleep for %d seconds.\n", holdTime);
	::sleep(holdTime);
	if (Sys::fstat(batchFd, sb) >= 0 && sb.st_mtime != lastmod) {
	    lastmod = sb.st_mtime;
	    if (trace)
		printf("The batch file mod time changed.\n");
	}
    } while (Sys::now() - lastmod < holdTime && Sys::now() - start < waitTime);
    if (trace)
	printf("Done waiting, time to do the update work.\n");

    /*
     * Time to do the work; relock the file and
     * proceed with the update operation.
     */
    if (flock(batchFd, LOCK_EX) < 0)
	fatal("Unable to re-lock batch file: %s", strerror(errno));
    (void) lseek(batchFd, SEEK_SET, 0);		// rewind
    FILE* fp = fdopen(batchFd, "r");
    if (fp == NULL)
	fatal("Unable to fdopen batch file");

    ReadWriteMLA* mla;
    if (update && Sys::isRegularFile(tocFile)) {
	mla = ReadWriteMLA::readMLA(tocFile);
	if (mla == NULL)
	    return (-1);
    } else {
	mla = new ReadWriteMLA(tocFile, maxmsgs, avgmsgsize, maxmsgsize);
    }
    mla->setOverwrite(overwrite);
    mla->setTrace(trace);
    mla->setCompactPool(compact);
    if (sep != NULL)
	mla->setMsgSeparator(sep);
    mla->load(fp, skip);			// load the batched messages

    /*
     * Remove the batch file so that the next
     * batch request will do an update.  Before
     * we unlink the file however, truncate it
     * to zero length so that any batch job waiting
     * for us to do an update for their message(s)
     * will reconsider whether or not they should do
     * the update work themselves (see above).
     */
    ftruncate(batchFd, 0);
    Sys::unlink(batchFile);
    if (trace)
	printf("Work purged from the batch file, update the MLA and exit.\n");
    return (mla->update() ? 0 : 1);
}

static void
copy(int dstfd, int srcfd)
{
    char buf[16*1024];
    int n;

    while ((n = read(srcfd, buf, sizeof (buf))) > 0) {
	int dn = write(dstfd, buf, n);
	if (dn < 0)
	    fatal("Batch file write error: %s", strerror(errno)); 
	else if (dn != n)
	    fatal("Batch file write short, requested %u, wrote %u", n, dn);
    }
}

static void
usage(void)
{
    printf("usage: %s [options] [files]\n", progName);
    printf("where options are:\n");
    printf("-c		create the archive, even if it exists\n");
    printf("-d dir		directory to write archive files\n");
    printf("-f		force overwriting of existing data\n");
    printf("-l #		set the maximum message size (default 3072)\n");
    printf("-m #		set the average message size (default 128)\n");
    printf("-n #		set the maximum number of messages (default 2500)\n");
    printf("-s sep		set the message separator string (default \"From \")\n");
    printf("-t		trace work (multiple times for more info)\n");
    printf("-T file		MLA table-of-contents database file\n");
    printf("-y		do not skip to the first message separator (default FALSE)\n");
    printf("-z		compact the database (default FALSE)\n");
    printf("Giving \"-\" for a file forces messages to be read from standard input\n");
    exit(1);
}

#include <stdarg.h>

void
fatal(const char* fmt ...)
{
    fflush(stdout);
    fprintf(stderr, "%s: ", progName);
    va_list ap;
    va_start(ap, fmt);
    vfprintf(stderr, fmt, ap);
    va_end(ap);
    fputs(".\n", stderr);
    exit(-1);
}
