/*	$Header: /usr/people/sam/irix/html/tools/MLA/RCS/Set.c++,v 1.3 1996/10/13 03:46:57 sam Exp $ */
/*
 * Copyright (c) 1990, 1991, 1992, 1993, 1994, 1995 Sam Leffler
 * Copyright (c) 1991, 1992, 1993, 1994, 1995 Silicon Graphics, Inc.
 *
 * Permission to use, copy, modify, distribute, and sell this software and 
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the names of
 * Sam Leffler and Silicon Graphics may not be used in any advertising or
 * publicity relating to the software without the specific, prior written
 * permission of Sam Leffler and Silicon Graphics.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 * 
 * IN NO EVENT SHALL SAM LEFFLER OR SILICON GRAPHICS BE LIABLE FOR
 * ANY SPECIAL, INCIDENTAL, INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND,
 * OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER OR NOT ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF 
 * LIABILITY, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE 
 * OF THIS SOFTWARE.
 */
#include "Set.h"

#define DEFAULTSIZE 31

fxIMPLEMENT_PtrArray(fxSetBuckets,fxSetBucket *);
fxIMPLEMENT_PtrArray(fxSetIters, fxSetIter *);

#define KEY(b) b->kvmem
#define KEYAT(b) b

fxSet::fxSet(u_int ksize, u_int initsize)
{
    if (initsize == 0) initsize = DEFAULTSIZE;
    buckets.resize(initsize);
    keysize = ksize;
    numKeys = 0;
}

fxSet::fxSet(const fxSet& a)
{
    for (int i = 0; i < a.buckets.length(); i++)
	for (const fxSetBucket* sb = a.buckets[i]; sb; sb = sb->next)
	    add(KEY(sb));
}

fxSet::~fxSet()
{
}

void fxSet::cleanup()
{
    u_int l = buckets.length();
    u_int i;
    for (i=0; i < l; i++) {
	fxSetBucket* bucket = buckets[i];
	while (bucket) {
	    fxSetBucket* bucket2 = bucket->next;
	    destroyKey(KEY(bucket));
	    invalidateIters(bucket);
	    delete bucket;
	    bucket = bucket2;
	    numKeys--;
	}
    }
    assert(numKeys == 0);
    l = iters.length();
    for (i=0; i < l; i++) {
	iters[i]->set = 0;
	iters[i]->node = 0;
    }
}

fxSetBucket::~fxSetBucket()
{
    delete kvmem;
}

void
fxSet::operator=(const fxSet& a)
{
    assert(keysize == a.getKeySize());
    if (this == &a) return;
    this->fxSet::~fxSet();
    for (int i = 0; i < a.buckets.length(); i++)
	for (const fxSetBucket* db = a.buckets[i]; db; db = db->next)
	    add(KEY(db));
}

fxBool
fxSet::contains(const void* key) const
{
    u_int index = (u_int)(hashKey(key) % buckets.length());
    const fxSetBucket* bucket = buckets[index];
    while (bucket && compareKeys(key,KEY(bucket)))
	bucket = bucket->next;
    return (bucket != 0);
}

void*
fxSet::find(const void *key) const
{
    u_int index = (u_int)(hashKey(key) % buckets.length());
    const fxSetBucket* bucket = buckets[index];
    while (bucket && compareKeys(key,KEY(bucket)))
	bucket = bucket->next;
    return (bucket ? KEY(bucket) : 0);
}

void
fxSet::add(const void* key)
{
    u_int index = (u_int)(hashKey(key) % buckets.length());
    fxSetBucket* bucket = buckets[index];
    while (bucket && compareKeys(key,KEY(bucket)))
	bucket = bucket->next;
    if (!bucket) {
	char* kvmem = (char*) malloc(keysize);
	copyKey(key,kvmem);
	fxSetBucket* bucket = new fxSetBucket(kvmem,buckets[index]);
	buckets[index] = bucket;
	numKeys++;
    }
}

void
fxSet::remove(void const* key)
{
    u_int index = (u_int) (hashKey(key) % buckets.length());
    fxSetBucket* bucket = buckets[index];
    fxSetBucket** prev = &buckets[index];
    while (bucket && compareKeys(key,KEY(bucket))) {
	prev = &bucket->next;
	bucket = bucket->next;
    }
    if (bucket) {
	*prev = bucket->next;
	destroyKey(KEY(bucket));
	invalidateIters(bucket);
	delete bucket;
	numKeys--;
    }
}

u_long
fxSet::hashKey(const void* key) const
{
    u_long u = 0;
    const u_long* p = (const u_long*) key;
    int l = (int) keysize;
    while (l >= sizeof (u_long)) {
	u ^= *p++;
	l -= sizeof (u_long);
    }
    return (u);
}

int
fxSet::compareKeys(const void* key1, const void* key2) const
{
    return ::memcmp(key1, key2, keysize);
}

void
fxSet::copyKey(const void* src, void* dst) const
{
    ::memmove(dst,src,keysize);
}

void
fxSet::destroyKey(void*) const
{}

void
fxSet::addIter(fxSetIter* i)
{
    iters.append(i);
}

void
fxSet::removeIter(fxSetIter* i)
{
    iters.remove(iters.find(i));
}

void
fxSet::invalidateIters(const fxSetBucket* db) const
{
    u_int l = iters.length();
    for (u_int i = 0; i < l; i++) {
	fxSetIter* di = (fxSetIter*) iters[i];
	if (di->node == db) {
	    (*di)++;
	    if (di->set)
		di->invalid = TRUE;
	}
    }
}

void
fxSet::operator|=(const fxSet& b)
{
    u_int l = b.buckets.length();
    for (u_int i = 0; i < l; i++)
	for (const fxSetBucket* sb = b.buckets[i]; sb; sb = sb->next)
	    add(KEY(sb));
}

void
fxSet::operator&=(const fxSet& b)
{
    u_int l = buckets.length();
    for (u_int i = 0; i < l; i++) {
	fxSetBucket* sb = buckets[i];
	fxSetBucket** prev = &buckets[i];
	while (sb) {
	    if (!b.contains(KEY(sb))) {
		fxSetBucket* next = sb->next;
		*prev = next;
		destroyKey(KEY(sb));
		invalidateIters(sb);
		delete sb;
		numKeys--;
		sb = next;
	    } else {
		prev = &sb->next;
		sb = sb->next;
	    }
	}
    }
}

void
fxSet::operator-=(const fxSet& b)
{
    u_int l = b.buckets.length();
    for (u_int i = 0; i < l; i++)
	for (const fxSetBucket* sb = b.buckets[i]; sb; sb = sb->next)
	    remove(KEY(sb));
}

fxSetIter::fxSetIter()
{
    set = 0;
    bucket = 0;
    node = 0;
    invalid = FALSE;
}

fxSetIter::fxSetIter(fxSet& d)
{
    set = &d;
    bucket = 0;
    node = d.buckets[bucket];
    invalid = FALSE;
    d.addIter(this);
    if (!node)
	advanceToValid();
}

fxSetIter::~fxSetIter()
{
    if (set)
	set->removeIter(this);
}

void
fxSetIter::operator=(fxSet& d)
{
    if (set)
	set->removeIter(this);
    set = &d;
    bucket = 0;
    node = d.buckets[bucket];
    invalid = FALSE;
    d.addIter(this);
    if (!node)
	advanceToValid();
}

void const*
fxSetIter::getKey() const
{
    return (invalid ? 0 : KEY(node));
}

void
fxSetIter::increment()
{
    if (set) {
	if (invalid)
	    invalid = FALSE;
	else if ((node = node->next) == NULL)
	    advanceToValid();
    }
}

void
fxSetIter::advanceToValid()
{
    u_int len = set->buckets.length();
    fxSetBucket* n;
    for(;;) {
	bucket++;
	assert(bucket<=len);
	if (bucket == len) {
	    set->removeIter(this);	// it's done with us
	    set = 0;
	    invalid = TRUE;
	    break;
	}
	if (n = set->buckets[bucket]) {	// intentional =
	    node = n;
	    invalid = FALSE;
	    break;
	}
    }
}
