#include "MailMsg.h"
#include "MLA.h"
#include "Str.h"
#include "Sys.h"

/*
 * Mail Message.
 */
void
MailMsg::setup(mnum_t m)
{
    msgnum = m;
    thread = 0;
    parent = 0;
    maybereply = FALSE;
    datetime = 0;
    name = MLA::nullstr;
    emailaddr = MLA::nullstr;
    replytoaddr = MLA::nullstr;
    to = MLA::nullstr;
    date = MLA::nullstr;
    subject = MLA::nullstr;
    msgid = MLA::nullstr;
    nreplies = 0;
    replynum = 0;
}

void
MailMsg::print(FILE* fd, const MLA& mla) const
{
    fprintf(fd, "---- Msg[%d]---\n", this - mla.getMsgTable());
    fprintf(fd, "Msgnum:  %-5d Thread: %d\n", msgnum, thread);
    fprintf(fd, "Parent:  %-5d Maybe:  %d\n",
	parent ? parent-1 : 0, maybereply);
    char date[80];
    struct tm* tm = gmtime(&datetime);
    strftime(date, sizeof (date), "%a, %d %b %Y %T", tm);
    fprintf(fd, "Time:    %s\n", date);
    printHeader(fd, mla);
    if (hasReplies()) {
	mla.printReplyNums(fd, *this);
	putc('\n', fd);
    }
    fputs("-------\n", fd);
}

void
MailMsg::printHeader(FILE* fd, const MLA& mla) const
{
    fprintf(fd, "Name:    \"%s\"\n", mla.checkstr(name));
    fprintf(fd, "Email:   \"%s\"\n", mla.checkstr(emailaddr));
    fprintf(fd, "ReplyTo: \"%s\"\n", mla.checkstr(replytoaddr));
    fprintf(fd, "To:      \"%s\"\n", mla.checkstr(to));
    fprintf(fd, "Date:    \"%s\"\n", mla.checkstr(date));
    fprintf(fd, "Subject: \"%s\"\n", mla.checkstr(subject));
    fprintf(fd, "MsgID:   \"%s\"\n", mla.checkstr(msgid));
}

#define	N(a)	(sizeof (a) / sizeof (a[0]))

const char* MailMsg::months[12] = {
    "Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul",
    "Aug", "Sep", "Oct", "Nov", "Dec"
};
const int MailMsg::monthdays[12] =
    { 31, 28, 31, 30, 31, 30, 31, 31, 30, 30, 30, 31 };

#define CENTURY        1900
#define BASEYEAR       1970
#define DAYSPERYEAR    365
#define SECSPERMIN     60
#define SECSPERHOUR    (60*60)
#define SECSPERDAY     (24*SECSPERHOUR)

static inline int isLeapYear(int y)
    { return (y > 1752 && (y%4 == 0 && (y%100 != 0 || y%400 == 0))); }

/*
 * Convert month/day/year items to a time_t.
 */
static time_t
getTicks(int month, int day, int year)
{
    int yday = day;				// day of year
    if (month > 2 && isLeapYear(year))		// add leap day in February
	yday++;
    do {
	if (isLeapYear(year))
	    yday++;
	yday += DAYSPERYEAR;			// full years
    } while (--year > BASEYEAR);
    month--;
    while (--month >= 0)
	yday += MailMsg::monthdays[month];	// full months

    return (yday*SECSPERDAY);
}

static int
getMonth(const char* s)
{
    int m = 0;
    while (m < N(MailMsg::months) && strncasecmp(s, MailMsg::months[m], 3))
	m++;
    return (m+1);
}

/*
 * Convert a From-line-style date string
 * to a time_t: Thu Jan  5 13:26:39 1995.
 */
time_t
MailMsg::cvtFromDate(const char* date)
{
    date += 4;					// skip day of week
    int month = getMonth(date);
    int d, y, h, m, s;
    if (sscanf(date, "%*3s %d %d:%d:%d %d", &d, &h, &m, &s, &y) == 5)
	return (getTicks(month, d-1, y) + (h*SECSPERHOUR) + (m*SECSPERMIN) + s);
    else
	return (0);				// XXX
}

/*
 * Convert a Devliery-Date:-style date string
 * to a time_t: Tue, 29 Nov 1994 15:49:19 PST
 */
time_t
MailMsg::cvtDeliveryDate(const char* date)
{
    date += 5;					// skip day of week
    int month = getMonth(date+3);
    int d, y, h, m, s;
    if (sscanf(date, "%d %*3s %d %d:%d:%d", &d, &y, &h, &m, &s) == 5)
	return (getTicks(month, d-1, y) + (h*SECSPERHOUR) + (m*SECSPERMIN) + s);
    else
	return (0);				// XXX
}

static inline int cvt(char a, char b) { return 10*(a-'0')+(b-'0'); }

/*
 * Split a MM/DD/YY-style date into
 * month, day, and year components.
 */
static void
splitdate(const char* date, int& month, int& day, int& year)
{
    if (date[0] == ' ' || date[0] == '0')
	month = cvt('0', date[1]);
    else if (date[1] == '/')
	month = cvt('0', date[0]);
    else if (date[2] == '/')
	month = cvt(date[0], date[1]);

    if (date[2] =='/') {
	if (date[5] == '/') {
	    day = cvt(date[3], date[4]);
	    year = cvt(date[6], date[7]);
	} else if (date[4] == '/') {
	    day = cvt('0', date[3]);
	    year = cvt(date[5], date[6]);
	}
    } else if (date[1] =='/') {
	if (date[4] == '/') {
	    day = cvt(date[2], date[3]);
	    year = cvt(date[5], date[6]);
	} else if (date[3] == '/') {
	    day = cvt('0', date[2]);
	    year = cvt(date[4], date[5]);
	}
    }
}

/*
 * Convert a MM/DD/YY-style date string to a time_t.
 */
time_t
MailMsg::cvtQueryDate(const fxStr& date)
{
    if (date.length() == 8) {
	int month, day, year;
	splitdate(date, month, day, year);
	year += CENTURY;
	return (getTicks(month, day, year));
    } else
	return (0);
}
