#ifndef _MLA_
#define	_MLA_

#include <stdarg.h>

#include "Str.h"
#include "MailMsg.h"
#include "zlib.h"

#define	MLA_MAGIC	0x4557		// magic number
#define	MLA_MAJOR	1		// major version
#define	MLA_MINOR	0		// minor version

#define	MLA_TOCNAME	"mla.toc"	// default toc db filename
#define	MLA_MSGNAME	"mla.msg"	// default msg db filename
#define	MLA_IXNAME	"mla.ix"	// default index db filename
#define	MLA_MSGSEP	"From "		// default inter-message separator

/*
 * On-disk database:
 *
 * fixed-size header		MLAHeader
 * message descriptor table	maxmsgs * sizeof (MailMsg)
 * thread table			maxthreads * sizeof (mnum_t)
 * reply spillover table	maxreplies * sizeof (mnum_t)
 * sorted message table		maxmsgs * sizeof (mnum_t)
 * hashed string pool		maxstrspace
 */
struct MLAHeader {
    u_short	magic;			// magic number
    u_short	version;		// format version number
    u_int	nmsgs;			// number of messages
    u_int	maxmsgs;		// max messages struct can handle
    u_int	nthreads;		// number of threads
    u_int	maxthreads;		// max threads struct can handle
    u_int	nreplies;		// size of reply spillover table
    u_int	maxreplies;		// max size of spillover table
    u_int	strcount;		// count of strings in string pool
    u_int	strspace;		// space used in string pool
    u_int	maxstrspace;		// max space of string pool
    off_t	msgseparator;		// pattern that separates messages
    off_t	msgfile;		// message database filename
    u_int	maxmsgsize;		// msgs larger are spilled to files
    off_t	ixfile;			// optional inverted index filename
    off_t	pad[12];		// spares for future expansion
    char	msgs[1];		// start of message table
    // thread table follows
    // reply spillover table follows
    // sorted message table follows
    // string pool follows
};

struct __db;
class ixDB;

/*
 * Mailing List Archive Database.
 */
class MLA {
public:
    struct strhash {
	strhash* next;		// linked list of hash buckets
	const char* data;	// string value
	off_t off;		// offset into hash pool
	u_int len;		// string length (w/o null)
    };
protected:
    int	fd;			// open descriptor for mapped file
    const char* data;		// mapped file
    size_t size;		// size of mapped file region
    u_int nmsgs;		// number messages in table
    u_int maxmsgs;		// size of message table
    const MailMsg* table;	// table of all messages
    const mnum_t* sorted;	// table sorted by date+time
    u_int nthreads;		// number of threads in table
    u_int maxthreads;		// size of thread table
    const mnum_t* threads;	// top-of-thread messages
    u_int nreplies;		// number of entries in spillover table
    u_int maxreplies;		// size of spillover table
    const mnum_t* replies;	// spillover table of message replies from db
    fxStr tocFile;		// toc filename
    fxStr msgFile;		// message body filename
    fxStr ixFile;		// keyword index filename, if present
    fxStr msgSeparator;		// message separator pattern to search for
    int trace;			// tracing

    strhash* buckets[128];	// hashed string pool
    size_t strspace;		// space in pool already allocated
    size_t maxstrspace;		// max space for string pool in db
    u_int strcount;		// total count of pooled strings
    const char* strpool;	// string pool, typically mapped from file
    strhash* strhashtab;	// hash entries for pooled strings read from db
    strhash* estrhashtab;	// end of strhashtab array
    strhash* strhashp;		// next available entry in strhashtab

    struct __db* msgdb;		// message body database
    u_int maxmsgsize;		// msgs larger are spilled to files
    ixDB* ixdb;			// keyword index database

    MLA(const char* toc, u_int maxmsgs, u_int avgmsgsize, u_int maxmsgsize);
    MLA(const char* toc);

    virtual void vprintMsg(const char* fmt, va_list ap) const;

    fxBool openMsgDB(u_int flags);
    fxBool openIXDB(u_int flags);

    static fxBool openMLA(const char* file, int flags, int& fd);
    fxBool setupMLA(int fd0, int prot, int share);
    void loadMLA(void);
    void createMLA(void);

    void decode(FILE* fout, z_stream& zstream, const MailMsg& msg) const;

    const char* hashstr(const char* s, fxBool isnew = TRUE);
    off_t stroff(const char* s) const;
    const char* getstr(off_t);
    void compact(char* cp, u_int& sc, u_int& ss);	// compact string pool
    fxBool isoffset(const char* s) const
	{ return (unsigned) s < maxstrspace; }
    const char* offstr(const char* s) const
	{ return (strpool + (off_t) s); }
    void purgeStringPool();
public:
    virtual ~MLA();

    virtual void warning(const char* fmt ...) const;
    virtual void error(const char* fmt ...) const;
    virtual void fatal(const char* fmt ...) const;

    static const char* nullstr;		// canonical null string

    const char* checkstr(const char* s) const
	{ return (isoffset(s) ? offstr(s) : s); }
    static void trimWS(fxStr&);

    const fxStr& getTOCFile(void) const		{ return tocFile; }
    const fxStr& getMsgFile(void) const		{ return msgFile; }
    const fxStr& getIXFile(void) const		{ return ixFile; }

    u_int getTrace(void) const			{ return trace; }
    void setTrace(u_int);

    const MailMsg* getMsgTable() const		{ return table; }
    u_int getMsgCount() const			{ return nmsgs; }
    const mnum_t* getSortedMsgTable() const	{ return sorted; }
    const MailMsg& getSortedMsg(mnum_t m) const	{ return table[sorted[m]]; }

    const mnum_t* getThreadTable() const	{ return threads; }
    u_int getThreadCount() const		{ return nthreads; }

    const mnum_t* getReplyTable() const		{ return replies; }
    u_int getReplyCount() const			{ return nreplies; }

    u_int getStrCount() const			{ return strcount; }
    size_t getStrSpace() const			{ return strspace; }

    void printHeader(FILE*) const;
    void printReplyNums(FILE* fd, const MailMsg& msg) const;
    void printMsgBody(FILE* fp, const MailMsg& msg) const;
};
#endif /* _MLA_ */
