#include "CGIForm.h"
#include "ReadOnlyMLA.h"
#include "Sys.h"
#include <sys/file.h>			// for LOCK_*
#include <sys/mman.h>
#include <errno.h>
#include <ctype.h>

#define	EODAY	(24*60*60-1)		// end-of-day (24 hours - 1 second)

class CGIReadOnlyMLA : public ReadOnlyMLA {
protected:
    CGIReadOnlyMLA(const char* toc);
public:
    virtual ~CGIReadOnlyMLA();

    static CGIReadOnlyMLA* readMLA(const char* file);

    virtual void vprintMsg(const char* fmt, va_list ap) const;
};

#define	N(a)	(sizeof (a) / sizeof (a[0]))

CGIForm::CGIForm(const char* name, const char* form)
    : appName(name)
    , scheme("thread")
    , tocFile(MLA_TOCNAME)
    , formFile(form)
    , dir(".")
{
    maxhits = 200;
    maxlevels = 3;
    beginTime = 0;
    endTime = 0;
    reverse = FALSE;
    ignoreCase = TRUE;
    isCGI = FALSE;
    msgnum = (mnum_t) -1;
    trace = 0;
}
CGIForm::~CGIForm() {}

static int
hex(int c)
{
    return isdigit(c) ? c-'0' : 10+(isupper(c) ? c-'A' : c-'a');
}

static void
fixup(char* cp)
{
    for (; *cp; cp++) {
	if (*cp == '+')
	    *cp = ' ';
	else if (*cp == '%') {
	    cp[0] = (hex(cp[1])<<4) + hex(cp[2]);
	    memmove(cp+1, cp+3, strlen(cp+3)+1);
	}
    }
}

static const char*
getscheme(const char* s)
{
    char c = tolower(*s);
    return (c == 'a' ?  "author" :
	    c == 's' ?  "subject" :
	    c == 'd' ?  "date" :
			"thread");
}

static fxBool
optbool(const char* v, fxBool def)
{
     return (v == NULL ? def :
	 (strcasecmp(v, "yes") == 0 || strcasecmp(v, "true") == 0));
}

#define	match(s)	(strcasecmp(tag, s) == 0)

void
CGIForm::parseQueryString(char* env)
{
    /*
     * Remove specification for this message number so
     * that formulated links have only one message spec
     * in them--this is so that browsers can identify
     * visited links properly.
     */
    char* cp = env;
    if (strncasecmp("Msg=", env, 4) == 0) {
	cp += 4;
	while (*cp && *cp++ != '&')
	    ;
    }
    queryString = cp;

    fxStr startDay, startMonth, startYear;
    fxStr endDay, endMonth, endYear;
    do {
	if (cp = strchr(env, '&'))
	    *cp++ = '\0';
	char* tag = env;
	char* value = strchr(env, '=');
	if (value) {
	    *value++ = '\0';
	    fixup(value);
	    if (*value == '\0')
		value = NULL;
	}
	fixup(tag);
	if (match("RV"))		reverse = optbool(value, TRUE);
	else if (match("IC"))		ignoreCase = optbool(value, TRUE);
	else if (value) {
		 if (match("From"))		fromPat = value;
	    else if (match("Subj"))		subjPat = value;
	    else if (match("Msg"))		msgnum = atoi(value);
	    else if (match("SM"))		startMonth = value;
	    else if (match("SD"))		startDay = value;
	    else if (match("SY"))		startYear = value;
	    else if (match("EM"))		endMonth = value;
	    else if (match("ED"))		endDay = value;
	    else if (match("EY"))		endYear = value;
	    else if (match("MH"))		maxhits = atoi(value);
	    else if (match("ML"))		maxlevels = atoi(value);
	    else if (match("SC"))		scheme = getscheme(value);
	    else if (match("Hits"))		hitset = value;
	}
    } while (env = cp);

    /*
     * Calculate begin+end dates for query in case the form uses them.
     */
    if (startDay != "" || startMonth != "" || startYear != "") {
	if (startDay != "") {
	    MLA::trimWS(startDay);
	    if (startDay.length() != 2)
		startDay.insert('0');
	} else
	    startDay = "01";
	beginTime = MailMsg::cvtQueryDate(
	    startMonth | "/" | startDay | "/" | startYear);
    }
    if (endDay != "" || endMonth != "" || endYear != "") {
	if (endDay == "") {			// last day of month
	    u_int m = atoi(endMonth);
	    endDay = MailMsg::monthdays[m < 12 ? m-1 : 11];
	} else {
	    MLA::trimWS(endDay);
	    if (endDay.length() != 2)
		endDay.insert('0');
	}
	endTime = MailMsg::cvtQueryDate(
	    endMonth | "/" | endDay | "/" | endYear) + EODAY;
    }

    /*
     * If possible, compress the query string to save space
     * in the generated HTML (it's emitted for each fetch).
     */
    if (fromPat == "" && (cp = strstr(queryString, "From=&")))
	queryString.remove(cp - (char*) queryString, 6);
    if (subjPat == "" && (cp = strstr(queryString, "Subj=&")))
	queryString.remove(cp - (char*) queryString, 6);
    for (u_int i = 0; i < queryString.length();) {
	if (queryString[i] == '+')
	    queryString.remove(i, queryString.skip(i,'+')-i);
	else
	    i++;
    }
}

void
CGIForm::parseArgcArgv(int argc, char* argv[])
{
    extern char *optarg;
    int c;
    while ((c = getopt(argc, argv, "b:c:d:e:f:h:il:m:rs:T:t")) != -1)
	switch (c) {
	case 'b':	beginTime = MailMsg::cvtQueryDate(optarg); break;
	case 'c':	scheme = optarg; break;
	case 'd':	dir = optarg; break;
	case 'e':	endTime = MailMsg::cvtQueryDate(optarg) + EODAY; break;
	case 'f':	fromPat = optarg; break;
	case 'h':	maxhits = atoi(optarg); break;
	case 'i':	ignoreCase = TRUE; break;
	case 'l':	maxlevels = atoi(optarg); break;
	case 'm':	msgnum = atoi(optarg); break;
	case 'r':	reverse = TRUE; break;
	case 's':	subjPat = optarg; break;
	case 't':	trace++; break;
	case 'T':	tocFile = optarg; break;
	case '?':	usage();
	}
}

void
CGIForm::setupArgs(int argc, char* argv[])
{
    char* env = getenv("QUERY_STRING");
    if (env != NULL) {
	isCGI = TRUE;
	printf("Content-type: text/html\n\n");

	parseQueryString(env);

	env = getenv("PATH_TRANSLATED");
	if (env && *env) {
	    dir = env;
	    dir.resize(dir.nextR(dir.length(), '/'));
	    if (env[dir.length()] != '\0')
		tocFile = env+dir.length();
	}
    } else {
	isCGI = FALSE;
	parseArgcArgv(argc, argv);
    }
    if (dir != "." && Sys::chdir(dir) < 0) {
	fprintf(isCGI ? stdout : stderr, "%s: %s: %s",
	    appName, (const char*) dir, strerror(errno));
	exit(-1);
    }
}

void
CGIForm::usage(void) const
{
    printf("usage: %s [options]\n", appName);
    printf("where options are:\n");
    printf("-b date		begin date for query\n");
    printf("-c scheme	set collation scheme\n");
    printf("-d dir		directory to write files\n");
    printf("-e date		end date for query\n");
    printf("-f regex	From: regex pattern\n");
    printf("-h n		max number of hits to report (default 200)\n");
    printf("-i		ignore upper/lower case distinction\n");
    printf("-l n		max number of sub-thread levels (default 3)\n");
    printf("-m num		set message number\n");
    printf("-r		reverse sort hit set\n");
    printf("-s regex	Subject: regex pattern\n");
    printf("-t		trace work (multiple times for more info)\n");
    printf("-T file		TOC database filename\n");
    printf("Unspecified search patterns match anything.\n");
    exit(1);
}

ReadOnlyMLA*
CGIForm::readMLA() const
{
    ReadOnlyMLA* mla = isCGI ?
	CGIReadOnlyMLA::readMLA(tocFile) : ReadOnlyMLA::readMLA(tocFile);
    if (mla != NULL)
	mla->setTrace(trace);
    return (mla);
}

void
CGIForm::readForm(const MLA& mla, const char*& data, size_t& size) const
{
    int fd = Sys::open(formFile, O_RDONLY);
    if (fd < 0)
	mla.fatal("%s: %s", (const char*) formFile, strerror(errno));
    struct stat sb;
    (void) Sys::fstat(fd, sb);
    size = sb.st_size;
    data = (const char*) mmap(NULL, size, PROT_READ, MAP_SHARED, fd, 0);
    if (data == (const char*) -1) {
	char* cp = new char[size];
	if (::read(fd, cp, size) != size)
	    mla.fatal("%s: read: %s", (const char*) formFile, strerror(errno));
	close(fd), fd = -1;
	data = cp;
    }
}

void
CGIForm::query(ReadOnlyMLA& mla, MailMsgArray& hits,
    RegEx* fromPat, RegEx* subjPat, fxBool& more)
{
    mla.query(hits, maxhits, beginTime, endTime, fromPat, subjPat, more);
}

void
CGIForm::printEscapes(FILE* fp, const MLA& mla, char code)
{
    const char* cp;

    switch (code) {
    case 'b':		// begining search date
	printDate(fp, *gmtime(&beginTime));
	break;
    case 'e':		// ending search date
	printDate(fp, *gmtime(&endTime));
	break;
    case 'c':		// collation scheme
	fputs(scheme, fp);
	break;
    case 'f':		// from pattern string
	if (fromPat != "")
	    printString(fp, fromPat);
	break;
    case 'h':		// host name
	if (cp = getenv("SERVER_NAME"))
	    fputs(cp, fp);
	break;
    case 'i':		// upper/lower case being ignored
	fputs(ignoreCase ?
	    "ignore upper/lower case" : "distinguish upper/lower case", fp);
	break;
    case 'm':		// count of messages
	fprintf(fp, "%u", mla.getMsgCount());
	break;
    case 'n':		// now
	{ time_t now = time(0);
	  printDateTime(fp, *localtime(&now));
	}
	break;
    case 'p':		// virtual pathname
	{ const char* pathname = getenv("PATH_INFO");
	  if (pathname) {
	    /*
	     * XXX somebody tacks an extra ``/'' onto the end;
	     * remove it to so URL's are recorded correctly as
	     * visited in the history database.
	     */
	    cp = strchr(pathname, '\0');
	    if (cp-pathname > 2 && cp[-1] == '/' && cp[-2] == '/')
		cp--;
	    fprintf(fp, "%.*s", cp-pathname, pathname);
	  }
	}
	break;
    case 'r':		// reverse collation enabled
	fputs(reverse ? "newest to front" : "oldest to front", fp);
	break;
    case 's':		// subject pattern string
	if (subjPat != "")
	    printString(fp, subjPat);
	break;
    case 't':		// count of threads
	fprintf(fp, "%u", mla.getThreadCount());
	break;
    case 'u':		// date+time of last archive update
	struct stat sb;
	Sys::stat(mla.getTOCFile(), sb);
	printDateTime(fp, *localtime(&sb.st_mtime));
	break;
    case 'v':		// MLA version
	fprintf(fp, "%u.%u", MLA_MAJOR, MLA_MINOR);
	break;
    case 'z':		// URL to MLA information
	fprintf(fp, "http://flake.asd/MLA/");			// XXX
	break;
    case '[':		// date of first message
	if (mla.getMsgCount() > 0)
	    printDate(fp, *gmtime(&mla.getSortedMsg(0).datetime));
	else
	    fprintf(fp, "<I>Never</I>");
	break;
    case ']':		// date of last message
	if (mla.getMsgCount() > 0)
	    printDate(fp,
		*gmtime(&mla.getSortedMsg(mla.getMsgCount()-1).datetime));
	else
	    fprintf(fp, "<I>Never</I>");
	break;

    case 'C':		// collation scheme
	printSchemeForm(fp, scheme);
	break;
    case 'F':		// form for From search string
	fputs("<INPUT TYPE=text NAME=From SIZE=30", fp);
	if (fromPat != "")
	    fprintf(fp, " VALUE=\"%s\"", (const char*) fromPat);
	fputs(">", fp);
	break;
    case 'H':		// max hits form component
	printMaxHitsForm(fp, maxhits);
	break;
    case 'I':		// ignore upper/lower case component
	printIgnoreCaseForm(fp, ignoreCase);
	break;
    case 'R':		// reverse collation component
	printCollateForm(fp, reverse);
	break;
    case 'S':		// form for Subject search string
	fputs("<INPUT TYPE=text NAME=Subj SIZE=30", fp);
	if (subjPat != "")
	    fprintf(fp, " VALUE=\"%s\"", (const char*) subjPat);
	fputs(">", fp);
	break;
    case 'T':		// max thread level form component
	printMaxLevelsForm(fp, maxlevels);
	break;
    case '<':		// form for starting search date
	if (mla.getMsgCount() > 0) {
	    tm tm1 = *gmtime(&mla.getSortedMsg(0).datetime);
	    time_t t = mla.getSortedMsg(mla.getMsgCount()-1).datetime;
	    tm tm2 = *gmtime(&t);
	    if (!beginTime)
		beginTime = t - 7*24*60*60;	// default to 1 week
	    tm tm3 = *gmtime(&beginTime);
	    printDateForm(fp, "S", tm1, tm2, tm3);
	} else {
	    if (!beginTime)
		beginTime = time(0);
	    tm tm1 = *localtime(&beginTime);
	    printDateForm(fp, "S", tm1, tm1, tm1);
	}
	break;
    case '>':		// form for ending search date
	if (mla.getMsgCount() > 0) {
	    tm tm1 = *gmtime(&mla.getSortedMsg(0).datetime);
	    tm tm2 = *gmtime(&mla.getSortedMsg(mla.getMsgCount()-1).datetime);
	    if (endTime) {
		tm tm3 = *gmtime(&endTime);
		printDateForm(fp, "E", tm1, tm2, tm3);
	    } else
		printDateForm(fp, "E", tm1, tm2, tm2);
	} else {
	    if (!endTime)
		endTime = time(0);
	    tm tm1 = *localtime(&endTime);
	    printDateForm(fp, "E", tm1, tm1, tm1);
	}
	break;
    default:
	fprintf(fp, "%c+%c", '%', code);
	break;
    }
}

void
CGIForm::printCollateForm(FILE* fd, fxBool rev)
{
    const char* fmt = "<INPUT TYPE=radio NAME=RV VALUE=%s%s> <B>%s</B>";
    fprintf(fd, fmt, "No", rev ? "" : " CHECKED", "Oldest to front");
    putc(' ', fd);
    fprintf(fd, fmt, "Yes", rev ? " CHECKED" : "", "Newest to front");
}

void
CGIForm::printIgnoreCaseForm(FILE* fd, fxBool ic)
{
    const char* fmt = "<INPUT TYPE=radio NAME=IC VALUE=%s%s> <B>%s</B>";
    fprintf(fd, fmt, "Yes", ic ? " CHECKED" : "", "Yes");
    putc(' ', fd);
    fprintf(fd, fmt, "No", ic ? "" : " CHECKED", "No");
}

void
CGIForm::printDate(FILE* fd, const tm& tm)
{
    char date[80];
    strftime(date, sizeof (date), "%B %e, %Y", &tm);
    fprintf(fd, "%s", date);
}

void
CGIForm::printDateTime(FILE* fd, const tm& tm)
{
    char date[80];
    strftime(date, sizeof (date), "%A, %B %e, %Y %T %Z", &tm);
    fprintf(fd, "%s", date);
}

static const char* months[] = {
    "January", "February", "March", "April", "May", "June",
    "July", "August", "September", "October", "November", "December",
};

void
CGIForm::printDateForm(FILE* fd, const char* name, const tm& t1, const tm& t2, const tm& sel)
{
    fprintf(fd, "<SELECT NAME=%sM>\n", name);
    if (t1.tm_year == t2.tm_year) {		// restrict months
	for (int month = t1.tm_mon; month <= t2.tm_mon; month++)
	    fprintf(fd, "<OPTION VALUE=%02d%s>%s\n", month+1,
		sel.tm_mon == month ? " SELECTED" : "", months[month]);
    } else {
	for (int month = 0; month < 12; month++)
	    fprintf(fd, "<OPTION VALUE=%02d%s>%s\n", month+1,
		sel.tm_mon == month ? " SELECTED" : "", months[month]);
    }
    fprintf(fd, "</SELECT>");
    fprintf(fd, " <INPUT TYPE=text NAME=%sD VALUE=\"%4d\" SIZE=4>",
	name, sel.tm_mday);
    fprintf(fd, "<SELECT NAME=%sY>\n", name);
    for (int year = t1.tm_year; year <= t2.tm_year; year++)
	fprintf(fd, "<OPTION VALUE=%d%s>%d\n", year,
	    sel.tm_year == year ? " SELECTED" : "", 1900+year);
    fprintf(fd, "</SELECT>");
}

void
CGIForm::printMaxHitsForm(FILE* fd, u_int maxhits)
{
    fprintf(fd, "<SELECT NAME=MH>\n");
    const int* hits;
    if (maxhits > 500) {
	static const int h1[] = { 50, 100, 250, 500, 750, 1000, 0 };
	hits = h1;
    } else if (maxhits > 250) {
	static const int h2[] = { 25, 50, 100, 200, 300, 400, 0 };
	hits = h2;
    } else if (maxhits > 125) {
	static const int h3[] = { 25, 50, 75, 100, 150, 200, 250, 0 };
	hits = h3;
    } else {
	// 10, 25, 50, 75, 100
	static const int h4[] = { 10, 25, 50, 75, 100, 0 };
	hits = h4;
    }
    for (; *hits && *hits <= maxhits; hits++)
	fprintf(fd, "<OPTION%s>%d\n",
	    maxhits == *hits ? " SELECTED" : "", *hits);
    if (*hits == 0)
	fprintf(fd, "<OPTION VALUE=\"-1\">All\n");
    fprintf(fd, "</SELECT>");
}

void
CGIForm::printMaxLevelsForm(FILE* fd, u_int levels)
{
    fprintf(fd, "<SELECT NAME=ML>\n");
    fprintf(fd, "<OPTION SELECTED>%d\n", levels);
    if (levels > 0) {
	do {
	    fprintf(fd, "<OPTION>%d\n", --levels);
	} while (levels > 1);
    }
    fprintf(fd, "</SELECT>");
}

void
CGIForm::printSchemeForm(FILE* fd, const char* def)
{
    fprintf(fd, "<SELECT NAME=SC>\n");
    static const char *schemes[] =
	{ "Thread", "Date", "Author", "Subject" };
    for (u_int i = 0; i < N(schemes); i++)
	fprintf(fd, "<OPTION VALUE=%c%s>%s\n", tolower(schemes[i][0]),
	     strcasecmp(def, schemes[i]) ? "" : " SELECTED", schemes[i]);
    fprintf(fd, "</SELECT>");
}

inline fxBool isMagic(int c)
    { return (c == '<' || c == '>' || c == '&'); }
void
CGIForm::printString(FILE* fp, const char* s)
{
    const char* cp = s;
    while (*cp && !isMagic(*cp))
	cp++;
    if (*cp) {
	do {
	    fwrite(s, cp-s, 1, fp);
	    switch (*cp++) {
	    case '<':	fputs("&lt;", fp); break;
	    case '>':	fputs("&gt;", fp); break;
	    case '&':	fputs("&amp;", fp); break;
	    }
	    for (s = cp; *cp && !isMagic(*cp); cp++)
		;
	} while (*cp);
	if (cp > s)
	    fwrite(s, cp-s, 1, fp);
    } else
	fputs(s, fp);
}

void
CGIForm::printFetchArgs(FILE* fp, mnum_t msgnum, const char* query)
{
    fprintf(fp, "Msg=%d&%s", msgnum, query);
}

CGIReadOnlyMLA::CGIReadOnlyMLA(const char* toc) : ReadOnlyMLA(toc)
{
}
CGIReadOnlyMLA::~CGIReadOnlyMLA() {}

CGIReadOnlyMLA*
CGIReadOnlyMLA::readMLA(const char* file)
{
    int fd;
    if (openMLA(file, O_RDONLY, fd)) {
	flock(fd, LOCK_SH);
	CGIReadOnlyMLA* mla = new CGIReadOnlyMLA(file);
	if (mla->setupMLA(fd, PROT_READ, MAP_SHARED))
	    return (mla);
	delete mla;
    }
    return (NULL);
}

void
CGIReadOnlyMLA::vprintMsg(const char* fmt, va_list ap) const
{
    fputs(getTOCFile() | ": ", stdout);
    vfprintf(stdout, fmt, ap);
    fputs(".\n", stdout);
}
