/*
 * Copyright 1995, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * Permission to use, copy, modify, distribute, and sell this software
 * and its documentation for any purpose is hereby granted without
 * fee, provided that (i) the above copyright notices and this
 * permission notice appear in all copies of the software and related
 * documentation, and (ii) the name of Silicon Graphics may not be
 * used in any advertising or publicity relating to the software
 * without the specific, prior written permission of Silicon Graphics.
 *
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * IN NO EVENT SHALL SILICON GRAPHICS BE LIABLE FOR ANY SPECIAL,
 * INCIDENTAL, INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY
 * DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER OR NOT ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY
 * THEORY OF LIABILITY, ARISING OUT OF OR IN CONNECTION WITH THE USE
 * OR PERFORMANCE OF THIS SOFTWARE.
 *
 * dbpage.c: simple database paging example
 *
 * $Revision: 1.3 $ $Date: 1996/05/18 07:36:53 $ 
 *
 */

#include <Performer/pf.h>
#include <Performer/pfdu.h>

typedef struct {
    int	fcnt;
    char **fnames;
    pfScene *scene;
    pfGroup *group;
} SharedData;
SharedData *shared;

void
dbfunc(void *data)
{
    pfNode *node;
    static pfNode *prev=0;
    static pfBuffer *buf=0;
    static int c=0;

    if (!buf)
    {
	buf = pfNewBuffer();
	pfSelectBuffer(buf);

	/* Open a window and context so loaders won't need to create */
	/* their own contexts each time. */
	pfOpenNewNoPortWin("Query", -1);
    }
    /* Load a model into scene. */
    if (!(node = pfdLoadFile(shared->fnames[c % shared->fcnt])))
    {
	pfNotify(PFNFY_NOTICE, PFNFY_PRINT,
	"cannot load file %s", shared->fnames[0]);
	exit(1);
    }
    pfBufferAddChild(shared->group, node);
    if (prev)
    {
	pfBufferRemoveChild(shared->group, prev);
	pfAsyncDelete(prev);
    }
    pfMergeBuffer();
    pfDBase();

    prev = node;
    c++;
}

int
main (int argc, char *argv[])
{
    pfPipe      *pipe;
    pfPipeWindow *pwin;
    pfChannel   *chan;
    pfGeoState  *gstate;
    pfScene	*scene;
    pfGroup     *group;    
    void	*arena;
    pfSphere	bsphere;
    float       t = 0.0f;
    int xs, ys;
    int i;    

    /* Initialize Performer */
    pfInit();	
    
    /* Get shared memory arena */
    arena = pfGetSharedArena();
    shared = (SharedData *) pfMalloc(sizeof(SharedData), arena);
    
    if (argc < 2)
    {
	printf("usage: %s file ... \n", argv[0]);
	exit(1);
    }
    shared->fnames = (char **) pfMalloc((argc-1)*sizeof(char*), arena);
    for (i=1; i < argc; i++)
	shared->fnames[i-1] = argv[i];
    shared->fcnt = argc-1;

    pfFilePath(".:/usr/share/Performer/data:../../../../data");
    pfMultiprocess(PFMP_APPCULLDRAW|PFMP_FORK_DBASE);			
    pfConfig();			
    pfDBaseFunc(dbfunc);

    shared->scene = pfNewScene();
    shared->group = pfNewGroup();
    pfAddChild(shared->scene, shared->group);
    
    /* Create a scene pfGeoState with lighting enabled */
    gstate = pfNewGState(arena);
    pfGStateMode(gstate, PFSTATE_ENLIGHTING, PF_ON);
    pfSceneGState(shared->scene, gstate);
    pfAddChild(shared->scene, pfNewLSource());

    /* Create and configure a pfChannel. */
    pipe = pfGetPipe(0);
    pwin = pfNewPWin(pipe);
    pfPWinOriginSize(pwin, 0, 0, 400, 400);
    pfOpenPWin(pwin);
    chan = pfNewChan(pipe);
    pfChanScene(chan, shared->scene);
    pfChanFOV(chan, 60.0f, 0.0f);

    while (t < 60.0f)
    {
	pfCoord	   view;
	float      s, c;

	pfSync();		

	/* determine extent of scene's geometry */
	pfGetNodeBSphere(shared->group, &bsphere);
	pfChanNearFar(chan, 1.0f, 10.0f * bsphere.radius);
	
	t = pfGetTime();
	pfSinCos(45.0f*t, &s, &c);
	pfSetVec3(view.hpr, 45.0f*t, -5.0f, 0);
	pfSetVec3(view.xyz, 2.0f * bsphere.radius * s, 
		-2.0f * bsphere.radius *c, 
		 0.3f * bsphere.radius);
	pfChanView(chan, view.xyz, view.hpr);

	pfFrame();
    }
    pfExit();
}
