/*
 * Copyright (c) 1994, 1995, 1996 Silicon Graphics, Inc.
 * 
 * Permission to use, copy, modify, distribute, and sell this software and
 * its documentation for any purpose is hereby granted without fee,
 * provided that (i) the above copyright notices and this permission
 * notice appear in all copies of the software and related documentation,
 * and (ii) the name of Silicon Graphics may not be used in any
 * advertising or publicity relating to the software without the specific,
 * prior written permission of Silicon Graphics.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
 * 
 * IN NO EVENT SHALL SILICON GRAPHICS BE LIABLE FOR ANY SPECIAL,
 * INCIDENTAL, INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY
 * DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER
 * OR NOT ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF
 * LIABILITY, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE
 * OF THIS SOFTWARE.
 */
/*
 *  videotex.c
 *
 *  Simple video texturing example.
 *
 *  Simon Hui, Silicon Graphics Inc., June 1996
 *
 *  $Revision: 1.7 $
 */

#include <stdlib.h>
#include <stdio.h>
#include <getopt.h>
#include <GL/gl.h>
#include <GL/glu.h>
#include <GL/glx.h>
#include <X11/keysym.h>
#include <device.h>
#include <vl/vl.h>
#include <vl/dev_sirius.h>

/* command line options */
char    *fmtarg;
GLenum  texfmt;		  /* internal texel format, derived from fmtarg */
int	sirfmt;		  /* sirius packing format, derived from fmtarg */
int	dofields = 1;
int	interlace = 0;

/* other defaults */
int	isrealityengine = 0;
int	border = 30;

/* VL resources */
VLControlValue	size, timing, texctl;
VLControlValue 	format;
VLControlValue 	cap_type;
VLServer	svr;
VLPath		path;
VLNode		src;
VLNode		drn;

/* GL and X resources */
Display		*dpy;
Window		win;
GLXContext	cx;
GLXVideoSourceSGIX videosource;

void usage(int, char **);
void initTexture(void);
void displayNewTexture(void);
void cleanup(void);
void processEvents(void);
void updateTimingFormat(void);
void makeWindow(void);
void resizeWindow(int w, int h);

/*
** Texel formats supported by different systems.
*/
#define REMASK 0x1
#define IRMASK 0x2
static struct {
    char *name;
    int sirfmt;
    int mask;
    int texfmt;
} formats[] = {
    "rgba4",	SIR_TEX_PACK_RGBA_4,	REMASK,	GL_RGBA4_EXT,
    "rgb5",	SIR_TEX_PACK_RGB_5,	REMASK, GL_RGB5_EXT,
    "rgba8",	SIR_TEX_PACK_RGBA_8,	REMASK, GL_RGBA8_EXT,
    "rgba6",	SIR_PACK_RGBA_6,	IRMASK, GL_RGBA8_EXT,
    "rgb8",	SIR_PACK_RGB_8,		IRMASK, GL_RGB8_EXT,
    "rgba12",	SIR_PACK_RGBA_12,	IRMASK, GL_RGBA12_EXT,
    "yyyy12",	SIR_PACK_YYYY_12,	IRMASK, GL_RGBA12_EXT,
};

static void
fatal(const char *msg) {
    fprintf(stderr, "%s\n", msg);
    exit(EXIT_FAILURE);
}

int
main(int argc, char **argv)
{

    int		c, insrc = VL_ANY;
    int 	device = VL_ANY;
    short	dev, val;
    int		scrn;
    int		i, n, mask;
    const GLubyte *renderstr;

    /* decide what machine we're running on, set defaults accordingly */
    makeWindow();
    renderstr = glGetString(GL_RENDERER);
    if (!strncmp(renderstr,"RE",2)) {
	isrealityengine = 1;
	interlace = 0;
	fmtarg = "rgb5";
	mask = REMASK;
    } else if (!strncmp(renderstr,"IR",2)) {
	isrealityengine = 0;
	dofields = 1;
	fmtarg = "rgb8";
	mask = IRMASK;
    } else {
	fatal("must run on RealityEngine or InfiniteReality");
	exit(1);
    }
    
    /* Parse the input string */
    while ((c = getopt(argc, argv, "n:t:v:hfi")) != EOF ) {
        switch (c) {
	  case 'i':
	    if (isrealityengine) fatal("RealityEngine cannot interlace");
	    interlace = 1;
	    break;
	  case 'f':
	    if (!isrealityengine) fatal("InfiniteReality cannot do frames");
	    dofields = 0;
	    break;
	  case 't':
	    fmtarg = optarg;
	    break;
	  case 'n':
	    device = atoi(optarg); /* user can override default device */
	    break;
	  case 'v':
	    insrc = atoi(optarg);
	    break;
	  case 'h':
	  default:
	    usage(argc, argv);
	    exit(1);
        }
    }
    /* match the texel format argument with our format table */
    n = sizeof(formats)/sizeof(formats[0]);
    for (i=0; i < n; i++) {
	if (!strcmp(formats[i].name, fmtarg)) {
	    if (!(mask & formats[i].mask)) {
		fatal("format not supported on this machine");
	    }
	    texfmt = formats[i].texfmt;
	    sirfmt = formats[i].sirfmt;
	    break;
	}
    }
    if (i==n) {
	fatal("cannot recognize format");
    }

    /* open the server */
    if (!(svr = vlOpenVideo(""))) {
	printf("couldn't open video\n");
	exit(1);
    }
    /* get the Video source */
    src = vlGetNode(svr, VL_SRC, VL_VIDEO, insrc);
    /* get the Texture drain */
    drn = vlGetNode(svr, VL_DRN, VL_TEXTURE, 0);

    /* create path   */
    path = vlCreatePath(svr, device, src, drn);
    if (path < 0) {
	vlPerror("vlCreatePath");
	exit(1);
    }
    /* setup path */
    if (vlSetupPaths(svr, (VLPathList)&path, 1, VL_SHARE, VL_SHARE) < 0) {
	vlPerror("vlSetupPaths");
	exit(1);
    }
    /* select the appropriate events */
    if (vlSelectEvents(svr, path, VLStreamPreemptedMask |
                            VLControlChangedMask ) < 0) {
            vlPerror("Select Events");
            exit(1);
    }
    /* only reality engine can capture noninterleaved frames */
    if (isrealityengine) {
	if(dofields)
	  cap_type.intVal = VL_CAPTURE_NONINTERLEAVED;
	else
	  cap_type.intVal = VL_CAPTURE_INTERLEAVED;
	if (vlSetControl(svr, path, drn, VL_CAP_TYPE, &cap_type) <0) {
	    vlPerror("VlSetControl");
	    exit(1);
	}
    }
    /* set the texture packing mode */
    texctl.intVal = sirfmt;
    if (vlSetControl(svr, path, drn, VL_PACKING, &texctl) <0) {
        vlPerror("VlSetControl");
        exit(1);
    }
    updateTimingFormat();
    vlBeginTransfer(svr, path, 0, NULL);

    /* init GL and X */
    scrn = DefaultScreen(dpy);
    if ((videosource = glXCreateGLXVideoSourceSGIX(dpy, scrn, svr, path,
						   VL_TEXTURE, drn)) == None) {
        fprintf(stderr, "can't create video source\n");
        exit(EXIT_FAILURE);
    }
    glXMakeCurrentReadSGI(dpy, win, videosource, cx);

    while (TRUE) {
	if (XEventsQueued(dpy, QueuedAfterReading)) {
	    XEvent event;

	    XNextEvent(dpy, &event);
	    switch (event.type) {
	      case KeyPress:
		{
		    char buf[16];
		    KeySym ks;
		    int rv;
		    
		    rv = XLookupString(&event.xkey, buf, sizeof(buf), &ks, 0);
		    switch (ks) {
		      case XK_Escape:
			exit(1);
			break;
		    }
		}
	      case ConfigureNotify:
		{
		    int width = event.xconfigure.width;
		    int height = event.xconfigure.height;
		    glViewport(0, 0, width, height);
		}
		break;
	    }
	}
	/* Check for VL events if any ... */
	processEvents();
	/* Next frame of data */
	displayNewTexture();
	glXSwapBuffers(dpy, win);
    }
}

void
usage(int argc, char **argv)
{
    fprintf(stderr, "Usage: %s [-t format][-i][-f][-n#] [-v#]\n", argv[0]);
    fprintf(stderr, "  -t format  texel format \n");
    fprintf(stderr, "  -i	  interlace \n");
    fprintf(stderr, "  -f	  frames instead of fields \n");
    fprintf(stderr, "  -n#	  video device number (as reported by vlinfo)\n");
    fprintf(stderr, "  -v#	  video input number (as reported by vlinfo)\n");
}

/*
 * initTexture: Create a texture map with correct attributes.
 */
int texwidth, texheight;

void
initTexture()
{
    float s_scale, t_scale;

    if (interlace) {
	const GLubyte *extensions = glGetString(GL_EXTENSIONS);
	int has_interlace;
	
	if (!extensions) {
	    printf("can't get GL_EXTENSIONS\n");
	}
	has_interlace = strstr((const char *) extensions,
			       "GL_SGIX_interlace") != NULL;
	glEnable(GL_INTERLACE_SGIX);
	if (!has_interlace || !glIsEnabled(GL_INTERLACE_SGIX)) {
	    fatal("can't enable interlace extension\n");
	}
    }

    /* choose texture size big enough to fit video */
    texwidth = 1024;
    if (size.xyVal.y > 512) {
	texheight = 1024;
    } else {
	texheight = 512;
    }
    s_scale = size.xyVal.x / (float)texwidth;
    t_scale = size.xyVal.y / (float)texheight;
    if (dofields || interlace) texheight /= 2;

    glEnable(GL_TEXTURE_2D);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    glTexImage2D(GL_TEXTURE_2D, 0, texfmt, texwidth, texheight, 0,
		 GL_RGBA, GL_UNSIGNED_BYTE, NULL);

    /* video texture comes in upside down, so compensate */
    glMatrixMode(GL_TEXTURE);
    glLoadIdentity();
    glTranslatef(0, t_scale, 0);
    glScalef(s_scale, -t_scale, 1);
    
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    glOrtho(-border, size.xyVal.x + border, -border, size.xyVal.y + border, -1, 1);
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();
    glTranslatef(0, 0, -0.5);
}

/*
 * displayNewTexture: Display a new texture. This routine is called at 30
 * Hz for 525 timing and 25Hz for 625 timing.
 */
void
displayNewTexture()
{
    static int even = 1;
    int width = size.xyVal.x;
    int height = size.xyVal.y;
    int yoffset = 0;

    if (interlace) {
	yoffset = !even;
    }
    if (dofields || interlace) {
	height /= 2;
    }
    if (isrealityengine) {
	width = 768;
    } else {
	/* infinite reality wants multiple of 8 */
	width = (width + 7) & ~0x7;
    }
    /* Load the texture from Sirius to Texture memory */
    glCopyTexSubImage2DEXT(GL_TEXTURE_2D, 0, 0, yoffset, 0, 0, width, height);

    glClear(GL_COLOR_BUFFER_BIT);
    /* Draw full frame of video */
    glBegin(GL_TRIANGLE_STRIP);
    glTexCoord2f(0,0); glVertex2f(0, 0);
    glTexCoord2f(1,0); glVertex2f(size.xyVal.x, 0);
    glTexCoord2f(0,1); glVertex2f(0, size.xyVal.y);
    glTexCoord2f(1,1); glVertex2f(size.xyVal.x, size.xyVal.y);
    glEnd();

    even = !even;
}

void
cleanup()
{
    vlEndTransfer(svr, path);
    vlDestroyPath(svr, path);
    vlCloseVideo(svr);
    exit(0);
}

void
processEvents()
{
    VLEvent ev;

    /* Check to see if any controls changed or if the video
     * path has been pre-empted.
     */
    if (vlCheckEvent(svr, VLControlChangedMask|
		    VLStreamPreemptedMask, &ev) == -1) {
	return;
    }
    switch(ev.reason) {
	case VLStreamPreempted:
	    cleanup();
	    exit(0);
	break;
	case VLControlChanged:
	    switch(ev.vlcontrolchanged.type) {
	      case VL_TIMING:
	      /* case VL_SIZE: */
	      /* case VL_PACKING: */
		updateTimingFormat();
		break;
	      default:
		break;
	    }
	break;
	default:
	break;
    }
}

void
updateTimingFormat()
{
    float s_scale, t_scale;

    /* Get the timing from input source */
    if (vlGetControl(svr, path, src, VL_TIMING, &timing) < 0) {
	vlPerror("vlGetControl");
	exit(1);
    }
    /* Set texture drain's timing to input source */
    if (vlSetControl(svr, path, drn, VL_TIMING, &timing) < 0) {
	vlPerror("vlSetControl");
	exit(1);
    }
    if (vlGetControl(svr, path, src, VL_SIZE, &size) < 0) {
	vlPerror("vlSetupPaths");
	exit(1);
    }
    printf("size:%dx%d format:%s interlace:%d fields:%d\n",
	   size.xyVal.x, size.xyVal.y, fmtarg, interlace, dofields);

    resizeWindow(size.xyVal.x + 2*border, size.xyVal.y + 2*border);
    XFlush(dpy);
    /* recompute texture based on new size */
    initTexture();
}

static int attributeList[] = { GLX_RGBA, GLX_RED_SIZE, 1, GLX_DOUBLEBUFFER, None };

static Bool
waitForNotify(Display *d, XEvent *e, char *arg) {
    return (e->type == MapNotify) && (e->xmap.window == (Window)arg);
}

void
makeWindow()
{
    XSetWindowAttributes swa;
    XVisualInfo *vi;

    dpy = XOpenDisplay(0);
    if (!dpy) fatal("can't open display");
    vi = glXChooseVisual(dpy, DefaultScreen(dpy), attributeList);
    if (!vi) fatal("no suitable visual");
    cx = glXCreateContext(dpy, vi, 0, GL_TRUE);
    swa.colormap = XCreateColormap(dpy, RootWindow(dpy, vi->screen),
				   vi->visual, AllocNone);
    swa.border_pixel = 0;
    swa.event_mask = ExposureMask | StructureNotifyMask | KeyPressMask;
    win = XCreateWindow(dpy, RootWindow(dpy, vi->screen), 0, 0, 100, 100,
			0, vi->depth, InputOutput, vi->visual,
			CWBorderPixel|CWColormap|CWEventMask, &swa);
    XStoreName(dpy, win, "videotex");
    glXMakeCurrent(dpy, win, cx);

    glClearColor(0,0,0,0);    
    glColor4f(1, 1, 1, 1);
    glEnable(GL_TEXTURE_2D);
}

void
resizeWindow(int w, int h) {
    static int mapped = 0;
    XEvent event;

    if (!mapped) {
	XMapWindow(dpy, win);
	XIfEvent(dpy, &event, waitForNotify, (char*)win);
	mapped = 1;
    }
    XResizeWindow(dpy, win, w, h);
    glViewport(0, 0, w, h);
}

