/*
 * Copyright (c) 1994 Silicon Graphics, Inc.
 *
 * Permission to use, copy, modify, distribute, and sell this software
 * and its documentation for any purpose is hereby granted without
 * fee, provided that (i) the above copyright notices and this
 * permission notice appear in all copies of the software and related
 * documentation, and (ii) the name of Silicon Graphics may not be
 * used in any advertising or publicity relating to the software
 * without the specific, prior written permission of Silicon Graphics.
 *
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * IN NO EVENT SHALL SILICON GRAPHICS BE LIABLE FOR ANY SPECIAL,
 * INCIDENTAL, INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY
 * DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER OR NOT ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY
 * THEORY OF LIABILITY, ARISING OUT OF OR IN CONNECTION WITH THE USE
 * OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

/*
 *	tpatt.c --  simple test pattern generator for SEOS Prodas S300HB
 *	Program by: Tom Fuke, The Reality Centre, Silicon Graphics Ltd
 *		    (tom@reading.sgi.com)
 *	Date:	    May 1994
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <get.h>
#include <gl/gl.h>
#include <gl/device.h>


#define HATCHSQUARES 8.0
#define DEFRECTSIZE 0.1	    /* default rectangle size is 10% of screen area */

#define FONTNC 127
#define FONTNR ((sizeof raster)/sizeof (unsigned short))
#define ASSIGN(fontch, of, wi, he, xof, yof, wid) \
    fontch.offset = of; \
    fontch.w = wi; \
    fontch.h = he; \
    fontch.xoff = xof; \
    fontch.yoff = yof; \
    fontch.width = wid
    
static Fontchar	lpoint_font[FONTNC];	/* character font for lightpoints */
static unsigned short raster[] = {0xE000, 0xE000, 0xE000};


static long mainmenu, patmenu, levelmenu;

static long x_win_origin, x_win_size, y_win_origin, y_win_size;
static short cur_pattern=0;	/* current pattern */
static short initial_pattern;
static char pat_name[100]="Black Field";
static short info_flag=0;	/* Show Pattern Information? */

/* pattern variables */
static float drive_level;   /* video drive level, between 0.0 and 1.0 */
static long rectv[4][2];	/* rectangle vertices */
static long rect_xl, rect_yl;	/* rectangle edge lengths */
				/* lengths are length = sqrt(rect_size) * (float) pixels */
static int gs_dir;		/* greyscale direction (0 = hor, 1 = vert) */
static int gs_gra;		/* greyscale intensity gradient (0 = increasing, 1 = decreasing) */
static int cb_order;		/* chequer board order (0=black first, 1=white first) */

/*********** SET PATTERN-SPECIFIC DEFAULTS *****************/
set_defaults()
{
/* 10% default area size */
size_rectangle(DEFRECTSIZE);

/* calculate rectangle vertices */
move_rectangle(x_win_size/2, y_win_size/2);

gs_dir = 0; gs_gra = 0; /* default greyscale  is horizontal increasing */
}


/****************** SIZE RECTANGLE ************************/
size_rectangle(f) /* where f is proportion of area 0.0 - 1.0 */
float f;
{
rect_xl = sqrt(f) * (float) x_win_size;
rect_yl = sqrt(f) * (float) y_win_size;

if(cur_pattern==8) rect_xl = 170;
}


/*************** MOVE RECTANGLE *******************/
move_rectangle(x, y)	/* where x and y are the centres */
long x, y;
{
long	xmin = rect_xl/2, xmax = x_win_size-(rect_xl/2), 
	ymin = rect_yl/2, ymax = y_win_size-(rect_yl/2);

if (x<xmin) x=xmin;
if (x>xmax) x=xmax;
if (y<ymin) y=ymin;
if (y>ymax) y=ymax;

rectv[0][0] = x-rect_xl/2;	/* bottom left */
rectv[0][1] = y-rect_yl/2;
rectv[1][0] = rectv[0][0];		/* top left */
rectv[1][1] = rectv[0][1] + rect_yl;
rectv[2][0] = rectv[0][0] + rect_xl;	/* top right */
rectv[2][1] = rectv[0][1] + rect_yl;
rectv[3][0] = rectv[0][0] + rect_xl;	/* bottom right */
rectv[3][1] = rectv[0][1];
}



/****************** DRAW PATTERN ********************/
draw_pattern(patno)
int patno;
/* patno:
 * 1 SPHEREMAP
 * 2 LINEAR CROSSHATCH
 * 3 GREYSCALE
 * 4 UNIFORM FIELD
 * 5 BLACK FIELD
 * 6 RECTANGLE
 * 7 CHEQUERBOARD
 * 8 RESOLUTION
 */
{
    float   x, y;
    float   cv[3];
    short   new_pat_flag;

    
    new_pat_flag = (patno!=cur_pattern);
    cur_pattern = patno;
	    
	    RGBcolor(0, 0, 0); clear();
	    
	    switch(patno) {
		case 1: strcpy(pat_name, "Spherically Mapped Dots");
		    if (new_pat_flag) drive_level=0.7;
		    cv[0]=drive_level; cv[1]=drive_level; cv[2]=drive_level;
		    set_perspective();
		    c3f(cv);
		    draw_sphere();
		    break;
		    
		case 2: strcpy(pat_name, "Linear Crosshatch");
		    if (new_pat_flag) drive_level=0.7;	/* If just selected, set dlevel to default 70% */
		    cv[0]=drive_level; cv[1]=drive_level; cv[2]=drive_level;
		    set_ortho();		    
		    c3f(cv);
		    for (x=0.0;x<=x_win_size;x+=((x_win_size-1)/HATCHSQUARES)) {
			move2(x, 0);
			draw2(x, y_win_size);
		    }
		    for (y=0.0;y<=x_win_size;y+=((y_win_size-1)/HATCHSQUARES)) {
			move2(0, y);
			draw2(x_win_size, y);
		    }
		    break;
		    
		case 3: strcpy(pat_name, "Greyscale");
		    if (new_pat_flag) drive_level=1.0;
		    set_ortho();
		    /*draw_gscale(1, 0);*/
		    draw_gscale(gs_dir, gs_gra);
		    break;
		    
		case 4: strcpy(pat_name, "Uniform Field");
		    if (new_pat_flag) drive_level=0.5;
		    cv[0]=drive_level; cv[1]=drive_level; cv[2]=drive_level;
		    set_ortho();
		    c3f(cv);
		    clear(); break;
		    
		case 5: strcpy(pat_name, "Black Field");
		    drive_level=0.0;
		    cv[0]=drive_level; cv[1]=drive_level; cv[2]=drive_level;
		    set_ortho();
		    c3f(cv);
		    clear(); break;
		    
		case 6: strcpy(pat_name, "Rectangle");
		    if (new_pat_flag) {drive_level=1.0; set_defaults();}
		    cv[0]=drive_level; cv[1]=drive_level; cv[2]=drive_level;
		    set_ortho();
		    c3f(cv);
		    bgnpolygon();
		    v2i(rectv[0]); v2i(rectv[1]); v2i(rectv[2]); v2i(rectv[3]);
		    endpolygon();
		    break;
		    
		case 7: strcpy(pat_name, "Chequerboard");
		    if(new_pat_flag) drive_level=0.7;
		    cv[0] = drive_level; cv[1]=drive_level; cv[2]=drive_level;
		    set_ortho();
		    c3f(cv);
		    draw_chequer();
		    break;
		    
		case 8: strcpy(pat_name, "Resolution");
		    if(new_pat_flag) {drive_level=0.5; set_defaults();}
		    cv[0]= drive_level; cv[1]=drive_level; cv[2]=drive_level;
		    set_ortho();
		    c3f(cv);
		    draw_resolution();
		    break;
		    
		default: break;
	    }
update_info();	   
}	    
	


/********** SET ORTHOGRAPHIC VIEW ****************/	   
set_ortho()
{
ortho2(-0.5, x_win_size-0.5, -0.5, y_win_size-0.5);
viewport(0, x_win_size-1, 0, y_win_size-1);
}    	    


/************** SET PERSPECTIVE VIEW ***************/
set_perspective()
{
    float hfov_left = -26.0;
    float hfov_right = 26.0;
    float vfov_top = 15.0;
    float vfov_bottom = -25.0;
    float near_clip = 0.1; /* bear in mind we draw a unit sphere */
    float far_clip = 10.0;
    double conv = 2.0 * M_PI / 360.0;	/* constant of conversion Rads per Degree */

    /*perspective(460, 1.2003667, 0.01, 10.0);*/
    /* use "window" for setting up asymmetric frusta */
    window( near_clip * ftan(hfov_left * conv),
	    near_clip * ftan(hfov_right * conv),
	    near_clip * ftan(vfov_bottom * conv),
	    near_clip * ftan(vfov_top * conv),
	    near_clip, far_clip);
    
    lookat(0.0, 0.0, 0.0, 0.0, 0.0, -1.0, 0.0);
    
  /*rotate(-500, 'y');*/
}



/*************** DRAW RESOLUTION **********************/
draw_resolution()
{
    int x;
    short line[2][2];
    
    
    /* draw circle */
    circ(x_win_size/2, y_win_size/2, (y_win_size-1)/2);
    
    
    line[0][1] = rectv[0][1];   /* y coord for start of line */
    line[1][1] = line[0][1] + rect_yl; /* y coord for end of line */
    
    /* 30 lines of 1 pixel on, 1 pixel off */
    for (x=0;x<60;x+=2) {
	line[0][0]=rectv[0][0]+x;   /* new x coord - start of line */
	line[1][0]=line[0][0];	    /*   ""          end of line */
	bgnline();
	v2s(line[0]); v2s(line[1]);
	endline();
    }
    
    
    /* 15 lines of 2 pixels on, 2 pixels off */
    for (;x<120;x+=4) {
	line[0][0]=rectv[0][0]+x;   /* new x coord - start of line */
	line[1][0]=line[0][0];	    /*   ""          end of line */
	bgnline();
	v2s(line[0]); v2s(line[1]);
	endline();
	line[0][0]+=1; line[1][0]=line[0][0];
	bgnline();
	v2s(line[0]); v2s(line[1]);
	endline();
    }
    
    
    /* 8 lines of 3 on, 3 off */
    for (;x<168;x+=6){
	line[0][0]=rectv[0][0]+x;   /* new x coord - start of line */
	line[1][0]=line[0][0];	    /*   ""          end of line */
	bgnline();
	v2s(line[0]); v2s(line[1]);
	endline();
	line[0][0]+=1; line[1][0]=line[0][0];
	bgnline();
	v2s(line[0]); v2s(line[1]);
	endline();
	line[0][0]+=1; line[1][0]=line[0][0];
	bgnline();
	v2s(line[0]); v2s(line[1]);
	endline();
    }
   
}





/******************** DRAW CHEQUERBOARD ****************/
draw_chequer()
{
    float x, y, xsize, ysize;
    float vec[2];
    float col[3];
    int on = cb_order;
    
    col[0] = drive_level; col[1] = drive_level; col[2] = drive_level;
    
    xsize = x_win_size/5.0; ysize = y_win_size/4.0;
    
    shademodel(FLAT);
    
 
    for (y=0.0; y<y_win_size; y+=ysize) {
	for (x=0.0; x<x_win_size; x+=xsize) {
	    if (on==0) RGBcolor(0, 0, 0);
	    if (on==1) c3f(col);
	    rectf(x, y, x+xsize, y+ysize);
	    on = !on;
	}
	
    }
}




/********************** DRAW SPHERE **********************/
draw_sphere()
{
    /* RADIUS OF SPHERE IS ARBITRARY (UNITY ) */
    double lambda, phi;	/* lambda is horiz angle, phi is vertical */
    double vertex[3];	/* cartesian vertex */
    double conv = 2.0 * M_PI / 360.0;	/* constant of conversion Rads per Degree */
    float   colour[3]; /* colour for dots */
    
    font(1);	/* light point fon */
    

/* HORIZONTAL LINES */
    
    for (phi=-25.0;phi<=15.0;phi+=5.0) {
	for (lambda=-75.0;lambda<=75.0;lambda+=1.0) {

	
	    vertex[2]=-fcos(phi*conv) * fcos(lambda*conv);
	    vertex[1]=fsin(phi*conv);
	    vertex[0]=fcos(phi*conv) * fsin(lambda*conv);
	    cmov(vertex[0],  vertex[1], vertex[2]);
	    if (phi==0.0) colour[0] = colour[1] = colour[2] = 1.0f;
	    else colour[0] = colour[1] = colour[2] = 0.7f;
	    c3f(colour);
	    charstr(".");
	}
    }


/* VERTICAL LINES */
    for (lambda=-75.0;lambda<=75.0;lambda+=5.0) {
	for (phi=-25.0;phi<=15.0;phi+=1.0) {
	    vertex[2]=-fcos(phi*conv) * fcos(lambda*conv);
	    vertex[1]=fsin(phi*conv);
	    vertex[0]=fcos(phi*conv) * fsin(lambda*conv);
	    cmov(vertex[0],  vertex[1], vertex[2]);
	    if (phi==0.0) colour[0] = colour[1] = colour[2] = 1.0f;
	    else colour[0] = colour[1] = colour[2] = 0.7f;
	    c3f(colour);
	    charstr(".");
	}
    }
    
    

    font(0);
}



/* DRAW GREYSCALE */
draw_gscale(int dir, int gra)
/* dir: 0 = hor, 1 = vert
 * gra: 0 = inc, 1 = dec
 */
{
    float x, y;
    float   v[2];
    int bars = 10;  /* ten bars */
    float intensity, cstep=drive_level/(float)bars;
    float col[3];
    int n=0;
    
    n = ((dir*2)+(gra*1)) + 1;
    
   
    shademodel(FLAT);
    clear();
    
    switch (n) {
	case 1: /* HORIZONTAL INC */
	    intensity = 0.0;
	    bgnqstrip();
	    for( x=0.0; x<=x_win_size; x+=(x_win_size/(float)bars) ) {
		col[0]=intensity; col[1]=intensity; col[2]=intensity;
		c3f(col);
		v[0]=x; v[1]=0.0;
		v2f(v);
		v[1]=(float)y_win_size-1.0;
		v2f(v);
		if (x!=0.0)intensity+=cstep;
	    }
	    endqstrip();
	    break;
	    
	case 2:	/* HORIZONTAL DEC */
	    intensity = 0.0;
	    bgnqstrip();
	    for( x=x_win_size; x>=0.0; x-=(x_win_size/(float)bars) ) {
		col[0]=intensity; col[1]=intensity; col[2]=intensity;
		c3f(col);
		v[0]=x; v[1]=0.0;
		v2f(v);
		v[1]=(float)y_win_size-1.0;
		v2f(v);
		if (x!=x_win_size)intensity+=cstep;
	    }
	    endqstrip();
	    break;
	    
	case 3: /* VERTICAL INC */
	    intensity = 0.0;
	    bgnqstrip();
	    for( y=0.0; y<=y_win_size+1; y+=(y_win_size/(float)bars) ) {
		col[0]=intensity; col[1]=intensity; col[2]=intensity;
		c3f(col);
		v[0]=0.0; v[1]=y;
		v2f(v);
		v[0]=(float)x_win_size-1.0;
		v2f(v);
		if (y!=0.0)intensity+=cstep;
	    }
	    endqstrip();
	    break;
	    
	case 4: /* VERTICAL DEC */
	    intensity = 0.0;
	    bgnqstrip();
	    for( y=y_win_size; y>=-1.0; y-=(y_win_size/(float)bars) ) {
		col[0]=intensity; col[1]=intensity; col[2]=intensity;
		c3f(col);
		v[0]=0.0; v[1]=y;
		v2f(v);
		v[0]=(float)x_win_size-1.0;
		v2f(v);
		if (y!=y_win_size)intensity+=cstep;
	    }
	    endqstrip();
	    break;
	    
	    
	    
	default:
	    break;
    }
    
}




/******************** UPDATE PATTERN INFO *****************/
update_info()
{
    char temp[10];
    sprintf(temp, "%3.1f", drive_level);
    
/*
    pushmatrix();
    drawmode(OVERDRAW);
    set_ortho();
    color(10);
    
 */

RGBcolor(255, 0, 0);
if (info_flag) {
		cmov2i(50, 60);
		charstr(pat_name);
		cmov2i(50, 30);
		charstr("Drive Level: "); charstr(temp);
    }
/*
    drawmode(NORMALDRAW);
    popmatrix();
*/

}


/************************* CHANGE DRIVE LEVEL ******************/
set_drivelevel(n)
int n;
{
    drive_level=(float)(n-1)/10.0;
    draw_pattern(cur_pattern);
}



/************ UPDATE MENU SETTINGS ************************/
update_menus()
{
int i;
float tmp;
    /* main menu */
    setpup(mainmenu, 3, (info_flag?PUP_CHECK:PUP_BOX));
    setpup(mainmenu, 2, ((cur_pattern==5)?PUP_GREY:PUP_NONE));
    
    /* drivelevel menu done at 'set_drivelevel' */
    for (i=1;i<=11;i++) {
	tmp=(float)(i-1)/10.0;
	setpup(levelmenu, i, ((tmp==drive_level)?PUP_GREY:PUP_NONE));
    }
    
    /* pattern menu */
    for (i=1;i<=8;i++) {
	setpup(patmenu, i, ((i==cur_pattern)?PUP_GREY:PUP_NONE));
    }
    
    }
    
     

/**************** OPEN WINDOW *****************/
open_window()
{

    if (getgdesc(GD_BITS_NORM_SNG_RED) == 0) {
	fprintf(stderr, "Single buffered RGB not available on this machine.\n");
	exit(-1);
    }
    
    blanktime(0);   /* DISABLE SCREEN BLANKING */
    
    prefposition(x_win_origin, x_win_origin + x_win_size-1,
		    y_win_origin, y_win_origin + y_win_size-1);
    noborder();
    winopen("");
    ortho2(-0.5, x_win_size-0.5, -0.5, y_win_size-0.5);
    viewport(0, x_win_size-1, 0, y_win_size-1);
    
    RGBmode();
    singlebuffer();
    gconfig();
    
/* OVERLAY PLANES SETUP
    drawmode(OVERDRAW);
    overlay(2);
    singlebuffer();
    cmode();
    mapcolor(10, 1.0, 0.0, 0.0);
    drawmode(NORMALDRAW);
    gconfig();
*/
    
    RGBcolor(0, 0, 0);
    clear();
    
    draw_pattern(initial_pattern);    /* initial field */
}






/******** CLOSE WINDOW, EXIT **************/
close_window()
{
    system("/usr/sbin/gamma 1.7");
    gexit();
    blanktime(60000);
    exit(0);
}


/************ SET LINEAR GAMMA ******************/
set_linear_gamma()
{
    int i;
    short tab[256];
    
    for (i=0;i<256;i++) tab[i]=i;
    gammaramp(tab, tab, tab);
}



/****************** DEFINE LIGHTPOINT RASTER *************/
/*
 * This is for the spherical dots - SEOS wanted dots that were
 * 3 x 3 pixels in size, centre on the middle pixel.
 * This is where I redefine the '.' character to do that.
 * In the draw_sphere() routine I move the character position
 * and then do a charstr("."); 
 */
define_lightpoint_raster()
{
    ASSIGN(lpoint_font['.'], 0, 3, 3, -1, -1, 4);
    defrasterfont(1, 3, FONTNC, lpoint_font, FONTNR, raster);   
}




/****************** MAIN ******************/
main (int argc, char *argv[])
{
    long dev, menuchoice;
    short val;
    long x_win_max, y_win_max;
    
    

    
    if (argc!=6) {
	printf("Usage: %s <n> <xo> <yo> <xs> <ys>\n", argv[0]);
	printf("where\tn = pattern number:\n");
	printf("\t1 SPHEREMAP\n\t2 LINEAR CROSSHATCH\n");
	printf("\t3 GREYSCALE\n\t4 UNIFORM FIELD\n");
	printf("\t5 BLACK FIELD\n\t6 RECTANGLE\n");
	printf("\t7 CHEQUERBOARD\n\t8 RESOLUTION\n");
	printf("and (xo, yo) is the origin of the window\n");
	printf("and (xs, ys) is the size of the window\n\n");
	exit(-1);
    }
    else {
	sscanf(argv[1],"%hd", &initial_pattern);
	sscanf(argv[2],"%ld", &x_win_origin);
	sscanf(argv[3],"%ld", &y_win_origin);
	sscanf(argv[4],"%ld", &x_win_size);
	sscanf(argv[5],"%ld", &y_win_size);
    }
    
    if ((initial_pattern<1)||(initial_pattern>8)) initial_pattern=5;
    
    /* Get Frame Buffer Size */
    x_win_max = getgdesc(GD_XPMAX);
    y_win_max = getgdesc(GD_YPMAX);
    (x_win_origin<0 ? 0 : x_win_origin);
    (y_win_origin<0 ? 0 : y_win_origin);
    (x_win_origin>x_win_max ? x_win_max : x_win_origin);
    (y_win_origin>y_win_max ? y_win_max : y_win_origin);
    if((x_win_origin + x_win_size)>x_win_max) x_win_size = x_win_max - x_win_origin;
    if((y_win_origin + y_win_size)>y_win_max) y_win_size = y_win_max - y_win_origin;
    
    open_window();
    set_linear_gamma();
    define_lightpoint_raster();
    cursoff();
    
    /*SETUP MENUS*/
    mainmenu = newpup();
    patmenu = newpup();
    levelmenu = newpup();
    addtopup(levelmenu, "%F0.0|0.1|0.2|0.3|0.4|0.5|0.6|0.7|0.8|0.9|1.0|", set_drivelevel);
    addtopup(patmenu, "%Fspheremap|crosshatch|greyscale|uniform|black|rectangle|chequer|resolution",
			    draw_pattern);
    addtopup(mainmenu, "Tomba's Pattern Generator %t|Patterns %m|Drive Level %m|show info|quit",
			    patmenu, levelmenu);
    setpup(mainmenu, 3, (info_flag?PUP_CHECK:PUP_BOX));
    
    set_defaults();	/* SET PATTERN SPECIFIC DEFAULTS */
    update_menus();
    
    
    
    qdevice(ESCKEY);
    qdevice(RIGHTMOUSE);
    qdevice(MIDDLEMOUSE);
    qdevice(LEFTMOUSE);
    qreset();    
    
    while((dev = qread(&val))!=ESCKEY){
    
	/* MENU BUTTON */
	if (dev == RIGHTMOUSE) {
	    if (val == 1) {
		menuchoice = dopup(mainmenu);
	    }    
	switch (menuchoice) {
	    case 3 :	/*info_flag = !info_flag;  toggle info */
			draw_pattern(cur_pattern);	
			break;
	    case 4 :	close_window(); break;	/* quit  */
	    default :	break;
	}
        }


    if (cur_pattern==7) {
	if (dev == LEFTMOUSE)
	    {if(val == 1) cb_order = !cb_order;}
	draw_pattern(cur_pattern);    
    }



	/* GREYSCALE */
    if (cur_pattern==3) {
	if (dev == LEFTMOUSE)
	    {if(val ==1) gs_gra = !gs_gra; qread(&val); qreset();}
	if (dev == MIDDLEMOUSE)
	    {if(val ==1) gs_dir = !gs_dir; qread(&val); qreset();}
	draw_pattern(cur_pattern);
    }
	
	
	/* RECTANGLE ACTIONS */
    if ((cur_pattern==6)||(cur_pattern==8)) {
	/* LEFT MOUSE */
	if (dev == LEFTMOUSE) {
	    if (val ==1) { /* downclick */
		curson();
		qread(&val);
		qreset();
		move_rectangle(getvaluator(MOUSEX), getvaluator(MOUSEY));
		draw_pattern(cur_pattern);
		cursoff();
	    }
	}
	
	/* MIDDLE MOUSE - RESET RECTANGLE DEFAULTS */
	if (dev == MIDDLEMOUSE) {
	    if (val == 1) { /* downclick */
		set_defaults();
		draw_pattern(cur_pattern);
	    }
	}
    }
    	

	/* REDRAW */
	if (dev == REDRAW) {
	    draw_pattern(cur_pattern);
	}
    update_menus();
    }
    
    close_window();
}

/***END OF MAIN******/
