//----------------------------------------------------------------------
// tsTopology.c++
//
// This implements the classes for keeping track of the topology of
// a scene graph
//----------------------------------------------------------------------
#include <Inventor/SoOutput.h>
#include <Inventor/actions/SoWriteAction.h>
#include "tsTopology.h"

//#define VERBOSE

//----------------------------------------------------------------------
// Methods for
// class tsTopGroup: public tsTopObject
//----------------------------------------------------------------------

//----------------------------------------------------------------------
tsTopGroup::tsTopGroup()
//----------------------------------------------------------------------
	{
	id=0;
	nofChilds=0;
	next=prev=first=0;
	}
//----------------------------------------------------------------------
tsTopGroup::tsTopGroup(uint32_t i)
//----------------------------------------------------------------------
	{
	id=i;
	nofChilds=0;
	next=prev=first=0;
	}
	
//----------------------------------------------------------------------
tsTopGroup::~tsTopGroup()
//----------------------------------------------------------------------
	{
	removeAll();
	}

//----------------------------------------------------------------------
int tsTopGroup::dump(int level)
//----------------------------------------------------------------------
	{
	int i;
	for (i=0;i<level;i++) fprintf(stderr, "  ");
	fprintf(stderr, "Group: %x\n", id);

	tsTopGroup* obj=first;
	i=nofChilds;
	while (i--)
		{
		obj->dump(level+1);
		obj=obj->next;
		}
	return 1;
	}
	
//----------------------------------------------------------------------
tsTopGroup* tsTopGroup::append(tsTopGroup* obj)
//----------------------------------------------------------------------
	{
	if (!obj) return 0;
	nofChilds++;
	if (!first) return first=obj->next=obj->prev=obj;

	obj->next=first;
	obj->prev=first->prev;
	first->prev->next=obj;
	return first->prev=obj;
	}
	
//----------------------------------------------------------------------
tsTopGroup* tsTopGroup::insert(tsTopGroup* obj, int pos)
//----------------------------------------------------------------------
	{
	int i=0;
	if (!obj) return 0;
	if (pos>=nofChilds) return append(obj);

	tsTopGroup* p=first;
	while (i++<pos) p=p->next;
	
	nofChilds++;
	obj->next=p;
	obj->prev=p->prev;
	p->prev->next=obj;
	if (!pos) first=obj;
	return p->prev=obj;
	}
	
//----------------------------------------------------------------------
tsTopGroup* tsTopGroup::replace(tsTopGroup* obj, int pos)
//----------------------------------------------------------------------
	{
	int i=0;
	if (!obj) return 0;
	if (pos>=nofChilds) return 0;

	tsTopGroup* p=first;
	while (i++<pos) p=p->next;
	
	obj->next=p->next;
	obj->prev=p->prev;
	p->prev->next=obj;
	p->next->prev=obj;
	delete p;
	if (!pos) first=obj;
	return obj;
	}
	
//----------------------------------------------------------------------
tsTopGroup*	tsTopGroup::remove(int pos)
//----------------------------------------------------------------------
	{
	int i=0;
	if (!first) return 0;
	if (nofChilds==1)
		{
		nofChilds=0;
		delete first;
		first=0;
		return 0;
		}	
	if (pos>=nofChilds) pos=nofChilds=1;

	tsTopGroup* p=first;
	tsTopGroup* ret;
	while (i++<pos) p=p->next;
	ret=p->next;
	p->prev->next=p->next;
	p->next->prev=p->prev;
	delete p;
	if (!pos) first=ret;
	nofChilds--;
	return ret;	
	}
	
//----------------------------------------------------------------------
int	tsTopGroup::removeAll()
//----------------------------------------------------------------------
	{
	tsTopGroup* obj=first;
	if (!first) return 0;
	while (nofChilds--)
		{
		first=first->next;
		delete obj;
		obj=first;
		}
	first=0;
	nofChilds=0;
	return 1;
	}
		
//----------------------------------------------------------------------
tsTopGroup* tsTopGroup::find(uint32_t iid)
//----------------------------------------------------------------------
	{
	tsTopGroup *ret, *ptr;
	if (id==iid) return this;
	int i=nofChilds;
	ptr=first;
	while (i--)
		{
		if (ret=ptr->find(iid)) return ret;
		ptr=ptr->next;
		}		
	return 0;
	}

			
//----------------------------------------------------------------------
// Methods for
// class tsTopology
//----------------------------------------------------------------------

//----------------------------------------------------------------------
tsTopology::tsTopology() 
//----------------------------------------------------------------------
	{
	root=0;
	isbinary=0;
	truncate();
	}
	
//----------------------------------------------------------------------
tsTopology::~tsTopology()
//----------------------------------------------------------------------
	{
	if (root) delete root;
	}	

//----------------------------------------------------------------------
int tsTopology::truncate() 
//----------------------------------------------------------------------
	{
	// truncate all paths
	commList.truncate(0);
	pathList.truncate(0);
	chldList.truncate(0);
	// but we have to free the mem of the graph-list
	for (int i=0;i<grphList.getLength();i++) if (grphList[i]) {free(grphList[i]); grphList[i]=0;}
	grphList.truncate(0);
	return 0;
	}
	
//----------------------------------------------------------------------
tsTopGroup* tsTopology::buildGroup(SoNode *node)
//----------------------------------------------------------------------
	{
	if (!node) return 0;
	tsTopGroup* obj=new tsTopGroup((uint32_t) node);
	if (node->isOfType(SoGroup::getClassTypeId()))
		{
		int i=0;
		SoGroup* grp=(SoGroup*) node;
		while (i<grp->getNumChildren())
			{
			obj->append(buildGroup(grp->getChild(i)));
			i++;
			}
		}
	return obj;
	}
		
//----------------------------------------------------------------------
int tsTopology::updateGroup(tsTopGroup *obj, SoPath *path, int record)
//----------------------------------------------------------------------
	{
	if (!path) return 0;
	if (!obj) return 0;

	// check if node is a group node
	if (!(path->getTail()->isOfType(SoGroup::getClassTypeId()))) return 0;

	// get group node
	SoGroup *grp=(SoGroup*) path->getTail();
	int i=0, j=0, nofc=grp->getNumChildren();
	uint32_t nid;
	if (nofc)
		{
		// if we havn't any children, we have to build them
		if (!obj->nofChilds)
			{
			while (i<nofc)
				{
				#ifdef VERBOSE
					fprintf(stderr, "new appending child #%d\n", i);
				#endif
				obj->append(buildGroup(grp->getChild(i)));
				if (record) addChange(TOP_INSERT_ID, path, i,grp->getChild(i) );
				i++;
				}
			return 1;
			}
		else
			{
			// we have to find the difference
			tsTopGroup *tg=obj->first;
			if (obj->nofChilds<nofc)
				{
				// nodes inserted
				while (i<nofc)
					{
					nid=(uint32_t) grp->getChild(i);
					if ((tg->id!=nid) || (i>=obj->nofChilds))
						{
						#ifdef VERBOSE
							fprintf(stderr, "node %x inserted\n", nid);
						#endif
						tg=obj->insert(buildGroup(grp->getChild(i)), i);
						if (record) addChange(TOP_INSERT_ID, path, i, grp->getChild(i));
						}
					tg=tg->next;
					i++;
					}
				}		
			else if (obj->nofChilds>nofc)
				{
				// nodes deleted
				while (j<obj->nofChilds)
					{
					nid=(j<nofc) ? (uint32_t) grp->getChild(j) : 0;
					if (tg->id!=nid)
						{
						#ifdef VERBOSE
							fprintf(stderr, "node %x deleted\n", tg->id);
						#endif
						tg=obj->remove(j);
						if (record) addChange(TOP_REMOVE_ID, path, j, 0);
						}
					else
						{
						tg=tg->next;
						j++;
						}
					}
				}
			else
				{
				// nodes replaced
				while (i<nofc)
					{
					nid=(uint32_t) grp->getChild(i);
					if (tg->id!=nid)
						{
						#ifdef VERBOSE
							fprintf(stderr, "node %x with node %x replaced\n", tg->id, nid);
						#endif
						tg=obj->replace(buildGroup(grp->getChild(i)), i);
						if (record) addChange(TOP_REPLACE_ID, path, i, grp->getChild(i));
						}
					tg=tg->next;
					i++;
					}
   				}
			return 1;
			}
		}
	else
		{
		// if we havn't any children too, nothing changed
		if (!obj->nofChilds) return 0;
		
		// otherwise all children have to be removed
		obj->removeAll();
		if (record) addChange(TOP_REMOVEALL_ID, path, 0, 0);	
		return 1;
		}
	}
	
//----------------------------------------------------------------------
int tsTopology::init(SoNode *node)
//----------------------------------------------------------------------
	{
	if (root) delete root;
	root=0;
	if (!node) return 0;
	root=buildGroup(node);
	truncate();
	return 1;
	}

//----------------------------------------------------------------------
int tsTopology::update(SoPath *path, int record)
//----------------------------------------------------------------------
	{
	if (!path) return 0;
	if (!root) return 0;
	
	tsTopGroup *tg=root->find((uint32_t) (path->getTail()));
	if (!tg)
		{
		fprintf(stderr, "Node not found!!!\n");
		return 0;
		}
	else
		{
		updateGroup(tg, path, record);
		}
	return 1;
	}

//----------------------------------------------------------------------
void *tsTopology::getGraphDesc(SoNode *nd)
//----------------------------------------------------------------------
	{
	SoOutput o;
	void *ptr=0;
	size_t  size=1024; 
	if (!nd) return 0;
	ptr=malloc(size);
	o.setBuffer(ptr, size, realloc, sizeof(int));
	o.setBinary(isbinary);
	SoWriteAction w(&o);
	w.apply(nd);
	o.getBuffer(ptr, size);
	
	((int*)ptr)[0]=size;
	return ptr;
	}
	
//----------------------------------------------------------------------
int tsTopology::addChange(int cmd, SoPath *path, int idx, SoNode *nd)
//----------------------------------------------------------------------
	{
	// if we have a removeall, we check the history back, if we deleted
	// any childs sequencially
	if (cmd==TOP_REMOVEALL_ID)
		{
		int i=pathList.getLength();
		while ((i) && (pathList[i-1]->getTail()==path->getTail()) && (commList[i-1]==TOP_REMOVE_ID)) i--;
		commList.truncate(i);
		pathList.truncate(i);
		chldList.truncate(i);
		// free memory in grph list
		for (int j=i;j<grphList.getLength();j++) if (grphList[j]) {free(grphList[j]); grphList[j]=0;}
		grphList.truncate(i);
		}
	commList.append(cmd);
	pathList.append(path);
	chldList.append(idx);
	grphList.append(getGraphDesc(nd));
	return 1;
	}	
	
//----------------------------------------------------------------------
int tsTopology::dump()
//----------------------------------------------------------------------
	{
	//if (root) return root->dump(0);
	int i;
	char *cmd[4]={"INSERT", "REMOVE", "REPLACE", "REMOVEALL"};
	
	for (i=0;i<pathList.getLength();i++)
		{
		fprintf(stderr, "%s %08x %d\n", cmd[(int)commList[i]], (uint32_t) pathList[i]->getTail(), chldList[i]);
		}
	return 0;	
	}
