/*
 * Copyright (C) 1996   Silicon Graphics, Inc.
 *
 _______________________________________________________________________
 ______________  S I L I C O N   G R A P H I C S   I N C .  ____________
 |
 |   $Revision: 1.2 $
 |
 |   multithreaded server
 |
 |   Author(s)          : Horst Vollhardt
 |
 ______________  S I L I C O N   G R A P H I C S   I N C .  ____________
 _______________________________________________________________________
 */

#include <errno.h>
#ifdef DEBUG2
#include <iostream.h>
#include <stdio.h>
#else
#include <syslog.h>
#endif
#include <signal.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/prctl.h>
#include <sys/socket.h>
#include <time.h>
#include <unistd.h>
#include <string.h>


#include <Inventor/SbPList.h>
#include <Inventor/SoDB.h>
#include <Inventor/SoInput.h>
#include <Inventor/SoOutput.h>
#include <Inventor/SoPath.h>
#include <Inventor/Xt/SoXt.h>
#include <Inventor/actions/SoWriteAction.h>


#include "HvServer.h"
#include "HvClientList.h"

#include "HvCommId.h"

static SbString dbDir("/var/tmp/");


////////////////////////////////////////////////////////////////////////
//
//   print usage message
//
////////////////////////////////////////////////////////////////////////
void print_usage()
{
#ifdef DEBUG2
   cerr << "usage: Serv [-f][-p port]\n";
#endif
}


////////////////////////////////////////////////////////////////////////
//
//   no-op signal handler
//
////////////////////////////////////////////////////////////////////////
void signal_handler(int)
{
   signal(SIGHUP, signal_handler);
}


////////////////////////////////////////////////////////////////////////
//
//   listen for client
//
////////////////////////////////////////////////////////////////////////
void listenClient(void *arg)
{
#ifdef DEBUG2
   cerr << "sproc ok (listen)!\n";
#endif

   HvClientList &list = *((HvClientList *)arg);

   fd_set readMask;
   HvNetBuffer buffer;

   int num;
   while (num = list.getNum()) {
      FD_ZERO(&readMask);
      for (int i = 0; i < num; i++) {
         FD_SET(list.getSocket(i), &readMask);
      }

      int len;
      if ((select(FD_SETSIZE, &readMask, NULL, NULL, NULL)) > 0) {
         for (i = 0; i < num; i++) {
	    int s = list.getSocket(i);
	    if (FD_ISSET(s, &readMask)) {
	       int r = list.receive(i);
	       if ((r > 0)&&(list.isComplete(i))) {
#ifdef DEBUG2
   cerr << i+1 << ".(" << list.getId() << "): ";
#endif
		  const char *data = (char *)list.get(i, len);
		  int id = *((int *)data);
#ifdef DEBUG2
   cerr << "id = " << id << "\n";
#endif
		  switch(id) {
		     case GRAPH_ID:
		        list.readSceneGraph(&data[S_INT]);
		     	break;
		     case PATH_ID:
		        list.readPath(&data[S_INT]);
		        list.sendPath(i, data, len);
		     	break;
		     case SELECT_ID:
		        list.sendPath(i, data, len);
		     	break;
		     case DESELECT_ID:
		        list.sendPath(i, data, len);
		     	break;
		     case CHANGE_ID:
			list.readChange((int*)&data[S_INT]);
		        list.sendPath(i, data, len);
		     	break;
		     default:
#ifdef DEBUG2
		        cerr << "\tUnknown id: " << id << "\n";
#endif
			// unknown packet but send anyway
		        list.sendPath(i, data, len);
			break;
		  }
	       list.clear(i);
               }
	       else if (r == 0) {
	          list.remove(i--);
		  num--;
	       }
	    }
	 }
      }
   }
   SbString file = dbDir;
   file += SbString(list.getId());
   file += ".civ";
   list.writeSceneGraph(file);
}



////////////////////////////////////////////////////////////////////////
//
//   handle new client connection
//
////////////////////////////////////////////////////////////////////////
void newClient(void *arg)
{
   static SbBool synchro = FALSE;  	// let's see if this poor-mans
   while(synchro);			// synchronization works
   synchro = TRUE;
   static SbPList clientListList;

   int sock = (int)arg;

   HvNetBuffer buffer(sock);

   // read ID
   int ret;
   while ((ret = buffer.receive()) != 0) {
      if (ret < 0) continue;  // may result in a infinite loop !
      if (buffer.isComplete()) break;
   }
   if (ret == 0) { close(sock); synchro = FALSE; exit(0); }

   int len;
   const char *data = (const char *)buffer.get(len);
   int id, dataId = *((int *)data);
   if (dataId == ID_ID) {
      id = *((int *)&data[S_INT]);
   }
   else if (dataId == SEND_LIST_ID) {	// send list of connected clients
      buffer.set(LIST_ID);
      int num = clientListList.getLength();
      buffer.append(num);
      for (int i = 0; i < num; i++) {
	 HvClientList &list = *((HvClientList *)clientListList[i]);
         int id = list.getId();
         int n = list.getNum();
         if (n == 0) {		// all clients have gone
            delete &list;
            clientListList.remove(i--);
  	    num--;
         }
         buffer.append(id);
         buffer.append(n);
      }
      int bytesSend = 0;
      while (bytesSend < buffer.getLength()) {		// wait until
         bytesSend += buffer.send(bytesSend);		// everything is send
      }
      buffer.clear();
      synchro = FALSE; exit(0);
   }
   else {
#ifdef DEBUG2
      cerr << "expected ID_ID, got: " << dataId << "\n";
#else
      syslog(LOG_ERR, "Serv(newClient): got unknown ID");
#endif
      close(sock); synchro = FALSE; exit(0);
   }

   // check if a thread for this ID already exists
   int pid, num = clientListList.getLength();
   for (int i = 0; i < num; i++) {
      HvClientList &list = *((HvClientList *)clientListList[i]);
      if (list.getNum() == 0) {		// all clients have gone
         delete &list;
         clientListList.remove(i--);
	 num--;
	 continue;
      }
      if (id == list.getId()) {		// put this client into the list
         list.append(sock);
	 list.sendSceneGraph(sock);	// this may take a while
	 pid = list.getPID();
	 kill(pid, SIGHUP);		// wakeup the thread 
	 synchro = FALSE; exit(0);
      }
   }

   // create new thread
   HvClientList *list = new HvClientList(id);
   clientListList.append(list);
   list->append(sock);
   list->requestSceneGraph(sock);
   if ((pid = sproc(listenClient, PR_SADDR | PR_NOLIBC | PR_SFDS, list)) < 0) {
#ifdef DEBUG2
      perror("Serv(newClient)");
#else
      syslog(LOG_ERR, "Serv(newClient): %m");
#endif
   }
   list->setPID(pid);

   synchro = FALSE;		// stop blocking this function

#ifdef DEBUG2
   cerr << "sproc ok (new)!\n";
#endif
}


////////////////////////////////////////////////////////////////////////
//
//  main loop waiting to accept client connections
//
////////////////////////////////////////////////////////////////////////
void main (int argc, char **argv)
{
   int c, port = 4077, len;
   extern char *optarg;
   extern int optind, opterr;
   SbBool backgrounding = TRUE;

   signal(SIGHUP, signal_handler);

   while ((c = getopt(argc, argv, "fp:d:")) != EOF) {
      switch (c) {
         case 'p':
	    port = atoi(optarg);
            break;
         case 'f':		// going not in background
	    backgrounding = FALSE;
            break;
         case 'd':
	    dbDir = optarg;
   	    len = dbDir.getLength();
            if ('/' != dbDir.getString()[len-1]) dbDir += "/";
            break;
         case '?':
	    print_usage();
	    exit(1);
	    break;
	 default:
#ifdef DEBUG2
	    cerr << "Serv: shouldn't show up here !?\n";
#endif
	    break;
      }
   }

   if (backgrounding) {
      _daemonize(0, -1, -1, -1);
/*
      char **a = new char*[argc+2];
      for (int i = 0; i < argc; i++)
         a[i] = strdup(argv[i]);
      a[i++] = "-f";
      a[i++] = NULL;
      execvp(argv[0], argv);
*/
   }

#ifdef DEBUG2
   SoXt::init(argv[0]);
#else
   SoXt::init(argv[0]);
   //ts//SoDB::init();
#endif

   HvServer server(port);

   int s = server.createSocket();
   if (s < 0) {
#ifdef DEBUG2
      perror("Serv(socket)");
#else
      syslog(LOG_ERR, "Serv(socket): %m");
#endif
      exit(1);
   }

   while ((s = server.acceptConnection()) >= 0) {
      if (sproc(newClient, PR_SADDR | PR_NOLIBC | PR_SFDS, s) < 0) {
#ifdef DEBUG2
         perror("Serv(sproc)");
#else
         syslog(LOG_ERR, "Serv(sproc): %m");
#endif
      }
   }

#ifdef DEBUG2
   cerr << "closing service\n";
#endif
   server.closeConnection();   
}
