/*
 * Copyright (C) 1996   Silicon Graphics, Inc.
 *
 _______________________________________________________________________
 ______________  S I L I C O N   G R A P H I C S   I N C .  ____________
 |
 |   $Revision: 1.5 $
 |
 |   Classes:
 |      HvWebIO
 |
 |   Author(s)          : Horst Vollhardt
 |
 ______________  S I L I C O N   G R A P H I C S   I N C .  ____________
 _______________________________________________________________________
 */

#include "HvWebIO.h"
#include "HvWebField.h"
#include "HvManager.h"

#include <iostream.h>


SO_NODE_SOURCE(HvWebIO);

////////////////////////////////////////////////////////////////////////
//
//  Initialize the class
//  
void HvWebIO::initClass()
//
////////////////////////////////////////////////////////////////////////
{
   SO_NODE_INIT_CLASS(HvWebIO,SoNode,"Node");
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    Constructor
//
// Use: public

HvWebIO::HvWebIO()
//
////////////////////////////////////////////////////////////////////////
{
   SO_NODE_CONSTRUCTOR(HvWebIO);

   SO_NODE_ADD_FIELD(type,	(OUTPUT));
   SO_NODE_ADD_FIELD(hidden,	(FALSE));
   SO_NODE_ADD_FIELD(hvInt,	(""));
   SO_NODE_ADD_FIELD(hvEnum,	(""));
   SO_NODE_ADD_FIELD(hvFloat,	(""));
   SO_NODE_ADD_FIELD(hvString,	(""));
   SO_NODE_ADD_FIELD(hvSlider,	(""));
   SO_NODE_ADD_FIELD(hvCheck,	(""));
   SO_NODE_ADD_FIELD(hvOption,	(""));
   SO_NODE_ADD_FIELD(hvList,	(""));
   SO_NODE_ADD_FIELD(hvField,	(""));

   SO_NODE_DEFINE_ENUM_VALUE (Type, INPUT);
   SO_NODE_DEFINE_ENUM_VALUE (Type, OUTPUT);
   SO_NODE_SET_SF_ENUM_TYPE (type, Type);

   parentNode = NULL;
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    find a node by node type name in the parent node
//
// Use: public

SoNode *
HvWebIO::findNode(SoNode *pNode, const char *name)
//
////////////////////////////////////////////////////////////////////////
{
   if (!pNode) return NULL;

   if (!strcmp(name, pNode->getTypeId().getName().getString()))
      return pNode; 

   if (! pNode->getTypeId().isDerivedFrom(SoGroup::getClassTypeId()))
      return NULL;

   SoGroup *group = (SoGroup *)pNode;
   int num = group->getNumChildren();
   for (int i = 0; i < num; i++) {
      SoNode *node = findNode(group->getChild(i), name);
      if (node) return node;
   }
   return NULL;
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//   find a specific field
//   returns NULL if it doesn't find one
//
SoField *HvWebIO::findField(SoNode *node, const char *name)
//
////////////////////////////////////////////////////////////////////////
{
return node->getField(SbName(name));
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    Analyses the strings in hvField returns a list of HvWebFields
//
// Use: public

SbPList
HvWebIO::analyseFields()
//
////////////////////////////////////////////////////////////////////////
{
   SbPList fieldList;

   int fieldType;
   if (type.getValue() == HvWebIO::INPUT) fieldType = HvWebField::INPUT_FIELD;
   else                                   fieldType = HvWebField::OUTPUT_FIELD;

   int num = hvField.getNum();
   for (int i = 0; i < num; i++) {
      if (! hvField[i]) continue;

      const char *string = hvField[i].getString();
      char *nodeName = NULL, *fieldName = NULL;

      if(!(string = findWord(string, nodeName))) {
         cerr << "analyseFields: empty string (" << string << ")\n";
         continue;
      }
      SoNode *node = findNode(parentNode, nodeName);
      if (!node) {
         cerr << "analyseFields: unknown node (" << nodeName << ")\n";
         continue;
      }

      if (!(string = strchr(string, '{'))) {
         cerr << "analyseFields: syntax error (" << string << ")\n";
         cerr << "\tmissing \'{\'\n";
	 continue;
      }
      string++;

      while (*string) { // find field names
         const char *str;
	 if (!(str = findWord(string, fieldName))) break;
	 string = str;

         if (node->getTypeId() == HvManager::getClassTypeId()) {
	    int sType;
            if (!strcmp("atom", fieldName))
	       sType = HvWebField::ATOM;
            else if (!strcmp("bond", fieldName))
	       sType = HvWebField::BOND;
	    else {
	       SoField *field = findField(node, fieldName);
               if (!field) {
                  cerr << "analyseFields: unknown field (" << fieldName << ")\n";
                  continue;
               }
               HvWebField *f = new HvWebField;
	       f->setField(field);
	       f->setType(fieldType);
	       f->setFieldName(fieldName);
	       f->setNodeType(node);
               f->setParentNode(parentNode);
	       f->setLabel(nodeName);
	       f->appendLabel("::");
	       f->appendLabel(fieldName);
               fieldList.append(f);
               continue;
	    }
            HvWebField *f = new HvWebField;

	    f->setType(HvWebField::SELECTION);
	    f->setFieldName("");

	    f->setSelectionType(sType);

            f->setParentNode(parentNode);

            fieldList.append(f);
	    continue;
         }

	 SoField *field = findField(node, fieldName);
         if (!field) {
            cerr << "analyseFields: unknown field (" << fieldName << ")\n";
            continue;
         }

         HvWebField *f = new HvWebField;

	 f->setField(field);

	 f->setType(fieldType);
	 f->setFieldName(fieldName);
	 f->setNodeType(node);

         f->setParentNode(parentNode);

	 f->setLabel(nodeName);
	 f->appendLabel("::");
	 f->appendLabel(fieldName);

         fieldList.append(f);
      }

      if (!(string = strchr(string, '}'))) {
         cerr << "analyseFields: syntax error (" << string << ")\n";
         cerr << "\tmissing \'{\'\n";
	 continue;
      }
   }

   return fieldList;
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    Analyses the strings in hvFloat returns a list of HvWebFields
//
// Use: public

SbPList
HvWebIO::analyseFloats()
//
////////////////////////////////////////////////////////////////////////
{
   SbPList fieldList;

   int fieldType;
   if (type.getValue() == HvWebIO::INPUT) fieldType = HvWebField::INPUT_FLOAT;
   else                                   fieldType = HvWebField::OUTPUT_FLOAT;

   int num = hvFloat.getNum();
   for (int i = 0; i < num; i++) {
      if (! hvFloat[i]) continue;

      const char *string = hvFloat[i].getString();
      char *label = NULL, *value = NULL;

      if (!(string = findWord(string, label))) {
//         cerr << "analyseFloats: empty string (" << string << ")\n";
         continue;
      }

      findWord(string, value);  // read optional default value;

      HvWebField *f = new HvWebField;

      f->setType(fieldType);

      f->setLabel(label); delete [] label;

      f->setParentNode(parentNode);

      if (value) {
         f->setValue(atof(value)); delete [] value;
      }

      fieldList.append(f);
   }

   return fieldList;
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    Analyses the strings in hvString returns a list of HvWebFields
//
// Use: public

SbPList
HvWebIO::analyseStrings()
//
////////////////////////////////////////////////////////////////////////
{
   SbPList fieldList;

   int fieldType;
   if (type.getValue() == HvWebIO::INPUT) fieldType = HvWebField::INPUT_STRING;
   else                                   fieldType = HvWebField::OUTPUT_STRING;

   int num = hvString.getNum();
   for (int i = 0; i < num; i++) {
      if (! hvString[i]) continue;

      const char *string = hvString[i].getString();
      char *label = NULL, *value = NULL;

      if (!(string = findWord(string, label))) {
//         cerr << "analyseStrings: empty string (" << string << ")\n";
         continue;
      }

      findWord(string, value);  // read optional default value;

      HvWebField *f = new HvWebField;

      f->setType(fieldType);

      f->setLabel(label); delete [] label;

      f->setParentNode(parentNode);

      if (value) {
         f->setString(value); delete [] value;
      }

      fieldList.append(f);
   }

   return fieldList;
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    Analyses the strings in hvSlider returns a list of HvWebFields
//
// Use: public

SbPList
HvWebIO::analyseSlider()
//
////////////////////////////////////////////////////////////////////////
{
   SbPList fieldList;

   int fieldType;
   fieldType = HvWebField::INPUT_SLIDER;

   int num = hvSlider.getNum();
   for (int i = 0; i < num; i++) {
      if (! hvSlider[i]) continue;

      const char *string = hvSlider[i].getString();
      char *label = NULL, *value = NULL;

      if (!(string = findWord(string, label))) {
//         cerr << "analyseSlider: empty string (" << string << ")\n";
         continue;
      }

      float smin = 0., smax = 1., sval = 0.5;
      string = findWord(string, value);  // read optional minimum;
      if (value) smin = atof(value);
      string = findWord(string, value);  // read optional maximum;
      if (value) smax = atof(value);
      string = findWord(string, value);  // read optional value;
      if (value) sval = atof(value);
      HvWebField *f = new HvWebField;

      f->setType(fieldType);

      f->setLabel(label); delete [] label;

      f->setSlider(smin, smax, sval);

      f->setParentNode(parentNode);

      if (value) delete [] value;

      fieldList.append(f);
   }

   return fieldList;
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    Analyses the strings in hvList returns a list of HvWebFields
//
// Use: public

SbPList
HvWebIO::analyseList()
//
////////////////////////////////////////////////////////////////////////
{
   SbPList fieldList;

   int fieldType;
   fieldType = HvWebField::INPUT_LIST;

   int num = hvList.getNum();
   for (int i = 0; i < num; i++) {
      if (! hvList[i]) continue;

      const char *string = hvList[i].getString();
      char *label = NULL, *value = NULL;

      if (!(string = findWord(string, label))) {
         continue;
      }

      HvWebField *f = new HvWebField;
      f->setType(fieldType);
      f->setLabel(label); delete [] label;
      f->setParentNode(parentNode);
      while (string = findWord(string, value)) {  // read list items;
         f->addListItem(value);
      }

      if (value) delete [] value;

      fieldList.append(f);
   }

   return fieldList;
}
