/*
 * Copyright (C) 1996   Silicon Graphics, Inc.
 *
 _______________________________________________________________________
 ______________  S I L I C O N   G R A P H I C S   I N C .  ____________
 |
 |   $Revision: 1.3 $
 |
 |   Classes:
 |      HvWebFunction
 |
 |   Author(s)          : Horst Vollhardt
 |
 ______________  S I L I C O N   G R A P H I C S   I N C .  ____________
 _______________________________________________________________________
 */

#include <stdlib.h>
#include <iostream.h>
#include <malloc.h>

#include <Xm/AtomMgr.h>
#include <Xm/Form.h>
#include <Xm/Frame.h>
#include <Xm/Label.h>
#include <Xm/MwmUtil.h>
#include <Xm/Protocols.h>
#include <Xm/PushB.h>
#include <Xm/RowColumn.h>
#include <Xm/Text.h>
#include <Xm/ToggleB.h>
#include <Xm/Separator.h>

#include <Inventor/Xt/SoXt.h>
#include <Inventor/SoDB.h>
#include <Inventor/SoInput.h>
#include <Inventor/SoType.h>
#include <Inventor/fields/SoSFFloat.h>
#include <Inventor/fields/SoMFFloat.h>
#include <Inventor/fields/SoMFVec3f.h>
#include <Inventor/fields/SoMFColor.h>
#include <Inventor/fields/SoMFEnum.h>
#include <Inventor/fields/SoMFInt32.h>
#include <Inventor/nodes/SoInfo.h>
#include <Inventor/nodes/SoSeparator.h>

#include "HvWebFunction.h"
#include "HvWebFunctionPixmap.h"
#include "HvWebField.h"
#include "HvManager.h"
#include "HvWebIO.h"


enum WidgetIds {
    DIALOG_SHELL,
    DIALOG_FORM,

    MAIN_FRAME,
    MAIN_LABEL,
    MAIN_FORM,

    INPUT_FRAME,
    INPUT_LABEL,
    INPUT_FORM,

    CONNECT_FRAME,
    CONNECT_LABEL,
    CONNECT_FORM,

    OUTPUT_FRAME,
    OUTPUT_LABEL,
    OUTPUT_FORM,
    OUTPUT_LOG_TEXT,

    HIDE_BUTTON,
    HELP_BUTTON,
    START_BUTTON,
    STOP_BUTTON,


    WIDGET_NUM, // must be last
};


#define HvUIOffset 5

////////////////////////////////////////////////////////////////////////
//
// Description:
//    Constructor
//
// Use: public

HvWebFunction::HvWebFunction()
//
////////////////////////////////////////////////////////////////////////
{
    mode = HvWebFunction::STANDBY;

    widgetList = new Widget[WIDGET_NUM];
    for (int i = 0; i < WIDGET_NUM; i++) {
        widgetList[i] = NULL;
    }
    webMenuButton = NULL;

    webFunctionDialogVisible = FALSE;
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    Destructor
//
// Use: public

HvWebFunction::~HvWebFunction()
//
////////////////////////////////////////////////////////////////////////
{
    // destroy widgets ?
    delete [] title;
    delete [] widgetList;
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//   Analyse the user interface definition
//
// Use: public

void
HvWebFunction::analyseURLBuffer(const char *s)
//
////////////////////////////////////////////////////////////////////////
{
   HvWebIO::initClass();

   SoInput in;
   in.setBuffer((void *)s, strlen(s));

   SoSeparator *root;
   if (!(root = SoDB::readAll(&in))) {
      cerr << "analyseURLBuffer: error while reading\n";
      return;
   }

   int n = root->getNumChildren();
   for (int i = 0; i < n; i++) {
      SoNode *node = root->getChild(i);

      if (node->getTypeId() == HvWebIO::getClassTypeId()) {
         HvWebIO *wio = (HvWebIO *)node;
	 wio->setParentNode(parentNode);
	 SbBool hidden = wio->hidden.getValue();

         SbPList list = wio->analyseStrings();
         int l = list.getLength();
	 for (int j = 0; j < l; j++) {
	    HvWebField *wf = (HvWebField *)list[j];
	    if (wf->getType() == HvWebField::OUTPUT_STRING) {
	       outputList.append(wf);
	       if (!hidden) visibleOutputList.append(wf);
	    }
	    else {
	       inputList.append(wf);
	       if (!hidden) visibleInputList.append(wf);
	    }

	    wf->setHidden(hidden);
	 }

         list = wio->analyseFloats();
         l = list.getLength();
	 for (j = 0; j < l; j++) {
	    HvWebField *wf = (HvWebField *)list[j];
	    if (wf->getType() == HvWebField::OUTPUT_FLOAT) {
	       outputList.append(wf);
	       if (!hidden) visibleOutputList.append(wf);
	    }
	    else {
	       inputList.append(wf);
	       if (!hidden) visibleInputList.append(wf);
	    }

	    wf->setHidden(hidden);
	 }

         list = wio->analyseSlider();
         l = list.getLength();
	 for (j = 0; j < l; j++) {
	    HvWebField *wf = (HvWebField *)list[j];
	    inputList.append(wf);
	    visibleInputList.append(wf);
	    wf->setHidden(FALSE);
	 }

         list = wio->analyseList();
         l = list.getLength();
	 for (j = 0; j < l; j++) {
	    HvWebField *wf = (HvWebField *)list[j];
	    inputList.append(wf);
	    visibleInputList.append(wf);
	    wf->setHidden(FALSE);
	 }

         list = wio->analyseFields();
         l = list.getLength();
	 for (j = 0; j < l; j++) {
	    HvWebField *wf = (HvWebField *)list[j];
	    if (wf->getType() == HvWebField::OUTPUT_FIELD) {
	       outputList.append(wf);
	       if (!hidden) visibleOutputList.append(wf);
	    }
	    else {
	       inputList.append(wf);
	       if (!hidden) visibleInputList.append(wf);
	    }

	    wf->setHidden(hidden);
	 }
      }
      else if (node->getTypeId() == SoInfo::getClassTypeId()) {
         SoInfo *info = (SoInfo *)node;
         const char *name = info->getName().getString();
	 const char *value = info->string.getValue().getString();

         if (!strcmp("Title", name)) {
	    title = new char[strlen(value)+1];
	    strcpy(title, value);
	 }
         else if (!strcmp("Port", name)) {
            daemon.setPort(atoi(value));
         }
      }
   }

   // if there are no visible fields, don't build gui
   int numVis = visibleInputList.getLength() + visibleOutputList.getLength();
   if (numVis == 0) mode = HvWebFunction::DIRECT;
}

////////////////////////////////////////////////////////////////////////
//  
// Description:
//    Callback for the web menu dialog button
//
// Use: private

void
HvWebFunction::webMenuDialogCB(Widget, HvWebFunction *wfn, void *)
//
////////////////////////////////////////////////////////////////////////
{
   // Check to see if we need a dialog
   if (wfn->mode == HvWebFunction::DIRECT) {
      startCB(NULL, wfn, NULL);
   }
   else {
      wfn->openWebFunctionDialog();
   }
}

////////////////////////////////////////////////////////////////////////
//  
// Description:
//    Build the web function dialog window
//
// Use: private

void
HvWebFunction::openWebFunctionDialog()
//
////////////////////////////////////////////////////////////////////////
{
   // Check to see if the dialog has already been built
   Widget dialog = widgetList[DIALOG_SHELL];
   if (dialog != NULL) {
      webFunctionDialogVisible = TRUE;
      SoXt::show(dialog);
      return;
   }

   Widget   shell = SoXt::getTopLevelWidget();
   Display  *display = XtDisplay(shell);

   int i, n = 0;
   Arg args[20];

   XtSetArg(args[n], XmNdeleteResponse, XmUNMAP); n++;
   XtSetArg(args[n], XtNtitle, "Web Function Dialog"); n++;

   // Remove the Iconize/Maximize/Quit functionality
   XtSetArg(args[n], XmNmwmDecorations,
      (MWM_DECOR_BORDER | MWM_DECOR_RESIZEH | MWM_DECOR_TITLE |
       MWM_DECOR_MENU)); n++;

   XtSetArg(args[n], XmNmwmFunctions,
      (MWM_FUNC_RESIZE | MWM_FUNC_MOVE | MWM_FUNC_CLOSE)); n++;

   widgetList[DIALOG_SHELL] = XtCreateWidget("HvWebFunctionDialog",
      topLevelShellWidgetClass, shell, args, n);

   // Add callback that handles the window closing
   Atom wmDeleteAtom = XmInternAtom(display, "WM_DELETE_WINDOW", False);
      XmAddWMProtocolCallback(widgetList[DIALOG_SHELL], wmDeleteAtom,
      (XtCallbackProc) HvWebFunction::webFunctionClosedCB, (caddr_t) this);

#ifdef ZIPPY
    // Add an event handler to receive map/unmap events on the main
    // shell window - this will allow us to hide this dialog when the
    // application iconifies.
    XtAddEventHandler(shell, StructureNotifyMask, FALSE,
        (XtEventHandler) HvManager::shellStructureNotifyCB,
        (XtPointer) this);
#endif

   Widget mainForm = widgetList[DIALOG_FORM] =
      XtVaCreateManagedWidget("mainForm", xmFormWidgetClass,
                     widgetList[DIALOG_SHELL], NULL);
   
   Widget hideButton = widgetList[HIDE_BUTTON] = 
      XtVaCreateManagedWidget("hideButton",
         xmPushButtonWidgetClass, mainForm,
         XmNrightAttachment,  XmATTACH_FORM,
         XmNrightOffset,      HvUIOffset,
         XmNbottomAttachment, XmATTACH_FORM,
         XmNbottomOffset,     HvUIOffset,
         XmNlabelString,      XmStringCreateSimple("Hide"),
         NULL);
   XtAddCallback(hideButton, XmNactivateCallback,
         (XtCallbackProc) HvWebFunction::unmanageCB,
	 (XtPointer) this); 

   Widget helpButton = widgetList[HELP_BUTTON] = 
      XtVaCreateManagedWidget("helpButton",
         xmPushButtonWidgetClass, mainForm,
         XmNrightAttachment,  XmATTACH_WIDGET,
         XmNrightOffset,      HvUIOffset,
         XmNrightWidget,      hideButton,
         XmNbottomAttachment, XmATTACH_FORM,
         XmNbottomOffset,     HvUIOffset,
         XmNlabelString,      XmStringCreateSimple("Help"),
         NULL);
   XtAddCallback(helpButton, XmNactivateCallback,
         (XtCallbackProc) HvWebFunction::helpCB,
	 (XtPointer) this); 

   Widget startButton = widgetList[START_BUTTON] =
      XtVaCreateManagedWidget("startButton",
         xmPushButtonWidgetClass, mainForm,
         XmNleftAttachment,   XmATTACH_FORM,
         XmNleftOffset,       HvUIOffset,
         XmNbottomAttachment, XmATTACH_FORM,
         XmNbottomOffset,     HvUIOffset,
         XmNlabelString,      XmStringCreateSimple("Start"),
         NULL);
   XtAddCallback(startButton, XmNactivateCallback,
         (XtCallbackProc) HvWebFunction::startCB,
	 (XtPointer) this); 

   Widget stopButton = widgetList[STOP_BUTTON] =
      XtVaCreateManagedWidget("stopButton",
         xmPushButtonWidgetClass, mainForm,
         XmNleftAttachment,   XmATTACH_WIDGET,
         XmNleftOffset,       HvUIOffset,
         XmNleftWidget,       startButton,
         XmNbottomAttachment, XmATTACH_FORM,
         XmNbottomOffset,     HvUIOffset,
         XmNlabelString,      XmStringCreateSimple ("Stop"),
         NULL);
   XtSetSensitive(stopButton, FALSE);

   Widget mainFrame = widgetList[MAIN_FRAME] =
      XtVaCreateManagedWidget("mainFrame",
         xmFrameWidgetClass, mainForm,
         XmNtopAttachment,    XmATTACH_FORM,
         XmNtopOffset,        HvUIOffset,
         XmNleftAttachment,   XmATTACH_FORM,
         XmNleftOffset,       HvUIOffset,
         XmNrightAttachment,  XmATTACH_FORM,
         XmNrightOffset,      HvUIOffset,
         XmNbottomAttachment, XmATTACH_WIDGET,
         XmNbottomWidget,     hideButton,
         XmNbottomOffset,     HvUIOffset,
         NULL);

   Widget mainLabel = widgetList[MAIN_LABEL] =
      XtVaCreateManagedWidget("mainLabel",
         xmLabelWidgetClass, mainFrame,
         XmNchildType,    XmFRAME_TITLE_CHILD,
         XmNlabelString,  XmStringCreateSimple(title),
         NULL);

   Widget mainBox = widgetList[MAIN_FORM] =
      XtVaCreateManagedWidget("mainBox", xmFormWidgetClass, mainFrame, NULL);

   int nInp = visibleInputList.getLength();
   if (nInp) {
      Widget inputFrame = widgetList[INPUT_FRAME] =
         XtVaCreateManagedWidget("inputFrame",
            xmFrameWidgetClass, mainBox,
            XmNleftAttachment,   XmATTACH_FORM,
            XmNleftOffset,       HvUIOffset,
            XmNrightAttachment,  XmATTACH_FORM,
            XmNrightOffset,      HvUIOffset,
            XmNtopAttachment,    XmATTACH_FORM,
            XmNtopOffset,        HvUIOffset,
            NULL);

      Widget inputLabel = widgetList[INPUT_LABEL] =
         XtVaCreateManagedWidget("inputLabel",
            xmLabelWidgetClass, inputFrame,
            XmNchildType,    XmFRAME_TITLE_CHILD,
            XmNlabelString,  XmStringCreateSimple ("Input"),
            NULL);

      Widget inputBox = widgetList[INPUT_FORM] =
         XtVaCreateManagedWidget("inputBox",
            xmFormWidgetClass, inputFrame,
	    NULL);

      Widget *list = new Widget[nInp];
      for (i = 0; i < nInp; i++) {
         HvWebField *f = (HvWebField*)visibleInputList[i];
         list[i] = f->buildWidgets(inputBox, this);
         XtVaSetValues(list[i],
            XmNrightAttachment, XmATTACH_FORM, XmNrightOffset, 0,
            XmNleftAttachment,  XmATTACH_FORM, XmNleftOffset,  0,
   	    NULL);
      }
      XtVaSetValues(list[0],
         XmNtopAttachment, XmATTACH_FORM, XmNtopOffset, 0, NULL);
      for (i = 1; i < (nInp-1); i++) {
         XtVaSetValues(list[i],
            XmNtopAttachment, XmATTACH_WIDGET, XmNtopOffset, 0,
            XmNtopWidget,     list[i-1],
	    NULL);
      }
      if (nInp == 1) {
         XtVaSetValues(list[0],
            XmNbottomAttachment,   XmATTACH_FORM, XmNbottomOffset,0,
            NULL);
      }
      else {
         XtVaSetValues(list[nInp-1],
            XmNtopAttachment,      XmATTACH_WIDGET, XmNtopOffset, 0,
            XmNtopWidget,          list[i-1],
            XmNbottomAttachment,   XmATTACH_FORM, XmNbottomOffset,0,
            NULL);
      }

      delete [] list;
   }

   int nOut = visibleOutputList.getLength();
   if (nOut) {
      Widget connectFrame = widgetList[CONNECT_FRAME] =
         XtVaCreateManagedWidget("connectFrame",
            xmFrameWidgetClass, mainBox,
            XmNleftAttachment,   XmATTACH_FORM,
            XmNleftOffset,       HvUIOffset,
            XmNrightAttachment,  XmATTACH_FORM,
            XmNrightOffset,      HvUIOffset,
            NULL);
      if (widgetList[INPUT_FRAME]) {
         XtVaSetValues(connectFrame,
            XmNtopAttachment,      XmATTACH_WIDGET, XmNtopOffset, HvUIOffset,
            XmNtopWidget,          widgetList[INPUT_FRAME],
            NULL);
      }
      else {
         XtVaSetValues(connectFrame,
            XmNtopAttachment,      XmATTACH_FORM, XmNtopOffset, HvUIOffset,
            NULL);
      }

      Widget connectLabel = widgetList[CONNECT_LABEL] =
         XtVaCreateManagedWidget("connectLabel",
            xmLabelWidgetClass, connectFrame,
            XmNchildType,    XmFRAME_TITLE_CHILD,
            XmNlabelString,  XmStringCreateSimple("Output"),
            NULL);

      Widget connectBox = widgetList[CONNECT_FORM] =
         XtVaCreateManagedWidget("connectBox",
            xmFormWidgetClass, connectFrame,
	    NULL);

      Widget *list = new Widget[nOut];
      for (i = 0; i < nOut; i++) {
         HvWebField *f = (HvWebField*)visibleOutputList[i];
         list[i] = (f->buildWidgets(connectBox, this));
         XtVaSetValues(list[i],
            XmNrightAttachment, XmATTACH_FORM, XmNrightOffset, 0,
            XmNleftAttachment,  XmATTACH_FORM, XmNleftOffset,  0,
	    NULL);
      }
      XtVaSetValues(list[0],
         XmNtopAttachment, XmATTACH_FORM, XmNtopOffset, 0,
         NULL);
      for (i = 1; i < (nOut-1); i++) {
         XtVaSetValues(list[i],
            XmNtopAttachment, XmATTACH_WIDGET, XmNtopOffset, 0,
            XmNtopWidget,     list[i-1],
   	    NULL);
      }
      if (nOut == 1) {
         XtVaSetValues(list[0],
            XmNbottomAttachment,   XmATTACH_FORM, XmNbottomOffset,0,
            NULL);
      }
      else {
         XtVaSetValues(list[nOut-1],
            XmNtopAttachment,      XmATTACH_WIDGET, XmNtopOffset, 0,
            XmNtopWidget,          list[i-1],
            XmNbottomAttachment,   XmATTACH_FORM, XmNbottomOffset,0,
            NULL);
      }

      delete [] list;
   }

   Widget outputFrame = widgetList[OUTPUT_FRAME] =
      XtVaCreateManagedWidget("outputFrame",
         xmFrameWidgetClass, mainBox,
         XmNleftAttachment,   XmATTACH_FORM,
         XmNleftOffset,       HvUIOffset,
         XmNrightAttachment,  XmATTACH_FORM,
         XmNrightOffset,      HvUIOffset,
         XmNbottomAttachment, XmATTACH_FORM,
         XmNbottomOffset,     HvUIOffset,
         NULL);
   if (widgetList[CONNECT_FRAME]) {
      XtVaSetValues(outputFrame,
         XmNtopAttachment,      XmATTACH_WIDGET, XmNtopOffset, HvUIOffset,
         XmNtopWidget,          widgetList[CONNECT_FRAME],
         NULL);
   }
   else if (widgetList[INPUT_FRAME]) {
      XtVaSetValues(outputFrame,
         XmNtopAttachment,      XmATTACH_WIDGET, XmNtopOffset, HvUIOffset,
         XmNtopWidget,          widgetList[INPUT_FRAME],
         NULL);
   }
   else {
      XtVaSetValues(outputFrame,
         XmNtopAttachment,      XmATTACH_FORM, XmNtopOffset, HvUIOffset,
         NULL);
   }

   Widget outputLabel = widgetList[OUTPUT_LABEL] =
      XtVaCreateManagedWidget("outputLabel",
         xmLabelWidgetClass, outputFrame,
         XmNchildType,    XmFRAME_TITLE_CHILD,
         XmNlabelString,  XmStringCreateSimple("Messages"),
         NULL);

   Widget outputBox = widgetList[OUTPUT_FORM] =
      XtVaCreateManagedWidget("outputBox", xmFormWidgetClass, outputFrame, NULL);

   n = 0;
   XtSetArg (args[n], XmNcolumns, 30); n++;
   XtSetArg (args[n], XmNrows, 4); n++;
   XtSetArg (args[n], XmNeditMode, XmMULTI_LINE_EDIT); n++;
   XtSetArg (args[n], XmNeditable, False); n++;
   XtSetArg (args[n], XmNcursorPositionVisible, False); n++;
   XtSetArg (args[n], XmNscrollBarDisplayPolicy, XmAS_NEEDED); n++;
   XtSetArg (args[n], XmNscrollingPolicy, XmAUTOMATIC); n++;
   XtSetArg (args[n], XmNvisualPolicy, XmCONSTANT); n++;

   XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
   XtSetArg (args[n], XmNtopOffset, HvUIOffset); n++;
   XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
   XtSetArg (args[n], XmNleftOffset, HvUIOffset); n++;
   XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
   XtSetArg (args[n], XmNrightOffset, HvUIOffset); n++;
   XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
   XtSetArg (args[n], XmNbottomOffset, HvUIOffset); n++;

//   XtSetArg (args[n], XmNvalue, "Demo Text\nNeue Zeile\n"); n++;

   Widget outputLog = widgetList[OUTPUT_LOG_TEXT] = 
      XmCreateScrolledText (outputBox, "outputLog", args, n);
   XtManageChild (outputLog);


   webFunctionDialogVisible = TRUE;
   SoXt::show(widgetList[DIALOG_SHELL]);
}

//////////////////////////////////////////////////////////////////////////
//
// Description:
//    Set the visiblity of the web function dialog to FALSE
//
// Usage: static
    
void
HvWebFunction::webFunctionClosedCB(Widget, void *info, XmAnyCallbackStruct *)
//
//////////////////////////////////////////////////////////////////////////
{
    HvWebFunction* wfn = (HvWebFunction*) info;
    wfn->webFunctionDialogVisible = FALSE;
}

//////////////////////////////////////////////////////////////////////////
//
// Description:
//    Set the visiblity of the web function dialog to FALSE
//
// Usage: static
    
void
HvWebFunction::unmanageCB(Widget, void *info, XmAnyCallbackStruct *)
//
//////////////////////////////////////////////////////////////////////////
{
    HvWebFunction &wfn = *(HvWebFunction*) info;
    wfn.webFunctionDialogVisible = FALSE;
    SoXt::hide(wfn.widgetList[DIALOG_SHELL]);
}


//////////////////////////////////////////////////////////////////////////
//
// Description:
//    start the application on the web server
//
// Usage: static
    
void
HvWebFunction::startCB(Widget, void *info, XmAnyCallbackStruct *)
//
//////////////////////////////////////////////////////////////////////////
{
    HvWebFunction &wfn = *(HvWebFunction*) info;

//    if (wfn.widgetList[STOP_BUTTON])
//       XtSetSensitive(wfn.widgetList[STOP_BUTTON], TRUE);
    if (wfn.widgetList[START_BUTTON])
       XtSetSensitive(wfn.widgetList[START_BUTTON], FALSE);

    wfn.state = HvWebFunction::FIRST_CALL;

    wfn.sensor.setFunction((SoSensorCB *)HvWebFunction::handleConnectionCB);
    wfn.sensor.setData(&wfn);
    wfn.sensor.attach(SoDB::getGlobalField("realTime"));

#ifdef UNDO
    wfn.makeBackup();
    wfn.wui->registerWebFunction(&wfn);
#endif /* UNDO */
}

//////////////////////////////////////////////////////////////////////////
//
// Description:
//    request a help message from the web server
//
// Usage: static
    
void
HvWebFunction::helpCB(Widget, void *info, XmAnyCallbackStruct *)
//
//////////////////////////////////////////////////////////////////////////
{
   HvWebFunction &wfn = *(HvWebFunction*) info;

   if (wfn.daemon.isConnected()) {         // connection is established
      while(! wfn.daemon.flush());         // first finish pending write

      wfn.daemon.set("HELP\n", 5);
      wfn.daemon.write(TRUE);
      while(! wfn.daemon.flush());  	   // send until finished
   }
   else {  // get message directly, not asynchronously
      while(! wfn.daemon.connect());   // wait until connection is established

      wfn.daemon.set("HELP\n", 5);
      wfn.daemon.write(TRUE);
      while(! wfn.daemon.flush());     // send until finished

      while(!wfn.daemon.read());       // wait until finished

      wfn.updateOutput();

      wfn.daemon.closeConnection();
   }
}

//////////////////////////////////////////////////////////////////////////
//
// Description:
//    append message to the log text
//
// Usage: private
    
void
HvWebFunction::appendLogText(char *text)
//
//////////////////////////////////////////////////////////////////////////
{
   XtVaSetValues(widgetList[OUTPUT_LOG_TEXT], XmNvalue, text, NULL);
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//   handle connection to application daemon
//
// Use: static

void
HvWebFunction::handleConnectionCB(HvWebFunction *wfn, SoSensor *)
//
////////////////////////////////////////////////////////////////////////
{
   // connect to server
   if (! wfn->daemon.isConnected()) {
      if (! wfn->daemon.connect()) return;
      wfn->state = HvWebFunction::FIRST_CALL;
   }

   // read from socket until the buffer is empty
   while (wfn->daemon.read()) {
      if (! wfn->daemon.isConnected()) {	// socket closed
      	 wfn->sensor.detach();
	 wfn->daemon.closeConnection();
         wfn->updateOutput();
	 if (wfn->widgetList[STOP_BUTTON])
            XtSetSensitive(wfn->widgetList[STOP_BUTTON], FALSE);
	 if (wfn->widgetList[START_BUTTON])
            XtSetSensitive(wfn->widgetList[START_BUTTON], TRUE);
	 wfn->state = FIRST_CALL;
	 return;
      }
      if (wfn->daemon.isComplete()) {
         wfn->updateOutput();
      }
   }

   if (! wfn->daemon.flush()) return;  // wait for pending write

   switch(wfn->state) {
      case HvWebFunction::FIRST_CALL:

      case HvWebFunction::SEND_START:
         wfn->daemon.set("START\n", 6);
         wfn->state = HvWebFunction::SEND_RECEIVE;
	 if (!wfn->daemon.write(TRUE)) return;

      case HvWebFunction::SEND_RECEIVE:
	 if (wfn->inputList.getLength() == 0) {  // skip "send fields"
	    wfn->state = HvWebFunction::SEND_SEND;
	    return;
	 }
         wfn->daemon.set("RECEIVE\n", 8);
         wfn->state = HvWebFunction::SEND_FIELD;
	 if (!wfn->daemon.write(TRUE)) return;

      case HvWebFunction::SEND_FIELD:
	 wfn->sendFields();
         wfn->state = HvWebFunction::SEND_SEND;
	 if (!wfn->daemon.write(FALSE)) return;

      case HvWebFunction::SEND_SEND:
         wfn->daemon.set("SEND\n", 5);
	 wfn->dataRead = FALSE;
         wfn->state = HvWebFunction::RECEIVING;
	 if (!wfn->daemon.write(TRUE)) return;

      case HvWebFunction::RECEIVING:
         if (wfn->dataRead) {	// send new request for data
	    wfn->state = HvWebFunction::SEND_SEND;
	 }
	 return;

      default:
         cerr << "Shouldn't show up here !\n";
         break;
   }
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//   set output fields in web function dialog
//
// Use: private

void
HvWebFunction::updateOutput()
//
////////////////////////////////////////////////////////////////////////
{
   if (!daemon.getReadLength()) return;

      int dataSize;
      const char *data = (char *)daemon.get(dataSize);
      int no, i, id = *((int *)data);
      const char *v = &data[2*S_INT];

      SbBool needUpdate = FALSE;

      switch (id) {
         case FORCE_ID:
            needUpdate = TRUE;
         case DATA_ID:
	    dataRead = TRUE;
            no = outputList.getLength();
            for (i = 0; i < no; i++) {
               v = ((HvWebField *)outputList[i])->update(v, needUpdate);
            }
   	    if (mode == HvWebFunction::DIRECT)
               state = HvWebFunction::LAST_CALL;
	    break;
         case HELP_ID:
         case MESS_ID:
            if (widgetList[OUTPUT_LOG_TEXT]) {
	       char *text;
	       XtVaGetValues(widgetList[OUTPUT_LOG_TEXT],
	          XmNvalue, &text, NULL);
	       int alen = strlen(text);
	       int tlen = dataSize - S_INT;
	       char *tmp = new char[alen + tlen + 1];
	       strncpy (tmp, text, alen);
	       strncpy (&tmp[alen], &data[S_INT], tlen);
	       tmp[alen + tlen] = '\0';
	       XtVaSetValues(widgetList[OUTPUT_LOG_TEXT],
	          XmNvalue, tmp, NULL);
	       delete [] text;
	       delete [] tmp;
	    }
	    else {
	       int tlen = dataSize - S_INT;
	       char *tmp = new char[tlen + 1];
	       strncpy (tmp, &data[S_INT], tlen); tmp[tlen] = '\0';
	       cerr << tmp;
	       delete [] tmp;
	    }
            break;
         default:
            cerr << "received packet of unknown type (" << id << ")\n";
	    break;
      }
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//	put the fields into the writeBuffer (binary)
//
// Use: private

void
HvWebFunction::sendFields()
//
////////////////////////////////////////////////////////////////////////
{
   int ni = inputList.getLength();
   if (ni == 0) return;

   int i, size = 2*S_INT;
   char *cBuff = new char[size];

   for (i = 0; i < ni; i++)
      ((HvWebField *)inputList[i])->pack(cBuff, size);

   int *iBuff = (int *)cBuff;
   iBuff[0] = 0;
   iBuff[1] = ni;
   daemon.set(cBuff, size);
   delete [] cBuff;
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//   build first version of the menu button
//
// Use: public

Widget
HvWebFunction::buildInitialButton(Widget parent)
//
////////////////////////////////////////////////////////////////////////
{
   if (webMenuButton == NULL) {
      webMenuButton = XtVaCreateManagedWidget("button",
         xmPushButtonWidgetClass, parent,
         XmNalignment, XmALIGNMENT_CENTER,
         NULL);
   }

   HvManager::buildColorPixmaps(webMenuButton, hvClockXpm, clockPixmap);
   HvManager::setPixmaps(webMenuButton, clockPixmap);

   return webMenuButton;
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//   Add a menu button to the dialog window
//
// Use: public

Widget
HvWebFunction::buildStartButton(Widget parent)
//
////////////////////////////////////////////////////////////////////////
{
   if (title == NULL) return NULL;

   if (webMenuButton == NULL) {
      webMenuButton = XtVaCreateManagedWidget(title,
         xmPushButtonWidgetClass, parent,
         XmNalignment, XmALIGNMENT_CENTER,
         NULL);
   }
   else {
      Arg args[3];
      Cardinal n = 0;

      XtSetArg(args[n], XmNlabelType, XmSTRING); n++;
      XtSetArg(args[n], XmNlabelString, XmStringCreateSimple(title)); n++;
      XtSetValues(webMenuButton, args, n);
   }

   XtAddCallback(webMenuButton, XmNactivateCallback,
      (XtCallbackProc) HvWebFunction::webMenuDialogCB,
      (XtPointer) this); 

   return webMenuButton;
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//
// Use: public

void
HvWebFunction::updateFieldReferences(void)
//
////////////////////////////////////////////////////////////////////////
{
    int iLen = inputList.getLength();
    int oLen = outputList.getLength();
    int i;

    for (i = 0; i < iLen; i++)
	((HvWebField *)inputList[i])->updateFieldReference();
    for (i = 0; i < oLen; i++)
	((HvWebField *)outputList[i])->updateFieldReference();
}


#ifdef UNDO
////////////////////////////////////////////////////////////////////////
//
// Description:
//   store actual values of all fields
//
// Use: public

void
HvWebFunction::makeBackup()
//
////////////////////////////////////////////////////////////////////////
{
   int no = outputList.getLength();
   for (int i = 0; i < no; i++) {
      ((HvWebField *)outputList[i])->makeBackup();
   }
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//   restore old values of all fields
//
// Use: public

void
HvWebFunction::undo()
//
////////////////////////////////////////////////////////////////////////
{
   int no = outputList.getLength();
   for (int i = 0; i < no; i++) {
      ((HvWebField *)outputList[i])->undo();
   }
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//   restore new values of all fields
//
// Use: public

void
HvWebFunction::redo()
//
////////////////////////////////////////////////////////////////////////
{
   int no = outputList.getLength();
   for (int i = 0; i < no; i++) {
      ((HvWebField *)outputList[i])->redo();
   }
}

#endif /* UNDO */
