/*
 * HvWebField.h
 *
 *	This file defines the HvWebField class.
 *
 * Copyright 1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
 */

#ident "$Revision: 1.3 $"

#ifndef  __HV_WEB_FIELD_H__
#define  __HV_WEB_FIELD_H__

#include <Xm/Xm.h>

#include <Inventor/SbString.h>

class SoField;
class SoNode;
class SoFieldSensor;
class SbString;
class HvWebFunction;

//////////////////////////////////////////////////////////////////////////////
//
//  Class: HvWebField
//
//////////////////////////////////////////////////////////////////////////////


class HvWebField {
  public:
    enum FieldType {
        INPUT_FLOAT,
        INPUT_STRING,
        INPUT_SLIDER,
        INPUT_CHECK,
        INPUT_OPTION,
        INPUT_LIST,
        INPUT_FIELD,
        OUTPUT_FLOAT,
        OUTPUT_STRING,
        OUTPUT_FIELD,
	SELECTION
    };

    enum SelectionType {
       ATOM,
       BOND
    };

    // Constructor / Destructor
    HvWebField();
    ~HvWebField();

    void setLabel(const char *str) { label = str; }
    void appendLabel(const char *str) { label += str; }

    void setField(SoField *f) { field = f; }

    void setValue(float v) { value = v; }

    void setString(const char *str) { string = str; }

    void setSlider(float smin = 0., float smax = 1., float sval = .5) {
       sliderMin = smin; sliderMax = smax; sliderValue = sval; }

    void setFieldName(const char *name) { fieldName = name; }

    void setNodeType(const SoNode *node) { nodeType = node->getTypeId(); }

    void addListItem(const char *str) {
       listItems.append(new SbString(str));
    }

    void setType(int t) { type = t; }
    int getType() { return type; }

    void setSelectionType(int t) { selectionType = t; }

    void   setConnected(SbBool c) { connected = c; }
    SbBool isConnected() { return connected; }

    void   setHidden(SbBool v) { hidden = v; if (v) connected = v; }
    SbBool isHidden() { return hidden; }

    Widget buildWidgets(Widget parent, HvWebFunction *wfn);

    // update connected fields
    const char *update(const char *v, SbBool NeedUpdate);

    // pack field values into a buffer
    // format:
    //   |header|data|
    //   header:	|(int)len|(int)messageType|(int)numFields|
    //   data:		|field[0]|field[1]| ... |field[n-1]|
    //   field:		|fieldHeader|fieldData|
    //   fieldHeader:	|(int)fieldLen|(int)fieldId|(int)numItem|
    //   fieldData:	|item[0]|item[1]| ... |item[n-1]|
    //   len/fieldLen:  length in bytes|chars
    void pack(char * &buff, int &s);

    void setParentNode(SoNode *pn) { parentNode = pn; };

    void updateFieldReference(void);

#ifdef UNDO
    void makeBackup();
    void undo();
    void redo();
#endif /* UNDO */

  private:
    int 		type;

    int			selectionType;

    Widget		*widgetList;

    SbString 		label;
    SoField		*field;
    SbString		fieldName;
    SoType		nodeType;
    float		value;    // for single fields
    SbString		string;   // for single strings
    float 		sliderMin, sliderMax, sliderValue;
    SbPList		listItems;

    int 		refID;

    SoNode	*parentNode;

    SbBool		connected;

    SbBool		hidden;

#ifdef UNDO
    SbString		undoCopy, redoCopy;
#endif /* UNDO */


    static void connectionChangedCB(Widget w, HvWebField *wf, void *);

};


// this class makes life easier in pack(), and update()
class HvTmpFloat {
   public:
      float	value;
      float getValue() { return value; }
};

#endif  /* !__HV_WEB_FIELD_H__ */
