/*
 * Copyright (C) 1996   Silicon Graphics, Inc.
 *
 _______________________________________________________________________
 ______________  S I L I C O N   G R A P H I C S   I N C .  ____________
 |
 |   $Revision: 1.0 $
 |
 |      Classes
 |         HvSocket: Encapsulates Networking Functions
 |
 |      This code doesn't contain any comments.
 |      It was hard to write, it should be hard to read.
 |
 |   Author(s)          : Horst Vollhardt
 |
 ______________  S I L I C O N   G R A P H I C S   I N C .  ____________
 _______________________________________________________________________
 */

#include "HvSocket.h"
#include <iostream.h>
#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <string.h>
#include <sys/ipc.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <errno.h>
#include <bstring.h>
#include <malloc.h>

#define BUF_SIZE  32678

#ifndef FALSE
#define FALSE 0
#endif

#ifndef TRUE
#define TRUE 1
#endif

////////////////////////////////////////////////////////////////////////
//
// Description:
//    Constructor
//
// Use: public

HvSocket::HvSocket()
//
////////////////////////////////////////////////////////////////////////
{
   actLength = maxLength = 0;
   buffer = NULL;
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    Constructor which analyses the given URL and
//    extracts hostname and file name part
//
// Use: public

HvSocket::HvSocket(const char *url)
//
////////////////////////////////////////////////////////////////////////
{
   actLength = maxLength = 0;
   buffer = NULL;

   URL = url;

   char *host = NULL, *file = NULL;
   splitURL(host, port, file);
   if (host) {
      hostname = host; delete [] host;
      if (file) {
         filename = file; delete [] file;
      }
   }
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    Destructor
//
// Use: public

HvSocket::~HvSocket()
//
////////////////////////////////////////////////////////////////////////
{
   delete [] buffer;
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    remove first block from buffer
//
// Use: public

void
HvSocket::popBuffer()
//
////////////////////////////////////////////////////////////////////////
{
      int len = *((int *)buffer);
      actLength -= len;
      memmove(buffer, &buffer[len], actLength);
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    append string to buffer
//
// Use: public

void
HvSocket::appendBuffer(const char *str, int len)
//
////////////////////////////////////////////////////////////////////////
{
   buffer = (char *)realloc(buffer, maxLength + len);
   bcopy(str, &buffer[actLength], len);
   maxLength += len;
   actLength += len;
}

////////////////////////////////////////////////////////////////////////
//
//  bind socket to name
//  returns -1 on error else new socket id
//  
// Use: public

int
HvSocket::bindSocket()
//
////////////////////////////////////////////////////////////////////////
{
   struct hostent *he;
   if (!(he = gethostbyname(hostname.getString()))) {
      perror("\tgethostbyname");
      return -1;
   }

   if ((sock = socket(he->h_addrtype, SOCK_STREAM, 0)) < 0) {
      perror("\tsocket");
      return -1;
   }

   bzero((caddr_t)&sin, sizeof(sin));
   sin.sin_family = he->h_addrtype;
   if (bind(sock, &sin, sizeof(sin)) < 0) {
      perror("\tbind");
      return -1;
   }
   bcopy(he->h_addr, &sin.sin_addr, he->h_length);
   sin.sin_port = port;

   return sock;
}

////////////////////////////////////////////////////////////////////////
//
//  connect socket to remote server
//  returns -1 on error, 0 on EINPROGRESS else socket id
//  
// Use: public

int
HvSocket::connectSocket()
//
////////////////////////////////////////////////////////////////////////
{
   if (connect(sock, &sin, sizeof(sin)) < 0) {
      if ((errno == EINPROGRESS)||(errno == EALREADY)) return 0;
      if (errno == EISCONN) return sock;
      perror("\tconnect");
      return -1;
   }

   return sock;
}

////////////////////////////////////////////////////////////////////////
//
//  write data to socket
//  returns -1 on error, 0 on success
//  
// Use: public

int
HvSocket::sendData(const char *data, int len)
//
////////////////////////////////////////////////////////////////////////
{
   if (write(sock, data, len) != len) {
      perror("\twrite");
      return -1;
   }

   return 0;
}

////////////////////////////////////////////////////////////////////////
//
//  read data from socket, increases the data buffer if necessary
//	size:    actual size of the data record
// 	maxSize: already allocated space for the data record
//  returns -1 on error, 0 on EOF
//
// Use: public

int
HvSocket::receiveData()
//
////////////////////////////////////////////////////////////////////////
{
   char buf[BUF_SIZE];
   int bytes = -1, newSize, ret;

   while ((ret = read(sock, buf, BUF_SIZE)) > 0) {
      bytes = ret;
      newSize = actLength + bytes;
      if (newSize > maxLength) {
         buffer = (char *)realloc(buffer, newSize);
	 maxLength = newSize;
      }
      bcopy(buf, &buffer[actLength], bytes);
      actLength = newSize;
   }
   if (ret == 0) return 0;

   return bytes;
}

////////////////////////////////////////////////////////////////////////
//
//  make socket nonblocking
//
// Use: public

int
HvSocket::makeNonblocking()
//
////////////////////////////////////////////////////////////////////////
{
   if (fcntl(sock, F_SETFL, FNDELAY) < 0) {
      perror("\tfcntl F_SETFL, FNDELAY");
      return -1;
   }
   return 0;
}

////////////////////////////////////////////////////////////////////////
//
//  split given URL into host, port and file
//  returns -1 on error
//  
// Use: public

int
HvSocket::splitURL(char* &host, int &port, char* &file)
//
////////////////////////////////////////////////////////////////////////
{
   char* http = "http://";
   int httpLen = strlen(http);
   const char *url = URL.getString();
   if (!SbString(url)) return -1;
   if (strncmp(http, url, httpLen)) return -1;  // no http protocol

   // get the host name
   const char *s;
   for (s = url + strlen(http) + 1; *s && (*s != ':') && (*s != '/'); s++);

   int hostnameLen = s - url;
   hostnameLen -= httpLen;
   host = new char[hostnameLen + 1];  // no error checking

   strncpy(host, url + httpLen, hostnameLen);
   host[hostnameLen] = '\0';

   // get port number
   if (*s == ':') {
      port = atoi(++s);
      while (*s && (*s != '/')) s++;
   }
   else { port = 80; }

   // get the file name
   if (*s == '/') {
      int filenameLen = strlen(s);
      file = new char[filenameLen + 1];
      strncpy(file, s, filenameLen);
      file[filenameLen] = '\0';
   }
   else {
      file = new char[2];
      file[0] = '/'; file[1] = '\0';
   }
return 0;
}

////////////////////////////////////////////////////////////////////////
//
//  test if the buffer is complete
//  first 4 bytes contain length (int)
//  
// Use: public

SbBool
HvSocket::isComplete()
//
////////////////////////////////////////////////////////////////////////
{
   if (actLength < 4) return FALSE;  // first 4 bytes contain "int"

   int size = *((int *)buffer);
   if (actLength < size) return FALSE;

   return TRUE;
}

////////////////////////////////////////////////////////////////////////
//
//  append the given string to the filename part of the URL
//  and returns it
//
//  URL    :=  http://HOST[:PORT][/FILE[?QUERY]]
//  QUERY  :=  [VAL1=]QUERY1[&VAL2=QUERY2][...]
//  
// Use: public

SbString
HvSocket::appendQueryToURL(const char *str)
//
////////////////////////////////////////////////////////////////////////
{
   SbString newURL;

   // no URL defined
   if (!URL) return newURL;

   // filename part of URL is empty
   if (!filename) return newURL;

   newURL = filename;

   // does the filename part already contain a query string ?
   if (strchr(filename.getString(), '?'))
      newURL += "&";
   else
      newURL += "?";

   newURL += str;

   return newURL;
}

