/*
 * Copyright (C) 1996   Silicon Graphics, Inc.
 *
 _______________________________________________________________________
 ______________  S I L I C O N   G R A P H I C S   I N C .  ____________
 |
 |   $Revision: 1.1 $
 |
 |      Classes
 |         HvNetBuffer: Encapsulates Networking Functions
 |
 |      This code doesn't contain any comments.
 |      It was hard to write, it should be hard to read.
 |
 |   Author(s)          : Horst Vollhardt
 |
 ______________  S I L I C O N   G R A P H I C S   I N C .  ____________
 _______________________________________________________________________
 */

#ifdef DEBUG
#include <iostream.h>
#endif

#include <unistd.h>
#include <malloc.h>
#include <string.h>
#include <stdio.h>
#include <errno.h>
#include <iostream.h>

#include "HvNetBuffer.h"


////////////////////////////////////////////////////////////////////////
//
// Description:
//    Constructor
//
// Use: public

HvNetBuffer::HvNetBuffer(int s)
//
////////////////////////////////////////////////////////////////////////
{
   maxLength = 0;
   buffer = NULL;
   numPipes=0;
   ispiped=0;
   clear();
   setSocket(s);
}


////////////////////////////////////////////////////////////////////////
//
// Description:
//    Destructor
//
// Use: public

HvNetBuffer::~HvNetBuffer()
//
////////////////////////////////////////////////////////////////////////
{
   free(buffer);
   
   // close pipes
   if (ispiped)
	  {
	   printf("server closing pipes\n");
	   for (int i=0;i<numPipes;i++)
		 close(pipe[i]);
	  }
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    takes a filename, opens the stream, and add it to the pipelist
//    return FALSE on error;
//
// Use: public

SbBool
HvNetBuffer::addPipe(SbString string)
//
////////////////////////////////////////////////////////////////////////
{
	int f;
	ispiped=1;
	fprintf(stderr, "server connect %s...\n", string.getString());
	f=open(string.getString(), O_WRONLY);
	if (f<0)
		{
		fprintf(stderr, "server         %s failed:", string.getString());
		perror(" ");
		return 0;
		}
	else
		fprintf(stderr, "server         %s ok\n", string.getString());
	
	pipe[numPipes++]=f;
	return 1;
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    clear buffer, allocate first bytes if necessary
//    return -1 on error;
//
// Use: public

int
HvNetBuffer::clear()
//
////////////////////////////////////////////////////////////////////////
{
   length = 0;
   if (maxLength < S_INT) {
      if ((buffer = (char *)realloc(buffer, INITIAL_LENGTH)) == NULL) {
         maxLength = 0;
         return -1;
      }
      else {
         maxLength = INITIAL_LENGTH;
      }
   }
   if (buffer == NULL) return -1;

   *((int *)buffer) = S_INT;

   return 0;
}


////////////////////////////////////////////////////////////////////////
//
// Description:
//  write buffer to socket
//  returns -1 on error otherwise number of bytes written
//  
// Use: public

int
HvNetBuffer::send(int offset, SbBool dataOnly)
//
////////////////////////////////////////////////////////////////////////
{
   if (buffer == NULL) return -1;

   int ret;
   if (dataOnly || (offset != 0)) { 
      ret = write(sock, &buffer[offset + S_INT], length - offset);
	  
	  // assuming that local file-system is better than network
	  // so we don't have to care about buffer-prolems, we
	  // just send as many bytes, as the network sent
	  for (int i=0;i<numPipes;i++)
		{
		//fprintf(stderr, "writing %d bytes to pipe %d\n",ret,  i);
		//  if (::write(pipe[i], &buffer[offset + S_INT], length - offset)!=ret)
		//	cerr << "couldn't write enough to pipe\n";
		}
   }
   else {
      ret = write(sock, buffer, length + S_INT);
	  for (int i=0;i<numPipes;i++)
		{
		//fprintf(stderr, "writing %d bytes to pipe %d\n", ret, i);
		//  if (::write(pipe[i], buffer, length + S_INT)!=ret)
		//	cerr << "couldn't write enough to pipe\n";
		}
   }

   if (ret == -1) {

#ifdef DEBUG
      perror("\twrite");
#endif
      if (errno == EAGAIN) return  0;
      else                 return -1;
   }

   if (dataOnly || (offset != 0)) return ret;
   else                           return ret - S_INT;   // problem if < 0
                                                        // it happens rarely
}


////////////////////////////////////////////////////////////////////////
//
// Description:
//  write buffer only to pipes
//  returns -1 on error otherwise number of bytes written
//  
// Use: public

int
HvNetBuffer::sendPipe(int offset, SbBool dataOnly)
//
////////////////////////////////////////////////////////////////////////
{
	if (buffer == NULL) return -1;

	int ret;
	if (dataOnly || (offset != 0))
		{ 
		for (int i=0;i<numPipes;i++)
			{
			ret=write(pipe[i], &buffer[offset + S_INT], length - offset);
			}
		}
	else
		{
		ret = write(sock, buffer, length + S_INT);
		for (int i=0;i<numPipes;i++)
			{
			ret=write(pipe[i], buffer, length + S_INT);
			}
		 }

	if (ret == -1)
		{

#ifdef DEBUG
      perror("\twrite");
#endif
		if (errno == EAGAIN) return  0;
		else                 return -1;
		}

	if (dataOnly || (offset != 0)) return ret;
	else                           return ret - S_INT;   // problem if < 0
                                                        // it happens rarely
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//  read data from socket, increases the data buffer if necessary
//  returns -1 on error, 0 on EOF, and bytes read otherwise 
//
// Use: public

int
HvNetBuffer::receive(SbBool binaryMode)
//
////////////////////////////////////////////////////////////////////////
{
	// determine how many bytes have to be read
	int bytesToRead, bytesRead;
	if (binaryMode)
		{
		if (length == 0)
			{		// first read complete header
			bytesToRead = S_INT;	 	// problem if only (1 - S_INT-1) bytes
										// read, it should happen seldom
                                        // return error in this case
			if ((bytesRead = reallyRead(bytesToRead, 0)) < bytesToRead)
				{
				if (bytesRead > 0) return -1;
				else 	       return bytesRead;
				}
			}
		bytesToRead = *((int *)buffer) - length - S_INT;
		}
	else
		{
		bytesToRead = 8192;   // reasonable value ?
		}

	bytesRead = reallyRead(bytesToRead, length+S_INT);

	if (!binaryMode)
		*((int *)buffer) = length + S_INT;

	return bytesRead;
}


////////////////////////////////////////////////////////////////////////
//
// Description:
//  read data from socket, increases the data buffer if necessary
//  returns -1 on error, 0 on EOF, and bytes read otherwise 
//
// Use: private

int
HvNetBuffer::reallyRead(int bytesToRead, int offset)
//
////////////////////////////////////////////////////////////////////////
{
	// realloc new memory if necessary
	if (maxLength < (offset + bytesToRead)) 
		{
		maxLength = offset + bytesToRead;
		buffer = (char *)realloc(buffer, maxLength);
		if (buffer == NULL)
			{
#ifdef DEBUG
         cerr << "reallyRead: couldn't allocate " << maxLength << " bytes\n";
#endif
			clear();
			return -1;
			}
		}

	int bytesRead = 0, ret;
	while ((ret = ::read(sock, &buffer[offset], bytesToRead)) > 0)
		{
		offset      += ret;
		bytesToRead -= ret;
		bytesRead   += ret;
		if (offset >= S_INT) length = offset - S_INT;
		if (bytesToRead <= 0) return bytesRead;
		}
		
	return ret;
}

 
////////////////////////////////////////////////////////////////////////
//
// Description:
//  test if the buffer is complete
//  first "4" bytes contain length (int)
//  
// Use: public

SbBool
HvNetBuffer::isComplete()
//
////////////////////////////////////////////////////////////////////////
{
   if (buffer == NULL) return FALSE;

   int size = *((int *)buffer);
   if ((length+S_INT) < size) return FALSE;

   return TRUE;
}


////////////////////////////////////////////////////////////////////////
//
// Description:
//  returns the data part
//  returns NULL on error
//
// Use: public

const void *
HvNetBuffer::get(int &len)
//
////////////////////////////////////////////////////////////////////////
{
   len = 0;
   if (buffer == NULL) return NULL;
   if (!isComplete())  return NULL;

   len = length;
   return &buffer[S_INT];
}


////////////////////////////////////////////////////////////////////////
//
// Description:
//  clear the buffer and store the given data in it
//  returns -1 on error
//
// Use: public

int
HvNetBuffer::set(const void *data, int len)
//
////////////////////////////////////////////////////////////////////////
{
   if (clear() == -1) return -1;

   if (len == 0) len = strlen((char *)data);

   return append(data, len);
}

#define HV_SET_SCALAR(TYPE)					\
int HvNetBuffer::set(const TYPE data) {				\
   if (clear() == -1) return -1;				\
   return append((const void *)&data, sizeof(TYPE)); }		\

HV_SET_SCALAR(char)
HV_SET_SCALAR(short)
HV_SET_SCALAR(int)
HV_SET_SCALAR(long)
HV_SET_SCALAR(float)
HV_SET_SCALAR(double)

#define HV_SET_VECTOR(TYPE)					\
int HvNetBuffer::set(const TYPE *data, int len) {		\
   if (clear() == -1) return -1;				\
   return append((const void *)data, len*sizeof(TYPE)); }	\

HV_SET_VECTOR(char)
HV_SET_VECTOR(short)
HV_SET_VECTOR(int)
HV_SET_VECTOR(long)
HV_SET_VECTOR(float)
HV_SET_VECTOR(double)


////////////////////////////////////////////////////////////////////////
//
// Description:
//  append the given data to the buffer
//  returns -1 on error
//
// Use: public

int
HvNetBuffer::append(const void *data, int len)
//
////////////////////////////////////////////////////////////////////////
{
   if (len == 0) len = strlen((char *)data);

   // realloc new memory if necessary
   if (maxLength < (length + len + S_INT)) {
      maxLength = length + len + S_INT;
      buffer = (char *)realloc(buffer, maxLength);
      if (buffer == NULL) {
         clear();
         return -1;
      }
   }

   // set data length to zero if buffer is not complete
   if (!isComplete()) {
      if (clear() == -1) return -1;
   }

   // allow overlapping regions
   memmove(&buffer[length+S_INT], data, len);
   length += len;
   *((int *)buffer) = length + S_INT;

   return 0;
}

#define HV_APPEND_SCALAR(TYPE)					\
int HvNetBuffer::append(const TYPE data) {			\
   return append((const void *)&data, sizeof(TYPE)); }		\

HV_APPEND_SCALAR(char)
HV_APPEND_SCALAR(short)
HV_APPEND_SCALAR(int)
HV_APPEND_SCALAR(long)
HV_APPEND_SCALAR(float)
HV_APPEND_SCALAR(double)

#define HV_APPEND_VECTOR(TYPE)					\
int HvNetBuffer::append(const TYPE *data, int len) {		\
   return append((const void *)data, len*sizeof(TYPE)); }	\

HV_APPEND_VECTOR(char)
HV_APPEND_VECTOR(short)
HV_APPEND_VECTOR(int)
HV_APPEND_VECTOR(long)
HV_APPEND_VECTOR(float)
HV_APPEND_VECTOR(double)

