/*
 * Copyright (C) 1996   Silicon Graphics, Inc.
 *
 _______________________________________________________________________
 ______________  S I L I C O N   G R A P H I C S   I N C .  ____________
 |
 |   $Revision: 1.4 $
 |
 |   Classes:
 |      HvManagerP
 |
 |   Author(s)          : Horst Vollhardt, Mark Benzel
 |
 ______________  S I L I C O N   G R A P H I C S   I N C .  ____________
 _______________________________________________________________________
 */

//----------------------------------------------------------------------
// Changes added by Tobias Strasser:
//
// Additional Field SoMFString pipes 
// this takes filenames to write out changes on the scene graph
//
//----------------------------------------------------------------------

#include <Inventor/SbLinear.h>  
#include <Inventor/SoPath.h>  
#include <Inventor/SoInput.h>  
#include <Inventor/actions/SoGLRenderAction.h>
#include <Inventor/actions/SoSearchAction.h>
#include <Inventor/actions/SoWriteAction.h>
#include <Inventor/elements/SoCacheElement.h>
#include <Inventor/misc/SoChildList.h>
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/SoXtComponent.h>
#include <Inventor/nodes/SoLabel.h>

#include <iostream.h>
#include <stdio.h>
#include <malloc.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <sys/types.h>
#include <errno.h>
#include <time.h>

#include "HvManagerP.h"

SO_NODE_SOURCE(HvManagerP);

////////////////////////////////////////////////////////////////////////
//
//  Initialize the class
//  
void HvManagerP::initClass()
//
////////////////////////////////////////////////////////////////////////
{
    SO_NODE_INIT_CLASS(HvManagerP,SoSeparator,"Separator");

}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    Constructor
//
// Use: public

HvManagerP::HvManagerP()
//
////////////////////////////////////////////////////////////////////////
{
   constructorCommon();
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    Constructor
//
// Use: public

HvManagerP::HvManagerP(int numChildren) : SoSeparator(numChildren)
//
////////////////////////////////////////////////////////////////////////
{
   constructorCommon();
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    common part of all constructor
//
// Use: private

void
HvManagerP::constructorCommon()
//
////////////////////////////////////////////////////////////////////////
{
    SO_NODE_CONSTRUCTOR(HvManagerP);

    SO_NODE_ADD_FIELD(pipe,		(""));
    SO_NODE_ADD_FIELD(monitor,	(FALSE));
    SO_NODE_ADD_FIELD(_grphstate,(0));

	isStarted=0;
	
    size_t size = 100;
    outputBuffer = malloc(size);
#ifndef DEBUG
    output.setBinary(TRUE);
#endif
    output.setBuffer(outputBuffer, size, (SoOutputReallocCB *)realloc);

    isBuiltIn = TRUE;

    // initialize graphs
    graphIn.setColor(0., 1., 1.);
    graphInSec.setColor(1., 1., 0.);
    graphInSec.setInterval(1000);

    // reference camera to check if the camera really changed
    refCamera = new SoOrthographicCamera;
    refCamera->ref();
 

}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    Destructor
//
// Use: private

HvManagerP::~HvManagerP()
//
////////////////////////////////////////////////////////////////////////
{

    refCamera->unref();
	
}


////////////////////////////////////////////////////////////////////////
//
// Description:
//   attaches the delaydSensor to the timer field
//   and opens the connection to the server
//
// Use: private

void
HvManagerP::attachDelayedSensor()
//
////////////////////////////////////////////////////////////////////////
{
   if (delayedSensor.getAttachedField() == NULL) {
      delayedSensor.setFunction((SoSensorCB *)HvManagerP::delayedCB);
      delayedSensor.setData(this);
      delayedSensor.attach(SoDB::getGlobalField("realTime"));
   }

   if (! delayedHandler.isConnected()) {
      delayedHandler.setPipename(pipe.getValue());

		delayedHandler.connect();
	  
 //     delayedHandler.set(ID_ID);
 //     delayedHandler.append(id.getValue());
 //     delayedHandler.write();
 //     while(! delayedHandler.flush());    // wait until written
   }
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//   detaches the delayedSensor from timer field
//
// Use: private

void
HvManagerP::detachDelayedSensor()
//
////////////////////////////////////////////////////////////////////////
{
   if (delayedSensor.getAttachedField() != NULL) {
      delayedSensor.detach();
   }
}



////////////////////////////////////////////////////////////////////////
//
// Description:
//   Stores all fields of the node and the fields of its children
//   in the list
//
// Use: private

void
HvManagerP::setNodeDefault(SoNode *node, SoFieldList &fieldList)
//
////////////////////////////////////////////////////////////////////////
{
   if (!node->affectsState()) return;
   node->getFields(fieldList);

#ifdef DEBUG
cerr << "set: " << node->getTypeId().getName().getString() << "\n";
#endif

   SoChildList *childList = node->getChildren();
   if (childList == NULL) return;
   int numChildren = childList->getLength();
   for (int i = 0; i < numChildren; i++)
      setNodeDefault((*childList)[i], fieldList);
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//   Set default flags for all fields of all nodes in the path to TRUE
//   Disables the notification of changes as well (SoWriteAction ?!!!)
//   Stores the original state in the list
//
// Use: private

void
HvManagerP::setAllDefault(SoPath *path, SoFieldList &fieldList,
                         SbIntList &saveList, SbIntList &notifyList)
//
////////////////////////////////////////////////////////////////////////
{
   int numNode = path->getLength();
   fieldList.truncate(0);
   saveList.truncate(0);

   for (int i = 0; i < numNode; i++) {
      SoNode *node = path->getNode(i);
      node->getFields(fieldList);
      notifyList.append(node->isNotifyEnabled());
      node->enableNotify(FALSE);

      // check for nodes within the same parent
      // which affect also the state, because they are
      // also written out with the SoWriteAction
      int idx;
      if (((idx = path->getIndex(i)) > 0)&&(i > 0)) {
	 SoNode *parent = path->getNode(i - 1);
	 SoChildList *childList = parent->getChildren();
         for (int j = 0; j < idx; j++) {
	    setNodeDefault((*childList)[j], fieldList);
	 }
      }
   }

   int numField = fieldList.getLength();
   for (i = 0; i < numField; i++) {
      SoField *field = fieldList[i];
      saveList.append(field->isDefault());
      field->setDefault(TRUE);
   }
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//   restore default flags for all fields of all nodes in the path
//
// Use: private

void
HvManagerP::resetAllDefault(SoPath *path, SoFieldList &fieldList,
                           SbIntList &saveList, SbIntList &notifyList)
//
////////////////////////////////////////////////////////////////////////
{
   int numNode = path->getLength();

   for (int i = 0; i < numNode; i++) {
      SoNode *node = path->getNode(i);
      node->enableNotify(notifyList[i]);
   }

   int numField = fieldList.getLength();
   for (i = 0; i < numField; i++) {
      fieldList[i]->setDefault(saveList[i]);
   }
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//   read scene graph from buffer and add it to the HvManagerP
//   returns -1 on error
//
// Use: private

int
HvManagerP::readSceneGraph(const char *buffer, int len)
//
////////////////////////////////////////////////////////////////////////
{
   SoInput input;
   input.setBuffer((void *)buffer, len);
   SoNode *root = NULL;
   if (! SoDB::read(&input, root)) return -1;
   HvManagerP *manager = (HvManagerP *) root;	// should be correct !
   int num = manager->getNumChildren();
   removeAllChildren();
   for (int i = 0; i < num; i++)
	{
    SoNode *child = manager->getChild(i);
    if (child->getTypeId().isDerivedFrom(SoCamera::getClassTypeId()))
		{
		//actualCamera->copyFieldValues(child);
		addChild(child);
#ifdef DEBUG
         cerr << "Camera replaced\n";
#endif
		}
    else
		{
        addChild(child);
		}
	}
	
   return 0;
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//   read path from buffer and change it in the HvManagerP
//   returns -1 on error
//
// Use: private

int
HvManagerP::readPath(const char *data)
//
////////////////////////////////////////////////////////////////////////
	{
	int numNodes = *((int *)data); data += S_INT;
	int camera=0;
	for (int i = 0; i < numNodes; i++)
		{
		// find node in the scene graph
		int level = *((int *)data); data += S_INT;
		SoNode *node = (SoNode *)this;
		
		for (int j = 0; j < level; j++)
			{
			int idx = *((int *)data); data += S_INT;
			SoGroup *group = (SoGroup *)node;
			if ((node = group->getChild(idx)) == NULL) return -1;
			}
		camera=(node->getTypeId().isDerivedFrom(SoCamera::getClassTypeId()));
			
		// get fields and set values
		int numFields = *((int *)data); data += S_INT;
		SoFieldList fieldList;
		node->getFields(fieldList);
		for (j = 0; j < numFields; j++)
			{
			int idx = *((int *)data);  data += S_INT;
			int len   = *((int *)data);  data += S_INT;
			if (!camera) // do not update fields of camera
				{
				SoField &field = *(fieldList[idx]);
				field.set(data);
				}
			data += len;
			}
		}
	return 0;
	}


////////////////////////////////////////////////////////////////////////
//
// Description:
//   reads a path from buffer and returns a SoPath
//
// Use: private

SoPath*
HvManagerP::readSoPath(int *buf)
//
////////////////////////////////////////////////////////////////////////
	{
	SoPath *path=new SoPath(this);
	path->ref();
	
	int level = *(buf++);
	SoNode *node = (SoNode *)this;

	for (int j =0; j < level; j++)
		{
		int idx = *(buf++);
		SoGroup *group = (SoGroup *)node;
		if ((node = group->getChild(idx)) == NULL) return path;
		path->append(node);
		}
	return path; 		
	}
	
////////////////////////////////////////////////////////////////////////
//
// Description:
//   reads a path from buffer and returns a SoPath
//
// Use: private

SoNode*
HvManagerP::readSubGraph(int *buf)
//
////////////////////////////////////////////////////////////////////////
	{
	int len=*(buf++);
	SoInput input;
	/*
	fprintf(stderr, "buf[0]=%x\n", buf[0]);
	fprintf(stderr, "buf[1]=%x\n", buf[1]);
	fprintf(stderr, "buf[2]=%x\n", buf[2]);
	fprintf(stderr, "buf[3]=%x\n", buf[3]);
	fprintf(stderr, "Len:%d Content:%s\n", len, buf);
	*/
	input.setBuffer((void *)buf, len);
	SoNode *node = NULL;
	if (! SoDB::read(&input, node)) return 0;
	return node;
	}	
	
////////////////////////////////////////////////////////////////////////
//
// Description:
//   decodes a CHANGE_ID chain
//
// Use: private

void
HvManagerP::readChange(int *buf)
//
////////////////////////////////////////////////////////////////////////
	{
	int nofChng=*(buf++);
	SoNode *sub;
	// process all changes
	// fprintf(stderr, "%d changed\n", nofChng);
	for (int i=0;i<nofChng;i++)
		{
		int	    cmd;
		SoPath *path;
		int		chld;
		SoGroup *grp;
		
		cmd=*(buf++);
		path=readSoPath(buf);
		buf+=buf[0]+1;
		chld=*(buf++);
		if (!path) continue;		
		grp=(SoGroup*) path->getTail();
		switch (cmd)
			{
			case TOP_INSERT_ID:
				//fprintf(stderr, "perform cmd INSERT\n");
				sub=readSubGraph(buf);
				buf+=(buf[0]+3)/4;
				grp->insertChild(sub, chld);
				topology.update(path, 0);
				break;
			case TOP_REMOVE_ID:
				//fprintf(stderr, "perform cmd REMOVE\n");
				grp->removeChild(chld);
				topology.update(path, 0);
				break;
			case TOP_REMOVEALL_ID:
				//fprintf(stderr, "perform cmd REMOVEALL\n");
				grp->removeAllChildren();
				topology.update(path, 0);
				break;
			case TOP_REPLACE_ID:
				//fprintf(stderr, "perform cmd REPLACE\n");
				sub=readSubGraph(buf);
				buf+=(buf[0]+3)/4;
				grp->replaceChild(chld, sub);
				topology.update(path, 0);
				break;
			default:
				fprintf(stderr, "Change-Cmd %d not valid\n", cmd);
				return;
			}
		}
	}

////////////////////////////////////////////////////////////////////////
//
// Description:
//   analyse the output coming from the server
//
// Use: private

void
HvManagerP::readServerOutput()
//
////////////////////////////////////////////////////////////////////////
{
   int len;
   const void *buffer = delayedHandler.get(len);

   if (monitor.getValue()) {
       graphIn.append(len);
       graphInSec.append(len);
   }

#ifdef DEBUG
cerr << "server returned: " << commId << "\n";
#endif

   int l;
   const void *buf;
   const char *cBuff;
   _grphstate.enableNotify(0);
   _grphstate=*((int *)buffer);     // set graphstate
   int commId =((int*) buffer)[1]; // get id
   //fprintf(stderr, "State:%d CommID:%d\n", _grphstate.getValue(), commId);
   switch (commId) {
      case ID_ID:
         //id.setValue(((int *)buffer)[2]);
		 break;
      case GRAPH_ID:
         l = ((int *)buffer)[2];
         readSceneGraph(&(((const char *)buffer)[3 * S_INT]), l);
		 break;
      case SEND_GRAPH_ID:
		 /* this ID comes from master, while redirecting his input
		fprintf(stderr, "shouldn't show up here");
         while(! delayedHandler.flush());  // wait for pending write

         size_t size;			  // write "complete" scene graph
         //buf = writeSceneGraph(this, size);
         delayedHandler.set(GRAPH_ID);
         delayedHandler.append((int)(size + S_INT));
         delayedHandler.append(buf, size);
         delayedHandler.write();
         while(! delayedHandler.flush());    // wait until written
		 */
		 break;
      case PATH_ID:
         cBuff = (const char *)buffer;
         readPath(&cBuff[2*S_INT]);   // notifyDisable should be faster ?
         break;
      case SELECT_ID:
		 /* be compatible */
         break;
      case DESELECT_ID:
		/* be compatible */
         break;
	  case CHANGE_ID:
         readChange( &(((int*)buffer)[2]));	    // notifyDisable should be faster ?
		 //topology.dump();
		 break;
      default:
#ifdef DEBUG
         cerr << "Unknown command: " << commId << "\n";
#endif
         break;
   }
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//   send changed nodes/fields to server and read changes from it
//
// Use: static private

void
HvManagerP::delayedCB(HvManagerP *manager, SoSensor *)
//
////////////////////////////////////////////////////////////////////////
{
   manager->synchronizeGraph();
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//   send changed nodes/fields to server and read changes from it
//
// Use: private

void
HvManagerP::synchronizeGraph()
//
////////////////////////////////////////////////////////////////////////
{
  // connect to server
   if (! delayedHandler.isConnected()) {
	  fprintf(stderr, "HvManagerP:  try to connect\n");
      if (! delayedHandler.connect()) return;
   }

   // read from socket until the buffer is empty
    detachDelayedSensor();
	if (delayedHandler.read())
		{
		do 
			{
	        if (delayedHandler.isComplete()) readServerOutput();
			} while (delayedHandler.read());
		}
   else {
       if (monitor.getValue()) {
           graphIn.append(0);
           graphInSec.append(0);
       }
   }
  attachDelayedSensor();

	// we don't have to write anything */
	return;

}


////////////////////////////////////////////////////////////////////////
//
// Description:
//    Does GL render action.
//
// Use: extender

void
HvManagerP::GLRenderBelowPath(SoGLRenderAction *action)
//
////////////////////////////////////////////////////////////////////////
{
    SoSeparator::GLRenderBelowPath(action);
    renderButton(action);
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    Does GL render action.
//
// Use: extender

void
HvManagerP::GLRenderInPath(SoGLRenderAction *action)
//
////////////////////////////////////////////////////////////////////////
{
    SoSeparator::GLRenderInPath(action);
    renderButton(action);
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    Does GL render action.
//
// Use: extender

void
HvManagerP::GLRenderOffPath(SoGLRenderAction *action)
//
////////////////////////////////////////////////////////////////////////
{
    SoSeparator::GLRenderOffPath(action);
    renderButton(action);
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    Does GL render action.
//
// Use: extender

void
HvManagerP::GLRender(SoGLRenderAction *action)
//
////////////////////////////////////////////////////////////////////////
{
    SoSeparator::GLRender(action);
    renderButton(action);
}


////////////////////////////////////////////////////////////////////////
//
// Description:
//    Does GL render action.
//
// Use: extender

void
HvManagerP::renderButton(SoGLRenderAction *action)
//
////////////////////////////////////////////////////////////////////////
{
#ifdef DEBUG
//cerr << "GLRenderAction\n";
#endif

    // render statistic plots
    if (monitor.getValue()) {
        graphIn.render();
        graphInSec.render();
    }

	// check if this is the first render, if so, attach the delayed handler
	if (!isStarted) 
		{
		// we also initialize the topology
		topology.init(this);
		isStarted=1;
		attachDelayedSensor();
		}

	return;	
}	
