/*
 * Copyright (C) 1996   Silicon Graphics, Inc.
 *
 _______________________________________________________________________
 ______________  S I L I C O N   G R A P H I C S   I N C .  ____________
 |
 |   $Revision: 1.4 $
 |
 |   Classes:
 |      HvManager
 |
 |   Author(s)          : Horst Vollhardt, Mark Benzel
 |
 ______________  S I L I C O N   G R A P H I C S   I N C .  ____________
 _______________________________________________________________________
 */

//----------------------------------------------------------------------
// Changes added by Tobias Strasser:
//
// Additional Field SoMFString pipes 
// this takes filenames to write out changes on the scene graph
//
//----------------------------------------------------------------------

#define MENUS_IN_POPUP

#include <X11/cursorfont.h>

#include <Xm/Xm.h>
#include <Xm/AtomMgr.h>
#include <Xm/Form.h>   
#include <Xm/Label.h>
#include <Xm/LabelG.h>
#include <Xm/Protocols.h>
#include <Xm/PushB.h>
#include <Xm/PushBG.h>
#include <Xm/RowColumn.h>
#include <Xm/Separator.h>
#include <Xm/SeparatoG.h>
#include <Xm/ToggleB.h>
#include <Xm/MwmUtil.h>
 
#include <GL/gl.h>
#include <GL/glx.h>

#include <Inventor/SbLinear.h>  
#include <Inventor/SoPath.h>  
#include <Inventor/SoInput.h>  
#include <Inventor/actions/SoGLRenderAction.h>
#include <Inventor/actions/SoSearchAction.h>
#include <Inventor/actions/SoWriteAction.h>
#include <Inventor/elements/SoCacheElement.h>
#include <Inventor/misc/SoChildList.h>
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/SoXtComponent.h>
#include <Inventor/nodes/SoLabel.h>
#include <Inventor/fields/SoSFNode.h>

#include "HvManager.h"
#include "HvManagerPixmap.h"
#include "HvURLHandler.h"
#include <iostream.h>
#include <stdio.h>
#include <malloc.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <errno.h>
#include <time.h>


// This SG_ routine is from /usr/src/X11/motif/overlay_demos.
extern "C" {
    iv_SG_getPopupArgs(Display *, int, ArgList, int *);
}

#ifdef IV2_0
static void     overlayMenuMappedCB(Widget w, Widget shell, XtPointer);
static void     overlayMenuUnmappedCB(Widget w, Widget shell, XtPointer);
#endif

enum WidgetIds {
    CUI_ROWCOL,

    WEB_BUTTON,

    // Menu dialog
    WEB_MENU,
    WEB_MENU_FORM,
    WEB_BUTTON_FORM,
    WEB_SEPARATOR1,
#ifdef UNDO
    WEB_SEPARATOR1,
    WEB_REDO_BUTTON,
    WEB_UNDO_BUTTON,
    WEB_UNDO_FORM,
#endif /* UNDO */

    WIDGET_NUM, // must be last
};

#define HvUIOffset 3

int HvManager::menuWidth = 0;
int HvManager::menuHeight = 0;
SbPList *HvManager::instanceList = NULL;

SO_NODE_SOURCE(HvManager);

////////////////////////////////////////////////////////////////////////
//
//  Initialize the class
//  
void HvManager::initClass()
//
////////////////////////////////////////////////////////////////////////
{
    SO_NODE_INIT_CLASS(HvManager,SoSeparator,"Separator");

}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    Constructor
//
// Use: public

HvManager::HvManager() : SoSeparator()
//
////////////////////////////////////////////////////////////////////////
{
   constructorCommon();
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    Constructor
//
// Use: public

HvManager::HvManager(int numChildren) : SoSeparator(numChildren)
//
////////////////////////////////////////////////////////////////////////
{
   constructorCommon();
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    common part of all constructor
//
// Use: private

void
HvManager::constructorCommon()
//
////////////////////////////////////////////////////////////////////////
{
    SO_NODE_CONSTRUCTOR(HvManager);

    SO_NODE_ADD_FIELD(url,		(""));
    SO_NODE_ADD_FIELD(server,		(""));
    SO_NODE_ADD_FIELD(id,		(-1));
    SO_NODE_ADD_FIELD(autoConnect,	(FALSE));
    SO_NODE_ADD_FIELD(monitor,	(FALSE));
	SO_NODE_ADD_FIELD(pipes,    (""));
    SO_NODE_ADD_FIELD(webButton,(FALSE));
    SO_NODE_ADD_FIELD(_grphstate,(0));
	_grphstate.enableNotify(0);
	pipes.deleteValues(0);
	url.deleteValues(0);
	
    widgetList = new Widget[WIDGET_NUM];
    for (int i = 0; i < WIDGET_NUM; i++) {
        widgetList[i] = NULL;
    }
    menuNeedToDeiconify = FALSE;
    removeEventHandler = FALSE;

    webMenuVisible = FALSE;

    size_t size = 100;
    outputBuffer = malloc(size);
#ifndef DEBUG
    output.setBinary(TRUE);
#endif
    output.setBuffer(outputBuffer, size, (SoOutputReallocCB *)realloc);

    if (HvManager::instanceList == NULL) {
        HvManager::instanceList = new SbPList;
    }
    HvManager::instanceList->append(this);

    isBuiltIn = TRUE;

    // initialize graphs
    graphIn.setColor(0., 1., 1.);
    graphInSec.setColor(1., 1., 0.);
    graphInSec.setInterval(1000);

    // reference camera to check if the camera really changed
    refCamera = new SoOrthographicCamera;
    refCamera->ref();
    
	//init graph state
	isStarted=0;
	actualCamera=0;
	sharedCamera=0;
	
	
#ifdef UNDO
    lastFunction = NULL;
#endif
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    Destructor
//
// Use: private

HvManager::~HvManager()
//
////////////////////////////////////////////////////////////////////////
{
    // Remove the event handler that we added to catch when the main
    // window resized.
    if (removeEventHandler) {
        XtRemoveEventHandler(SoXt::getTopLevelWidget(), StructureNotifyMask,
            FALSE, (XtEventHandler) HvManager::shellStructureNotifyCB,
            (XtPointer) this);
    }

    // Remove the event handler that we added to catch when the main
    // window iconified.
    if (widgetList[WEB_MENU]) {
        XtRemoveEventHandler(SoXt::getTopLevelWidget(), StructureNotifyMask,
            FALSE, (XtEventHandler) HvManager::shellStructureNotifyCB,
            (XtPointer) this);
    }

    // Remove the event handlers and callbacks we added on the GLX widgets
    // and also delete all the glx buttons...
    for (int i=0; i < buttonGlxList.getLength(); i++) {
 
        XtRemoveEventHandler((Widget) glxList[i], StructureNotifyMask, FALSE,
            (XtEventHandler) HvManager::glxStackingOrderChangeCB,
            (XtPointer) buttonGlxList[i]);
    
        XtRemoveCallback((Widget) glxList[i], XmNdestroyCallback,
            (XtCallbackProc) HvManager::glxDestroyedCB,
            (XtPointer) this);
    
        XtRemoveCallback((Widget) buttonGlxList[i], XmNdestroyCallback,
            (XtCallbackProc) HvManager::glxButtonsDestroyedCB,
            (XtPointer) this);
    
        XtDestroyWidget( (Widget) buttonGlxList[i] );
    }
    windowList.truncate(0);
    glxList.truncate(0);
    buttonGlxList.truncate(0);
    parentGlxList.truncate(0);
 
    // Destroy our widgets
    if (widgetList[WEB_MENU])
        XtDestroyWidget(widgetList[WEB_MENU]);
    delete [] widgetList;

    int n = webFunctionList.getLength();
    for (i = 0; i < n; i++) delete (HvWebFunction *)webFunctionList[i];

    // Remove ourselves from the instanceList
    int idx;
    if ((idx = HvManager::instanceList->find(this)) != -1) {
        HvManager::instanceList->remove(idx);
    }

    // Layout the rest of the instances if any remain
    if (HvManager::instanceList->getLength() > 0) {
        layoutInstances();
    }
    else {
        delete HvManager::instanceList;
        HvManager::instanceList = NULL;
    }

    refCamera->unref();
	
}


////////////////////////////////////////////////////////////////////////
//
// Description:
//    callback when something changes in the scene graph
//
// Use: private

void
HvManager::nodeSensorCB(void *data, SoSensor *s)
//
////////////////////////////////////////////////////////////////////////
	{
//   static int counter = 0;
//   cerr << "*** " << counter++ << " ***\n";

	HvManager &manager   = *(HvManager *)data;
	SoNodeSensor &sensor = *(SoNodeSensor *)s;

	SoNode  *node  = sensor.getTriggerNode();
	if (node == NULL) return;  // this sensor gets triggered by a NULL node ?

	SoField *field = sensor.getTriggerField();
 
	SbString string = node->getTypeId().getName().getString();

	if (field)
		{
		// exception 1: don't send lassoPoints (SoSFNode) (be more specific)
		if (field->isOfType(SoSFNode::getClassTypeId())) return;

		// exception 2: check if camera has really changed
		//	      redraw used to touch one field
		if (node->getTypeId().isDerivedFrom(SoCamera::getClassTypeId()))
			{
			SbName fieldName;
			node->getFieldName(field, fieldName);
			SoField *f = manager.refCamera->getField(fieldName);
			if (f != NULL)
				{
				//	    if (*f == *field) return;  // doesn't work, precission is to high
			    SbString ov, nv;
				f->get(ov); field->get(nv);
				if (ov == nv) return;
				f->set(nv.getString());
				}
			}
		//check if the node is already queued
		int num = manager.delayedNodeList.getLength();
		for (int i = 0; i < num; i++) if (node == manager.delayedNodeList[i]) break;
	
		// append field to the existing field list for this node
		if (i < num)
			{
			// check if this field is already queued
			SoFieldList &list = *((SoFieldList *)manager.delayedNodeFieldList[i]);
			int nf = list.getLength();
			for (int j = 0; j < nf; j++) if (field == list[j]) break;
	
			if (j == nf) list.append(field);	// append new field
			return;
			}
	
		// add field and path to the delayed transmission queue
		SoSearchAction search;
		search.setNode(node);
		search.apply(&manager);
		SoPath *nodePath = search.getPath();
		if (nodePath == NULL)
			{
	#ifdef DEBUG
			 cerr << "changed node (" << string.getString() << ") not found\n";
	#endif
			return;
			}
		nodePath->ref();
		manager.delayedPathList.append(nodePath);
		manager.delayedNodeList.append(node);
		SoFieldList *list = new SoFieldList;
		manager.delayedNodeFieldList.append(list);
		list->append(field);
		}
		
	else // it was only a node, which was changed, probably a group node
		{
		// if so, we have to update the topology
		if (node->isOfType(SoGroup::getClassTypeId()))
				{
				SoPath *path=new SoPath(&manager);
				if (node!=&manager) path->append(node);
				manager.topology.update(path, 1);
				//manager.topology.dump();
				return; // we don't have to queue the node, because theese
						// changes will be treatened seperatly
				}
		}
	}


////////////////////////////////////////////////////////////////////////
//
// Description:
//   attaches the delaydSensor to the timer field
//   and opens the connection to the server
//
// Use: private

void
HvManager::attachDelayedSensor()
//
////////////////////////////////////////////////////////////////////////
{
   if (delayedSensor.getAttachedField() == NULL) {
      delayedSensor.setFunction((SoSensorCB *)HvManager::delayedCB);
      delayedSensor.setData(this);
      delayedSensor.attach(SoDB::getGlobalField("realTime"));
   }

   if (! delayedHandler.isConnected()) {
      delayedHandler.setHostname(server.getValue().getString());
      delayedHandler.setPort(4077);

	if (! delayedHandler.isPiped())
		{
		  for (int i=0;i<pipes.getNum();i++)
			  {
			  delayedHandler.pipeBuffer.addPipe(pipes[i]);
			  delayedHandler.addPipe(pipes[i]);
			  }
		}
	
      while(! delayedHandler.connect());    // wait until connected
#ifdef DEBUG
cerr << "Connetced to " << delayedHandler.getHostname() << "\n";
#endif
      delayedHandler.set(ID_ID);
      delayedHandler.append(id.getValue());
      delayedHandler.write();
      while(! delayedHandler.flush());    // wait until written
   }

}

////////////////////////////////////////////////////////////////////////
//
// Description:
//   detaches the delayedSensor from timer field
//
// Use: private

void
HvManager::detachDelayedSensor()
//
////////////////////////////////////////////////////////////////////////
{
   if (delayedSensor.getAttachedField() != NULL) {
      delayedSensor.detach();
   }
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//   attaches the nodeSensor to the HvManager
//
// Use: private

void
HvManager::attachNodeSensor()
//
////////////////////////////////////////////////////////////////////////
{
   if (nodeSensor.getAttachedNode() == NULL) {
      nodeSensor.setFunction((SoSensorCB *)nodeSensorCB);
      nodeSensor.setData(this);
      nodeSensor.setPriority(0);
      nodeSensor.attach(this);
   }
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//   detaches the nodeSensor from the HvManager
//
// Use: private

void
HvManager::detachNodeSensor()
//
////////////////////////////////////////////////////////////////////////
{
   if (nodeSensor.getAttachedNode() != NULL) {
      nodeSensor.detach();
   }
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//   Stores all fields of the node and the fields of its children
//   in the list
//
// Use: private

void
HvManager::setNodeDefault(SoNode *node, SoFieldList &fieldList)
//
////////////////////////////////////////////////////////////////////////
{
   if (!node->affectsState()) return;
   node->getFields(fieldList);

#ifdef DEBUG
cerr << "set: " << node->getTypeId().getName().getString() << "\n";
#endif

   SoChildList *childList = node->getChildren();
   if (childList == NULL) return;
   int numChildren = childList->getLength();
   for (int i = 0; i < numChildren; i++)
      setNodeDefault((*childList)[i], fieldList);
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//   Set default flags for all fields of all nodes in the path to TRUE
//   Disables the notification of changes as well (SoWriteAction ?!!!)
//   Stores the original state in the list
//
// Use: private

void
HvManager::setAllDefault(SoPath *path, SoFieldList &fieldList,
                         SbIntList &saveList, SbIntList &notifyList)
//
////////////////////////////////////////////////////////////////////////
{
   int numNode = path->getLength();
   fieldList.truncate(0);
   saveList.truncate(0);

   for (int i = 0; i < numNode; i++) {
      SoNode *node = path->getNode(i);
      node->getFields(fieldList);
      notifyList.append(node->isNotifyEnabled());
      node->enableNotify(FALSE);

      // check for nodes within the same parent
      // which affect also the state, because they are
      // also written out with the SoWriteAction
      int idx;
      if (((idx = path->getIndex(i)) > 0)&&(i > 0)) {
	 SoNode *parent = path->getNode(i - 1);
	 SoChildList *childList = parent->getChildren();
         for (int j = 0; j < idx; j++) {
	    setNodeDefault((*childList)[j], fieldList);
	 }
      }
   }

   int numField = fieldList.getLength();
   for (i = 0; i < numField; i++) {
      SoField *field = fieldList[i];
      saveList.append(field->isDefault());
      field->setDefault(TRUE);
   }
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//   restore default flags for all fields of all nodes in the path
//
// Use: private

void
HvManager::resetAllDefault(SoPath *path, SoFieldList &fieldList,
                           SbIntList &saveList, SbIntList &notifyList)
//
////////////////////////////////////////////////////////////////////////
{
   int numNode = path->getLength();

   for (int i = 0; i < numNode; i++) {
      SoNode *node = path->getNode(i);
      node->enableNotify(notifyList[i]);
   }

   int numField = fieldList.getLength();
   for (i = 0; i < numField; i++) {
      fieldList[i]->setDefault(saveList[i]);
   }
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//   read scene graph from buffer and add it to the HvManager
//   returns -1 on error
//
// Use: private

int
HvManager::readSceneGraph(const char *buffer, int len)
//
////////////////////////////////////////////////////////////////////////
{
   SoInput input;
   input.setBuffer((void *)buffer, len);
   SoNode *root = NULL;
   if (! SoDB::read(&input, root)) return -1;
   HvManager *manager = (HvManager *) root;	// should be correct !
   int num = manager->getNumChildren();
   if (sharedCamera) sharedCamera->ref();//sharedCamera=(SoCamera*) sharedCamera->copy();
   removeAllChildren();

   for (int i = 0; i < num; i++)
	{
    SoNode *child = manager->getChild(i);
    if (child->getTypeId().isDerivedFrom(SoCamera::getClassTypeId()))
		{
		if (strcmp(child->getName().getString(), "SHARED_CAMERA") || (!sharedCamera))
			{
			if (actualCamera)
				{
				actualCamera->copyFieldValues(child);
				addChild(actualCamera);
				#ifdef DEBUG
					cerr << "Camera replaced\n";
				#endif
				}
			else
				{
				fprintf(stderr, "Oops,  no camera\n");
				addChild(child);
				}
			}
		else
			{
			fprintf(stderr, "try to replace the shared camera\n");
			sharedCamera->copyFieldValues(child);
			addChild(sharedCamera);
			//#ifdef DEBUG
				cerr << "Shared Camera replaced\n";
			//#endif
			}
		}
     else 
		{
        addChild(child);
		}
	}
   for(i = 0; i < webFunctionList.getLength(); i++) {
	((HvWebFunction *)webFunctionList[i])->updateFieldReferences();
   }

   return 0;
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//   read path from buffer and change it in the HvManager
//   returns -1 on error
//
// Use: private

int
HvManager::readPath(const char *data)
//
////////////////////////////////////////////////////////////////////////
	{
	int numNodes = *((int *)data); data += S_INT;
	int camera=0;
	for (int i = 0; i < numNodes; i++)
		{		
		// find node in the scene graph
		int level = *((int *)data); data += S_INT;
		SoNode *node = (SoNode *)this;
		
		for (int j = 0; j < level; j++)
			{
			int idx = *((int *)data); data += S_INT;
			SoGroup *group = (SoGroup *)node;
			if ((node = group->getChild(idx)) == NULL) return -1;
			}
		if (node->getTypeId().isDerivedFrom(SoCamera::getClassTypeId())) camera=1;
		
		// get fields and set values
		int numFields = *((int *)data); data += S_INT;
		SoFieldList fieldList;
		node->getFields(fieldList);
		for (j = 0; j < numFields; j++)
			{
			int idx = *((int *)data);  data += S_INT;
			int len   = *((int *)data);  data += S_INT;
			SoField &field = *(fieldList[idx]);
			field.set(data); data += len;
			}
		}
	if (camera&&(numNodes<2)) return 1;
	return 0;
	}


////////////////////////////////////////////////////////////////////////
//
// Description:
//   write scene graph into buffer and return it
//
// Use: private

const void *
HvManager::writeSceneGraph(SoNode *node, size_t &size)
//
////////////////////////////////////////////////////////////////////////
{
   output.resetBuffer();
   SoWriteAction write(&output);
   write.apply(node);
   if (! output.getBuffer(outputBuffer, size)) {
      cerr << "Error on write\n";
   }

   return outputBuffer;
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//   write path into buffer and return it
//
// Use: private

void
HvManager::writePath(const SoPath *path, SbIntList &list)
//
////////////////////////////////////////////////////////////////////////
{
   int numNode = path->getLength();
   list.truncate(0);
   list[0] = numNode - 1;  // don't care about head node

   for (int i = 1; i < numNode; i++) {
      list[i] = path->getIndex(i);
   }

/*  doesn't work so well, as I expected

   output.resetBuffer();
   SoWriteAction write(&output);
   write.apply(path);
   if (! output.getBuffer(outputBuffer, size)) {
      cerr << "Error on write\n";
   }

   return outputBuffer;
*/
}
////////////////////////////////////////////////////////////////////////
//
// Description:
//   reads a path from buffer and returns a SoPath
//
// Use: private

SoPath*
HvManager::readSoPath(int *buf)
//
////////////////////////////////////////////////////////////////////////
	{
	SoPath *path=new SoPath(this);
	path->ref();
	
	int level = *(buf++);
	SoNode *node = (SoNode *)this;

	for (int j =0; j < level; j++)
		{
		int idx = *(buf++);
		SoGroup *group = (SoGroup *)node;
		if ((node = group->getChild(idx)) == NULL) return path;
		path->append(node);
		}
	return path; 		
	}
	
////////////////////////////////////////////////////////////////////////
//
// Description:
//   reads a path from buffer and returns a SoPath
//
// Use: private

SoNode*
HvManager::readSubGraph(int *buf)
//
////////////////////////////////////////////////////////////////////////
	{
	int len=*(buf++);
	SoInput input;
	/*
	fprintf(stderr, "buf[0]=%x\n", buf[0]);
	fprintf(stderr, "buf[1]=%x\n", buf[1]);
	fprintf(stderr, "buf[2]=%x\n", buf[2]);
	fprintf(stderr, "buf[3]=%x\n", buf[3]);
	fprintf(stderr, "Len:%d Content:%s\n", len, buf);
	*/
	input.setBuffer((void *)buf, len);
	SoNode *node = NULL;
	if (! SoDB::read(&input, node)) return 0;
	return node;
	}	
	
////////////////////////////////////////////////////////////////////////
//
// Description:
//   decodes a CHANGE_ID chain
//
// Use: private

void
HvManager::readChange(int *buf)
//
////////////////////////////////////////////////////////////////////////
	{
	int nofChng=*(buf++);
	SoNode *sub;
	int len;
	// process all changes
	// fprintf(stderr, "%d changed\n", nofChng);
	for (int i=0;i<nofChng;i++)
		{
		int	    cmd;
		SoPath *path;
		int		chld;
		SoGroup *grp;
		
		cmd=*(buf++);
		path=readSoPath(buf);
		buf+=buf[0]+1;
		chld=*(buf++);
		if (!path) continue;		
		grp=(SoGroup*) path->getTail();
		switch (cmd)
			{
			case TOP_INSERT_ID:
				//fprintf(stderr, "perform cmd INSERT\n");
				sub=readSubGraph(buf);
				buf+=(buf[0]+3)/4;
				grp->insertChild(sub, chld);
				topology.update(path, 0);
				break;
			case TOP_REMOVE_ID:
				//fprintf(stderr, "perform cmd REMOVE\n");
				grp->removeChild(chld);
				topology.update(path, 0);
				break;
			case TOP_REMOVEALL_ID:
				//fprintf(stderr, "perform cmd REMOVEALL\n");
				grp->removeAllChildren();
				topology.update(path, 0);
				break;
			case TOP_REPLACE_ID:
				//fprintf(stderr, "perform cmd REPLACE\n");
				sub=readSubGraph(buf);
				buf+=(buf[0]+3)/4;
				grp->replaceChild(chld, sub);
				topology.update(path, 0);
				break;
			default:
				fprintf(stderr, "Change-Cmd %d not valid\n", cmd);
				return;
			}
		}
	}

////////////////////////////////////////////////////////////////////////
//
// Description:
//   analyse the output coming from the server
//
// Use: private

void
HvManager::readServerOutput()
//
////////////////////////////////////////////////////////////////////////
{
   int len;
   const void *buffer = delayedHandler.get(len);
   int commId = *((int *)buffer);

   if (monitor.getValue()) {
       graphIn.append(len);
       graphInSec.append(len);
   }

#ifdef DEBUG
cerr << "server returned: " << commId << "\n";
#endif

   int l;
   const void *buf;
   const char *cBuff;

   switch (commId) {
      case ID_ID:
         id.setValue(((int *)buffer)[1]);
		 break;
      case GRAPH_ID:
         l = ((int *)buffer)[1];
         readSceneGraph(&(((const char *)buffer)[2 * S_INT]), l);
		 fprintf(stderr, "reading entire scene-graph\n");
		 // reinit the topology
		 topology.init(this);
		 break;
      case SEND_GRAPH_ID:
		 fprintf(stderr, "sending entire graph to server\n");
         while(! delayedHandler.flush());  // wait for pending write

         size_t size;			  // write "complete" scene graph
         buf = writeSceneGraph(this, size);
         delayedHandler.set(GRAPH_ID);
         delayedHandler.append((int)(size + S_INT));
         delayedHandler.append(buf, size);
		 buffer=delayedHandler.getWrite(len);
         delayedHandler.write();
         while(! delayedHandler.flush());    // wait until written
		 break;
      case PATH_ID:
         cBuff = (const char *)buffer;
		 detachNodeSensor();	    // avoid infinite loops (you know why ?)
         if (readPath(&cBuff[S_INT])==1) len=0; // if only camera changed, don't
												// say it to children
 		 attachNodeSensor();	    // but then the scene doesn't get redrawn
		 break;
      case SELECT_ID:
		 /* this was used for ChemSelections */ 
         break;
      case DESELECT_ID:
		 /* this was used for ChemSelections */ 
         break;
	  case CHANGE_ID:
		 detachNodeSensor();			// avoid infinite loops (you know why ?)
         readChange( &(((int*)buffer)[1]));	    // notifyDisable should be faster ?
 		 attachNodeSensor();	    // but then the scene doesn't get redrawn
		 topology.dump();
		 break;
      default:
#ifdef DEBUG
         cerr << "Unknown command: " << commId << "\n";
#endif
         break;
		}
	 if ((pipes.getNum()>0) && (len))
		{
		_grphstate.enableNotify(FALSE);
		_grphstate=_grphstate.getValue()+1;
		while(! delayedHandler.flushPipe());  // wait for pending write
		delayedHandler.pipeBuffer.set((int) _grphstate.getValue()); // set state
		delayedHandler.pipeBuffer.append(buffer, len); //append rest
		delayedHandler.writePipe();
		while(! delayedHandler.flushPipe());    // wait until written
		}
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//   send changed nodes/fields to server and read changes from it
//
// Use: static private

void
HvManager::delayedCB(HvManager *manager, SoSensor *)
//
////////////////////////////////////////////////////////////////////////
{
   manager->synchronizeGraph();
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//   send changed nodes/fields to server and read changes from it
//
// Use: private

void
HvManager::synchronizeGraph()
//
////////////////////////////////////////////////////////////////////////
{
   // connect to server
   if (! delayedHandler.isConnected()) {
      if (! delayedHandler.connect()) return;
   }

   // read from socket until the buffer is empty
   if (delayedHandler.read()) {
      do {
         if (! delayedHandler.isConnected()) {	// socket closed
            detachDelayedSensor();
			delayedHandler.closeConnection();
            readServerOutput();			
		    return;
         }
         if (delayedHandler.isComplete()) {
            readServerOutput();
	 }
      } while (delayedHandler.read());
   }
   else {
       if (monitor.getValue()) {
           graphIn.append(0);
           graphInSec.append(0);
       }
   }

   if (! delayedHandler.flush()) return;  // wait for pending write

   int numNode = delayedNodeList.getLength();
   int numChng = topology.commList.getLength();

   if (numChng != 0)
	{
	// ok, send all changes
	delayedHandler.set(CHANGE_ID);
	
	// how many change we have ?
	delayedHandler.append(numChng);
	
	// now write out each change
	for (int i=0;i<numChng;i++)
		{
		// first send the command
		int cmd=topology.commList[i];
		delayedHandler.append(cmd);
		
		// then the path to the group-node
		SbIntList indexList;
		writePath(topology.pathList[i], indexList);
		
		int numIndex = indexList.getLength();
		for (int j = 0; j < numIndex; j++)
			delayedHandler.append(indexList[j]);

		// send the index of the child changed
		delayedHandler.append(topology.chldList[i]);
		
		// if we had a REPLACE or an INSERT, send the subgraph
		if ((cmd==TOP_INSERT_ID) || (cmd==TOP_REPLACE_ID))	
			{
			int *buf=(int*) topology.grphList[i];
			int  len=(buf[0]+3)/4;
			delayedHandler.append(buf, len);
			}
		}		

	// if we have pipes, write to them first
	if (pipes.getNum()>0)
		{
		int len;
		const void *buffer = delayedHandler.getWrite(len);
		// we increment our graph-state
		_grphstate.enableNotify(0);
		_grphstate=_grphstate.getValue()+1;
		while(! delayedHandler.flushPipe());  // wait for pending write
		delayedHandler.pipeBuffer.set((int) _grphstate.getValue()); // set state
		delayedHandler.pipeBuffer.append(buffer, len); //append rest
		delayedHandler.writePipe();
		while(! delayedHandler.flushPipe());    // wait until written
		}

	// truncate topology and send buffer
	topology.truncate();
	delayedHandler.write();    // don't wait until written
	}
else if (numNode != 0)
	{
	int notifyPipes=0;
	delayedHandler.set(PATH_ID);
	// prepare all nodes for writing to the bufferList
	delayedHandler.append(numNode);
	SbIntList indexList;

	for (int i = 0; i < numNode; i++)
		{
		SoNode *node = delayedNodeList[i];
		SoPath *path = delayedPathList[i];

#ifdef DEBUG
cerr << i << ".: " << node->getTypeId().getName().getString() << "\n";
#endif
		if (!node->getTypeId().isDerivedFrom(SoCamera::getClassTypeId())) notifyPipes=1;

		writePath(path, indexList);
		int numIndex = indexList.getLength();
		for (int j = 0; j < numIndex; j++)
			delayedHandler.append(indexList[j]);

		SoFieldList *listP = (SoFieldList *)delayedNodeFieldList[i];
		int numField = listP->getLength();
		delayedHandler.append(numField);

		SoFieldList fieldList;
		node->getFields(fieldList);
		int nf = fieldList.getLength();
		SbString value;

		for (j = 0; j < numField; j++)
			{
			SoField *field = (*listP)[j];
			for (int k = 0; k < nf; k++) 
				{  // find index of this field
				if (field == fieldList[k]) break;
				}
			delayedHandler.append(k);
			field->get(value);
			int len = value.getLength() + 1;
			int lenPlus = (len&0x3) ? (4-(len&0x3)) : 0;
			delayedHandler.append(len + lenPlus);
			delayedHandler.append(value.getString(), len);
			if (lenPlus > 0) delayedHandler.append("\0\0\0\0", lenPlus);

#ifdef DEBUG
//cerr << "Field(" << j << ") = " << value.getString() << "\n";
#endif
			}
	
		// clear memory, etc.
		path->unref();
		listP->truncate(0);
		}
	delayedNodeList.truncate(0);
	delayedPathList.truncate(0);
	delayedNodeFieldList.truncate(0);

	//delayedSensor.detach();
	
	// if we have pipes, write to them first
	if ((pipes.getNum()>0) && (notifyPipes))
		{
		int len;
		const void *buffer = delayedHandler.getWrite(len);
		// fprintf(stderr,"start writing %d path-bytes...",len);
		// we increment our graph-state 
		_grphstate.enableNotify(0);
		_grphstate=_grphstate.getValue()+1;
 		
		// fprintf(stderr, "1-");	
		while(! delayedHandler.flushPipe());  // wait for pending write
		//fprintf(stderr, "2-");	
		delayedHandler.pipeBuffer.set((int) _grphstate.getValue()); // set state
		//fprintf(stderr, "3-");	
		delayedHandler.pipeBuffer.append(buffer, len); //append rest
		//fprintf(stderr, "4-");	
		delayedHandler.writePipe();
		//fprintf(stderr, "5-");	
		while(! delayedHandler.flushPipe());    // wait until written
		//fprintf(stderr, "done\n");
		}

	delayedHandler.write();    // don't wait until written
	}
}


////////////////////////////////////////////////////////////////////////
//
// Description:
//    Does GL render action.
//
// Use: extender

void
HvManager::GLRenderBelowPath(SoGLRenderAction *action)
//
////////////////////////////////////////////////////////////////////////
{
    SoSeparator::GLRenderBelowPath(action);
    renderButton(action);
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    Does GL render action.
//
// Use: extender

void
HvManager::GLRenderInPath(SoGLRenderAction *action)
//
////////////////////////////////////////////////////////////////////////
{
    SoSeparator::GLRenderInPath(action);
    renderButton(action);
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    Does GL render action.
//
// Use: extender

void
HvManager::GLRenderOffPath(SoGLRenderAction *action)
//
////////////////////////////////////////////////////////////////////////
{
    SoSeparator::GLRenderOffPath(action);
    renderButton(action);
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    Does GL render action.
//
// Use: extender

void
HvManager::GLRender(SoGLRenderAction *action)
//
////////////////////////////////////////////////////////////////////////
{
    SoSeparator::GLRender(action);
    renderButton(action);
}


////////////////////////////////////////////////////////////////////////
//
// Description:
//    Does GL render action.
//
// Use: extender

void
HvManager::renderButton(SoGLRenderAction *action)
//
////////////////////////////////////////////////////////////////////////
{
#ifdef DEBUG
//cerr << "GLRenderAction\n";
#endif

    // render statistic plots
    if (monitor.getValue()) {
        graphIn.render();
        graphInSec.render();
    }

	// find the shared camera
	sharedCamera=(SoCamera*) SoNode::getByName("SHARED_CAMERA");
	
    // find the topmost camera node
    if (actualCamera == NULL) {
        SoNode *node;
        if ((node = action->getNodeAppliedTo()) != NULL) {
	   SoSearchAction search;
	   search.setType(SoCamera::getClassTypeId());
	   search.apply(node);
	   SoPath *path = search.getPath();
	   if (path != NULL) {
  	      actualCamera = (SoCamera *)path->getTail();
	      cameraParent = (SoGroup *)path->getNodeFromTail(1);
	      cameraRoot = (SoGroup *)path->getHead();
#ifdef DEBUG
	   cerr << "found camera\n";
#endif
	   }
	}
    }

	// check if we have to render the webbutton
	if (!webButton.getValue())
		{
		// check if we started the sensors
		if (!isStarted)
			{
			// we also initialize the topology
			topology.init(this);
			isStarted=1;
		    attachNodeSensor();
			attachDelayedSensor();
			}
		return;
		}


    // Prevent caching at this level to make sure we get called
    // for every rendering so that we can properly cache our list
    // of windows
    SoState *state = action->getState();
//    SoCacheElement::invalidate(state);

    //
    // Keep a list of all the windows this node is being rendered into to
    // make sure we have built the UI within the parent of this
    // window.
    //
   
    Window window = glXGetCurrentDrawable();
    
    // Check if we already encountered this window
    if (windowList.find(window) >= 0)
        return;

    // Else, this is a new window so check to see whether we already have
    // built the buttons within the parent of this GLX window (this would
    // be the case when switching from double to single buffer).
    Widget glx = XtWindowToWidget(SoXt::getDisplay(), window);

    // If glx is NULL, then return as there is nowhere to build the buttons.
    if (glx == NULL) return;

    Widget parent = XtParent(glx);
    
    if (parentGlxList.find(parent) >= 0) {
        return;
    }

    //
    // OK, so this is a brand new window and we haven't built the buttons
    // for that widget yet, so create the buttons and add those widgets to
    // our list.

    Widget button = createGlxWindowButtons(parent);
  
    windowList.insert(window, 0);
    glxList.insert(glx, 0);
    buttonGlxList.insert(button, 0);
    parentGlxList.insert(parent, 0);

    // Add a structure notify to the current window to be told when the
    // stacking order changes - to make sure our buttons are on top.
    //
    // StructureNotify events are called for changes in window size,
    // position, border or stacking order.
    XtAddEventHandler(glx, StructureNotifyMask, FALSE,
        (XtEventHandler) HvManager::glxStackingOrderChangeCB,
        (XtPointer) button);

    // Add a structure notify for when the window resizes.
    XtAddEventHandler(SoXt::getTopLevelWidget(), StructureNotifyMask, FALSE,
        (XtEventHandler) HvManager::shellStructureNotifyCB, (XtPointer) this);
  
    // Add destroy callbacks so that we can remove those windows
    // from our list.
    XtAddCallback(glx, XmNdestroyCallback,
        (XtCallbackProc) HvManager::glxDestroyedCB,
        (XtPointer) this);
    XtAddCallback(button, XmNdestroyCallback,
        (XtCallbackProc) HvManager::glxButtonsDestroyedCB,
        (XtPointer) this);

    if (autoConnect.getValue())
       iconCB(widgetList[WEB_BUTTON], WEB_BUTTON, NULL);
}

    
//////////////////////////////////////////////////////////////////////////
//
// Description:
//    This builds the main window buttons within the given widget.
//
// Usage: private

Widget
HvManager::createGlxWindowButtons(Widget parent)
//
//////////////////////////////////////////////////////////////////////////
{
    Arg args[12];
    int n;
    unsigned short width, height;

    int whichInstance = HvManager::instanceList->getLength() - 1;

    // Create a row column to hold the multiple buttons
    n = 0;
    XtSetArg(args[n], XmNorientation, XmHORIZONTAL); n++;
    XtSetArg(args[n], XmNspacing, 0); n++;
    XtSetArg(args[n], XmNmarginHeight, 0); n++;
    XtSetArg(args[n], XmNmarginWidth, 0); n++;
    Widget rowColumn = widgetList[CUI_ROWCOL] =
        XmCreateRowColumn(parent, "iconRowCol", args, n);

    // Position the rowColumn
    positionMenu(this, parent, whichInstance);

    // Create the icon for web menu dialog
    Widget webButton = buildColorButton(rowColumn, WEB_BUTTON, 
        hvWebButtonXpm);
    XtManageChild(webButton);

    // Build the remaining pixmaps
    buildColorPixmaps(webButton, hvWebGoButtonXpm,   goPixmap);
    buildColorPixmaps(webButton, hvWebStopButtonXpm, stopPixmap);

    XtManageChild(rowColumn);

    // Get the width and height of rowColumn
    if (whichInstance == 0) {
        n = 0;
        XtSetArg(args[0], XmNwidth, &width);
        XtSetArg(args[1], XmNheight, &height);
        XtGetValues(rowColumn, args, 2);
        HvManager::menuWidth = width;
        HvManager::menuHeight = height;
    }

    // Define the standard arrow cursor to prevent the viewer cursor
    // from showing up when the mouse moves over the small icons.
    static Cursor defaultCursor = 0;
    Display *display = XtDisplay(rowColumn);
    if (! defaultCursor) {
        XColor foreground, background;
        foreground.red = 65535;
        foreground.green = foreground.blue = 0;
        background.red = background.green = background.blue = 65535;

        defaultCursor = XCreateFontCursor(display, XC_left_ptr);
        XRecolorCursor(display, defaultCursor, &foreground, &background);
    }
    XDefineCursor(display, XtWindow(rowColumn), defaultCursor);

    return rowColumn;
}
    
//////////////////////////////////////////////////////////////////////////
//
// Description:
//    If the stacking order changes, we need to make sure our widgets are
//    on top
//
// Usage: private

void
HvManager::glxStackingOrderChangeCB(Widget, Widget buttons, 
                                 XAnyEvent *, Boolean *)
//
//////////////////////////////////////////////////////////////////////////
{
    XRaiseWindow(XtDisplay(buttons), XtWindow(buttons));
}
    
//////////////////////////////////////////////////////////////////////////
//
// Description:
//    Called when the glx widget gets destroyed.  This removes the entries
//    from the lists AND destroys the buttons to make sure that we rebuild
//    new buttons (and add a structure notify for the new window).
//
// Usage: private

void
HvManager::glxDestroyedCB(Widget w, HvManager *p, XtPointer)
//
//////////////////////////////////////////////////////////////////////////
{
    int idx = p->glxList.find(w);
    
    // remove our info from the list (if we haven't already done so)
    if (idx < 0)
        return;
    
    // remove our buttons too....
    XtDestroyWidget( (Widget) p->buttonGlxList[idx] );

    p->windowList.remove(idx);
    p->glxList.remove(idx);
    p->buttonGlxList.remove(idx);
    p->parentGlxList.remove(idx);
}
    
//////////////////////////////////////////////////////////////////////////
//
// Description:
//    Called when the button form widget gets destroyed.  This removes 
//    the entries from the lists.
//
// Usage: private

void
HvManager::glxButtonsDestroyedCB(Widget w, HvManager *p, XtPointer)
//
//////////////////////////////////////////////////////////////////////////
{
    int idx = p->buttonGlxList.find(w);

    // remove our info from the list (if we haven't already done so)
    if (idx < 0)
        return;

    p->windowList.remove(idx);
    p->glxList.remove(idx);
    p->buttonGlxList.remove(idx);
    p->parentGlxList.remove(idx);
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//  	creates a motif push button with the given pixmap
//
Widget
HvManager::buildColorButton(Widget parent, int id, char **data)
//
////////////////////////////////////////////////////////////////////////
{
    Arg args[8];
    int n;
    
    // Create the push button
    n = 0;
    XtSetArg(args[n], XmNuserData, this); n++;
    XtSetArg(args[n], XmNhighlightThickness, 0); n++;
    XtSetArg(args[n], SgNpixmapLocateHighlight, True); n++; // use highlight pixmap
    XtSetArg(args[n], XmNmarginHeight, 0); n++;
    XtSetArg(args[n], XmNmarginWidth, 0); n++;
    // ??? this cannot be a push button Gadget or adding an event handler will fail
    Widget button = XmCreatePushButton(parent, NULL, args, n);
    widgetList[id] = button;
    
    XtAddCallback(button, XmNactivateCallback, 
		(XtCallbackProc) HvManager::iconCB, (XtPointer) id);
    
    Pixmap pixmaps[3];
    buildColorPixmaps(button, data, pixmaps);

    setPixmaps(button, pixmaps);
    
    return button;
}


////////////////////////////////////////////////////////////////////////
//
// Description:
//  	applies a pixmap to a button
//
void
HvManager::setPixmaps(Widget button, Pixmap pixmaps[3])
//
////////////////////////////////////////////////////////////////////////
{
    Arg args[8];
    int n;

    // assign the pixmaps to the push buttons
    n = 0;
    XtSetArg(args[n], XmNlabelType, XmPIXMAP); n++;
    XtSetArg(args[n], XmNlabelPixmap, pixmaps[0]); n++;
    XtSetArg(args[n], SgNlocatePixmap, pixmaps[1]); n++;
    XtSetArg(args[n], XmNlabelInsensitivePixmap, pixmaps[2]); n++;
    XtSetValues(button, args, n);
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//  	creates a colored pixmap for the given button
//
void
HvManager::buildColorPixmaps(Widget button, char **data, Pixmap pixmaps[3])
//
////////////////////////////////////////////////////////////////////////
{
    Display	*display = XtDisplay(button);
    Drawable	d = DefaultRootWindow(display);
    Pixel	bg, hbg;
    
    // get the color for the push button background & highlight background
    XtVaGetValues(button, XmNbackground, &bg, NULL);
    hbg = SgGetLocatePixel(button, bg);
    
    XpmColorSymbol symbols[2];
    
    XpmAttributes attributes;
    attributes.valuemask = NULL;
    attributes.valuemask |= XpmColorSymbols;
    attributes.colorsymbols = symbols;
    
    // normal pixmap
    symbols[0].name  = XmNbackground;
    symbols[0].pixel = bg;
    symbols[0].value = NULL;
    attributes.numsymbols = 1;
    XpmCreatePixmapFromData(display, d, data, &pixmaps[0], NULL, &attributes);
    
    // highlighted pixmap
    symbols[0].name  = XmNbackground;
    symbols[0].pixel = hbg;
    symbols[0].value = NULL;
    attributes.numsymbols = 1;
    XpmCreatePixmapFromData(display, d, data, &pixmaps[1], NULL, &attributes);
    
    // insensitive pixmap
    symbols[0].name  = XmNbackground;
    symbols[0].pixel = bg;
    symbols[0].value = NULL;
    symbols[1].name  = "other";
    symbols[1].pixel = 0;
    symbols[1].value = "#707070";
    attributes.numsymbols = 2;
    XpmCreatePixmapFromData(display, d, data, &pixmaps[2], NULL, &attributes);
    
    XpmFreeAttributes(&attributes);
}


//////////////////////////////////////////////////////////////////////////
//
// Description:
//    Builds the web menu
//
// Usage: private

Widget
HvManager::buildWebButton(Widget parent)
//
//////////////////////////////////////////////////////////////////////////
{
    int     n;
    Arg     args[12];

    n = 0;
    XtSetArg(args[n], XmNhighlightThickness, 0); n++;
    XtSetArg(args[n], XmNmarginWidth, 0); n++;
    XtSetArg(args[n], XmNmarginHeight, 0); n++;
    XtSetArg(args[n], XmNuserData, this); n++;

    widgetList[WEB_BUTTON] = XtCreateManagedWidget("Web Functions",
        xmPushButtonGadgetClass, parent, args, n);
    XtAddCallback(widgetList[WEB_BUTTON], XmNactivateCallback,
        (XtCallbackProc) HvManager::webButtonCB, (XtPointer) WEB_BUTTON); 

    return widgetList[WEB_BUTTON];
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//      called when any of the UI buttons get pressed
//
// static

void
HvManager::iconCB(Widget w, int id, void *)
//
////////////////////////////////////////////////////////////////////////
{
    HvManager *wui;
    XtVaGetValues(w, XmNuserData, &wui, NULL);

    switch (id) {
       case WEB_BUTTON:
	  wui->openWebMenu();
          break;
       default:
	  cerr << "Unknown iconCB (" << id << ")\n";
	  break;
    }
}

#ifdef UNDO
////////////////////////////////////////////////////////////////////////
//
// Description:
//      called when any of the undo buttons get pressed
//
// static

void
HvManager::webMenuUndoCB(Widget, void *info, void *)
//
////////////////////////////////////////////////////////////////////////
{
   HvManager &wui = *(HvManager *)info;

   if (wui.lastFunction) {
      XtSetSensitive(wui.widgetList[WEB_REDO_BUTTON], TRUE);
      XtSetSensitive(wui.widgetList[WEB_UNDO_BUTTON], FALSE);
      wui.lastFunction->undo();
   }
}
    

////////////////////////////////////////////////////////////////////////
//
// Description:
//      called when any of the redo buttons get pressed
//
// static

void
HvManager::webMenuRedoCB(Widget, void *info, void *)
//
////////////////////////////////////////////////////////////////////////
{
   HvManager &wui = *(HvManager *)info;

   if (wui.lastFunction) {
      XtSetSensitive(wui.widgetList[WEB_UNDO_BUTTON], TRUE);
      XtSetSensitive(wui.widgetList[WEB_REDO_BUTTON], FALSE);
      wui.lastFunction->redo();
   }
}
#endif /* UNDO */


//////////////////////////////////////////////////////////////////////////
//
// Description:
//    Callback for the web menu
//
// Usage: private

void
HvManager::webButtonCB(Widget w, int, void *)
//
//////////////////////////////////////////////////////////////////////////
{
    HvManager *wui;
    XtVaGetValues(w, XmNuserData, &wui, NULL);

    wui->openWebMenu();
}


////////////////////////////////////////////////////////////////////////
//  
// Description:
//    Build the web menu dialog window
//
// Use: private

void
HvManager::openWebMenu()
//
////////////////////////////////////////////////////////////////////////
{
    int i;

    // Check to see if the dialog has already been built
    Widget dialog = widgetList[WEB_MENU];
    if (dialog != NULL) {
        webMenuVisible = TRUE;
        SoXt::show(dialog);
        return;
    }

    int numButton = url.getNum();
    if (numButton < 1)
		{
		// we also initialize the topology
		topology.init(this);
	    attachNodeSensor();
	    attachDelayedSensor();
		return;
		}

    // display a dialog and dynamically add buttons
    // while loading the UI definitions
    buildWebMenuDialog();

    webFunctionList.truncate(0);
    wfnLoadList.truncate(0);
    urlList.truncate(0);
    for (i = 0; i < numButton; i++) {
       urlList.append(new HvURLHandler(url[i].getString()));
       HvURLHandler *u = (HvURLHandler *)urlList[i];
       u->setBinaryOutput(TRUE);

       webFunctionList.append(new HvWebFunction);
       HvWebFunction *wfn = (HvWebFunction *)webFunctionList[i];
       wfnLoadList.append(wfn);
       wfn->setWebUI(this);
       wfn->setParentNode(this);
       wfn->setHostname(u->getHostname());
       wfn->setPort(u->getPort());

       wfn->buildInitialButton(widgetList[WEB_BUTTON_FORM]);
    }

    timeSensor.setFunction((SoSensorCB *)HvManager::readWebFunctionInterfaceCB);
    timeSensor.setData(this);
    timeSensor.attach(SoDB::getGlobalField("realTime"));

    attachNodeSensor();
    attachDelayedSensor();
	// we also initialize the topology
	topology.init(this);
	}


//////////////////////////////////////////////////////////////////////////
//
// Description:
//    Set the visiblity of the web menu dialog to FALSE
//
// Usage: static
    
void
HvManager::webMenuClosedCB(Widget, void *info, XmAnyCallbackStruct *)
//
//////////////////////////////////////////////////////////////////////////
{
    HvManager* wui = (HvManager*) info;
    wui->webMenuVisible = FALSE;
}

//////////////////////////////////////////////////////////////////////////
//
// Description:
//    Map/unmap the preferences dialog if the application is deiconified
//    or iconified
//
// Usage: static

void
HvManager::shellStructureNotifyCB(Widget, HvManager *wui, XEvent *xe, Boolean *)
//
//////////////////////////////////////////////////////////////////////////
{
    if (xe->type == MapNotify) {
        if (wui->menuNeedToDeiconify) {
            Widget shell = wui->widgetList[WEB_MENU];
            XMapWindow(XtDisplay(shell), XtWindow(shell));
            wui->menuNeedToDeiconify = TRUE;   
        }
    }
    else if (xe->type == UnmapNotify) {
        if (wui->webMenuVisible) {
            Widget shell = wui->widgetList[WEB_MENU];
            XUnmapWindow(XtDisplay(shell), XtWindow(shell));
            wui->menuNeedToDeiconify = TRUE;
        }
    }
    else if (xe->type == ConfigureNotify) {
        int whichInstance = HvManager::instanceList->find(wui);
        const Widget &parent = (Widget)wui->parentGlxList[0];
        positionMenu(wui, parent, whichInstance);
    }
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    Layouts the remaining menus after a HvManager has been destroyed
//
// Usage: static private

void
HvManager::layoutInstances()
//
////////////////////////////////////////////////////////////////////////
{
    Arg args[2];
    int length = HvManager::instanceList->getLength();
    HvManager *wui;

    for (int i = 0; i < length; i++) {
        wui = (HvManager *)((*HvManager::instanceList)[i]);
        const Widget &parent = (Widget)wui->parentGlxList[0];
        positionMenu(wui, parent, i);

        // Change the title on the Web Menu
        if (wui->widgetList[WEB_MENU] != NULL) {
            if (i > 0) {
                SbString theTitle("Web Functions (");
                theTitle += SbString(i + 1);
                theTitle += ")";
                XtSetArg(args[0], XtNtitle, theTitle.getString());
            }
            else {
                XtSetArg(args[0], XtNtitle, "Web Functions"); 
            }
            XtSetValues(wui->widgetList[WEB_MENU], args, 1);
        }
    }
}
 
////////////////////////////////////////////////////////////////////////
//
// Description:
//    Positions a menu depending on the width of the window and which
//    instance of the menu it is.
//
// Usage: static private

void
HvManager::positionMenu(HvManager *wui, const Widget &parent, int whichInstance)
//
////////////////////////////////////////////////////////////////////////
{
    int n;
    Arg args[12];
    unsigned short width, height;
    int numX, modX, modY;

    // Get the width and height of the parent
    XtSetArg(args[0], XtNwidth, &width);
    XtSetArg(args[1], XtNheight, &height);
    XtGetValues(parent, args, 2);
        
    // Figure out where to put the rowColumn.  Each rowColumn has a
    // width of 30 and a height of 30.  Add 15 pixels of padding and
    // we get 45 x 45.
    if (whichInstance <= 0) {
        modX = 0;
        modY = 0;
    }
    else {
        numX = (width - 15) / HvManager::menuWidth;
        modX = whichInstance % numX;
        modY = whichInstance / numX;
    }

    n = 0;
#ifdef IV2_0
    XtSetArg(args[n], XtNx, width - (45 + HVWebUI::menuWidth*modX)); n++;
    XtSetArg(args[n], XtNy, height - (45 + HvManager::menuHeight*modY)); n++;
#else
    XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM); n++;
    XtSetArg(args[n], XmNleftOffset, (15 + HvManager::menuWidth*modX)); n++;
    XtSetArg(args[n], XmNrightAttachment, XmATTACH_NONE); n++;
    XtSetArg(args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
    XtSetArg(args[n], XmNbottomOffset, (15 + HvManager::menuHeight*modY)); n++;
#endif
    XtSetValues(wui->widgetList[CUI_ROWCOL], args, n);
}

#ifdef IV2_0
 
////////////////////////////////////////////////////////////////////////
//
// Description:
//    Convenience routine for creating menus in the popup planes.
//
// Usage: static private

void
HvManager::getPopupArgs(Display *d, int scr, ArgList args, int *n)
//
////////////////////////////////////////////////////////////////////////
{
    iv_SG_getPopupArgs(d, scr, args, n);
}

////////////////////////////////////////////////////////////////////////
//
// register map/unmap callbacks to load/unload the given color maps
// when the pulldown menu (or popup menu) gets mapped.
//
// Use: static public

void
HvManager::registerColormapLoad(Widget w, Widget shell)
//
////////////////////////////////////////////////////////////////////////
{
    if (! w || ! shell || ! XtIsShell(shell))
        return;
    XtAddCallback(w, XmNmapCallback, (XtCallbackProc) overlayMenuMappedCB,
                  shell);
    XtAddCallback(w, XmNunmapCallback, (XtCallbackProc) overlayMenuUnmappedCB,
                  shell);
}

// 
// Called when an overlay menu (pulldown or popup) is about to be
// mapped - this will load the correct color map on the window.
//
static void
overlayMenuMappedCB(Widget w, Widget shell, XtPointer)
{
    HvManager::addColormapToShell(w, shell);
}

//
// Called when an overlay menu (pulldown or popup) is no longer
// mapped - this will unload the color map to make sure that the OGL
// overlay (really popup planes on 8 bit machine) colormap gets used.
//
static void 
overlayMenuUnmappedCB(Widget w, Widget shell, XtPointer)
{
    HvManager::removeColormapFromShell(w, shell);
} 

////////////////////////////////////////////////////////////////////////
//
// Convenience routine to load the given colormap
//
// Use: static public

void
HvManager::addColormapToShell(Widget w, Widget shell)
//  
////////////////////////////////////////////////////////////////////////
{ 
    if (! w || ! shell || ! XtIsShell(shell))
        return;
    
    // load the color map RIGH AWAY to reduce pulldown menu flickers
    Colormap map;
    XtVaGetValues(w, XmNcolormap, &map, NULL);
    XInstallColormap(XtDisplay(w), map);
    
    // check to see if there is already a property
    Window *windowsReturn;
    int countReturn;
    Status status = XGetWMColormapWindows(XtDisplay(shell),
        XtWindow(shell), &windowsReturn, &countReturn);
     
    // if no property, just create one
    if (!status)
    {
        Window windows[2];
        windows[0] = XtWindow(w);
        windows[1] = XtWindow(shell);
        XSetWMColormapWindows(XtDisplay(shell), XtWindow(shell),
            windows, 2);
    }
    // else there was a property, add myself to the beginning
    else {
        Window *windows = (Window *)XtMalloc((sizeof(Window))*
                                             (countReturn+1));
        windows[0] = XtWindow(w);
        for (int i=0; i<countReturn; i++)
            windows[i+1] = windowsReturn[i];
        XSetWMColormapWindows(XtDisplay(shell), XtWindow(shell),
            windows, countReturn+1);
        XtFree((char *)windows);
        XtFree((char *)windowsReturn);
    }
}

////////////////////////////////////////////////////////////////////////
// 
// Convenience routine to remove the given colormap
//

void
HvManager::removeColormapFromShell(Widget w, Widget shell)
//
////////////////////////////////////////////////////////////////////////
{
    if (! w || ! shell || ! XtIsShell(shell))
        return;
    
    // check to see if there is a property
    Window *windowsReturn;
    int countReturn;
    Status status = XGetWMColormapWindows(XtDisplay(shell),
        XtWindow(shell), &windowsReturn, &countReturn);
    
    // if no property, just return.  If there was a property, continue
    if (status)
    {
        // search for a match
        for (int i=0; i<countReturn; i++)
        {
            if (windowsReturn[i] == XtWindow(w))
            {
                // we found a match, now copu the rest down
                for (i++; i<countReturn; i++)
                    windowsReturn[i-1] = windowsReturn[i];
     
                XSetWMColormapWindows(XtDisplay(w), XtWindow(shell),
                                        windowsReturn, countReturn-1);
                break;  // from outer for
            }
        }
        XtFree((char *)windowsReturn);   
    }
}

#endif



////////////////////////////////////////////////////////////////////////
//
// Description:
//   Build the web menu dialog
//
void HvManager::buildWebMenuDialog()
//
////////////////////////////////////////////////////////////////////////
{
   Widget   shell = SoXt::getTopLevelWidget();
   Display  *display = XtDisplay(shell);

   int n = 0;
   Arg args[12];

   XtSetArg(args[n], XmNdeleteResponse, XmUNMAP); n++;
   XtSetArg(args[n], XtNtitle, "Web Functions"); n++;

   // Remove the Iconize/Maximize/Quit functionality
   XtSetArg(args[n], XmNmwmDecorations,
      (MWM_DECOR_BORDER | MWM_DECOR_RESIZEH | MWM_DECOR_TITLE |
       MWM_DECOR_MENU)); n++;

   XtSetArg(args[n], XmNmwmFunctions,
      (MWM_FUNC_RESIZE | MWM_FUNC_MOVE | MWM_FUNC_CLOSE)); n++;

   XtSetArg(args[n], XmNallowShellResize, TRUE); n++;

   widgetList[WEB_MENU] = XtCreateWidget("HvManagerWebFunctions",
      topLevelShellWidgetClass, shell, args, n);

   // Add callback that handles the window closing
   Atom wmDeleteAtom = XmInternAtom(display, "WM_DELETE_WINDOW", False);
      XmAddWMProtocolCallback(widgetList[WEB_MENU], wmDeleteAtom,
      (XtCallbackProc) HvManager::webMenuClosedCB, (caddr_t) this);

#ifdef ZIPPY
    // Add an event handler to receive map/unmap events on the main
    // shell window - this will allow us to hide this dialog when the
    // application iconifies.
    XtAddEventHandler(shell, StructureNotifyMask, FALSE,
        (XtEventHandler) HvManager::shellStructureNotifyCB,
        (XtPointer) this);
#endif

   // Create a form to hold everything
   Widget form = widgetList[WEB_MENU_FORM] =
      XmCreateForm(widgetList[WEB_MENU], "webMenuForm", NULL, 0);
   XtManageChild(form);


#ifdef UNDO

   // Create a form to hold the undo/redo button
   Widget rowCol2 = widgetList[WEB_UNDO_FORM] =
      XtVaCreateManagedWidget("webUndoRowCol",
         xmRowColumnWidgetClass, widgetList[WEB_MENU_FORM],
         XmNleftAttachment, XmATTACH_FORM,
         XmNleftOffset, HvUIOffset,
         XmNrightAttachment, XmATTACH_FORM,
         XmNrightOffset, HvUIOffset,
         XmNbottomAttachment, XmATTACH_WIDGET,
         XmNbottomOffset, HvUIOffset,
         XmNbottomWidget, sep1,
	 XmNpacking, XmPACK_COLUMN,
	 XmNorientation, XmHORIZONTAL,
	 XmNnumColumns, 1,
	 XmNisAligned, TRUE,
	 XmNentryAlignment, XmALIGNMENT_CENTER,
	 NULL);

   Widget undoButton = widgetList[WEB_UNDO_BUTTON] =
      XtVaCreateManagedWidget("Undo",
         xmPushButtonWidgetClass, widgetList[WEB_UNDO_FORM],
	 NULL);
   XtAddCallback(undoButton, XmNactivateCallback,
      (XtCallbackProc) HvManager::webMenuUndoCB,
      (XtPointer) this); 
   XtSetSensitive(undoButton, FALSE);

   Widget redoButton = widgetList[WEB_REDO_BUTTON] =
      XtVaCreateManagedWidget("Redo",
         xmPushButtonWidgetClass, widgetList[WEB_UNDO_FORM],
	 NULL);
   XtAddCallback(redoButton, XmNactivateCallback,
      (XtCallbackProc) HvManager::webMenuRedoCB,
      (XtPointer) this); 
   XtSetSensitive(redoButton, FALSE);

   Widget sep1 = widgetList[WEB_SEPARATOR1] =
      XtVaCreateManagedWidget("webSeparator",
         xmSeparatorWidgetClass, widgetList[WEB_MENU_FORM],
         XmNleftAttachment, XmATTACH_FORM,
         XmNleftOffset, HvUIOffset,
         XmNrightAttachment, XmATTACH_FORM,
         XmNrightOffset, HvUIOffset,
         XmNbottomAttachment, XmATTACH_WIDGET,
         XmNbottomOffset, HvUIOffset,
         XmNbottomWidget, rowCol2,
	 NULL);

#endif /* UNDO */

   // Create a form to hold the button
   Widget rowCol = widgetList[WEB_BUTTON_FORM] =
      XtVaCreateManagedWidget("webMenuRowCol",
         xmRowColumnWidgetClass, widgetList[WEB_MENU_FORM],
         XmNtopAttachment, XmATTACH_FORM,
         XmNtopOffset, HvUIOffset,
         XmNleftAttachment, XmATTACH_FORM,
         XmNleftOffset, HvUIOffset,
         XmNrightAttachment, XmATTACH_FORM,
         XmNrightOffset, HvUIOffset,
#ifdef UNDO
         XmNbottomAttachment, XmATTACH_WIDGET,
         XmNbottomOffset, HvUIOffset,
         XmNbottomWidget, sep1,
#else
         XmNbottomAttachment, XmATTACH_FORM,
         XmNbottomOffset, HvUIOffset,
#endif /* UNDO */
	 XmNpacking, XmPACK_COLUMN,
	 XmNorientation, XmVERTICAL,
	 XmNnumColumns, 1,
	 XmNisAligned, TRUE,
	 XmNentryAlignment, XmALIGNMENT_CENTER,
	 NULL);

   if (autoConnect.getValue()) {
      webMenuVisible = TRUE;
//      SoXt::show(widgetList[WEB_MENU]);
   }
   else {
      webMenuVisible = FALSE;
      SoXt::show(widgetList[WEB_MENU]);
   }
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//   Read the definitions of the web interface (asynchronously)
//
void HvManager::readWebFunctionInterfaceCB(HvManager *wui, SoSensor *)
//
////////////////////////////////////////////////////////////////////////
{
   int len, numURL = wui->urlList.getLength();
   for (int i = 0; i < numURL; i++) {
      HvURLHandler *huh = (HvURLHandler *)wui->urlList[i];

      if (huh->run("WFN_CMD=init")) { // finished
         HvWebFunction *wfn = (HvWebFunction *)wui->wfnLoadList[i];
	 huh->append("\0", 1);
         wfn->analyseURLBuffer((const char*)huh->get(len));
         wfn->buildStartButton(wui->widgetList[WEB_BUTTON_FORM]);

	 delete huh;
	 wui->urlList.remove(i);
	 wui->wfnLoadList.remove(i);
	 i--; numURL--;
      }   
   }
   
   if (numURL <= 0) {
      wui->setPixmaps(wui->widgetList[WEB_BUTTON], wui->goPixmap);
      wui->timeSensor.detach();
      return;
   }


   // variables for the blinking button (shouldn't be static !)
   static int counter = 0, blinkTime = 10;
   static SbBool phase = TRUE;

   blinkTime = 2*numURL;
   if (blinkTime > 10) blinkTime = 10;
   counter--;
   if (counter <= 0) {
      counter = blinkTime;
      if (phase) {
	 wui->setPixmaps(wui->widgetList[WEB_BUTTON], wui->goPixmap);
	 phase = FALSE;
      }
      else {
	 wui->setPixmaps(wui->widgetList[WEB_BUTTON], wui->stopPixmap);
	 phase = TRUE;
      }
   }
}

#ifdef UNDO
////////////////////////////////////////////////////////////////////////
//
// Description:
//   set the last called webFunction
//
void HvManager::registerWebFunction(HvWebFunction *wfn)
//
////////////////////////////////////////////////////////////////////////
{
   lastFunction = wfn;

   XtSetSensitive(widgetList[WEB_REDO_BUTTON], FALSE);
   XtSetSensitive(widgetList[WEB_UNDO_BUTTON], TRUE);
}
#endif /* UNDO */
