/*
 * Copyright (C) 1996   Silicon Graphics, Inc.
 *
 _______________________________________________________________________
 ______________  S I L I C O N   G R A P H I C S   I N C .  ____________
 |
 |   $Revision: 1.4 $
 |
 |   Classes:
 |      HvGraph
 |
 |   Author(s)          : Horst Vollhardt
 |
 ______________  S I L I C O N   G R A P H I C S   I N C .  ____________
 _______________________________________________________________________
 */


#include <sys/time.h>
#include <stdio.h>
#include <iostream.h>
#include <GL/gl.h>

#include <Inventor/SoLists.h>

#include "HvGraph.h"

#define HV_INDEX(a)	((a)&(HV_MAX_DATA-1))

#define HV_MSEC(a)	((a.tv_sec-time0)*1000 + a.tv_usec/1000)

SbPList *HvGraph::instanceList = NULL;

////////////////////////////////////////////////////////////////////////
//
// Description:
//    Constructor
//
// Use: public

HvGraph::HvGraph()
//
////////////////////////////////////////////////////////////////////////
{

    struct timeval tim;
    gettimeofday(&tim);
    time0 = tim.tv_sec;
    for (int i = 0; i < HV_MAX_DATA; i++) { x[i] = -1; y[i] = 0; }
    x[HV_MAX_DATA-1] = 0;
    yMaxIndex = 0;
    average = 0;

    start = 0;

    interval = -1;

    setColor(1., 1., 1.);

    if (HvGraph::instanceList == NULL)
        HvGraph::instanceList = new SbPList;

    HvGraph::instanceList->append(this);
}


////////////////////////////////////////////////////////////////////////
//
// Description:
//    Destructor
//
// Use: public

HvGraph::~HvGraph()
//
////////////////////////////////////////////////////////////////////////
{
    // Remove ourselves from the instanceList
    int index;
    if ((index = HvGraph::instanceList->find(this)) != -1) {
        HvGraph::instanceList->remove(index);
    }
}


////////////////////////////////////////////////////////////////////////
//
// Description:
//    append a new data value
//    the data array is cyclic, the first value will be removed
//
// Use: public

void HvGraph::append(long value)
//
////////////////////////////////////////////////////////////////////////
{
    struct timeval tim;
    gettimeofday(&tim);
    long msec = HV_MSEC(tim);
    int pos = start;
    int yMax = y[yMaxIndex];

    if (interval > 0) {
        msec /= interval;
        msec *= interval;
	int pos1 = HV_INDEX(start - 1);
	if (x[pos1] == msec) {
	   pos = pos1;
	   value += y[pos];
	}
    }

    average = (double)(HV_MAX_DATA * average - y[pos] + value) / HV_MAX_DATA;
    x[pos] = msec;
    y[pos] = value;

    // set new max value if necessary
    if ((pos == yMaxIndex)&&(value < yMax)) {
        // search for a new maximum
	yMax = y[0]; yMaxIndex = 0;
	for (int i = 1; i < HV_MAX_DATA; i++) {
	    if (y[i] > yMax) {
	        yMax = y[i]; yMaxIndex = i;
	    }
	}
    }
    else if (value >= yMax) { yMaxIndex = pos; }

    start = HV_INDEX(pos + 1);
}


static GLubyte digits[10][8] = {
{ 0x70, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x70 },
{ 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x60, 0x20 },
{ 0xf8, 0x80, 0x40, 0x30, 0x08, 0x08, 0x88, 0x70 },
{ 0x70, 0x88, 0x08, 0x08, 0x30, 0x08, 0x88, 0x70 },
{ 0x10, 0x10, 0xf8, 0x90, 0x50, 0x50, 0x30, 0x10 },
{ 0x70, 0x88, 0x08, 0x08, 0x70, 0x40, 0x40, 0x78 },
{ 0x70, 0x88, 0x88, 0x88, 0xf0, 0x80, 0x88, 0x70 },
{ 0x40, 0x40, 0x40, 0x20, 0x20, 0x10, 0x08, 0xf8 },
{ 0x70, 0x88, 0x88, 0x88, 0x70, 0x88, 0x88, 0x70 },
{ 0x70, 0x88, 0x08, 0x78, 0x88, 0x88, 0x88, 0x70 } };

GLuint HvGraph::fontOffset = 0;

////////////////////////////////////////////////////////////////////////
//
// Description:
//    render the graph
//
// Use: public

void HvGraph::render()
//
////////////////////////////////////////////////////////////////////////
{
    // get our number
    int nr = HvGraph::instanceList->find(this);

    // make raster font for digits
    if (HvGraph::fontOffset == 0) {
        GLint align;
	glGetIntegerv(GL_UNPACK_ALIGNMENT, &align);
        glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
        HvGraph::fontOffset = glGenLists(10);
        for (GLuint i = 0; i < 10; i++) {
            glNewList(i+HvGraph::fontOffset, GL_COMPILE);
                glBitmap(5, 8, 0., 0., 7., 0., digits[i]);
            glEndList();
        }
        glPixelStorei(GL_UNPACK_ALIGNMENT, align);
    }

    // check if we are in RGBA mode
    GLint mode;
    glGetIntegerv(GL_RGBA_MODE, &mode);
    if (!mode) return;
    
    // calculate ortho projection
    GLfloat xmin = x[start];
    GLfloat xmax = x[HV_INDEX(start - 1)];
    GLfloat ymax = y[yMaxIndex]; if (ymax == 0) ymax = 1;
    GLfloat dx = xmax - xmin;
    GLfloat dy = ymax;
    GLfloat oxmin = xmin - 0.02*dx;
    GLfloat oxmax = xmax + 0.02*dx;
    GLfloat oymin = -(1.6*(9-nr) + 0.3)*dy;
    GLfloat oymax = (1.6*nr + 1.3)*dy;
    GLfloat bxmin = xmin - 0.01*dx;
    GLfloat bxmax = xmax + 0.01*dx;
    GLfloat bymin = -0.15*dy;
    GLfloat bymax = ymax + 0.15*dy;

    // render graph
    glPushAttrib(GL_LINE_BIT    | GL_TRANSFORM_BIT | GL_ENABLE_BIT |
                 GL_CURRENT_BIT | GL_LIST_BIT      | GL_COLOR_BUFFER_BIT |
		 GL_DEPTH_BUFFER_BIT);
      glDisable(GL_LIGHTING);
      glMatrixMode(GL_MODELVIEW);
      glPushMatrix();
        glLoadIdentity();
        glMatrixMode(GL_PROJECTION);
        glPushMatrix();
          glLoadIdentity();
          glOrtho(oxmin, oxmax, oymin, oymax, -1., 1.);
	  glDisable(GL_DEPTH_TEST);
	  glDepthMask(0);
          glEnable(GL_BLEND);
	  glBlendFunc(GL_SRC_ALPHA, /*GL_ONE,*/ GL_ONE_MINUS_SRC_ALPHA);
          glEnable(GL_LINE_SMOOTH);
	  glLineWidth(1);
	  glColor4f(color[0], color[1], color[2], 1.);

	  glBegin(GL_LINE_LOOP);
	    glVertex3f(bxmin, bymin, 0.9999);
	    glVertex3f(bxmax, bymin, 0.9999);
	    glVertex3f(bxmax, bymax, 0.9999);
	    glVertex3f(bxmin, bymax, 0.9999);
	  glEnd();

	  glBegin(GL_LINE_STRIP);
	    int j = start;
	    for (int i = 0; i < HV_MAX_DATA; i++, j++) {
	       int k = HV_INDEX(j);
	       glVertex3f(x[k], y[k], 0.9999);
	    }
	  glEnd();

          // print maximum and average value
	  glColor4f(1., 1., 1., 1.);
          glListBase(HvGraph::fontOffset-'0');
          sprintf(maxText, "%ld", y[yMaxIndex]);
	  glRasterPos3f(xmin, 0.5*ymax + 0.15*dy, 0.9999);
          glCallLists(strlen(maxText), GL_UNSIGNED_BYTE, (GLubyte *) maxText);
          sprintf(maxText, "%ld", (long)average);
	  glRasterPos3f(xmin, 0.15*dy, 0.9999);
          glCallLists(strlen(maxText), GL_UNSIGNED_BYTE, (GLubyte *) maxText);

          glMatrixMode(GL_PROJECTION);
        glPopMatrix();
        glMatrixMode(GL_MODELVIEW);
      glPopMatrix();
    glPopAttrib();

}


/*
#include "aux.h"

HvGraph graph;

void myinit(void)
{
    glShadeModel(GL_FLAT);
}

void display(void)
{
    GLfloat white[3] = { 1.0, 1.0, 1.0 };
    glClear(GL_COLOR_BUFFER_BIT);
    glColor3fv(white);
    glRasterPos2i(20, 100);
    graph.render();
    glFlush ();
}

void myReshape(GLsizei w, GLsizei h)
{
    glViewport(0, 0, w, h);
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    glOrtho (0.0, w, 0.0, h, -1.0, 1.0);
    glMatrixMode(GL_MODELVIEW);
}

void main(int, char** argv)
{
    auxInitDisplayMode (AUX_SINGLE | AUX_RGBA);
    auxInitPosition (0, 0, 500, 500);
    auxInitWindow (argv[0]);
    myinit();
    auxReshapeFunc (myReshape);
    auxMainLoop(display);
}
*/
