/*
 * Copyright (C) 1996   Silicon Graphics, Inc.
 *
 _______________________________________________________________________
 ______________  S I L I C O N   G R A P H I C S   I N C .  ____________
 |
 |   $Revision: 1.3 $
 |
 |   Classes:
 |      HvClientList
 |
 |   Author(s)          : Horst Vollhardt
 |
 ______________  S I L I C O N   G R A P H I C S   I N C .  ____________
 _______________________________________________________________________
 */

#include <fcntl.h>
#include <iostream.h>
#include <malloc.h>
#include <sys/types.h>
#include <sys/prctl.h>
#include <signal.h>

#include <Inventor/SbString.h>
#include <Inventor/SoDB.h>
#include <Inventor/SoInput.h>
#include <Inventor/SoOutput.h>
#include <Inventor/SoLists.h>
#include <Inventor/actions/SoWriteAction.h>
#include <Inventor/fields/SoField.h>
#include <Inventor/nodes/SoNode.h>
#include <Inventor/nodes/SoGroup.h>
#include <Inventor/nodes/SoSeparator.h>

#include "HvClientList.h"
#include "HvManager.h"


#ifdef DEBUG2
static ViewerPID = 0; // -1 means show debugging viewer
#endif

////////////////////////////////////////////////////////////////////////
//
// Description:
//    Constructor
//
// Use: public

HvClientList::HvClientList(unsigned long id)
//
////////////////////////////////////////////////////////////////////////
{
   setId(id);
   sceneGraph = NULL;
}


////////////////////////////////////////////////////////////////////////
//
// Description:
//    Destructor
//
// Use: public

HvClientList::~HvClientList()
//
////////////////////////////////////////////////////////////////////////
{
   closeAllConnections();

   int num = bufferList.getLength();
   for (int i = 0; i < num; i++) {
      HvNetBuffer *buff = (HvNetBuffer *)bufferList[0];
      delete buff;
      bufferList.remove(0);
   }
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    write the final scene graph
//
// Use: public

void
HvClientList::writeSceneGraph(const SbString &file)
//
////////////////////////////////////////////////////////////////////////
{
   SoOutput out;
   out.openFile(file.getString());
   SoWriteAction wa(&out);
   wa.apply(sceneGraph);
}


////////////////////////////////////////////////////////////////////////
//
// Description:
//    Destructor
//
// Use: public

void
HvClientList::closeAllConnections()
//
////////////////////////////////////////////////////////////////////////
{
   int num = socketList.getLength();
   for (int i = 0; i < num; i++)
      closeConnection(socketList[i]);
}


////////////////////////////////////////////////////////////////////////
//
// Description:
//    Destructor
//
// Use: public

void
HvClientList::remove(int idx)
//
////////////////////////////////////////////////////////////////////////
{
   if ((idx < 0)||(idx >= socketList.getLength())) return;

   closeConnection(socketList[idx]);
   socketList.remove(idx);

   HvNetBuffer *buff = (HvNetBuffer *)bufferList[idx];
   delete buff;
   bufferList.remove(idx);
}


////////////////////////////////////////////////////////////////////////
//
//  make socket nonblocking
//
// Use: private

int
HvClientList::makeNonblocking(int s)
//
////////////////////////////////////////////////////////////////////////
{
   if (fcntl(s, F_SETFL, FNDELAY) < 0) {
      return -1;
   }
   return 0;
}


////////////////////////////////////////////////////////////////////////
//
//  append a client connection to the list
//
// Use: public

int
HvClientList::append(int s)
//
////////////////////////////////////////////////////////////////////////
{
   socketList.append(s);
   bufferList.append(new HvNetBuffer(s));

//   return makeNonblocking(s);
   return 0;
}


////////////////////////////////////////////////////////////////////////
//
//  read scene graph from data
//  returns -1 on error
//
// Use: public

int
HvClientList::readSceneGraph(const char *data)
//
////////////////////////////////////////////////////////////////////////
{
   if (sceneGraph != NULL) return -1;  // shouldn't be called twice

   int len = *((int *)data);

   SoInput input;
   input.setBuffer((void *)&data[S_INT], len - S_INT);
   sceneGraph = SoDB::readAll(&input);

   if (sceneGraph == NULL) {
#ifdef DEBUG2
      cerr << "\tCouldn't read buffer\n";
#endif
      return -1;
   }

   sceneGraph->ref();

#ifdef DEBUG2
   SoWriteAction wa;
   wa.apply(sceneGraph);

   if (ViewerPID != -1) return 0;
   if ((ViewerPID = sproc(viewGraph, PR_SADDR | PR_NOLIBC | PR_SFDS,
			  sceneGraph)) < 0) {
      perror("Serv(view)");
   }
#endif

   return 0;
}


////////////////////////////////////////////////////////////////////////
//
//  send scene graph to one client
//  returns -1 on error
//
// Use: public

int
HvClientList::sendSceneGraph(int s)
//
////////////////////////////////////////////////////////////////////////
{
   if (sceneGraph == NULL) return -1;  // there should be a scene graph

#ifdef DEBUG2
cerr << "Sending scene graph\n";
#endif


   SoOutput output;
   size_t size = 100;
   output.setBuffer(malloc(size), size, (SoOutputReallocCB *)realloc);
   SoWriteAction write(&output);
   write.apply(sceneGraph);

   void *buffer;
   if (! output.getBuffer(buffer, size)) {
      cerr << "Error on write (sendSceneGraph)\n";
   }

   writeBuffer.setSocket(s);
   writeBuffer.set(GRAPH_ID);
   writeBuffer.append((int)(size + S_INT));
   writeBuffer.append(buffer, size);
   int bytesSend = 0;
   while (bytesSend < writeBuffer.getLength()) {	// wait until
      bytesSend += writeBuffer.send(bytesSend);		// everything is send
   }
   writeBuffer.clear();

#ifdef DEBUG2
cerr << bytesSend << " bytes of scene graph send\n";
#endif

   return 0;
}


////////////////////////////////////////////////////////////////////////
//
//  send request for the scene graph of the client
//  returns -1 on error
//
// Use: public

int
HvClientList::requestSceneGraph(int s)
//
////////////////////////////////////////////////////////////////////////
{
   writeBuffer.setSocket(s);
   writeBuffer.set(SEND_GRAPH_ID);
   int bytesSend = 0;
   while (bytesSend < writeBuffer.getLength()) {	// wait until
      int ret = writeBuffer.send(bytesSend);		// everything is send
      if (ret > 0) bytesSend += ret;
   }
   writeBuffer.clear();

#ifdef DEBUG2
   cerr << "requestSG: " << bytesSend << " bytes sent\n";
#endif

   return 0;
}


////////////////////////////////////////////////////////////////////////
//
//  read (several) SoPath from data
//  returns -1 on error
//
// Use: public

int
HvClientList::readPath(const char *data)
//
////////////////////////////////////////////////////////////////////////
{
   if (sceneGraph == NULL) return -1;  // nothing to update

   int numNodes = *((int *)data); data += S_INT;

   for (int i = 0; i < numNodes; i++) {

      // find node in the scene graph
      int level = *((int *)data); data += S_INT;
      SoNode *node = (SoNode *)sceneGraph;

      for (int j = 0; j < level; j++) {
         int idx = *((int *)data); data += S_INT;
	 SoGroup *group = (SoGroup *)node;
         if ((node = group->getChild(idx)) == NULL) return -1;
      }

      // get fields and set values
      int numFields = *((int *)data); data += S_INT;
      SoFieldList fieldList;
      node->getFields(fieldList);
      for (j = 0; j < numFields; j++) {
         int idx = *((int *)data);  data += S_INT;
         int len = *((int *)data);  data += S_INT;
	 fieldList[idx]->set(data); data += len;
      }
   }
#ifdef DEBUG2
   if (ViewerPID > 1) kill(ViewerPID, SIGHUP);
#endif
   return 0;



/* didn't work, don't know why !

   int num = *((int *)data);
   data += S_INT;

   for (int i = 0; i < num; i++) {
      int len = *((int *)data);
#ifdef DEBUG2
         cerr << i << "> " << len << " bytes to read\n";
#endif

      SoInput input;
      SoPath  *path = NULL;
      input.setBuffer((void *)&data[S_INT], len - S_INT);
      if (SoDB::read(&input, path) == FALSE) {
#ifdef DEBUG2
         cerr << "\tError while reading path\n";
#endif
         path = NULL;
      }

      if (path == NULL) {
#ifdef DEBUG2
      cerr << "\tEmpty path ?!\n";
#endif
      }
      else {
         path->ref();
#ifdef DEBUG2
         SoWriteAction writeAction;
         writeAction.apply(path);
#endif
         // path->unref();   // don't destroy path, => Seg. Fault
      }

      data += len;
   }

   return 0;
*/
}
////////////////////////////////////////////////////////////////////////
//
// Description:
//   reads a path from buffer and returns a SoPath
//
// Use: private

SoPath*
HvClientList::readSoPath(int *buf)
//
////////////////////////////////////////////////////////////////////////
	{
	SoPath *path=new SoPath(sceneGraph);
	path->ref();
	
	int level = *(buf++);
	SoNode *node = (SoNode *)sceneGraph;

	for (int j =0; j < level; j++)
		{
		int idx = *(buf++);
		SoGroup *group = (SoGroup *)node;
		if ((node = group->getChild(idx)) == NULL) return path;
		path->append(node);
		}
	return path; 		
	}
	
////////////////////////////////////////////////////////////////////////
//
// Description:
//   reads a path from buffer and returns a SoPath
//
// Use: private

SoNode*
HvClientList::readSubGraph(int *buf)
//
////////////////////////////////////////////////////////////////////////
	{
	int len=*(buf++);
	SoInput input;
	/*
	fprintf(stderr, "buf[0]=%x\n", buf[0]);
	fprintf(stderr, "buf[1]=%x\n", buf[1]);
	fprintf(stderr, "buf[2]=%x\n", buf[2]);
	fprintf(stderr, "buf[3]=%x\n", buf[3]);
	fprintf(stderr, "Len:%d Content:%s\n", len, buf);
	*/
	input.setBuffer((void *)buf, len);
	SoNode *node = NULL;
	if (! SoDB::read(&input, node)) return 0;
	return node;
	}	
	
////////////////////////////////////////////////////////////////////////
//
// Description:
//   decodes a CHANGE_ID chain
//
// Use: private

void
HvClientList::readChange(int *buf)
//
////////////////////////////////////////////////////////////////////////
	{
 	if (sceneGraph == NULL) return;  // nothing to update
	int nofChng=*(buf++);
	SoNode *sub;
	int len;
	// process all changes
	// fprintf(stderr, "%d changed\n", nofChng);
	for (int i=0;i<nofChng;i++)
		{
		int	    cmd;
		SoPath *path;
		int		chld;
		SoGroup *grp;
		
		cmd=*(buf++);
		path=readSoPath(buf);
		buf+=buf[0]+1;
		chld=*(buf++);
		if (!path) continue;		
		grp=(SoGroup*) path->getTail();
		switch (cmd)
			{
			case TOP_INSERT_ID:
				//fprintf(stderr, "perform cmd INSERT\n");
				sub=readSubGraph(buf);
				buf+=(buf[0]+3)/4;
				grp->insertChild(sub, chld);
				//topology.update(path, 0);
				break;
			case TOP_REMOVE_ID:
				//fprintf(stderr, "perform cmd REMOVE\n");
				grp->removeChild(chld);
				//topology.update(path, 0);
				break;
			case TOP_REMOVEALL_ID:
				//fprintf(stderr, "perform cmd REMOVEALL\n");
				grp->removeAllChildren();
				//topology.update(path, 0);
				break;
			case TOP_REPLACE_ID:
				//fprintf(stderr, "perform cmd REPLACE\n");
				sub=readSubGraph(buf);
				buf+=(buf[0]+3)/4;
				grp->replaceChild(chld, sub);
				//topology.update(path, 0);
				break;
			default:
				fprintf(stderr, "Change-Cmd %d not valid\n", cmd);
				return;
			}
		}
	}

////////////////////////////////////////////////////////////////////////
//
//  send path to the clients except the given client
//  returns -1 on error
//
// Use: public

int
HvClientList::sendPath(int client, const void *data, int len)
//
////////////////////////////////////////////////////////////////////////
{
#ifdef DEBUG2
   cerr << "sending path to all but " << client << "\n";
#endif

   writeBuffer.set(data, len);

   int num = getNum();
   fd_set writeMask;
   int *bytesSend = new int[num];
   for (int i = 0; i < num; i++) bytesSend[i] = 0;

   int finished = 0;		// no. of clients finished
   while (finished < (num-1)) {
      FD_ZERO(&writeMask);
      for (int i = 0; i < num; i++) {
         if (i == client) continue;
	 if (bytesSend[i] >= len) continue;
         FD_SET(getSocket(i), &writeMask);
      }
//      int ret;
//      if ((ret = select(FD_SETSIZE, NULL, &writeMask, NULL, NULL)) > 0) {
         for (i = 0; i < num; i++) {
            if (i == client) continue;
	    if (bytesSend[i] >= len) continue;
 	    int s = getSocket(i);
//	    if (FD_ISSET(s, &writeMask)) {
               writeBuffer.setSocket(s);
               int r = writeBuffer.send(bytesSend[i]);
	       if (r > 0) {
#ifdef DEBUG2
   cerr << r << " bytes sent to " << i << "\n";
#endif

	          bytesSend[i] += r;
		  if (bytesSend[i] >= len) finished++;
	       }
//	    }
         }
//      }
   }

   delete [] bytesSend;
   return 0;
}



#ifdef DEBUG2
////////////////////////////////////////////////////////////////////////
//
// Use: private
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/nodes/SoSphere.h>

void
HvClientList::viewGraph(void *arg)
//
////////////////////////////////////////////////////////////////////////
{
   SoNode *node = (SoNode *)arg;
   Widget window = SoXt::getTopLevelWidget();
   SoXtExaminerViewer *viewer = new SoXtExaminerViewer(window);
   SoSeparator *root = new SoSeparator;
   root->addChild(node);
   viewer->setSceneGraph(root);
   viewer->show();
   SoXt::show(window);
   SoXt::mainLoop();
}
#endif
