/*
 * Copyright (C) 1996   Silicon Graphics, Inc.
 *
 _______________________________________________________________________
 ______________  S I L I C O N   G R A P H I C S   I N C .  ____________
 |
 |   $Revision: 1.1 $
 |
 |   Classes:
 |      HvClientHandler
 |
 |   Author(s)          : Horst Vollhardt
 |
 ______________  S I L I C O N   G R A P H I C S   I N C .  ____________
 _______________________________________________________________________
 */

#include <iostream.h>

#include "HvClientHandler.h"


////////////////////////////////////////////////////////////////////////
//
// Description:
//    Constructor
//
// Use: public

HvClientHandler::HvClientHandler(const char *host, int p)
               : HvClient(host, p)
//
////////////////////////////////////////////////////////////////////////
{
   state = FIRST_CALL;
   readState = FIRST_CALL;
   writeState = FIRST_CALL;
   bytesSend = 0;
   writePipeState = FIRST_CALL;
   bytesSendPipe = 0;
   connected = FALSE;
}


////////////////////////////////////////////////////////////////////////
//
// Description:
//    Destructor
//
// Use: public

HvClientHandler::~HvClientHandler()
//
////////////////////////////////////////////////////////////////////////
{
}


////////////////////////////////////////////////////////////////////////
//
// Description:
//    connects to server
//    returns TRUE if completed
//    (there are intentionally no breaks used)
//
// Use: public

SbBool
HvClientHandler::connect()
//
////////////////////////////////////////////////////////////////////////
{
   if (isConnected()) return TRUE;

   switch (state) {
      case HvClientHandler::FIRST_CALL:
         state = HvClientHandler::BINDING;

      case HvClientHandler::BINDING:
         if (bindSocket() < 0)      return FALSE;
         state = HvClientHandler::NON_BLOCKING;

      case HvClientHandler::NON_BLOCKING:
         if (makeNonblocking() < 0) return FALSE;
         state = HvClientHandler::CONNECTING;

      case HvClientHandler::CONNECTING:
         if (connectSocket() <= 0) return FALSE;
	 setSocket(getSocket());
         state = HvClientHandler::SET_BUFFER;

      case HvClientHandler::SET_BUFFER:
	 if (clear() == -1)            return FALSE;
	 if (readBuffer.clear() == -1) return FALSE;
	 if (pipeBuffer.clear() == -1) return FALSE;
         state = HvClientHandler::FINISHING;

      case HvClientHandler::FINISHING:
         connected = TRUE;
         state = HvClientHandler::FIRST_CALL;
	 return TRUE;

      case HvClientHandler::LAST_CALL:

      default:
         cerr << "You shouldn't show up here !\n";
         break;
   }

   // shouldn't show up here
   return TRUE;
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    read from server
//    returns TRUE if buffer is completed or EOF is reached
//    (there are intentionally no breaks used)
//
// Use: public

SbBool
HvClientHandler::read(SbBool binaryMode)
//
////////////////////////////////////////////////////////////////////////
{
   if (!isConnected()) return FALSE;

   int ret;
   switch (readState) {
      case HvClientHandler::FIRST_CALL:
         readState = HvClientHandler::SET_BUFFER;

      case HvClientHandler::SET_BUFFER:
         readState = HvClientHandler::CLEAR_BUFFER;

      case HvClientHandler::CLEAR_BUFFER:
      	 if (readBuffer.clear() == -1) return FALSE;
         readState = HvClientHandler::RECEIVING;

      case HvClientHandler::RECEIVING:
         ret = readBuffer.receive(binaryMode);
         if (ret < 0) {
	    return FALSE;
	 }
         else if (ret > 0) {
	    if (binaryMode) {
   	       if (!isComplete()) return FALSE;
	    }
	    else {
	       return TRUE;
	    }
	 }
	 else {
	    closeConnection();
	 }
         readState = HvClientHandler::FINISHING;

      case HvClientHandler::FINISHING:
         readState = HvClientHandler::FIRST_CALL;
	 return TRUE;

      case HvClientHandler::LAST_CALL:

      default:
         cerr << "You shouldn't show up here !\n";
         break;
   }

   // should show up here
   return TRUE;
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    send buffer to server
//    returns TRUE if session is completed
//    (there are intentionally no breaks used)
//
// Use: public

SbBool
HvClientHandler::write(SbBool dataOnly)
//
////////////////////////////////////////////////////////////////////////
{
   if (!isConnected()) return TRUE;	// nowhere to write

   writeMode = dataOnly;

   switch (writeState) {
      case HvClientHandler::FIRST_CALL:
         bytesSend = 0;
         writeState = HvClientHandler::SENDING;

      case HvClientHandler::SENDING:
         bytesSend += send(bytesSend, dataOnly);
	 if (bytesSend < getLength()) return FALSE;
         writeState = HvClientHandler::FINISHING;

     case HvClientHandler::FINISHING:
         writeState = HvClientHandler::FIRST_CALL;
	 clear();
	 return TRUE;

      case HvClientHandler::LAST_CALL:

      default:
         cerr << "You shouldn't show up here !\n";
         break;
   }

   // should show up here
   return TRUE;
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    send buffer to pipe
//    returns TRUE if session is completed
//    (there are intentionally no breaks used)
//
// Use: public

SbBool
HvClientHandler::writePipe(SbBool dataOnly)
//
////////////////////////////////////////////////////////////////////////
{
   if (!isConnected()) return TRUE;	// nowhere to write

   writePipeMode = dataOnly;

   switch (writePipeState) {
      case HvClientHandler::FIRST_CALL:
         bytesSendPipe = 0;
         writePipeState = HvClientHandler::SENDING;

      case HvClientHandler::SENDING:
         bytesSendPipe += pipeBuffer.sendPipe(bytesSendPipe, dataOnly);
		 if (bytesSendPipe < pipeBuffer.getLength()) return FALSE;
         writePipeState = HvClientHandler::FINISHING;

     case HvClientHandler::FINISHING:
         writePipeState = HvClientHandler::FIRST_CALL;
		 pipeBuffer.clear();
		 return TRUE;

      case HvClientHandler::LAST_CALL:

      default:
         cerr << "You shouldn't show up here !\n";
         break;
   }

   // should show up here
   return TRUE;
}


////////////////////////////////////////////////////////////////////////
//
// Description:
//    flush the write buffer if it is not empty
//    returns TRUE if the buffer is flushed
//    (there are intentionally no breaks used)
//
// Use: public

SbBool
HvClientHandler::flush()
//
////////////////////////////////////////////////////////////////////////
{
   if (!isConnected()) return TRUE;	// nowhere to write

   switch (writeState) {
      case HvClientHandler::FIRST_CALL:
         return TRUE;

      case HvClientHandler::SENDING:
         return write(writeMode);

      case HvClientHandler::FINISHING:
      case HvClientHandler::LAST_CALL:
      default:
	 return TRUE;
   }
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    flush the write-pipe buffer if it is not empty
//    returns TRUE if the buffer is flushed
//    (there are intentionally no breaks used)
//
// Use: public

SbBool
HvClientHandler::flushPipe()
//
////////////////////////////////////////////////////////////////////////
{
   if (!isConnected()) return TRUE;	// nowhere to write

   switch (writePipeState) {
      case HvClientHandler::FIRST_CALL:
         return TRUE;

      case HvClientHandler::SENDING:
         return writePipe(writeMode);

      case HvClientHandler::FINISHING:
      case HvClientHandler::LAST_CALL:
      default:
	 return TRUE;
   }
}
