/*
 * Copyright (C) 1997   Silicon Graphics, Inc.
 *
 _______________________________________________________________________
 ______________  S I L I C O N   G R A P H I C S   I N C .  ____________
 |
 |   $Revision: 1.2 $
 |
 |   Classes:
 |      HvChart
 |
 |   Author(s)          : Horst Vollhardt
 |
 ______________  S I L I C O N   G R A P H I C S   I N C .  ____________
 _______________________________________________________________________
 */

#include <iostream.h>

#include <Inventor/SbBox.h>
#include <Inventor/SoPrimitiveVertex.h>
#include <Inventor/actions/SoGLRenderAction.h>
#include <Inventor/bundles/SoMaterialBundle.h>
#include <Inventor/elements/SoGLTextureCoordinateElement.h>
#include <Inventor/elements/SoLightModelElement.h>
#include <Inventor/elements/SoLazyElement.h>

#include "HvChart.h"


SO_NODE_SOURCE(HvChart);

////////////////////////////////////////////////////////////////////////
//
//  Initialize the class
//  
void HvChart::initClass()
//
////////////////////////////////////////////////////////////////////////
{
    SO_NODE_INIT_CLASS(HvChart,SoShape,"Shape");
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    Constructor
//
// Use: public

HvChart::HvChart()
//
////////////////////////////////////////////////////////////////////////
{
    SO_NODE_CONSTRUCTOR(HvCHart);

    SO_NODE_ADD_FIELD(width,		(1.));
    SO_NODE_ADD_FIELD(height,		(1.));
    SO_NODE_ADD_FIELD(depth,		(1.));
    SO_NODE_ADD_FIELD(spacing,		(0.3));
    SO_NODE_ADD_FIELD(numRows,		(0));
    SO_NODE_ADD_FIELD(numColumns,	(0));
    SO_NODE_ADD_FIELD(numSlices,	(0));
    SO_NODE_ADD_FIELD(slice,		(0.));
    SO_NODE_ADD_FIELD(heights,		(0.));
    SO_NODE_ADD_FIELD(colors,		(0));
    SO_NODE_ADD_FIELD(selected,		(0));
    SO_NODE_ADD_FIELD(url,		(""));
    SO_NODE_ADD_FIELD(rowLabel,		(NULL));
    SO_NODE_ADD_FIELD(columnLabel,	(NULL));
    SO_NODE_ADD_FIELD(sectionLabel,	(NULL));
    SO_NODE_ADD_FIELD(description,	(NULL));

    cubeNorm[0][0] =  0.; cubeNorm[0][1] = -1.; cubeNorm[0][2] =  0.;
    cubeNorm[1][0] =  0.; cubeNorm[1][1] =  1.; cubeNorm[1][2] =  0.;
    cubeNorm[2][0] =  0.; cubeNorm[2][1] =  0.; cubeNorm[2][2] =  1.;
    cubeNorm[3][0] =  1.; cubeNorm[3][1] =  0.; cubeNorm[3][2] =  0.;
    cubeNorm[4][0] =  0.; cubeNorm[4][1] =  0.; cubeNorm[4][2] = -1.;
    cubeNorm[5][0] = -1.; cubeNorm[5][1] =  0.; cubeNorm[5][2] =  0.;
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    Destructor
//
// Use: public

HvChart::~HvChart()
//
////////////////////////////////////////////////////////////////////////
{
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    Compute bounding box
//
// Use: protected virtual

void
HvChart::computeBBox(SoAction *action, SbBox3f &box, SbVec3f &center)
//
////////////////////////////////////////////////////////////////////////
{
    box.setBounds(0., 0., 0.,
	width.getValue(), height.getValue(), depth.getValue());
    center.setValue(width.getValue()/2., height.getValue()/2.,
	depth.getValue()/2.);
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    Generate triangle representation of the chart
//
// Use: protected virtual

void
HvChart::generatePrimitives(SoAction *action)
//
////////////////////////////////////////////////////////////////////////
{
    SoPrimitiveVertex pv;

    SoState *state = action->getState();

    SbBool useTexFunc =
	(SoTextureCoordinateElement::getType(state) ==
	 SoTextureCoordinateElement::FUNCTION);

    const SoTextureCoordinateElement *tce;
    SbVec4f texCoord;
    if(useTexFunc)
    	tce = SoTextureCoordinateElement::getInstance(state);
    else { texCoord[2] = 0.; texCoord[3] = 0.; }

    SbVec3f  point;

#define GEN_VERTEX(pv, coord, s, t, normal)     \
      point.setValue(coord);	                \
      if (useTexFunc)                           \
         texCoord = tce->get(point, normal);    \
      else {                                    \
         texCoord[0] = s;                       \
         texCoord[1] = t;                       \
      }                                         \
      pv.setPoint(point);                       \
      pv.setNormal(normal);                     \
      pv.setTextureCoords(texCoord);            \
      shapeVertex(&pv)

    int nr = numRows.getValue();
    int nc = numColumns.getValue();
    int sl = (int)(slice.getValue() * numSlices.getValue());
    sl = (sl < numSlices.getValue()) ? sl : numSlices.getValue()-1;
    float w = width.getValue();
    float h = height.getValue();
    float d = depth.getValue();
    float space = spacing.getValue();
    float dw = w / ((float)nc - space); float cw = (1.-space) * dw;
    float dd = d / ((float)nr - space); float cd = (1.-space) * dd;
    const float *v = heights.getValues(sl*nr*nc);
    const short *c = colors.getValues(sl*nr*nc);

    cubeCoord[0][1] = cubeCoord[1][1] =
    cubeCoord[2][1] = cubeCoord[3][1] = 0.;

    float z = 0.;
    for(int i = 0; i < nr; i++, z += dd) {
	cubeCoord[0][2] = cubeCoord[1][2] =
	cubeCoord[4][2] = cubeCoord[5][2] = z;
	cubeCoord[2][2] = cubeCoord[3][2] =
	cubeCoord[6][2] = cubeCoord[7][2] = z + cd;

    	float x = 0.;
        for(int j = 0; j < nc; j++, v++, c++, x += dw) {
	    cubeCoord[0][0] = cubeCoord[2][0] =
	    cubeCoord[4][0] = cubeCoord[6][0] = x;
	    cubeCoord[1][0] = cubeCoord[3][0] =
	    cubeCoord[5][0] = cubeCoord[7][0] = x + cw;

	    cubeCoord[4][1] = cubeCoord[5][1] =
	    cubeCoord[6][1] = cubeCoord[7][1] = *v * h;

	    pv.setMaterialIndex(*c);

	    beginShape(action, TRIANGLE_FAN);
	      GEN_VERTEX(pv, cubeCoord[5], .25,  0.0, cubeNorm[2]);
	      GEN_VERTEX(pv, cubeCoord[1], .25,  0.0, cubeNorm[2]);
	      GEN_VERTEX(pv, cubeCoord[0], .25,  0.0, cubeNorm[2]);
	      GEN_VERTEX(pv, cubeCoord[4], .25,  0.0, cubeNorm[2]);
	      GEN_VERTEX(pv, cubeCoord[6], .25,  0.0, cubeNorm[1]);
	      GEN_VERTEX(pv, cubeCoord[7], .25,  0.0, cubeNorm[1]);
	      GEN_VERTEX(pv, cubeCoord[3], .25,  0.0, cubeNorm[3]);
	      GEN_VERTEX(pv, cubeCoord[1], .25,  0.0, cubeNorm[3]);
	    endShape();

	    beginShape(action, TRIANGLE_FAN);
	      GEN_VERTEX(pv, cubeCoord[2], .25,  0.0, cubeNorm[4]);
	      GEN_VERTEX(pv, cubeCoord[6], .25,  0.0, cubeNorm[4]);
	      GEN_VERTEX(pv, cubeCoord[4], .25,  0.0, cubeNorm[4]);
	      GEN_VERTEX(pv, cubeCoord[0], .25,  0.0, cubeNorm[4]);
	      GEN_VERTEX(pv, cubeCoord[1], .25,  0.0, cubeNorm[0]);
	      GEN_VERTEX(pv, cubeCoord[3], .25,  0.0, cubeNorm[0]);
	      GEN_VERTEX(pv, cubeCoord[7], .25,  0.0, cubeNorm[5]);
	      GEN_VERTEX(pv, cubeCoord[6], .25,  0.0, cubeNorm[5]);
	    endShape();
	}
    }
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    Generate triangle representation of the chart
//
// Use: protected virtual

void
HvChart::GLRender(SoGLRenderAction *action)
//
////////////////////////////////////////////////////////////////////////
{
    if(! shouldGLRender(action)) return;

    SoState *state = action->getState();

    beginSolidShape(action);
    SoMaterialBundle mb(action);
    mb.sendFirst();

    glPushAttrib(GL_ALL_ATTRIB_BITS);

    glShadeModel(GL_FLAT);
    glEnable(GL_COLOR_MATERIAL);
    glColorMaterial(GL_FRONT, GL_DIFFUSE);


    SbBool sendNormals =
	(SoLightModelElement::get(state) !=
	 SoLightModelElement::BASE_COLOR);

    int nr = numRows.getValue();
    int nc = numColumns.getValue();
    int sl = (int)(slice.getValue() * numSlices.getValue());
    sl = (sl < numSlices.getValue()) ? sl : numSlices.getValue()-1;
    float w = width.getValue();
    float h = height.getValue();
    float d = depth.getValue();
    float space = spacing.getValue();
    float dw = w / ((float)nc - space); float cw = (1.-space) * dw;
    float dd = d / ((float)nr - space); float cd = (1.-space) * dd;
    const float *v = heights.getValues(sl*nr*nc);
    const short *c = colors.getValues(sl*nr*nc);

    cubeCoord[0][1] = cubeCoord[1][1] =
    cubeCoord[2][1] = cubeCoord[3][1] = 0.;

    float z = 0.;
    for(int i = 0; i < nr; i++, z += dd) {
	cubeCoord[0][2] = cubeCoord[1][2] =
	cubeCoord[4][2] = cubeCoord[5][2] = z + cd;
	cubeCoord[2][2] = cubeCoord[3][2] =
	cubeCoord[6][2] = cubeCoord[7][2] = z;

    	float x = 0.;
        for(int j = 0; j < nc; j++, v++, c++, x += dw) {
	    cubeCoord[0][0] = cubeCoord[2][0] =
	    cubeCoord[4][0] = cubeCoord[6][0] = x;
	    cubeCoord[1][0] = cubeCoord[3][0] =
	    cubeCoord[5][0] = cubeCoord[7][0] = x + cw;

	    cubeCoord[4][1] = cubeCoord[5][1] =
	    cubeCoord[6][1] = cubeCoord[7][1] = *v * h;

	    const SbColor color = SoLazyElement::getDiffuse(state, *c);
	    glColor3fv(color.getValue());

	    glBegin(GL_TRIANGLE_FAN);
	      if(sendNormals) glNormal3fv(cubeNorm[3]);
	      glVertex3fv(cubeCoord[5]);
	      glVertex3fv(cubeCoord[1]);
	      glVertex3fv(cubeCoord[3]);
	      glVertex3fv(cubeCoord[7]);
	      if(sendNormals) glNormal3fv(cubeNorm[1]);
	      glVertex3fv(cubeCoord[6]);
	      glVertex3fv(cubeCoord[4]);
	      if(sendNormals) glNormal3fv(cubeNorm[2]);
	      glVertex3fv(cubeCoord[0]);
	      glVertex3fv(cubeCoord[1]);
	    glEnd();


	    glBegin(GL_TRIANGLE_FAN);
	      if(sendNormals) glNormal3fv(cubeNorm[4]);
	      glVertex3fv(cubeCoord[2]);
	      glVertex3fv(cubeCoord[6]);
	      glVertex3fv(cubeCoord[7]);
	      glVertex3fv(cubeCoord[3]);
	      if(sendNormals) glNormal3fv(cubeNorm[0]);
	      glVertex3fv(cubeCoord[1]);
	      glVertex3fv(cubeCoord[0]);
	      if(sendNormals) glNormal3fv(cubeNorm[5]);
	      glVertex3fv(cubeCoord[4]);
	      glVertex3fv(cubeCoord[6]);
	    glEnd();

	}
    }
    glPopAttrib();

    endSolidShape(action);
}
