/*
 * This file contains the MVPANECLASS window class code for the example player.
 */

#include <stdlib.h>
#include <windows.h>
#include <windowsx.h>
#include <commdlg.h>
#include <string.h>
#include <stdarg.h>

/* MediaView Include Files */
#include <medv14.h>

/* Application Include Files */
#include "topic.h"
#include "player.h"
#include "pane.h"
#include "resrc1.h"
#include "proto.h"

/* Global Variables for Panes */
HWND hCaptureWnd = 0;                           /* initially un-captured */

/* are we currently selecting? */
int selecting = FALSE;
HWND hSelectCapture = 0;

/*****************************************************************************
 **	FUNCTION: Pane_Register													**
 **	PURPOSE: Register the window class for the Pane							**
 **	COMMENTS:																**
 **		The MVPaneClass contains extra bytes for holding Pane data.			**
 **		See pane.h for details.												**
 *****************************************************************************/
BOOL Pane_Register(HINSTANCE hInstance)
	{
	WNDCLASS wc;

	/* create the Pane class */
	wc.style = CS_DBLCLKS;
	wc.lpfnWndProc = PaneWndProc;
	wc.cbClsExtra = 0;
	wc.cbWndExtra = EXTRA_PANE_SIZE;
	wc.hInstance = hInstance;
  	wc.hIcon = LoadIcon (hInstance, "myicon");
	wc.hCursor = 0;
	wc.hbrBackground = GetStockObject(WHITE_BRUSH); 
	wc.lpszMenuName =  0;
	wc.lpszClassName = MVPANECLASS;
	if (RegisterClass(&wc) == 0)
		return(FALSE);

	return(TRUE);
	}

/*****************************************************************************
 **	FUNCTION: Pane_Create													**
 **	PURPOSE: Create a new Pane.												**
 **	COMMENTS:																**
 **		This call creates a MediaView (MV) and a window for displaying		**
 **		it.																	**
 *****************************************************************************/
HWND Pane_Create(HWND hParentWnd, HINSTANCE hInstance, HTITLE hTitle, int type, int paneFlags, PANEHOTPROC fpHotProc)
	{
	HWND hWnd;
	LPMV lpMV;
	ERR err;
	LPPANE lpP;

	/* Create the PANE Window */
	hWnd = CreateWindowEx(
#if defined(WIN32) && !defined(NT350)
			WS_EX_CLIENTEDGE,
#else
			0,
#endif
			MVPANECLASS,
			"Pane",
			WS_CHILD|WS_VISIBLE|WS_HSCROLL|WS_VSCROLL,
			CW_USEDEFAULT,
			CW_USEDEFAULT,
			CW_USEDEFAULT,
			CW_USEDEFAULT,
			hParentWnd,
			0,
			hInstance,
			0 );
	if (hWnd == 0)
		return(0);

	/* make it initially invisible without scroll bars */
	ShowWindow(hWnd, SW_HIDE);
	ShowScrollBar(hWnd, SB_BOTH, FALSE);

	/* now Create the MediaView (MV) */
	lpMV = lpMVNew(hTitle, &err);
	if (lpMV == 0)
		return(0);

	/* bind MV to the Window handle */
	if (!fMVSetWindow(lpMV, hWnd, &err))
		return(0);

	/* allocate a structure to store the PANE data */
	if ((lpP = (LPPANE)GlobalAllocPtr(GHND, sizeof(PANE))) == 0)
		{
		MVDelete(lpMV);
		DestroyWindow(hWnd);
		return(0);
		}

	/* store away the MV and the type */
	SetPaneMV(hWnd, lpMV);
	SetPaneData(hWnd, lpP);

	/* initialize PANE data */
	lpP->iType =  type;
	lpP->flags = paneFlags;
	
	/* if there is no Hotspot callback pointer, use the default */
	if (fpHotProc == 0)
		fpHotProc = Pane_Hotspot;
	lpP->fpHotProc = fpHotProc;

	/*
	 * The Windows GetParent function will make the main window the parent
	 * of the first Popup.  We need the parent to be the base Pane,
	 * so we track all of this outside Windows.
	 */
	lpP->hWndParent =  hParentWnd;

	/* allow selection of embedded windows */
	fMVAllowEWSelection(lpMV, TRUE);

	return(hWnd);
	}


/*****************************************************************************
 **	FUNCTION: Pane_Destroy													**
 **	PURPOSE: Destroy the Pane.												**
 **	COMMENTS:																**
 *****************************************************************************/
void Pane_Destroy(HWND hWnd)
	{
	LPMV lpMV = GetPaneMV(hWnd);
	LPPANE lpP = GetPaneData(hWnd);
	HTHLITE hHits;
	
	if (lpMV)
		{
		hHits = hMVGetHighlights(lpMV);

		/* undo any highlights */
		if (hHits)
			{
			GlobalFree(hHits);
			hHits = 0;
			}
		}

	if (lpP)
		GlobalFreePtr(lpP);

	/* destroy the MV */
	if (lpMV)
		MVDelete(lpMV);

	/* and destroy the window */
	DestroyWindow(hWnd);

	return;
	}

/*****************************************************************************
 **	FUNCTION: PaneWndProc													**
 **	PURPOSE: Process messages sent to the Pane.								**
 **	COMMENTS:																**
 *****************************************************************************/
long WINAPI PaneWndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
	{
	PAINTSTRUCT ps;
	HDC hDC;
	LPPANE lpP = GetPaneData(hWnd);

	switch (message) 
		{

		case WM_KEYDOWN:
			Pane_KeyDown(hWnd, wParam, lParam);
			break;

		case WM_SIZE:
			/* if sizing is not suppressed, do the layout */
			if (lpP && !(lpP->flags & P_NOSIZE))
				Pane_Layout(hWnd, 0);
			break;

		case WM_PAINT:
			/* draw the Pane on the display */
			hDC = BeginPaint (hWnd, &ps);
        	Pane_Draw(hWnd, hDC, &(ps.rcPaint));
			EndPaint (hWnd, &ps);
			break;

		case WM_MOUSEMOVE:
			Pane_MouseMove(hWnd, wParam, lParam);
			break;

		case WM_LBUTTONUP:
			Pane_MouseLButtonUp(hWnd, wParam, lParam);
			break;
	/*******************************************************/
	/******** TUTORIAL: Insert Lesson 22 code here. ********/
	/*******************************************************/
			
		default:
			return (DefWindowProc(hWnd, message, wParam, lParam));
		}
	return (0);
	}

/*****************************************************************************
 **	FUNCTION: Pane_Draw														**
 **	PURPOSE: Repaint the pane.												**
 **	COMMENTS:																**
 *****************************************************************************/
int Pane_Draw(HWND hWnd, HDC hDC, LPRECT lpR)
	{
	ERR err;
	RECT rect;
	LPMV lpMV = GetPaneMV(hWnd);
	LPPANE lpP = GetPaneData(hWnd);

	if (lpMV == 0)
		return(ERR_FAILED);

	if (lpR == NULL)
		{
		GetClientRect (hWnd, &rect);
		lpR = &rect;
		}
	if (!fMVApplyToDC (lpMV, hDC, lpR, &err))
		return(err);

	/* the non-scrolling region needs a border at the bottom */
	if (lpP->iType == NSR_PANE && fMVHasSR(lpMV))
		{
		/* be sure to use the whole window rect, not just the update rect */
		GetClientRect (hWnd, &rect);
		MoveToEx(hDC, rect.left, rect.bottom-1, NULL);
		LineTo(hDC, rect.right, rect.bottom-1);
		}
	return(0);
	}

/*****************************************************************************
 **	FUNCTION: Pane_Layout													**
 **	PURPOSE: Realize the Pane ... this does not actually paint, that		**
 **		is done handling the WM_PAINT message.								**
 **	COMMENTS:																**
 *****************************************************************************/
void Pane_Layout(HWND hWnd, LPRECT lpR)
	{
	LPMV lpMV = GetPaneMV(hWnd);
	LPPANE lpP = GetPaneData(hWnd);
	ERR err;

	if (lpMV == 0)
		return;

	/*******************************************************/
	/******** TUTORIAL: Insert Lesson 22 code here. ********/
	/*******************************************************/
		{
		/* this does the MediaView layout of the subTopic */
		fMVRealize(lpMV, lpR, (LPERR)&err);
		}

	return;
	}

/*****************************************************************************
 **	FUNCTION: Pane_CheckHorz												**
 **	PURPOSE: Check whether horizontal scrollbar is needed.					**
 **	COMMENTS:																**
 **		MediaView only does a layout on what is visible in the window, so	**
 **		as we scroll vertically we might uncover the need for a horizontal	**
 **		scrollbar later in the topic. For example, a wide table in the		**
 **		middle of text is only recognized to need scrollbars when it gets a	**
 **		layout. This code checks for just such a condition and adds a HORZ	**
 **		scroll bar if needed.												**
 *****************************************************************************/
void Pane_CheckHorz(HWND hWnd, LPMV lpMV)
	{
	int x, minx;
	POINT pt;
	ERR err;

	/*******************************************************/
	/******** TUTORIAL: Insert Lesson 22 code here. ********/
	/*******************************************************/
	}

	 
/*****************************************************************************
 **	FUNCTION: Pane_ScrollVert												**
 **	PURPOSE: handle vertical scrolling.										**
 **	COMMENTS:																**
 **		these messages only come to a scrolling Pane						**
 *****************************************************************************/
void Pane_ScrollVert(HWND hWnd, WPARAM wParam, LPARAM lParam)
	{
	LPMV lpMV = GetPaneMV(hWnd);
	LPPANE lpP = GetPaneData(hWnd);
	POINT pt;
	ERR err;
	int y;
	int yTop = 0, yBottom = 0;
	RECT rc;
	WORD pos, code;
	
	/* make sure it is a scrolling Pane */
	if (lpMV == 0 ||lpP->iType != SR_PANE)
		return;
	
	code = GET_WM_VSCROLL_CODE(wParam, lParam);
	pos = GET_WM_VSCROLL_POS(wParam, lParam);

	switch (code)
		{
		case SB_TOP:
		case SB_BOTTOM:
		case SB_LINEUP:
		case SB_LINEDOWN:
		case SB_PAGEUP:
		case SB_PAGEDOWN:
	/*******************************************************/
	/******** TUTORIAL: Insert Lesson 22 code here. ********/
	/*******************************************************/
			break;

		case SB_THUMBPOSITION:
	/*******************************************************/
	/******** TUTORIAL: Insert Lesson 22 code here. ********/
	/*******************************************************/
			break;

		case SB_THUMBTRACK:
	/*******************************************************/
	/******** TUTORIAL: Insert Lesson 22 code here. ********/
	/*******************************************************/
			break;
		}

	}                
	
/*****************************************************************************
 **	FUNCTION: Pane_ScrollHorz												**
 **	PURPOSE: handle horizontal scrolling (these messages only come			**
 **	to the scrolling region.												**	
 **	COMMENTS:																**
 *****************************************************************************/
void Pane_ScrollHorz(HWND hWnd, WPARAM wParam, LPARAM lParam)
	{
	LPMV lpMV = GetPaneMV(hWnd);
	LPPANE lpP = GetPaneData(hWnd);
	POINT pt;
	int err;
	int x;
	WORD pos, code;
	
	/* make sure it is a scrolling Pane */
	if (lpMV == 0 || lpP->iType != SR_PANE)
		return;
	
	code = GET_WM_HSCROLL_CODE(wParam, lParam);
	pos = GET_WM_HSCROLL_POS(wParam, lParam);

	switch (code)
		{
		case SB_TOP:
		case SB_BOTTOM:
		case SB_LINEUP:
		case SB_LINEDOWN:
		case SB_PAGEUP:
		case SB_PAGEDOWN:
	/*******************************************************/
	/******** TUTORIAL: Insert Lesson 22 code here. ********/
	/*******************************************************/
			break;

		case SB_THUMBPOSITION:
			/* set the horizontal scroll position in MediaView */
	/*******************************************************/
	/******** TUTORIAL: Insert Lesson 22 code here. ********/
	/*******************************************************/
			break;

		case SB_THUMBTRACK:
	/*******************************************************/
	/******** TUTORIAL: Insert Lesson 22 code here. ********/
	/*******************************************************/
			break;
		}
	}

/*****************************************************************************
 **	FUNCTION: Pane_ScrollRanges												**
 **	PURPOSE: Calculate the new scroll ranges								**
 **	COMMENTS:																**
 **		Adding a vertical scrollbar uses up horizontal space, and vice		**
 **		versa. When a scroll bar is added, check whether this creates a		**
 **		need for one in the other direction.								**
 *****************************************************************************/
void Pane_ScrollRanges(LPMV lpMV)
	{
	POINT pt;
	int x, y;
	HWND hWnd = hwndMVGetWindow(lpMV);
	int flags;
	int didVert = FALSE;
	ERR err;
	LPPANE lpP = GetPaneData(hWnd);

	/*******************************************************/
	/******** TUTORIAL: Insert Lesson 22 code here. ********/
	/*******************************************************/
	}

/*****************************************************************************
 **	FUNCTION: Pane_MouseMove												**
 **	PURPOSE: handle the changing of the cursor over hotspots				**
 **	COMMENTS:																**
 *****************************************************************************/
void Pane_MouseMove(HWND hWnd, WPARAM wParam, LPARAM lParam)
	{
	POINT pt;
	MVPOINTINFO mvpi;
	LPMV lpMV = GetPaneMV(hWnd);
	LPPANE lpP = GetPaneData(hWnd);
	HINSTANCE hInst = GetPaneInstance(lpMV);
	ERR err;
	extern HCURSOR curT, curB, curE, curU, curArrow, curIbeam;

	if (lpMV == 0)
		return;

#ifdef WIN32
	pt.x = MAKEPOINTS(lParam).x;
	pt.y = MAKEPOINTS(lParam).y;
#else
	pt.x = LOWORD(lParam);
	pt.y = HIWORD(lParam);
#endif


	/*
	 * If we are positioned over a hotspot, set the cursor
	 * according to the hotspot type.  Otherwise, use the IBEAM over
	 * text, or the ARROW over everything else.
	 */
	mvpi.dwSize = sizeof(mvpi);
	if (fMVGetPointInfo (lpMV, pt, &mvpi, NULL, &err))
		{
		switch (mvpi.ct)
			{
			case CONTENT_TEXT:
				SetCursor(curT);
				break;
			case CONTENT_BITMAP:
				SetCursor(curB);
				break;
			case CONTENT_WINDOW:
				SetCursor(curE);
				break;
			default:
				/* includes CONTENT_UNKNOWN */
				SetCursor(curU);
				break;
			}
		}
	else
		{
		/* not a hot spot ... is it over text? */
		if (mvpi.ct == CONTENT_TEXT)
			SetCursor(curIbeam);
		else
			SetCursor(curArrow);
		}
	}

/*****************************************************************************
 **	FUNCTION: Pane_MouseLButtonUp											**
 **	PURPOSE: handle mouse up												**
 **	COMMENTS:																**
 **		This is a completed "click", so tell MediaView.It also				**
 **		completes a selection, so process that too.							**
 *****************************************************************************/
void Pane_MouseLButtonUp(HWND hWnd, WPARAM wParam, LPARAM lParam)
	{
	POINT pt;
	LPMV lpMV = GetPaneMV(hWnd);
	LPPANE lpP = GetPaneData(hWnd);
	ERR err;
	char buff[256];		/* NOTE: 256 is the longest allowed hotspot string. */
	MVHOTINFO mvhi;
	MVPOINTINFO mvpi;

	if (lpMV == 0)
		return;

#ifdef WIN32
	pt.x = MAKEPOINTS(lParam).x;
	pt.y = MAKEPOINTS(lParam).y;
#else
	pt.x = LOWORD(lParam);
	pt.y = HIWORD(lParam);
#endif

 
		/* get hotspot data if it exists */
		mvpi.dwSize = sizeof(mvpi);
		mvhi.dwSize = sizeof(mvhi);
		mvhi.lpBuffer = buff;
	  	mvhi.dwBufSize = sizeof(buff);
		fMVGetPointInfo (lpMV, pt, &mvpi, &mvhi, &err);

		/* if we are hot and there was enough space to copy the hotspot data */
		if (mvhi.dwBufSize >= mvhi.dwSizeNeeded)
			{
			/* process the hotspot data in the installable hotspot handler */
			(lpP->fpHotProc)(lpMV, (LPMVHOTINFO)&mvhi);
			}
	}



/*****************************************************************************
 **	FUNCTION: Pane_Hotspot													**
 **	PURPOSE: This is the default PANE hotspot handler hotspot handler		**
 **	COMMENTS:																**
 **		This is what is called if the Pane_Open was passed a NULL			**
 **		pointer for the fpHotspotProc argument.								**
 *****************************************************************************/
BOOL CALLBACK Pane_Hotspot(LPMV lpMV, LPMVHOTINFO lpHI)
	{
	VA va;
	HTITLE hTitle;
	HWND hWnd = hwndMVGetWindow(lpMV);

	if (lpMV == 0)
		return(FALSE);
	
	switch (lpHI->nType)
		{

		case HOTSPOT_HASH:
			/* convert the hash data into a VA */
			hTitle = hMVGetTitle(lpMV, NULL);
			va = vaMVConvertHash(hTitle, (HASH)(lpHI->hash));

			Pane_SetAddress(hWnd, va, 0);
			Pane_Layout(hWnd, 0);
			InvalidateRect(hWnd, 0, TRUE);
			break;

		case HOTSPOT_POPUPHASH:
			break;

		case HOTSPOT_STRING:
			/* put string hotspot handling here */
			break;
			
		case HOTSPOT_UNKNOWN:
		default:
			/* handle unknown hotspots here */
			return(FALSE);
			break;
		}
	return(TRUE);
	}
 
/*****************************************************************************
 **	FUNCTION: Pane_SetAddress												**
 **	PURPOSE: Set the MediaView address.										**
 **	COMMENTS:																**
 *****************************************************************************/
int Pane_SetAddress(HWND hWnd, VA va, long scroll)
	{
	ERR err;
	LPMV lpMV = GetPaneMV(hWnd);
	LPPANE lpP = GetPaneData(hWnd);
	int iSubTopic;

	/* Set the address. We retrieve the subTopic from the hWnd */
	iSubTopic = lpP->iType;
	if (!fMVSetAddress(lpMV, va, iSubTopic, scroll, (LPERR)&err))
		{
			return(err);
		}

	/* the display update is done from the calling routine */
	return(0);
	}
 
/*****************************************************************************
 **	FUNCTION: Pane_Popup													**
 **	PURPOSE: manifest a topic in a POPUP window								**
 **	COMMENTS:																**
 **		Popups will cascade. If the mouse is clicked outside an active		**
 **		popup, this code will clean them up back to the window with the		**
 **		click.																**
 *****************************************************************************/
int Pane_Popup(LPMV lpParentMV, LPRECT lpR, VA va)
	{


	return(TRUE);
	}

/*****************************************************************************
 **	FUNCTION: Pane_CloseAllPopups											**
 **	PURPOSE: If any Popups are in effect, close them.						**
 **	COMMENTS:																**
 *****************************************************************************/
HWND Pane_CloseAllPopups(HWND hWnd)
	{
	return(hWnd);
	}



/*****************************************************************************
 **	FUNCTION: Pane_KeyDown													**
 **	PURPOSE: Process the keydown message.									**
 **	COMMENTS:																**
 *****************************************************************************/
void Pane_KeyDown(HWND hWnd, WPARAM wParam, LPARAM lParam)
	{
	ERR err;
	LPMV lpMV = GetPaneMV(hWnd);
	LPPANE lpP = GetPaneData(hWnd);
	
	switch (wParam)
		{
		case VK_TAB:
			/* Tabbing through hot spots */
			fMVMoveFocus(lpMV, !(GetKeyState(VK_SHIFT) & 0x8000), &err);
			break;

		case VK_RETURN:
			{
			/* simulate "clicking" on a hot spot */
			BYTE  buff[256];			/* 256 is longest hotspot string */	
			MVHOTINFO mvhi;

			mvhi.dwSize = sizeof(mvhi);
			mvhi.lpBuffer = buff;
		  	mvhi.dwBufSize = sizeof(buff);

			/* if we are hot and there was enough space to copy the hotspot data */
			if (fMVFocusInfo (lpMV, &mvhi, &err) && (mvhi.dwBufSize >= mvhi.dwSizeNeeded))
				{
				(lpP->fpHotProc)(lpMV, (LPMVHOTINFO)&mvhi);
				}
			break;
			}


	/*******************************************************/
	/******** TUTORIAL: Insert Lesson 22 code here. ********/
	/*******************************************************/

		}
	}


/*****************************************************************************
 **	FUNCTION: Pane_ProportionScroll											**
 **	PURPOSE: Create Windows95 proportional thumb on the scrollbar.			**
 **	COMMENTS:																**
 *****************************************************************************/
void Pane_ProportionScroll(HWND hWnd, int iType)
	{

/* if you are using NT 3.50 or earlier, be sure to define NT350 for the preprocessor */
#if defined(WIN32) && !defined(NT350)
	RECT rc;
	SCROLLINFO si;
	int iTopic,iRange, iPage;
	POINT ptRange, ptTopic;
	LPMV lpMV = GetPaneMV(hWnd);
	
	/*******************************************************/
	/******** TUTORIAL: Insert Lesson 22 code here. ********/
	/*******************************************************/
#endif
	}
