/*
 * The ANNOTATION data base. 
 *
 * Application must set the lpfnHotspotCallback.
 * 
 * To add an annotation:
 *     Application calls AddAnnotation()
 *     Application calls hMVInsertEmbeddedWindow  (MediaView)
 * At topic entry
 *     Application calls GetAnnotations N+1 times
 *     Repeat N times
 *        Application calls hMVInsertEmbeddedWindow (MediaView)
 * At topic exit
 *     Application calls hMVRemoveInsertedEW
 * To Delete
 *     Delete the annotation from the database
 *     Call hMVRemoveInsertedEW
 *     Application then executes topic entry code above to reinsert the ones left
 *     
 */
  
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <windows.h>
#include <windowsx.h>
#include <medv14.h>
#include <mvew14.h>
#include "topic.h"
#include "player.h"
#include "pane.h"
#include "annotate.h"
#include "resrc1.h"
#include "proto.h"

#ifndef WIN32
#define FILE_BEGIN 0
#define FILE_CURRENT 1
#define FILE_END 2
#endif

/*
 * These global variables are used throughout the annotation database.
 */
long CurrentTopicNum = 0;
VA CurrentVa = 0;
DWORD CurrentLch = 0;
LPMV CurrentMV = 0;
int CurrentSelection = 0;
extern HCURSOR curArrow;
 

/****************************************************************************
 **     FUNCTION: AnnotateDlgProc                                          **
 **     PURPOSE: Manipulate the annotation text.                           **
 **     COMMENTS:                                                          **
 ****************************************************************************/
BOOL WINAPI AnnotateDlgProc(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
	{
	UINT cmd;
	long topicNum;
	VA va;
	DWORD lch;
	long offset, nextOffset;
	LPSTR szTopicName, szTopicText;
	int item = 0;
	static int changedAnn = FALSE;
	extern LPTOPIC lpTopic;

	switch (message) 
		{
		case WM_INITDIALOG:
			changedAnn = FALSE;
			offset = 0;
			topicNum = ENDMARK;
			while ((nextOffset = GetNextRecord(hfAnnotation, &offset, &topicNum, &va, &lch)) != ENDMARK)
				{
				ReadRecord(hfAnnotation, offset, &szTopicName, &szTopicText);
				szTopicName = CheckDuplicate(GetDlgItem(hDlg, IDC_LIST1), szTopicName);
				SendDlgItemMessage(hDlg, IDC_LIST1, LB_ADDSTRING, 0, (long)(LPSTR)szTopicName);
				
				/*
				 * Is this the selected annotation?  Large records
				 * might take a while, so change to WAIT cursor.
				 */
				if (topicNum == CurrentTopicNum && va == CurrentVa && lch == CurrentLch)
					{
					SetCursor(LoadCursor(0, IDC_WAIT));
					SendDlgItemMessage(hDlg, IDC_LIST1, LB_SETCURSEL, item, 0);
					SendDlgItemMessage(hDlg, IDC_EDIT1, WM_SETTEXT, 0, (long)szTopicText);
					CurrentSelection = item;
					SetCursor(curArrow);
					}
				++item;
				topicNum = ENDMARK;
					
				/* clean up allocated memory */	
				GlobalFreePtr(szTopicName);
				GlobalFreePtr(szTopicText);
				szTopicName = szTopicText = 0;
				offset = nextOffset;
				}
			SetFocus(GetDlgItem(hDlg, IDC_EDIT1));
			break;

		case WM_CLOSE:
			EndDialog(hDlg, FALSE);
			return(TRUE);
			break;

		case WM_COMMAND:
			cmd = GET_WM_COMMAND_ID(wParam, lParam);
			switch (cmd)
				{
				case IDOK:
					if (changedAnn)
						CatchEdits(hDlg);
					changedAnn = FALSE;
					EndDialog(hDlg, TRUE);
					return(TRUE);

				case IDC_DELETE:
					ReadRecordNum(hfAnnotation, CurrentSelection, &topicNum, &va, &lch, 0, &szTopicText);
					DeleteRecord(hfAnnotation, topicNum, va, lch);
					SendDlgItemMessage(hDlg, IDC_LIST1, LB_DELETESTRING, CurrentSelection, 0);
					if (CurrentSelection != 0)
						--CurrentSelection;
					SendDlgItemMessage(hDlg, IDC_LIST1, LB_SETCURSEL, CurrentSelection, 0);
					MV_ReDisplay(lpTopic);
					EndDialog(hDlg, FALSE);
					return(TRUE);

				case IDC_LIST1:
					if (GET_WM_COMMAND_CMD(wParam, lParam) == LBN_SELCHANGE)
						{
						/* catch any text changes */
						if (changedAnn)
							CatchEdits(hDlg);
						changedAnn = FALSE;

						/*
						 * Update the display to the new record. Large records
						 * might take a while, so change to WAIT cursor.
						 */
						SetCursor(LoadCursor(0, IDC_WAIT));
						CurrentSelection = (int)SendDlgItemMessage(hDlg, IDC_LIST1, LB_GETCURSEL, 0, 0);
						ReadRecordNum(hfAnnotation, CurrentSelection, &topicNum, &va, &lch, 0, &szTopicText);
						SendDlgItemMessage(hDlg, IDC_EDIT1, WM_SETTEXT, 0, (long)szTopicText);
						GlobalFreePtr(szTopicText);
						szTopicText = 0;
						SetCursor(curArrow);
						break;
						}
					else if (GET_WM_COMMAND_CMD(wParam, lParam) != LBN_DBLCLK)
						break;
					/* drop through to Go To on double click */;
				
				case IDC_GOTO:
					/* save any changed text */
					if (changedAnn)
						CatchEdits(hDlg);
					changedAnn = FALSE;

					/* change the topic in the TOPIC window */
					ReadRecordNum(hfAnnotation, CurrentSelection, &topicNum, &va, &lch, 0, 0);
					MV_AddToHistory(Topic_ValidMV(lpTopic));
					MV_ChangeAddress(0, lpTopic, va, lch);					
					EndDialog(hDlg, FALSE);
					return(TRUE);

				case IDC_EDIT1:
					/* track whether we actually change anything */
					if (GET_WM_COMMAND_CMD(wParam, lParam) == EN_CHANGE)
						changedAnn = TRUE;
					break;
				}
			break;
		}
	return (FALSE);
	}

/****************************************************************************
 **     FUNCTION: CatchEdits                                               **
 **     PURPOSE: Get the text from the IDC_EDIT1 into the database.        **
 **     COMMENTS:                                                          **
 ****************************************************************************/
 void CatchEdits(HWND hDlg)
 	{
	LPSTR szTopicText, szTopicName;
	long topicNum;
	VA va;
	DWORD lch;
	int size;

	size = (int)SendDlgItemMessage(hDlg, IDC_EDIT1, WM_GETTEXTLENGTH, 0, 0);
	szTopicText = GlobalAllocPtr(GHND, size+1);

	SendDlgItemMessage(hDlg, IDC_EDIT1, WM_GETTEXT, size+1, (long)szTopicText);
	szTopicText[size] = 0;
	ReadRecordNum(hfAnnotation, CurrentSelection, &topicNum, &va, &lch, &szTopicName, 0);
	WriteRecord(hfAnnotation, topicNum, va, lch, szTopicName, szTopicText);
	GlobalFreePtr(szTopicText);
	GlobalFreePtr(szTopicName);
	szTopicText = szTopicName = 0;
	}
	   
	
/****************************************************************************
 **     FUNCTION: CheckDuplicate                                           **
 **     PURPOSE: If there is already an annotation for this topic, add an  **
 **        extension.                                                      **
 **     COMMENTS:                                                          **
 ****************************************************************************/
LPSTR CheckDuplicate(HWND hList, LPSTR szTopicName)
	{
	char buff[256];
	int index = (int)SendMessage(hList, LB_GETCOUNT, 0, 0);
	LPSTR p, q;
	int len, num;
	
	while (index != 0)
		{
		len = strlen(szTopicName);
		SendMessage(hList, WM_GETTEXT, index-1, (long)buff);
		if (strncmp(buff, szTopicName, len) == 0)
			{
			/* yes ... there already is one */
			p = GlobalAllocPtr(GHND, len + 5);
			
			/* is this one a duplicate too? Look for [<number>] at end */
			q = strrchr(buff, '[');
			num = 2;
			if (q && q[2] == ']' && q[3] == 0)
				num = (int)string2DW(q+1) + 1;
	
			/* format up the new instance with a number at the end */
			wsprintf(p, "%s[%d]", szTopicName, num);
			GlobalFreePtr(szTopicName);
			szTopicName = 0;
			return(p);
			}
		--index;
		}
	return(szTopicName);
	}

/****************************************************************************
 **     FUNCTION: string2DW                                                **
 **     PURPOSE: Convert a string into a number                            **
 **     COMMENTS:                                                          **
 **        The strtoul or atol functions are not available to NT programs. **
 ****************************************************************************/
 DWORD string2DW(LPSTR sz)
 	{
	DWORD dw = 0;
	LPSTR p;

	p = sz;
	while (*p)
		dw = 10*dw +  (*p++ - '0');

	return(dw);
	}
	
	
/***********  ANNOTATION DATABASE FUNCTIONS  *************/

		
/****************************************************************************
 **     FUNCTION: WriteRecord                                              **
 **     PURPOSE: Write an annotation record to the data base.              **
 **     COMMENTS:                                                          **
 **       This example uses a crude database .... the records are          **
 **       sequential and looked up linearly.  Each record consists of      **
 **            topic number (long)                                         **
 **            totalRecordsize(WORD)                                       **
 **            VA(DWORD)                                                   **
 **            lch  (DWORD)                                                **
 **            namesize(WORD)                                              **
 **            datasize(WORD)                                              **
 **            ... topic name ...                                          **
 **             ... data ...                                               **
 **                                                                        **
 **        Because there is no way to shrink a file without changing the   **
 **        handle, we keep an  end of of file record (ENDMARK)             **
 ****************************************************************************/
int WriteRecord(HFILE hf, long topicNum, VA va, DWORD lch, LPSTR szTopicName, LPSTR szData)
	{
	long offset, end, lVal;
	WORD size1, size2;
	HFILE hfTmp;
	char szTmp[32];
			
	/* if the record exists, delete it by copying to a temp file and back without it */
	if (FindRecord(hf, topicNum, va, lch, &offset))
		{ 
		wsprintf(szTmp, "~XXX");
		hfTmp = _lcreat(szTmp, 0);
		if (hfTmp == HFILE_ERROR)
			return(-1);
			
		/* get the size of the record to be deleted */
		_llseek(hf, offset + sizeof(long) + sizeof(VA) + sizeof(long), FILE_BEGIN);
		_lread(hf, &size1, sizeof(size1));
		_lread(hf, &size2, sizeof(size2));
		end = _llseek(hf, size1 + size2, FILE_CURRENT);
				
		/* copy up to the old record */
		_llseek(hf, 0, FILE_BEGIN);
        CopyFileBytes(hfTmp, hf, offset);
        
        /* skip the old record and copy the rest */
		offset = _llseek(hf, end, FILE_BEGIN);
		end = CopyToEnd(hfTmp, hf);
		
		/* now copy the whole thing (less the deleted record) back to the original */
		offset = _llseek(hf, 0, FILE_BEGIN);
		_llseek(hfTmp, 0, FILE_BEGIN);
		CopyFileBytes(hf, hfTmp, end);
		_lclose(hfTmp);
	  	}
	else
		{
		/* offset is positioned at the end of the file */
		_llseek(hf, offset, FILE_BEGIN);
		}

	/* now write out the record */
	offset = _llseek(hf, 0, FILE_CURRENT);
	_lwrite(hf, (LPCSTR)&topicNum, sizeof(topicNum));
	_lwrite(hf, (LPCSTR)&va, sizeof(VA));
	_lwrite(hf, (LPCSTR)&lch, sizeof(lch));
	size1 = strlen(szTopicName)+1;
	_lwrite(hf, (LPCSTR)&size1, sizeof(size1));
	size2 = strlen(szData)+1;
	_lwrite(hf, (LPCSTR)&size2, sizeof(size2));
	_lwrite(hf, (LPCSTR)szTopicName, size1);
	_lwrite(hf, (LPCSTR)szData, size2);

	/* Write the EOF marker */
	lVal = ENDMARK;
	_lwrite(hf, (LPCSTR)&lVal, sizeof(lVal));
	return(0);
	}
	
/*
 * Copy bytes from one file to another.  Return the end file position
 */
long CopyFileBytes(HFILE hfDst, HFILE hfSrc, long count)
	{
	char buff[BUFSIZ];
	int num;
	
	while (count)
		{ 
		if (count > BUFSIZ)
			num = BUFSIZ;
		else
			num = (int)count;
			
		count -= num;
			
		num = _lread(hfSrc, buff, num);
		if (num > 0)
			_lwrite(hfDst, buff, num);
		}
	if (num == HFILE_ERROR)
		return(0);
		
	return(_llseek(hfDst, 0, FILE_CURRENT));
	}

/*
 * Copy to end ... copy records until the EOF marker.
 */
long CopyToEnd(HFILE hfDst, HFILE hfSrc)
 	{
	long topicNum;
	VA va;
	DWORD lch;
	WORD size1, size2;
	
	while (1)
		{
		size1 = _lread(hfSrc, &topicNum, sizeof(topicNum));
		if (size1 == 0 || topicNum == ENDMARK)
			return(_llseek(hfDst, 0, FILE_END));
		_lread(hfSrc, &va, sizeof(va));
		_lread(hfSrc, &lch, sizeof(lch));
		_lread(hfSrc, &size1, sizeof(size1));
		_lread(hfSrc, &size2, sizeof(size2));

		_lwrite(hfDst, (LPCSTR)&topicNum, sizeof(topicNum));
		_lwrite(hfDst, (LPCSTR)&va, sizeof(va));
		_lwrite(hfDst, (LPCSTR)&lch, sizeof(lch));
		_lwrite(hfDst, (LPCSTR)&size1, sizeof(size1));
		_lwrite(hfDst, (LPCSTR)&size2, sizeof(size2));

		CopyFileBytes(hfDst, hfSrc, size1);
		CopyFileBytes(hfDst, hfSrc, size2);
		}
	}

/****************************************************************************
 **     FUNCTION: DeleteRecord                                             **
 **     PURPOSE: Remove a record from the database.                        **
 **     COMMENTS:                                                          **
 ****************************************************************************/
BOOL DeleteRecord(HFILE hf, long topicNum, VA va, DWORD lch)
	{
	long offset, end, lVal;
	WORD size1, size2;
	HFILE hfTmp;
	char szTmp[32];
			
	/* if the record exists, delete it by copying to a temp file and back without it */
	if (!FindRecord(hf, topicNum, va, lch, &offset))
		return(FALSE);

	wsprintf(szTmp, "~XXX");
	hfTmp = _lcreat(szTmp, 0);
	if (hfTmp == HFILE_ERROR)
		return(-1);
			
	/* get the size of the record to be deleted */
	_llseek(hf, offset + sizeof(long) + sizeof(VA) + sizeof(long), FILE_BEGIN);
	_lread(hf, &size1, sizeof(size1));
	_lread(hf, &size2, sizeof(size2));
	end = _llseek(hf, size1 + size2, FILE_CURRENT);
				
	/* copy up to the old record */
	_llseek(hf, 0, FILE_BEGIN);
    CopyFileBytes(hfTmp, hf, offset);
        
    /* skip the old record and copy the rest */
	offset = _llseek(hf, end, FILE_BEGIN);
	end = _llseek(hf, 0, FILE_END) - sizeof(DWORD);	// endmark
	_llseek(hf, offset, FILE_BEGIN);
	end = CopyFileBytes(hfTmp, hf, end - offset);
		
	/* now copy the whole thing (less the deleted record) back to the original */
	offset = _llseek(hf, 0, FILE_BEGIN);
	_llseek(hfTmp, 0, FILE_BEGIN);
	CopyFileBytes(hf, hfTmp, end);
	_lclose(hfTmp);

	/* Write the EOF marker */
	lVal = ENDMARK;
	_lwrite(hf, (LPCSTR)&lVal, sizeof(lVal));
	return(0);
	}
	
/****************************************************************************
 **     FUNCTION: FindRecord                                               **
 **     PURPOSE: Is there a specific record in the database?.              **
 **     COMMENTS:                                                          **
 ****************************************************************************/
BOOL FindRecord(HFILE hf, long topicNum, VA va, DWORD lch, LPLONG lpOffset)
	{
	VA r_va;
	DWORD r_lch;
	long offset = 0;
	
	_llseek(hf, 0, FILE_BEGIN);
	*lpOffset = 0;
	while ((offset = GetNextRecord(hf, lpOffset, &topicNum, &r_va, &r_lch)) != ENDMARK)
		{
		if (va == r_va && lch == r_lch)
			{
			return(TRUE);
			}
		*lpOffset = offset;
		}
     return(FALSE);
	}

/****************************************************************************
 **     FUNCTION: GetNextRecord                                            **
 **     PURPOSE:Get the offset to the next annotation record.              **
 **     COMMENTS:                                                          **
 **        Each time this routine is succsessful it returns the offset of  **
 **        the next record. The *lpOffset, etc. are filled in with the     **
 **        values for the current record.							       **
 ****************************************************************************/
long EXPORT PASCAL GetNextRecord(HFILE hf, LPLONG lpOffset, LPLONG lptopicNum, LPVA lpva, LPDWORD lplch)
	{
	long r_topicNum;
	WORD size1, size2;
	long offset;
	
	/* begin searching at the passed in offset */
	_llseek(hf, *lpOffset, FILE_BEGIN);
	
	while (1)
		{
		_lread(hf, &r_topicNum, sizeof(r_topicNum));
		if (r_topicNum == ENDMARK)
			return(ENDMARK);
		_lread(hf, lpva, sizeof(*lpva));
		_lread(hf, lplch, sizeof(*lplch));
		_lread(hf, &size1, sizeof(size1));
		_lread(hf, &size2, sizeof(size2));
					
		/* do we have a match? */
		if (*lptopicNum == ENDMARK || *lptopicNum == r_topicNum)
			{
			*lptopicNum = r_topicNum;
			offset = _llseek(hf, 0, FILE_CURRENT) + size1 + size2;
			return(offset);
			}
			
		/* advance to the next record */
		*lpOffset = _llseek(hf, size1 + size2, FILE_CURRENT);			
		}
	return(ENDMARK);
	}

/****************************************************************************
 **     FUNCTION: ReadRecordNum                                            **
 **     PURPOSE:Get record number "num".                                   **
 **     COMMENTS:                                                          **
 ****************************************************************************/
BOOL EXPORT PASCAL ReadRecordNum(HFILE hf, int num, LPLONG lptopicNum, LPVA lpva, LPDWORD lplch, LPSTR * lpszTopicName, LPSTR * lpszTopicText)
	{
	WORD size1, size2;
	int i = 0;
	LPSTR lp;
	
	/* begin searching at the passed in offset */
	_llseek(hf, 0, FILE_BEGIN);
	
	while (1)
		{
		_lread(hf, lptopicNum, sizeof(*lptopicNum));
		if (*lptopicNum == ENDMARK)
			return(FALSE);
		_lread(hf, lpva, sizeof(*lpva));
		_lread(hf, lplch, sizeof(*lplch));
		_lread(hf, &size1, sizeof(size1));

		/* are all these reads succeeding? */
		if (_lread(hf, &size2, sizeof(size2)) != sizeof(size2))
			return(FALSE);
					
		/* is this the one? */
		if (i == num)
			{
			lp = GlobalAllocPtr(GHND, size1);
			_lread(hf, lp, size1);
			if (lpszTopicName)
				*lpszTopicName = lp;
			else
				{
				GlobalFreePtr(lp);
				lp = 0;
				}

			lp = GlobalAllocPtr(GHND, size2);
			_lread(hf, lp, size2);
			if (lpszTopicText)
				*lpszTopicText = lp;
			else
				{
				GlobalFreePtr(lp);
				lp = 0;
				}
			return(TRUE);
			}
			
		/* otherwise advance to the next record */
		_llseek(hf, size1 + size2, FILE_CURRENT);
		++i;			
		}
	return(ENDMARK);
	}


/****************************************************************************
 **     FUNCTION: ReadRecord                                               **
 **     PURPOSE: Read an annotation record from the data base.             **
 **     COMMENTS:                                                          **
 **        Assumes that offset is at the beginning of the record           **
 **        from FindRecord or GetAnnotation)                               **
 **        This returns only the text of the record.  The caller must      **
 **        use GlobalFreePtr to free the memory.                           **
 ****************************************************************************/
BOOL ReadRecord(HFILE hf, long offset, LPSTR * lpszTopicName, LPSTR * lpszData)
	{
	long topic;
	WORD size1, size2;
	VA va;
	DWORD lch;
	
	/* begin reading at the passed in offset */
	_llseek(hf, offset, FILE_BEGIN);
	
	/* read the key information */
	_lread(hf, &topic, sizeof(topic));
	if (topic == ENDMARK)
		return(FALSE);
	_lread(hf, &va, sizeof(va));
	_lread(hf, &lch, sizeof(lch));
	_lread(hf, &size1, sizeof(size1));
	_lread(hf, &size2, sizeof(size2));
	
	/* space for the data */
	*lpszTopicName = GlobalAllocPtr(GHND, size1);
		if (!*lpszTopicName)
		return(FALSE);
	*lpszData = GlobalAllocPtr(GHND, size2);
	if (!*lpszData)
		{
		GlobalFreePtr(*lpszTopicName);
		*lpszTopicName = 0;
		return(FALSE);
		}
		
	/* and read the title and data from the file */
	_lread(hf, *lpszTopicName, size1);
	_lread(hf, *lpszData, size2);
	return(TRUE);
	}

/****************************************************************************
 **     FUNCTION: OpenAnnotation                                           **
 **     PURPOSE: Derive the file name from the title name and open it.     **
 **     COMMENTS:                                                          **
 ****************************************************************************/
 HFILE EXPORT PASCAL OpenAnnotation(LPSTR szFilename)
 	{
	char buff[_MAX_PATH], *p;
	HFILE hFile;
	long lVal;

	/* derive the annotation file name */
	strcpy(buff, szFilename);
	if ((p = strrchr(buff, '\\')) == 0)
		p = buff;
	if ((p = strchr(p, '.')) == 0)
		strcat(buff, ".ann");
	else
		strcpy(p, ".ann");
	hFile = _lopen(buff, OF_READWRITE);
	if (hFile == HFILE_ERROR)
		{
		hFile = _lcreat(buff, 0);
		if (hFile == HFILE_ERROR)
			{
			return(0);
			}
		/* provide it with an end-of-file marker */
		lVal = ENDMARK;
		_lwrite(hFile, (LPCSTR)&lVal, sizeof(lVal));
		_lclose(hFile);
		hFile = _lopen(buff, OF_READWRITE);
		}

	return(hFile);
	}

/****************************************************************************
 **     FUNCTION: AddAnnotation                                            **
 **     PURPOSE:The application is adding an empty annotation to the topic **
 **     COMMENTS:                                                          **
 ****************************************************************************/
int EXPORT PASCAL AddAnnotation(LPMV lpMV, HFILE hFile, long topicNum, VA va, DWORD lch, LPSTR szTopicName)
	{
	long offset;
	HINSTANCE hInstance = GetPaneInstance(lpMV);

	/* if there is not already one, write an empty annotation record */
	if (FindRecord(hfAnnotation, topicNum, va, lch, &offset))
		return (-1);
	WriteRecord(hfAnnotation, topicNum, va, lch, szTopicName, "");
	
	/* put up the main annotation dialog */
	CurrentTopicNum = topicNum;
	CurrentVa = va;
	CurrentLch = lch;
	DialogBox(hInstance, "ANNOTATEDLG", hMainWnd, AnnotateDlgProc);
	return(0);
	}

/****************************************************************************
 **     FUNCTION: GetAnnotation                                            **
 **     PURPOSE:Get the offset to the next annotation record.              **
 **     COMMENTS:                                                          **
 **        Each time this routine is succsessful it returns the offset of  **
 **        the next record. The lpva, etc. are filled in with the  values  **
 **        for the current record.
 ****************************************************************************/
long EXPORT PASCAL GetAnnotation(HFILE hf, LPLONG lpOffset, LPLONG lptopicNum, LPVA lpva, LPDWORD lplch)
	{
	return(GetNextRecord(hf, lpOffset, lptopicNum, lpva, lplch));
	}

/****************************************************************************
 **     FUNCTION: AnnotationCallback                                       **
 **     PURPOSE: This is the routine that the ANNEW DLL calls back with    **
 **       Hotspot information.                                             **
 **     COMMENTS:                                                          **
 **        Each time this routine is succsessful it returns the offset of  **
 **        the next record. The lpva, etc. are filled in with the  values  **
 **        for the current record.
 ****************************************************************************/
void EXPORT PASCAL AnnotationCallback(LPMV lpMV, long topicNum, VA va, DWORD lch)
	{
	/*******************************************************/
	/******** TUTORIAL: Insert Lesson 16 code here. ********/
	/*******************************************************/
	}

/****************************************************************************
 **     FUNCTION: UpdateAnnotations                                        **
 **     PURPOSE: Insert annotation markers for the current topic.          **
 **     COMMENTS:                                                          **
 **       In this example we are limiting annotations to the scrolling     **
 **       region if one exists)                                            **
 ****************************************************************************/
BOOL UpdateAnnotations(LPMV lpMV)
	{
	long topicNum;
	VA va;
	DWORD lch;
	long offset, nextOffset;
	/*******************************************************/
	/******** TUTORIAL: Insert Lesson 16 code here. ********/
	/*******************************************************/
	return(TRUE);
	}

/****************************************************************************
 **     FUNCTION: InsertAnnotations                                        **
 **     PURPOSE: Insert a single annotation marker.                        **
 **     COMMENTS:                                                          **
 ****************************************************************************/
 BOOL InsertAnnotation(LPMV lpMV, long topicNum, VA va, DWORD lch)
 	{
	char buff[100];
	ERR err;
	/*******************************************************/
	/******** TUTORIAL: Insert Lesson 16 code here. ********/
	/*******************************************************/
	return(TRUE);
	}
