/*
 * This file contains miscellaneous MediaView handling code.
 */

#include <stdlib.h>
#include <windows.h>
#include <windowsx.h>
#include <commdlg.h>
#include <mmsystem.h>
#include <string.h>
#include <memory.h>
#include <ctype.h>
#include <stdarg.h>

/* MediaView Include Files */
#include <medv14.h>
#include <mverror.h>

/* Application Include Files */
#include "topic.h"
#include "player.h"
#include "pane.h"
#include "annotate.h"
#include "resrc1.h"
#include "proto.h"

/* The history list. */
typedef struct tagHISTORY
	{
	int inUse;				/* in use flag ... hName might be NULL */
	VA va;					/* topic address */
	long scroll;			/* the scroll position */
	int isName;				/* is the hName a name handle or a topic number? */
	HANDLE hName;			/* topic name */
	HTITLE hTitle;			/* title handle (for interfile jumps) */
	} HISTORY;
	
HISTORY History[NUMHISTORY] = {0};
int iH = -1;		/* History index */
int iB = -1;		/* back history index */

HISTORY Bookmarks[NUMBOOKMARKS] = {0};
int iBM = -1;		/* next  free Bookmark */

LPPRINTMARK PrintMarks = 0;
typedef struct tagPRINTTOPIC
	{
	VA va;
	HTITLE hTitle;
	} PRINTTOPIC, far * LPPRINTTOPIC;

int iPM = -1;

/* when the keyword search has more than one hit, save the adress vector here */
ADDR * KeyAddrs = 0;
int NumKeyTopics;

/* for scrolling to next search hit, we need to save some state information */
int nHitsInTopic = 0;
int CurrentHitShown = 0;


/****************************************************************************
 **     FUNCTION: MV_Search                                                **
 **     PURPOSE: Full text search for the string pattern.                  **
 **     COMMENTS:                                                          **
 ****************************************************************************/
int MV_Search(LPMV lpMV, int scope, int andFlag, char ** groupList, char * szSearch, HWND hDlg)
	{
	int flag;
	HTLIST htList;
	HTLIST htList1, htList2;
	QUERYERR qe;
	long topicNumber;
	HTITLE hTitle = hMVGetTitle(lpMV, NULL);

	/* clean up from previous search */
	if (hTopicList)
		MVTopicListDestroy(hTopicList);

	/* AND or OR multiple search words? */
	if (andFlag)
		flag = IMPLICIT_AND;
	else
		flag = IMPLICIT_OR;

	switch (scope)
		{
		case SEARCH_TOPIC:
			/*
			 * search current topic: get the current topic number,
			 * create a 1 topic list, and search
			 */
			topicNumber = lMVTopicNumber(lpMV);
			htList = hMVTopicListFromTopicNo(hTitle, topicNumber);
			hTopicList = hMVTopicListFromQuery(	hTitle, 
												(WORD)flag,
												szSearch,
												htList,
												0,
												&hHighlights,
												&qe);

			/* clean up the interim topic list */
			MVTopicListDestroy(htList);
			break;

		case SEARCH_GROUPS:
			/*
			 * Search all chosen groups:
			 * for each group in the groupList
			 *    get the topic list and merge into the main list
			 * do the search on the entire combined topic list
			 */
			htList = htList1 = htList2 = 0;

			/* find first group with topics */
			while (*groupList && htList == 0)
				htList = hMVTopicListLoad(hTitle, *groupList++);
			while (*groupList)
				{
				htList1 = htList;
				/* if there are any topics, combine them */
				if ((htList2 = hMVTopicListLoad(hTitle, *groupList)) != 0)
					{
					htList = hMVTopicListCombine(TL_OR, htList1, htList2);

					/* be sure to clean up the intermediate lists */
					MVTopicListDestroy(htList1);
					MVTopicListDestroy(htList2);
					}
				++groupList;
				}

			/* if there are any topics to search, then do so */
			if (htList)
				{
				hTopicList = hMVTopicListFromQuery(	hTitle,
													(WORD)flag,
													szSearch,
													htList,
													0,
													&hHighlights,
													&qe);
				MVTopicListDestroy(htList);
				}
			else
				{
				/* otherwise set up the error return */
				hTopicList = 0;
				return(ERR_FAILED);
				}
			break;

		case SEARCH_ALL:
			/* search all topics in the title */
			hTopicList = hMVTopicListFromQuery(	hTitle,
												(WORD)flag,
												szSearch,
												0,
												0,
												&hHighlights,
												&qe);
			break;
		}

	/* return 0 if there were any Search hits */
	if (hTopicList == 0)
		return(qe.iError);
	else if (lMVTopicListLength(hTopicList) == 0)
		return(ERR_FAILED);
	else
		return(0);
	}

/****************************************************************************
 **     FUNCTION: MV_SearchInterrupt                                       **
 **     PURPOSE: Callback routine for searching.                           **
 **     COMMENTS:                                                          **
 **       MediaView will call this routine periodically to see if the      **
 **       search should be canceled. It is installed with MVTitleOpenEx.   **
 **       The EXPORT must be set because it is called back from within the **
 **       MediaView DLL.                                                   **
 ****************************************************************************/
BOOL EXPORT PASCAL MV_SearchInterrupt(LPVOID lpData, LPVOID lpOther)
 	{
	extern int gStopSearch;
 	MSG  msg;      

	/* Each time we are called, dispatch any pending messages */
 	while (PeekMessage(&msg, 0, 0, 0, PM_NOREMOVE))
		{
		GetMessage(&msg, 0, 0, 0); 
		TranslateMessage(&msg);
		DispatchMessage(&msg); 
		}
	
	/* initially 0, this variable set if "Stop Search" button is pushed */
	return (gStopSearch);     
	}

/****************************************************************************
 **     FUNCTION: MV_ShowSearchResults                                     **
 **     PURPOSE: Load up the listbox with the search results.              **
 **     COMMENTS:                                                          **
 **        The caller must make sure that the Search results dialog is     **
 **        active.
 ****************************************************************************/
void MV_ShowSearchResults(LPMV lpMV)
	{
	long count, i, topicNumber;
	char buff[512];
	HWND hList;
	HTITLE hTitle = hMVGetTitle(lpMV, NULL);
	
	/* get the handle of the hits display list */
	hList = GetDlgItem(hSearch, IDC_LIST1);

	SendMessage(hList, LB_RESETCONTENT, 0, 0);

	/* has there been a search? */
	if (hTopicList == 0)
		{
		/* disable GOTO button */
		EnableWindow(GetDlgItem(hSearch, IDC_GOTO), FALSE);
		return;
		}
	
	/* for each topic number */
	count = lMVTopicListLength(hTopicList);
	for (i = 0; i < count; ++i)
		{
		topicNumber = lMVTopicListLookup(hTopicList, i);
		lMVTitleGetInfo(hTitle, TTLINF_TOPICTITLE, topicNumber, (long)(LPSTR)buff);

		/* and add it to the list box */
		SendMessage(hList, LB_ADDSTRING, 0, (long)(LPSTR)buff);
		}

	/* default selection to first item */
	SendMessage(hList, LB_SETCURSEL, 0, 0);
	EnableWindow(GetDlgItem(hSearch, IDC_GOTO), TRUE);
			
	}

/****************************************************************************
 **     FUNCTION: MV_GoToFromSearchResults                                 **
 **     PURPOSE: Get the results from the list box and set the topic       **
 **     COMMENTS:                                                          **
 **       This relies on the order of topic titles in the listbox          **
 **       corresponding to the title number order in the search results    **
 **       topic list.                                                      **
 ****************************************************************************/
void MV_GoToFromSearchResults(HWND hDlg, LPMV lpMV, LPVA lpVA)
	{
	int selNumber;
	long topicNumber;
	HTITLE hTitle = hMVGetTitle(lpMV, NULL);

	/* convert selection -> topic number -> VA */
	selNumber = (int)SendMessage( hDlg, LB_GETCURSEL, 0, 0);
	topicNumber = lMVTopicListLookup(hTopicList, selNumber);
	*lpVA = vaMVConvertTopicNumber(hTitle, topicNumber);
	}

/****************************************************************************
 **     FUNCTION: MV_ScrollToSearchHighlight                               **
 **     PURPOSE: Scroll to the indicated search highlight.                 **
 **     COMMENTS:                                                          **
 ****************************************************************************/
int MV_ScrollToSearchHighlight(LPMV lpMV, int sNum)
	{
	ERR err = 0;

	if (lpMV == 0)
		return(0);
		
	if (!fMVHasSR(lpMV))
		return(0);
	
	/* make it jump to the highlight */
	fMVScrollToHighlight(lpMV, sNum, &err);

	/* and reset the scroll bars afterwards */
	Pane_ScrollRanges(lpMV);

	CurrentHitShown = sNum + 1;

	/* force the Pane to update the display */
	InvalidateRect(hwndMVGetWindow(lpMV), 0, TRUE);

	return(err);
	}
	
/****************************************************************************
 **     FUNCTION: MV_UpdateSearchHits                                      **
 **     PURPOSE: Update the variables that track search hits in the        **
 **       current topic.                                                   **
 **     COMMENTS:                                                          **
 **       For this example we are only calling this routine with a valid   **
 **       MV for a topic that has a scrolling region.                      **
 ****************************************************************************/
void MV_UpdateSearchHits(LPMV lpMV)
    {
	HTHLITE hHits;
	ERR err;
	
	if (!hHighlights)
		return;

	/* Update the lpMV with the HTHLITE array for the current topic */
	nHitsInTopic = 0;
	if (showHits)
		{
		/* free any previous topic search hit list */
		hHits = hMVGetHighlights(lpMV);
		if (hHits)
			{
			GlobalFree(hHits);
			hHits = 0;
			}

		/* get a list of hits in this topic from the Global hit list */
		hHits = hMVHighlightsInTopic(hHighlights, lMVTopicNumber(lpMV));
	  	hMVSetHighlights(lpMV, hHits, TRUE, &err);

		/* how many hits in this topic? */
		nHitsInTopic = (int)lMVTopicHighlightCount(hHits);
		CurrentHitShown = 0;
		}
	else
	  	{
	  	/* Otherwisw turn them off */
	  	hMVSetHighlights(lpMV, 0, TRUE, &err);
		}

	UI_NextHighlight(nHitsInTopic ? TRUE : FALSE);
	}
		
/****************************************************************************
 **     FUNCTION: MV_ScrollToNextHighlight                                 **
 **     PURPOSE: Scroll the next search highlight into view.               **
 **     COMMENTS:                                                          **
 **       If search hits are being shown (and there are any), scroll to    **
 **       to the next.  Wrap at end of file.                               **
 ****************************************************************************/
void MV_ScrollToNextHighlight(LPMV lpMV)
	{
	if (showHits && nHitsInTopic != 0)
		{
		/* wrap back to zero if at end */
		if (CurrentHitShown == nHitsInTopic)
			CurrentHitShown = 0;

		/* this routine will increment CurrentHitShown */
		MV_ScrollToSearchHighlight(lpMV, CurrentHitShown);

		}
	}

/****************************************************************************
 **     FUNCTION: MV_Keywords                                              **
 **     PURPOSE: Load up the dialog with Keywords from the selected group  **
 **     COMMENTS:                                                          **
 **        The keywords for group 'a' are kept in Word Wheel '|a', etc.    **
 ****************************************************************************/
int MV_Keywords(LPMV lpMV, HWND hList, HWND hGroups)
	{
	char buff[_MAX_PATH+1];
	int numWords, j, index;
	long dw;
	char * WWfile = buff;
	HTITLE hTitle = hMVGetTitle(lpMV, NULL);
   
	/* open the WordWheel associated with the KeyIndex group */
	index = (int)SendMessage(hGroups, CB_GETCURSEL, 0, 0);
	SendMessage(hList, LB_RESETCONTENT, 0, 0);

	/* there may not be any keyword groups */
	if (index == -1)
		return(-1);

	/* close the previously open WW file */
	if (hWordWheel)
		MVWordWheelClose(hWordWheel);

	/* one at a time is kept open: the name is of the form "|?" where the group index name subs for the '?' */
	dw = SendMessage(hGroups, CB_GETITEMDATA, index, 0);
	WWfile[0] = '|';
	if (index == -1)
		WWfile[1] = '0';
	else
		WWfile[1] = (char)dw;
	WWfile[2] = 0;

	/* the keyword lists are stored in a Word Wheel, so open one */
   	if ((hWordWheel = hMVWordWheelOpenTitle(hTitle, WWfile)) == 0)
		return(ERR_FAILED);

	/* load up list box with the keywords from this group */
	numWords = (int)lMVWordWheelLength(hWordWheel);
	for (j = 0; j < numWords; ++j)
		{
		/* get each keyword (they are already alphabatized) */
		nMVWordWheelLookup(hWordWheel, j, buff, sizeof(buff));
		SendMessage(hList, LB_ADDSTRING, 0, (long)(LPSTR)buff);
		}
	return(index);
	}

/****************************************************************************
 **     FUNCTION: MV_IndexLookup                                           **
 **     PURPOSE: Create a list of topics containing the keyword            **
 **     COMMENTS:                                                          **
 ****************************************************************************/
int MV_IndexLookup(HWND hMainWnd, HWND hList, HWND hSearch, LPTOPIC lpT)
	{
	VA va;
	int selNumber;
	char buff[512];
	LPMV lpMV = Topic_ValidMV(lpT);
	HTITLE hTitle = hMVGetTitle(lpMV, NULL);
	HINSTANCE hInstance = GetPaneInstance(lpMV);
	long lRet;

	/* get a buffer big enough for all the KeyIndex ADDRs */
	selNumber = (int)SendMessage(hList, LB_GETCURSEL, 0, 0);
	SendMessage(hList, LB_GETTEXT, selNumber, (long)(LPSTR)buff);
	NumKeyTopics = nMVKeyIndexGetCount(hWordWheel, buff);
	KeyAddrs = (ADDR *)GlobalAllocPtr(GHND, NumKeyTopics * sizeof(ADDR));
	if (!KeyAddrs)
		return(wERRS_OOM);

	/* now get all the ADDRs and load up the search results display */
	lRet = lMVKeyIndexGetAddrs(hWordWheel, buff, 0, NumKeyTopics, (LPBYTE)KeyAddrs);

	/* was it successful? */
	if (lRet > 0)
		{
	
		/* if there is only one, then go to it */
		if (NumKeyTopics == 1)
			{
			va = vaMVConvertAddr(hTitle, *(KeyAddrs));
			MV_AddToHistory(Topic_ValidMV(lpT));
			MV_ChangeAddress(0, lpT, va, 0);
			}
		else
			{
			/* otherwise load up a dialog box with it */
			DialogBox(hInstance, "IndexResultsDlg", hMainWnd, IndexResultsDlgProc);
			}
		}

	/* now clean up the list */
	GlobalFreePtr(KeyAddrs);
	KeyAddrs = 0;
	return(0);
	}

/****************************************************************************
 **     FUNCTION: MV_LoadIndexHits                                         **
 **     PURPOSE: Load a list box with the topic titles of key index search **
 **       hits.                                                            **
 **     COMMENTS:                                                          **
 ****************************************************************************/
void MV_LoadIndexHits(HWND hWnd, LPMV lpMV)
	{
	int i;
	VA va, tmpVa;
	int subTopic;
	long scroll;
	HANDLE hName;
	LPSTR lp;
	ERR err;
	HTITLE hTitle = hMVGetTitle(lpMV, NULL);
	
	/* remember the address upon entry */
	MVGetAddress(lpMV, &va, &subTopic, &scroll);

	for (i = 0; i < NumKeyTopics; ++i)
		{
		/* set the address of each one so we can get the topic name */
		tmpVa = vaMVConvertAddr(hMVGetTitle(lpMV, 0), KeyAddrs[i]);
		if (!fMVSetAddress(lpMV, tmpVa, NSR_PANE, 0, &err))
			fMVSetAddress(lpMV, tmpVa, SR_PANE, 0, &err);

		/* get the topic name */
		hName = hMVGetName(lpMV);
		lp = GlobalLock(hName);

		/* set it into the list box */
		
		SendMessage(hWnd, LB_ADDSTRING, 0, (long)lp);
		GlobalFree(hName);
		hName = 0;
		}

	/* restore the address at exit */
	fMVSetAddress(lpMV, va, subTopic, scroll, &err);
	}

/****************************************************************************
 **     FUNCTION: GoToFromIndex                                            **
 **     PURPOSE: One of the displayed index key search hits was selected.  **
 **        Use it to determine the goto address.
 **     COMMENTS:                                                          **
 ****************************************************************************/
void MV_GoToFromIndex(HWND hWnd, LPMV lpMV)
	{
	VA va;
	int index;
	extern LPTOPIC lpTopic;

	index = (int)SendMessage(hWnd, LB_GETCURSEL, 0, 0);
	va = vaMVConvertAddr(hMVGetTitle(lpMV, 0), KeyAddrs[index]);
	MV_AddToHistory(Topic_ValidMV(lpTopic));
	MV_ChangeAddress(0, lpTopic, va, 0);
	}

/****************************************************************************
 **     FUNCTION: MV_IndexSelect                                           **
 **     PURPOSE: Set the list selection according to the character match.  **
 **     COMMENTS:                                                          **
 ****************************************************************************/
int MV_IndexSelect(HWND hwnd, LPSTR szKey)
	{
	long index;

	/* this tells us the closest match in the keyword list */
	index = lMVWordWheelPrefix(hWordWheel, szKey);
	SendMessage(hwnd, LB_SETCURSEL, (WPARAM)index, 0);
	return(0);
	}

/****************************************************************************
 **     FUNCTION: MV_AddToHistory                                          **
 **     PURPOSE: Add the current topic to the history list                 **
 **     COMMENTS:                                                          **
 **       Location 0 always contains the oldest entry.  If the array is	   **
 **       full, move everything down and add the new entry to the end.     **
 ****************************************************************************/
void MV_AddToHistory(LPMV lpMV)
	{

	/* is the History buffer full? If so, make room for more */
	if (iH == DIMENSION(History)-1)
		{
		if (History[0].isName)
			GlobalFree(History[0].hName);
		/* shift everything down by one */
		memmove(History, History+1, sizeof(HISTORY) * (iH));
		}
	else
		{
		/* otherwise bump the current History index */
		++iH;
		}

	History[iH].inUse = TRUE;
	MV_HistoryGetTopic(&History[iH], lpMV);
	History[iH].hTitle = hMVGetTitle(lpMV, NULL);

	/*
	 * update the Back pointer ... the strategy is to take it backward
	 * each time Back is performed, but to reset it to the History
	 * pointer when the SetTopic comes from any action the moves
	 * forward (such as a hotspot jump).
	 */
	iB = iH;

	/* if the history display is active, update it */
	if (hHist)
		MV_ShowHistory(lpMV, GetDlgItem(hHist, IDC_LIST1));
	}

/****************************************************************************
 **     FUNCTION: MV_HistoryGetTopic                                       **
 **     PURPOSE: Put the topic address, etc. into the HISTORY structure    **
 **     COMMENTS:                                                          **
 ****************************************************************************/
void MV_HistoryGetTopic(HISTORY * pH, LPMV lpMV)
	{
	int subTopic;

	MVGetAddress(lpMV, &pH->va, &subTopic, &pH->scroll);
	if ((pH->hName = hMVGetName(lpMV)) != 0)
		pH->isName = TRUE;
	else
		{
		pH->hName = (HANDLE)lMVTopicNumber(lpMV);
		pH->isName = FALSE;
		}
	}

/****************************************************************************
 **     FUNCTION: MV_ShowHistory                                           **
 **     PURPOSE: Write the history strings into the display                **
 **     COMMENTS:                                                          **
 ****************************************************************************/
void MV_ShowHistory(LPMV lpMV, HWND hwnd)
	{
	LPSTR lp;
	char buff[20];
	long topicNumber;
	int i;

	SendMessage(hwnd, LB_RESETCONTENT, 0, 0);
	if (iH < 0)
		return;

	for (i = iH; i >= 0; --i)
		{
		/* if there is a Topic Name, display it */
		if (History[i].isName)
			lp = GlobalLock(History[i].hName);
		else
			{
			/* otherwise just create a topic number string */
			topicNumber = lMVTopicNumber(lpMV);
			wsprintf(buff, "Topic #%d", (int)History[i].hName);
			lp = buff;
			}
		SendMessage(hwnd, LB_ADDSTRING, 0, (long)lp);
		}
	}

/****************************************************************************
 **     FUNCTION: MV_GoToFromHistory                                       **
 **     PURPOSE: Go to a new topic according to the history selection      **
 **     COMMENTS:                                                          **
 **       Watch out for interfile references.                              **
 ****************************************************************************/
void MV_GoToFromHistory(int index, LPVA lpVA, LPHTITLE lphTitle, LPLONG lpScroll)
	{
	*lpVA = History[index].va;
	*lphTitle = History[index].hTitle;
	*lpScroll = History[index].scroll;
	}

/****************************************************************************
 **     FUNCTION: MV_CleanupHistory                                        **
 **     PURPOSE: Free all of the History name strings                      **
 **     COMMENTS:                                                          **
 ****************************************************************************/
void MV_CleanupHistory()
	{
	int i;

	for (i = 0; i < DIMENSION(History); ++i)
		{
		if (History[i].inUse && History[i].isName)
			GlobalFree(History[i].hName);
		}
	memset(History, 0, sizeof(History));
	iH = 0;
	}

/****************************************************************************
 **     FUNCTION: MV_Back                                                  **
 **     PURPOSE: Go back one in the history list.                          **
 **     COMMENTS:                                                          **
 **        See MV_GoToFromHistory for comments on handling the back        **
 **        index.                                                          **
 ****************************************************************************/
int MV_Back(LPHTITLE lphTitle, LPVA lpVA, LPLONG lpScroll)
	{
	if (iB == -1)
		return(iB);


	MV_GoToFromHistory(iB, lpVA, lphTitle, lpScroll);
	return(--iB);
	}

/*
 * When we added the Back jump to the history list, it reset the Back pointer.
 * Undo that here.
 */
void MV_BackAdjust1(int oldB)
	{
	iB = oldB;
	}

/* catch the wrap around on the back pointer */
void MV_BackAdjust2()
	{

	/* if the list is full, everything moved down one to add the new entry */
	if (iH == DIMENSION(History)-1)
		--iB;
	}

/****************************************************************************
 **     FUNCTION: MV_AddBookmark                                           **
 **     PURPOSE: Add an entry to the Bookmark menu.                        **
 **     COMMENTS:                                                          **
 **       This version allows up to 10.                                    **
 ****************************************************************************/
int MV_AddBookmark(HWND hwnd, LPMV lpMV, LPSTR szName)
	{
	HMENU hMenu;

	if (iBM == DIMENSION(Bookmarks)-1)
		return(FALSE);
	
	/* get the handle of the Bookmark subMenu */
	hMenu = GetMenu(hwnd);
	hMenu = GetSubMenu(hMenu, 2);

	/* Add the seperator if this is the first one */
	if (++iBM == 0)
		AppendMenu(hMenu, MF_SEPARATOR, 0, 0);

	/* now add the new menu item */
	AppendMenu(hMenu, MF_STRING|MF_ENABLED, ID_BOOKMARKBASE+iBM, szName);

	/* save the current topic address */
	MV_HistoryGetTopic(&Bookmarks[iBM], lpMV);
	return(TRUE);
	}

/****************************************************************************
 **     FUNCTION: MV_ShowBookmarks                                         **
 **     PURPOSE: Write the history strings into the display                **
 **     COMMENTS:                                                          **
 ****************************************************************************/
void MV_ShowBookmarks(HWND hwnd, HWND hMainWnd)
	{
	int i;
	HMENU  hMenu;
	char buff[512];

	/* get the handle of the Bookmark subMenu */
	hMenu = GetMenu(hMainWnd);
	hMenu = GetSubMenu(hMenu, 2);

	SendMessage(hwnd, LB_RESETCONTENT, 0, 0);
	if (iBM == -1)
		return;

	/* for each Bookmark, write the string into the list box */
	for (i = 0; i <= iBM; ++i)
		{
		/* there is the "Define" and "SEPERATOR" first, so +2 */
		GetMenuString(hMenu, 2+i, buff, sizeof(buff), MF_BYPOSITION);
		SendMessage(hwnd, LB_ADDSTRING, 0, (long)(LPSTR)buff);
		}
	}

/****************************************************************************
 **     FUNCTION: MV_GoToFromBookmark                                      **
 **     PURPOSE: Go to a new topic according to the bookmark selection     **
 **     COMMENTS:                                                          **
 ****************************************************************************/
void MV_GoToFromBookmark(int bm, LPHTITLE lphTitle, LPVA lpVA, LPLONG lpScroll)
	{
	*lpVA = Bookmarks[bm].va;
	*lphTitle = Bookmarks[bm].hTitle;
	*lpScroll = Bookmarks[bm].scroll;
	}

/****************************************************************************
 **     FUNCTION: MV_DelBookmark                                           **
 **     PURPOSE: Delete a bookmark.                                        **
 **     COMMENTS:                                                          **
 ****************************************************************************/
int MV_DelBookmark(HWND hwnd, HWND hMainWnd)
	{
	HMENU  hMenu;
	int i;
	char buff[512];
	
	/* delete the string from the list box */
	i = (int)SendMessage(hwnd, LB_GETCURSEL, 0, 0);
	SendMessage(hwnd, LB_DELETESTRING, i, 0);

	/* get the handle of the Bookmark subMenu */
	hMenu = GetMenu(hMainWnd);
	hMenu = GetSubMenu(hMenu, 2);

	/* delete the string from the menu */
	DeleteMenu(hMenu, i+2, MF_BYPOSITION);

	/* delete the entry from the Bookmarks array */
	if (i < iBM)
		memmove(Bookmarks+i, Bookmarks+i+1, (iBM-i) * sizeof(HISTORY));
	--iBM;

	/* now run through the menu and reset the IDs to be a contiguous block */
	for (i = 0; i <= iBM; ++i)
		{
		/* allow for "Define" and SEPARATOR */
		GetMenuString(hMenu, i+2, buff, sizeof(buff), MF_BYPOSITION);
		ModifyMenu(hMenu, i+2, MF_BYPOSITION|MF_STRING, i+ID_BOOKMARKBASE, buff);
		}
		
	/* last bookmark delete also deletes separator */
	if (iBM == -1)
		DeleteMenu(hMenu, 1, MF_BYPOSITION);

	return(TRUE);
	}
 

/****************************************************************************
 **     FUNCTION: MV_AddPrintMark                                          **
 **     PURPOSE: Add a new print mark.                                     **
 **     COMMENTS:                                                          **
 ****************************************************************************/
int MV_AddPrintMark(LPMV lpMV, HWND hMarks, HWND hGroups, int markType)
	{
	INT subTopic;
	int  selNumber;
	long scroll;
	long topicNumber;
	HANDLE hName;
	LPSTR lp;
	LPGROUPUI lpG;
	char buff[512];

	/* allocate a new PrintMark */
	++iPM;
	if (PrintMarks == 0)
		PrintMarks = (LPVOID)GlobalAllocPtr(GHND, sizeof(PRINTMARK));
	else
		PrintMarks = (LPVOID)GlobalReAllocPtr(PrintMarks, (iPM+1) * sizeof(PRINTMARK), GHND);

	PrintMarks[iPM].type = markType;
 	PrintMarks[iPM].hTitle = hMVGetTitle(lpMV, NULL);

	switch (markType)
		{
		case IDC_TOPIC:
			/* save away the current address information */
			MVGetAddress(lpMV, &PrintMarks[iPM].va, &subTopic, &scroll);

			hName = hMVGetName(lpMV);
			if (hName)
				{
				/* build a string with the topic name */
				lp = GlobalLock(hName);
				/* 8 == "Topic: " + \0 */
				PrintMarks[iPM].lpName = GlobalAllocPtr(GHND, strlen(lp) + 8);
				wsprintf(PrintMarks[iPM].lpName, "Topic: %s", lp);
				GlobalFree(hName);
				}
			else
				{
				/* otherwise just create a topic number string */
				topicNumber = lMVTopicNumber(lpMV);
				PrintMarks[iPM].lpName = GlobalAllocPtr(GHND, 32);
				wsprintf(PrintMarks[iPM].lpName, "Topic #%d", topicNumber);
				}
			break;
		case IDC_GROUPS:
			selNumber = (int)SendMessage(hGroups, LB_GETCURSEL, 0, 0);
			SendMessage(hGroups, LB_GETTEXT, selNumber, (long)(LPSTR)buff);
			/* 8 == "Group: " + \0 */
			PrintMarks[iPM].lpName = GlobalAllocPtr(GHND, strlen(buff) + 8);
			wsprintf(PrintMarks[iPM].lpName, "Group: %s", buff);

			/* get the Group Name from the title */
			for (lpG = lpGroups; lpG && lpG->title; ++lpG)
				{
				/* 7 == "Group: " */
				if (stricmp(lpG->title, buff) == 0)
					break;
				}
			PrintMarks[iPM].lpGroupName = GlobalAllocPtr(GHND, strlen(lpG->name)+1);
			strcpy(PrintMarks[iPM].lpGroupName, lpG->name);
			break;
		case IDC_ALL:
			break;
		}
	SendMessage(hMarks, LB_ADDSTRING, 0, (long)(LPSTR)PrintMarks[iPM].lpName);
	return(0);
	}

/****************************************************************************
 **     FUNCTION: MV_ShowMarkList                                          **
 **     PURPOSE: Load the list box with the current mark list              **
 **     COMMENTS:                                                          **
 ****************************************************************************/
int MV_ShowMarkList(HWND hWnd)
	{
	int i;

	SendMessage(hWnd, LB_RESETCONTENT, 0, 0);

	for (i = 0; i <= iPM; ++i)
		{
		SendMessage(hWnd, LB_ADDSTRING, 0, (long)(LPSTR)PrintMarks[i].lpName);
		}
	return(0);
	}
	
/****************************************************************************
 **     FUNCTION: MV_DelPrintMark                                          **
 **     PURPOSE: Delete a print mark                                       **
 **     COMMENTS:                                                          **
 ****************************************************************************/
int MV_DelPrintMark(HWND hMarks)
	{
	int selNumber;

	/* which one to delete? */
	selNumber = (int)SendMessage(hMarks, LB_GETCURSEL, 0, 0);

	/* take it out of the PrintMark list. No need to move if the
	 * last one is being deleted.
	 */
	GlobalFreePtr(PrintMarks[selNumber].lpName);
	if (selNumber < iPM)
		memmove(PrintMarks + selNumber,
					 PrintMarks + selNumber + 1,
					 sizeof(PRINTMARK) * (iPM - selNumber));

	--iPM;

	/* and refresh the display */
	MV_ShowMarkList(hMarks);
	return(0);
	}

/****************************************************************************
 **     FUNCTION: MV_FreePrintMarks                                        **
 **     PURPOSE: Free up all the allocated info about print marks          **
 **     COMMENTS:                                                          **
 ****************************************************************************/
int MV_FreePrintMarks()
	{
	while (iPM != -1)
		{
		GlobalFreePtr(PrintMarks[iPM].lpName);
		--iPM;
		}
	if (PrintMarks)
		GlobalFreePtr(PrintMarks);
	PrintMarks = 0;
	return(0);
	}

/****************************************************************************
 **     FUNCTION: MV_PrintAll                                              **
 **     PURPOSE: Print all topics                                          **
 **     COMMENTS:                                                          **
 ****************************************************************************/
int MV_PrintAll(LPMV lpMV, HDC hdc)
	{
	long i, numTopics;
	LPPRINTTOPIC lpPT;
	HTITLE hTitle = hMVGetTitle(lpMV, NULL);
	int err;

	/* create a list of VAs of all topics in the title */
	numTopics = lMVTitleGetInfo(hTitle, TTLINF_NUMTOPICS, 0, 0);
	lpPT = (LPVOID)GlobalAllocPtr(GHND, numTopics * sizeof(PRINTTOPIC));
	if (!lpPT)
		return(wERRS_OOM);
	for (i = 0; i < numTopics; ++i)
		{
		lpPT[i].va = vaMVConvertTopicNumber(hTitle, i);
		lpPT[i].hTitle = hTitle;
		}
	
	/* print the list */
	err = MV_PrintTopicList(lpMV, hdc, lpPT, numTopics);
	GlobalFreePtr(lpPT);
	return(err);
	}

/****************************************************************************
 **     FUNCTION: MV_PrintMarkList                                         **
 **     PURPOSE: Print all the marked topics                               **
 **     COMMENTS:                                                          **
 ****************************************************************************/
int MV_PrintMarkList(LPMV lpMV, HDC hdc)
	{
	int num, err;

	LPPRINTTOPIC lpPT = MV_CreateTopicList(lpMV, &num);

	if (!lpPT)
		return(wERRS_OOM);
	err = MV_PrintTopicList(lpMV, hdc, lpPT, num);
	GlobalFreePtr(lpPT);
	return(err);
	}

/****************************************************************************
 **     FUNCTION: MV_CreateTopicList                                       **
 **     PURPOSE: Create a list of PRINTTOPIC structures from the PrintMark **
 **          list                                                          **
 **     COMMENTS:                                                          **
 ****************************************************************************/
LPPRINTTOPIC MV_CreateTopicList(LPMV lpMV, LPINT lpNum)
	{
	int i, j;
	HTLIST ht;
	LPPRINTTOPIC lpPT;
	long topicNum, topicLength, ii;
	HTITLE hTitle = hMVGetTitle(lpMV, NULL);

	/* how many topics do we need? */
	*lpNum = 0;
	for (i = 0; i <= iPM; ++i)
		switch (PrintMarks[i].type)
			{
			case IDC_TOPIC:
				++*lpNum;
				break;
			case IDC_GROUPS:
				/* get number of topics in the group */
				ht = hMVTopicListLoad(hTitle, PrintMarks[i].lpGroupName);
				if (ht)
					{
					topicLength = lMVTopicListLength(ht);
					*lpNum += (int)topicLength;
					MVTopicListDestroy(ht);
					}
				break;
			}

	/* allocate space for the list */
	lpPT = (LPVOID)GlobalAllocPtr(GHND, *lpNum * sizeof(PRINTTOPIC));
	if (!lpPT)
		return(0);

	/* and copy each VA to it */
	for (i = 0, j = 0; i <= iPM; ++i )
		switch (PrintMarks[i].type)
			{
			case IDC_TOPIC:
				lpPT[j].va = PrintMarks[i].va;
				lpPT[j++].hTitle = PrintMarks[i].hTitle;
				break;
			case IDC_GROUPS:
				/* get the HTLIST again */
				ht = hMVTopicListLoad(hTitle, PrintMarks[i].lpGroupName);
				if (!ht)
					continue;
				topicLength = lMVTopicListLength(ht);

				/* step through and get the VA for each topic in the group */
				for (ii = 0; ii < topicLength; ++ii)
					{
					topicNum = lMVTopicListLookup(ht, ii);
					lpPT[j].va = vaMVConvertTopicNumber(hTitle, topicNum);
					lpPT[j++].hTitle = PrintMarks[i].hTitle;
					}
				MVTopicListDestroy(ht);
				break;
			}

	return(lpPT);
	}

/****************************************************************************
 **     FUNCTION: MV_PrintCurrentTopic                                     **
 **     PURPOSE: Print the current topic                                   **
 **     COMMENTS:                                                          **
 ****************************************************************************/
int MV_PrintCurrentTopic(LPMV lpMV, HDC hdc)
	{
	PRINTTOPIC pt;
	INT subTopic;
	long scroll;

	MVGetAddress(lpMV, &pt.va, &subTopic, &scroll);
	pt.hTitle = hMVGetTitle(lpMV, NULL);
	return(MV_PrintTopicList(lpMV, hdc, &pt, 1));
	}

/****************************************************************************
 **     FUNCTION: MV_PrintTopicList                                        **
 **     PURPOSE: Print topics (both subTopics) from a list.                **
 **     COMMENTS:                                                          **
 ****************************************************************************/
int MV_PrintTopicList(LPMV lpMV, HDC hdc, LPPRINTTOPIC lpPT, long numTopics)
	{
	ERR err = wERRS_BADPRINT;
  	RECT rctPrint, rctPage;
	POINT pt;
	HCURSOR hCursor;
	DOCINFO di;
	HANDLE hName;
	int xScroll;
	int i;
	HWND hWnd = hwndMVGetWindow(lpMV);
	HTITLE hTitle = hMVGetTitle(lpMV, NULL);
	extern char far gErrBuff[];

	/* this might take a while */
	hCursor = SetCursor(LoadCursor(0, IDC_WAIT));

	/*
	 * Printing needs an lpMV for stepping through the topics, and a window
	 * for managing the layout.  Create a dummy Pane for this.
	 */
	hWnd = Pane_Create(hWnd, GetPaneInstance(lpMV), hTitle, SR_PANE, P_NOSIZE, 0);
	if (hWnd == 0)
		goto PrintError;
	lpMV = GetPaneMV(hWnd);
    
	/* Get the physical page size for this printer. */
	if( Escape(hdc, GETPHYSPAGESIZE, 0, NULL, (LPSTR)&pt) <= 0 )
		goto PrintError;

	/* start the logical print document */
	hName = hMVGetName(lpMV);
	di.cbSize = sizeof(DOCINFO);
	di.lpszOutput = NULL;
	di.lpszDocName = hName ? (LPSTR)GlobalLock(hName) : "";

	/* Provide a StartPage and EndPage for each topic.  MediaView will
	 * call back through plfnPageCallBack to ask for Start/End Page
	 * for any interim page breaks inside a topic.
	 */
	if (StartDoc(hdc, (DOCINFO FAR *)&di) == SP_ERROR)
		{
		err = wERRS_BADPRINT;
		goto PrintError;
		}

	/* for each topic in the topic List */
	for (i = 0; i < numTopics; ++i)
		{
		/* start the first page ... add header here if you want one */
		if (StartPage(hdc) <= 0)
			goto PrintError;

	  /* Set up the printing rectangle for each page (Topic). 
	   * With MM_TEXT mode, the following gives 1 inch margins.
		*/
		rctPage.left = GetDeviceCaps(hdc, LOGPIXELSX);
		rctPage.right = pt.x - rctPage.left;
		rctPage.top = GetDeviceCaps(hdc, LOGPIXELSY);
		rctPage.bottom = pt.y - rctPage.top;
		CopyRect((LPRECT)&rctPrint, (LPRECT)&rctPage);

		/* Get one region or the other */
		MV_SetAddressInTitle(lpPT[i].hTitle, lpMV, lpPT[i].va);

		/* if there is a non-scrolling region, print it */
		if (fMVHasNSR(lpMV))
			{
			xScroll = 0;
			if (!fMVPrintMedia(lpMV, hdc, FALSE, &rctPage, &rctPrint,
				&xScroll, MV_PageCallBack, 0L, &err))
				goto PrintError;
			}

		/* rctPrint describes the remaining space on the page for the scrolling region */
		if (fMVHasSR(lpMV))
			{
			fMVSetAddress(lpMV, lpPT[i].va, SR_PANE, 0, &err);
			xScroll = 0;
			if (!fMVPrintMedia(lpMV, hdc, FALSE, &rctPage,
				&rctPrint, &xScroll, MV_PageCallBack, 0L, &err))
				goto PrintError;
			}

		/* close the final page (of the topic) */
		if (EndPage(hdc) < 0)
			goto PrintError;
		}

	/* close the logical print document */
	EndDoc(hdc);
	err = 0;

	/* clean up and get out */
	PrintError:
	if (err != 0)
		{
		AbortDoc(hdc);

		/* FORMAT THE ERROR MESSAGE */
		if ((hName = hMVGetName(lpMV)) != 0)
			wsprintf(gErrBuff, "Print error %d in topic: %s", err, GlobalLock(hName));
		else
			wsprintf(gErrBuff, "Print error %d in topic #%ld", err, lMVTopicNumber(lpMV));
		}

	if (hCursor)
		SetCursor(hCursor);
	DeleteDC(hdc);
	if (hName)
		GlobalFree(hName);
 	if (hWnd)
		Pane_Destroy(hWnd);
	return err;
	}

/****************************************************************************
 **     FUNCTION: MV_PageCallBack                                          **
 **     PURPOSE: Print a topic (both subTopics).                           **
 **     COMMENTS:                                                          **
 **       This could be used to put headers or footers on each page        **
 ****************************************************************************/
BOOL _export far PASCAL MV_PageCallBack(LONG lNotUsed, HDC hdc, RECT rectPrinter,
                                             LPRECT lpPage, BOOL fStartPage)
	{
	/* MediaView tells when each interim EndPage or StartPage should occur */
	if( fStartPage )
		return (BOOL)(StartPage(hdc) > 0);
	else
		return (BOOL)(EndPage(hdc) >= 0);
	}

/****************************************************************************
 **     FUNCTION: MV_UpdateOptions                                         **
 **     PURPOSE: Propagate global conditions to an MV                      **
 **     COMMENTS:                                                          **
 ****************************************************************************/
BOOL MV_UpdateOptions(LPMV lpMV)
	{
	ERR err;
	
	/* turn on hotspot highlighting if appropriate */
	fMVHighlightHotspots(lpMV, bShowHotspots, &err);

	/* if there is a Kerning Boundary set, do it here */
	if (iKerningBoundary)
		MVSetKerningBoundary(lpMV, iKerningBoundary);

	/* don't display partial lines */
	MVHidePartialLines(lpMV, TRUE);

	return(TRUE);
	}

/****************************************************************************
 **     FUNCTION: MV_SetMagnifier                                          **
 **     PURPOSE: Set the magnification factor for the topic.               **
 **     COMMENTS:                                                          **
 **       fMVSetMagnifier destroys the VA in the lpMV, so we need to save  **
 **       it around the calls.                                             **
 ****************************************************************************/
int MV_SetMagnifier(LPMV lpMV, int mag)
	{
	ERR err = 0;
	VA va;
	int subTopic;
	long scroll;

	/* remember the VA */
	MVGetAddress(lpMV, &va, &subTopic, &scroll);

	if (!fMVSetMagnifier(lpMV, mag, &err))
		goto earlyExit;

	/* reset the address. */
	fMVSetAddress(lpMV, va, subTopic, scroll, (LPERR)&err);

	earlyExit:
	return(err);
	}


/****************************************************************************
 **     FUNCTION: MV_LoadKeyIndexes                                        **
 **     PURPOSE: Create a list of Keyword Index groups   .                 **
 **     COMMENTS:                                                          **
 ****************************************************************************/
int MV_LoadKeyIndexes(LPMV lpMV)
	{
	char buff[256];
	int count = 0, i;
	long c;
	HTITLE hTitle;
	LPGROUPUI lpGroup;

	/* how many groups are there? */
	hTitle = hMVGetTitle(lpMV, NULL);
	while (0 != lMVTitleGetInfo(hTitle, TTLINF_KEYINDEX, MAKELONG(sizeof(buff), count),
						(LONG)(LPSTR)buff))
		++count;

	if (count == 0)
		{
		lpKeyIndex = 0;
		return(0);
		}

	/* allocate 1 extra for a marker */
	lpGroup = (LPVOID)GlobalAllocPtr(GHND, sizeof(GROUPUI) * (count+1));
	if (!lpGroup)
		return(wERRS_OOM);

	/* read in the groups */
	for (i = 0; i < count; ++i)
		{
		c = lMVTitleGetInfo(hTitle, TTLINF_KEYINDEX, MAKELONG(sizeof(buff), i),
						(LONG)(LPSTR)buff);
		lpGroup[i].name = GlobalAllocPtr(GHND, 2);
		lpGroup[i].name[0] = (char)c;
		lpGroup[i].name[1] = 0;
		lpGroup[i].title = GlobalAllocPtr(GHND, strlen(buff) + 1);
		strcpy(lpGroup[i].title, buff);
		}

	/* leave an empty entry to mark the end of the array */
	lpGroup[count].name = lpGroup[count].title = 0;
	lpKeyIndex = lpGroup;
	return(0); 
	
	}

/****************************************************************************
 **     FUNCTION: MV_LoadGroups                                            **
 **     PURPOSE: Create a list of Search Groups in memory.                 **
 **     COMMENTS:                                                          **
 ****************************************************************************/
void MV_LoadGroups(LPMV lpMV)
	{
	char buff[256];
	int count = 0, i, j;
	HTITLE hTitle;
	LPGROUPUI lpGroup;
	char *p, *q;

	/* how many groups are there? */
	hTitle = hMVGetTitle(lpMV, NULL);
	while (-1 != lMVTitleGetInfo(hTitle, TTLINF_GROUPS,	MAKELONG(sizeof(buff), count),
						(LONG)(LPSTR)buff))
		++count;
 	if (count == 0)
		{
		lpGroups = 0;
		return;
		}

	/* allocate 1 extra for a marker */
	lpGroup = (LPVOID)GlobalAllocPtr(GHND, sizeof(GROUPUI) * (count+1));

	/* read in the groups */
	for (i = 0, j = 0; i < count; ++i)
		{
		lMVTitleGetInfo(hTitle, TTLINF_GROUPS,	MAKELONG(sizeof(buff), i),
						(LONG)(LPSTR)buff);

		/* 
		 * parse out the group name and title. The format is
		 * <GROUPNAME>.GRP "<GROUPTITLE>"
		 * There is a single space between the name and title.
		 * If there is no search string (i.e., the group is not searchable)
		 * then the <GROUPTITLE> is 0 length inside the quotes.
		 */
		 p = strchr(buff, '.');
		 *p = 0;

		/* make sure there actually is a title (group is searchable) */
		p = strchr(p+1, '\"');
		if (p && p[1] != '"')
			{
			/* copy the group name */
			lpGroup[j].name = GlobalAllocPtr(GHND, strlen(buff) + 1);
			strcpy(lpGroup[j].name, buff);

			/* now copy the group title */
			p++;
			q = strchr(p, '\"');
			*q = 0;
			lpGroup[j].title = GlobalAllocPtr(GHND, strlen(p) + 1);
			strcpy(lpGroup[j].title, p);

			++j;
			}
		}

	/* leave an empty entry to mark the end of the array */
	lpGroup[j].name = lpGroup[j].title = 0;
	lpGroups = lpGroup;
	}

/****************************************************************************
 **     FUNCTION: MV_ErrorMessage                                          **
 **     PURPOSE: Map the error number to a message   .                     **
 **     COMMENTS:                                             .            **
 ****************************************************************************/
void MV_ErrorMessage(WORD err, LPBYTE lpBuf, int cbBuff)
	{
	extern HINSTANCE ghInstance;
	LoadString(ghInstance, err, lpBuf, cbBuff);
	}

/****************************************************************************
 **     FUNCTION: MV_Hotspot                                               **
 **     PURPOSE: This is the default PANE hotspot handler hotspot handler  **
 **     COMMENTS:                                                          **
 **        This hotspot handler gets installed instead of the default      **
 **        handler (Pane_Hotspot).  It processes string hotspots as well   **
 **        as the usual HASH hotspots. All jumps are routed to the topic.  **
 ****************************************************************************/
BOOL CALLBACK MV_Hotspot(LPMV lpMV, LPMVHOTINFO lpHI)
	{
	VA va;
	HANDLE hTitle;
	HWND hWnd;
	extern LPTOPIC lpTopic;

	if (lpMV == 0)
		return(FALSE);
	
	switch (lpHI->nType)
		{
		case HOTSPOT_STRING:
			break;
		case HOTSPOT_HASH:
			/* a jump closes all open popups */
			hWnd = Pane_CloseAllPopups(hwndMVGetWindow(lpMV));

			/* this hWnd is the base Pane. Use its MV. */
			lpMV = GetPaneMV(hWnd);

			/* convert the hash data into a VA */
			hTitle = hMVGetTitle(lpMV, NULL);
			va = vaMVConvertHash(hTitle, (HASH)(lpHI->hash));
			/* in this model we handle the hotspot at the Topic level */
			MV_AddToHistory(Topic_ValidMV(lpTopic));
			if (!MV_ChangeAddress(0, lpTopic, va, 0))
				return(FALSE);
			break;
		case HOTSPOT_POPUPHASH:
			/* convert the hash data into a VA */
			hTitle = hMVGetTitle(lpMV, NULL);
			va = vaMVConvertHash(hTitle, (HASH)(lpHI->hash));

			if (!Pane_Popup(lpMV, &(lpHI->r), va))
				return(FALSE);
			break;

		case HOTSPOT_UNKNOWN:
		default:
			/* handle unknown hotspots here */
			return(FALSE);
		}
		
	return(TRUE);
	}


	
/****************************************************************************
 **     FUNCTION: MV_SetAddressInTitle                                     **
 **     PURPOSE: Set address in the MV watching out for an inter-file      **
 **        jump                                                            **
 **     COMMENTS:                                                          **
 **        Make sure that the address reflects a valid subTopic.           **
 ****************************************************************************/
LPMV MV_SetAddressInTitle(HTITLE hTitle, LPMV lpMV, VA va)
	{
	ERR err;
	
	/* check for invalid VA */
	if (va == vaNil)
		return(0);

	/* is this an interfile jump? */
	if (hTitle != hMVGetTitle(lpMV, 0))
		{
		if (!fMVSetTitle(lpMV, hTitle, 0, &err))
			return(0);
		}

	/* jump to the new topic ... be sure to get a valid subtopic */
	if (!fMVSetAddress(lpMV, va, NSR_PANE, 0, &err))
		if (!fMVSetAddress(lpMV, va, SR_PANE, 0, &err))
			return(0);

	return(lpMV);
	}

/****************************************************************************
 **     FUNCTION: MV_ChangeAddress                                         **
 **     PURPOSE: Update the display with new MV topics.                    **
 **     COMMENTS:                                                          **
 **       The model in this application is to only update the TOPIC window **
 **       (the TOC window stays constant).                                 **
 ****************************************************************************/
BOOL MV_ChangeAddress(HTITLE hTitle, LPTOPIC lpT, VA va, long scroll)
	{
	int statusP, statusN, statusC;
	LPMV lpMV = Topic_ValidMV(lpT);
	
	/* check for invalid VA */
	if (va == vaNil)
		return(0);

	/* are we crossing into another file? */
	if (hTitle && hTitle != hMVGetTitle(lpMV, NULL))
		{
		/* get the new title assigned to the Topic and TOC pane */
		lpT = UI_OpenFile(0, GetPaneInstance(lpMV), hTitle, 0);
		lpMV = Topic_ValidMV(lpT);
		}
 	Topic_SetTopic(lpT, va, scroll);

 	/*
 	 * If there is anything to do between setting the address and
	 * actually doing the layout, this is the spot.	In this
	 * application it is:
	 * 1. insert annotation markers for the new topic (scrolling region only).
	 * 2. process "macro" commands
	 * 3. Determine whether there are any visible search hits.
	 */

	if (fMVHasSR(GetPaneMV(lpT->hSR)))
		MV_UpdateSearchHits(GetPaneMV(lpT->hSR));
	if (fMVHasNSR(GetPaneMV(lpT->hNSR)))
		MV_UpdateSearchHits(GetPaneMV(lpT->hNSR));

 	/* now update the display */
	Topic_Layout(lpT);

	/*
	 * determine whether there are any Next or Prev topics in the group
	 * and grey out the UI appropriately. If there is a contents button,
	 * indicate visible/grey status.
	 */
	statusP = (addrMVGetPrev(lpMV) != addrNil);
	statusN = (addrMVGetNext(lpMV) != addrNil);
	statusC = (vaMVGetContents(hMVGetTitle(lpMV, NULL)) != va);
	UI_UpdateEnable(statusP, statusN, statusC);

	return(TRUE);
	}
	
/****************************************************************************
 **     FUNCTION: MV_NextPrev                                              **
 **     PURPOSE: Jump to the next or previous topic in the browse group    **
 **     COMMENTS:                                                          **
 ****************************************************************************/
BOOL MV_NextPrev(LPTOPIC lpT, int type)
	{
	LPMV lpMV = Topic_ValidMV(lpT);
	ADDR addr;
	VA va;
	HTITLE hTitle = hMVGetTitle(lpMV, NULL);

	/* if there is a Next or Prev, jump to it */
	if (type == IDC_NEXT)
		{
		if ((addr = addrMVGetNext(lpMV)) == addrNil)
			return(FALSE);
		}
	else
		{
		if ((addr = addrMVGetPrev(lpMV)) == addrNil)
			return(FALSE);
		}

	MV_AddToHistory(Topic_ValidMV(lpT));

	/* we have an address, convert to a VA and set the topic */
	va = vaMVConvertAddr(hTitle, addr);
	MV_ChangeAddress(0, lpT, va, 0);
	return(TRUE);
	}

