/*
 * This file contains the MVTopicClass code for the sample player.
 */

#include <stdlib.h>
#include <windows.h>
#include <windowsx.h>
#include <commdlg.h>
#include <string.h>
#include <stdarg.h>
#include <malloc.h>

/* MediaView Include Files */
#include <medv14.h>

/* Application Include Files */
#include "topic.h"
#include "player.h"
#include "pane.h"
#include "annotate.h"
#include "resrc1.h"
#include "proto.h"

/****************************************************************************
 **     FUNCTION: Topic_Register                                           **
 **     PURPOSE: Register the MVTopicClass                                 **
 **     COMMENTS:                                                          **
 ****************************************************************************/
BOOL Topic_Register(HANDLE hInstance)
	{
	WNDCLASS  wc;

	wc.style = 0;
	wc.lpfnWndProc = TopicWndProc;
	wc.cbClsExtra = 0;
	wc.cbWndExtra = EXTRA_TOPIC_SIZE;
	wc.hInstance = hInstance;
	wc.hIcon = LoadIcon (hInstance, "myicon");
	wc.hCursor = 0;
	wc.hbrBackground = GetStockObject(WHITE_BRUSH); 
	wc.lpszMenuName = 0;
	wc.lpszClassName = MVTOPICCLASS;
	if (RegisterClass(&wc) == 0)
		return(FALSE);
	}

/****************************************************************************
 **     FUNCTION: TopicWndProc                                             **
 **     PURPOSE: Handle top level messages to the Topic window             **
 **     COMMENTS:                                                          **
 ****************************************************************************/
long WINAPI TopicWndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
	{
	LPTOPIC lpT = GetTopicPtr(hWnd);
	HWND hWndPane;
	ERR err;
	extern HCURSOR curArrow;

	switch (message) 
		{
		case WM_SIZE:
			Topic_Layout(lpT);
			break;
		case WM_MOUSEMOVE:
			/*
			 * if we get this message, it is because the mouse is
			 * in a region of the Topic that has no Pane (Topic is bigger
			 * than the sum of the Panes. Be sure to set the cursor back to
			 * the arrow.
			 */
			SetCursor(curArrow);
			break;
		case WM_KEYDOWN:
			/* send keystrokes to the most recently clicked in region */
			if (lpT->hActive)
				Pane_KeyDown(lpT->hActive, wParam, lParam);
			break;
		case UWM_PANE_FOCUS:
			/* 
			 * One of the Panes got the "focus". Clear any
			 * selection from the non-active Pane.
			 */       
			hWndPane = (HWND)lParam;
			if (lpT->hActive && hWndPane != lpT->hActive)
				fMVClearSelection(GetPaneMV(lpT->hActive), &err);
			lpT->hActive = hWndPane;

			/* 
			 * Similarly, we notify the main window applications
			 * window procedure that this TOPIC has the TOPIC_FOCUS.
			 */
			SendMessage(lpT->hParent, UWM_TOPIC_FOCUS, 0, (long)hWnd);
			break;    
		default:
			return (DefWindowProc(hWnd, message, wParam, lParam));
		}
	return (0);
	}

/****************************************************************************
 **     FUNCTION: Topic_Create                                             **
 **     PURPOSE: Create a TOPIC and pass all of the necessary information  **
 **        to the children (Panes).                                        **
 **     ARGUMENTS:                                                         **
 **     RETURNS: 0 if successful, or a MediaView error code                **
 **     COMMENTS:                                                          **
 **        This routine creates a TOPIC window and binds it to each of     **
 **        a scrolling and non-scrolling region                            **
 **        (Panes) by doing a Pane_Create on each one.                     **
 **        The MV associated structures are created by the Panes.          **
 ****************************************************************************/
LPTOPIC Topic_Create(HINSTANCE hInstance, HTITLE hTitle, HWND hParent, DWORD style, LPSTR szInitialTopic, PANEHOTPROC fpHotProc)
	{
	VA va;
	LPTOPIC lpT = (LPTOPIC)GlobalAllocPtr(GHND, sizeof(TOPIC));

	/* create the TOPIC structure */
	if (!lpT)
		return(0);
	memset(lpT, 0, sizeof(TOPIC));

	/* create the topic window */
	lpT->hTopic = CreateWindow(
			MVTOPICCLASS,
			0,
        	style,
			CW_USEDEFAULT,
			CW_USEDEFAULT,
			CW_USEDEFAULT,
			CW_USEDEFAULT,
			hParent,
			0,
			hInstance,
			0 );
	
	/* Create the scrolling and non-scrolling region panes */
	if ((lpT->hSR = Pane_Create(lpT->hTopic, 
								hInstance,
								hTitle, 
								SR_PANE,
								P_NOSIZE,
								fpHotProc)) == 0)
		return(0);
  	if ((lpT->hNSR = Pane_Create(lpT->hTopic,
  								hInstance,
  								hTitle, 
  								NSR_PANE,
  								P_NOSIZE,
  								fpHotProc)) == 0)
		return(0);

	/* Set the Topic data */
	SetTopicPtr(lpT->hTopic, lpT);
	lpT->hParent = hParent;
 	lpT->isOpen = TRUE;

	/*
	 * If there is an initial topic, set the address. Otherwise
	 * set to the Contents topic.
	 */
	if (szInitialTopic)
		{
		va = vaMVConvertContextString(hTitle, szInitialTopic);
		if (va != vaNil)
			Topic_SetTopic(lpT, va, 0);
		else
			Topic_Contents(lpT);
		}
	else
		Topic_Contents(lpT);

	return(lpT);
	}

/****************************************************************************
 **     FUNCTION: Topic_Destroy                                            **
 **     PURPOSE: Destroy the Topic.  This includes closing each of the     **
 **       associated Panes                                                 **
 **     COMMENTS:                                                          **
 ****************************************************************************/
void Topic_Destroy(LPTOPIC lpT)
	{
	HWND hWnd = lpT->hTopic;

	/* is the topic fully open? */
	if (!lpT->isOpen)
		return;

	/* cleanup things that are open or allocated */
	Pane_Destroy(lpT->hSR);
	Pane_Destroy(lpT->hNSR);
	lpT->isOpen = FALSE;

	/* and destroy the TOPIC and Topic Window */
	GlobalFreePtr(lpT);
	DestroyWindow(hWnd);
	return;
	}

/****************************************************************************
 **     FUNCTION: Topic_Layout                                             **
 **     PURPOSE: Layout the topic in the Topic. This includes figuring out **
 **        how the window needs to be sized and the SR and NSR regions     **
 **        laid out.                                                       **
 **     COMMENTS:                                                          **
 ****************************************************************************/
int Topic_Layout(LPTOPIC lpT)
	{
	LPMV lpMVsr, lpMVnsr;
	POINT ptNSR;
	RECT rect;
	int x, y, yNSR;

	/* has the Topic been opened? */
	if (lpT == 0 || !lpT->isOpen)
		return(ERR_FAILED);

	/* get the MV pointers for each Pane */
	lpMVsr = GetPaneMV(lpT->hSR);
	lpMVnsr = GetPaneMV(lpT->hNSR);

	/* get the size of the Topic window */
	GetClientRect(lpT->hTopic, &rect);
	x = rect.right - rect.left;
	y = rect.bottom - rect.top;
	yNSR = 0;

	/*
	 * If the window is minimized, or if either dimension goes to 0,
	 * don't bother to Lay it out.  Furthermore, things break if you do ...
	 * the Pane size goes to 0 and the MediaView ptGetMVSize call pins to 0.
	 */
	if (x == 0 || y == 0)
		return(0);

	/* turn off updates until all the screen work is done */
	SendMessage(lpT->hTopic, WM_SETREDRAW, FALSE, 0);
 
	/* layout the NSR */
	if (fMVHasNSR(lpMVnsr))
		{
		/*
		 * We cannot get the NSR size without doing a layout (Realize) first.
		 * Let the NSR use up what it needs of the client rectangle, then
		 * allocate the rest to the SR.
		 */
		Pane_Layout(lpT->hNSR, &rect);
		ptNSR = ptMVGetSize(lpMVnsr);
		yNSR = min(y, ptNSR.y);
		MoveWindow(lpT->hNSR, 0, 0, x, yNSR, TRUE);
		ShowWindow(lpT->hNSR, SW_SHOW);
		}
	else
		ShowWindow(lpT->hNSR, SW_HIDE);

	/* size the scrolling region, allowing for a possible NSR */
	if (fMVHasSR(lpMVsr))
		{
		MoveWindow(lpT->hSR, 0, yNSR, x, y - yNSR, TRUE);

		/* the TRUE argument causes the fMVRealize to happen */
		Pane_Layout(lpT->hSR, 0);
		ShowWindow(lpT->hSR, SW_SHOW);
		}
	else
		ShowWindow(lpT->hSR, SW_HIDE);

	/* force the repainting now */
	SendMessage(lpT->hTopic, WM_SETREDRAW, TRUE, 0);
	InvalidateRect(lpT->hTopic, 0, TRUE);
	return(0);
	}

/****************************************************************************
 **     FUNCTION: Topic_SetTopic                                           **
 **     PURPOSE: Given a VA (address), set the topic to that address       **
 **     COMMENTS:                                                          **
 **       Setting the topic at the topic level keeps the Scrolling and     **
 **       non-Scrolling Panes in synch.                                    **
 ****************************************************************************/
BOOL Topic_SetTopic(LPTOPIC lpT, VA va, long scroll)
	{

	/* is this a valid address? */
	if (va == vaNil)
		return(FALSE);

	/* jump to the new topic */
	Pane_SetAddress(lpT->hNSR, va, scroll);
	Pane_SetAddress(lpT->hSR, va, scroll);
	return(TRUE);
	}

/****************************************************************************
 **     FUNCTION: Topic_Contents                                           **
 **     PURPOSE: Set the topic to the contents topic.                      **
 **     COMMENTS:                                                          **
 ****************************************************************************/
void Topic_Contents(LPTOPIC lpT)
	{
	VA va;
	LPMV lpMV = Topic_ValidMV(lpT);
	HTITLE hTitle = hMVGetTitle(lpMV, NULL);

	if (!lpT->isOpen)
		return;

	/* get the VA of the Contents topic */
	va = vaMVGetContents(hTitle);

	/* set the address; this will also do the layout */
	Topic_SetTopic(lpT, va, 0);
	}

/****************************************************************************
 **     FUNCTION: Topic_HotspotHighlights                                  **
 **     PURPOSE: Set hotspot highlighting for the Topic                    **
 **     COMMENTS:                                                          **
 ****************************************************************************/
void Topic_HotspotHighlights(LPTOPIC lpT, int OnOff)
	{
	ERR err;

	/* set hotspot highlighting for each Pane */
	fMVHighlightHotspots(GetPaneMV(lpT->hSR), OnOff, &err);
	fMVHighlightHotspots(GetPaneMV(lpT->hNSR), OnOff, &err);
	}



/****************************************************************************
 **     FUNCTION: Topic_HighlightLook                                      **
 **     PURPOSE: Set the search highlight color                            **
 **     COMMENTS:                                                          **
 ****************************************************************************/
void Topic_HighlightLook(LPTOPIC lpT, HBRUSH hH)
	{
	ERR err;

	/* set the brush for Search highlights */
	hMVSetHighlightLook(GetPaneMV(lpT->hNSR), hH, R2_XORPEN, &err);
	hMVSetHighlightLook(GetPaneMV(lpT->hSR), hH, R2_XORPEN, &err);

	/* force the repainting */
	Topic_Layout(lpT);
	}


/****************************************************************************
 **     FUNCTION: Topic_ValidMV                                            **
 **     PURPOSE: Get a valid MV pointer from the Topic                     **
 **     COMMENTS:                                                          **
 ****************************************************************************/
LPMV Topic_ValidMV(LPTOPIC lpT)
	{

	/*
	 * In order to get a valid VA from MediaView, we need to use
	 * a "valid" lpMV.  This means that we ask with an lpMV for
	 * a subTopic that actually exists (use hSR if it has a scrolling
	 * region, etc.). We are guaranteed that we have at least one
	 * subTopic (sometimes both).
	 */
	if (fMVHasSR(GetPaneMV(lpT->hSR)))
		return(GetPaneMV(lpT->hSR));
	else
		return(GetPaneMV(lpT->hNSR));
	}

