/*
 * This file contains miscellaneous MediaView handling code.
 */

#include <stdlib.h>
#include <windows.h>
#include <windowsx.h>
#include <commdlg.h>
#include <mmsystem.h>
#include <string.h>
#include <memory.h>
#include <ctype.h>
#include <stdarg.h>

/* MediaView Include Files */
#include <medv14.h>
#include <mverror.h>

/* Application Include Files */
#include "topic.h"
#include "player.h"
#include "pane.h"
#include "annotate.h"
#include "resrc1.h"
#include "proto.h"

/* The history list. */
typedef struct tagHISTORY
	{
	int inUse;				/* in use flag ... hName might be NULL */
	VA va;					/* topic address */
	long scroll;			/* the scroll position */
	int isName;				/* is the hName a name handle or a topic number? */
	HANDLE hName;			/* topic name */
	HTITLE hTitle;			/* title handle (for interfile jumps) */
	} HISTORY;
	
HISTORY History[NUMHISTORY] = {0};
int iH = -1;		/* History index */
int iB = -1;		/* back history index */






/****************************************************************************
 **     FUNCTION: MV_SearchInterrupt                                       **
 **     PURPOSE: Callback routine for searching.                           **
 **     COMMENTS:                                                          **
 **       MediaView will call this routine periodically to see if the      **
 **       search should be canceled. It is installed with MVTitleOpenEx.   **
 **       The EXPORT must be set because it is called back from within the **
 **       MediaView DLL.                                                   **
 ****************************************************************************/
BOOL EXPORT PASCAL MV_SearchInterrupt(LPVOID lpData, LPVOID lpOther)
 	{
	extern int gStopSearch;

	/* initially 0, this variable set if "Stop Search" button is pushed */
	return (gStopSearch);     
	}



/****************************************************************************
 **     FUNCTION: MV_AddToHistory                                          **
 **     PURPOSE: Add the current topic to the history list                 **
 **     COMMENTS:                                                          **
 **       Location 0 always contains the oldest entry.  If the array is	   **
 **       full, move everything down and add the new entry to the end.     **
 ****************************************************************************/
void MV_AddToHistory(LPMV lpMV)
	{

	/* is the History buffer full? If so, make room for more */
	if (iH == DIMENSION(History)-1)
		{
		if (History[0].isName)
			GlobalFree(History[0].hName);
		/* shift everything down by one */
		memmove(History, History+1, sizeof(HISTORY) * (iH));
		}
	else
		{
		/* otherwise bump the current History index */
		++iH;
		}

	History[iH].inUse = TRUE;
	MV_HistoryGetTopic(&History[iH], lpMV);
	History[iH].hTitle = hMVGetTitle(lpMV, NULL);

	/*
	 * update the Back pointer ... the strategy is to take it backward
	 * each time Back is performed, but to reset it to the History
	 * pointer when the SetTopic comes from any action the moves
	 * forward (such as a hotspot jump).
	 */
	iB = iH;

	/* if the history display is active, update it */
	if (hHist)
		MV_ShowHistory(lpMV, GetDlgItem(hHist, IDC_LIST1));
	}

/****************************************************************************
 **     FUNCTION: MV_HistoryGetTopic                                       **
 **     PURPOSE: Put the topic address, etc. into the HISTORY structure    **
 **     COMMENTS:                                                          **
 ****************************************************************************/
void MV_HistoryGetTopic(HISTORY * pH, LPMV lpMV)
	{
	int subTopic;

	MVGetAddress(lpMV, &pH->va, &subTopic, &pH->scroll);
	if ((pH->hName = hMVGetName(lpMV)) != 0)
		pH->isName = TRUE;
	else
		{
		pH->hName = (HANDLE)lMVTopicNumber(lpMV);
		pH->isName = FALSE;
		}
	}

/****************************************************************************
 **     FUNCTION: MV_ShowHistory                                           **
 **     PURPOSE: Write the history strings into the display                **
 **     COMMENTS:                                                          **
 ****************************************************************************/
void MV_ShowHistory(LPMV lpMV, HWND hwnd)
	{
	LPSTR lp;
	char buff[20];
	long topicNumber;
	int i;

	SendMessage(hwnd, LB_RESETCONTENT, 0, 0);
	if (iH < 0)
		return;

	for (i = iH; i >= 0; --i)
		{
		/* if there is a Topic Name, display it */
		if (History[i].isName)
			lp = GlobalLock(History[i].hName);
		else
			{
			/* otherwise just create a topic number string */
			topicNumber = lMVTopicNumber(lpMV);
			wsprintf(buff, "Topic #%d", (int)History[i].hName);
			lp = buff;
			}
		SendMessage(hwnd, LB_ADDSTRING, 0, (long)lp);
		}
	}

/****************************************************************************
 **     FUNCTION: MV_GoToFromHistory                                       **
 **     PURPOSE: Go to a new topic according to the history selection      **
 **     COMMENTS:                                                          **
 **       Watch out for interfile references.                              **
 ****************************************************************************/
void MV_GoToFromHistory(int index, LPVA lpVA, LPHTITLE lphTitle, LPLONG lpScroll)
	{
	*lpVA = History[index].va;
	*lphTitle = History[index].hTitle;
	*lpScroll = History[index].scroll;
	}

/****************************************************************************
 **     FUNCTION: MV_CleanupHistory                                        **
 **     PURPOSE: Free all of the History name strings                      **
 **     COMMENTS:                                                          **
 ****************************************************************************/
void MV_CleanupHistory()
	{
	int i;

	for (i = 0; i < DIMENSION(History); ++i)
		{
		if (History[i].inUse && History[i].isName)
			GlobalFree(History[i].hName);
		}
	memset(History, 0, sizeof(History));
	iH = 0;
	}

/****************************************************************************
 **     FUNCTION: MV_Back                                                  **
 **     PURPOSE: Go back one in the history list.                          **
 **     COMMENTS:                                                          **
 **        See MV_GoToFromHistory for comments on handling the back        **
 **        index.                                                          **
 ****************************************************************************/
int MV_Back(LPHTITLE lphTitle, LPVA lpVA, LPLONG lpScroll)
	{
	if (iB == -1)
		return(iB);


	MV_GoToFromHistory(iB, lpVA, lphTitle, lpScroll);
	return(--iB);
	}

/*
 * When we added the Back jump to the history list, it reset the Back pointer.
 * Undo that here.
 */
void MV_BackAdjust1(int oldB)
	{
	iB = oldB;
	}

/* catch the wrap around on the back pointer */
void MV_BackAdjust2()
	{

	/* if the list is full, everything moved down one to add the new entry */
	if (iH == DIMENSION(History)-1)
		--iB;
	}

 





/****************************************************************************
 **     FUNCTION: MV_ErrorMessage                                          **
 **     PURPOSE: Map the error number to a message   .                     **
 **     COMMENTS:                                             .            **
 ****************************************************************************/
void MV_ErrorMessage(WORD err, LPBYTE lpBuf, int cbBuff)
	{
	extern HINSTANCE ghInstance;
	LoadString(ghInstance, err, lpBuf, cbBuff);
	}

/****************************************************************************
 **     FUNCTION: MV_Hotspot                                               **
 **     PURPOSE: This is the default PANE hotspot handler hotspot handler  **
 **     COMMENTS:                                                          **
 **        This hotspot handler gets installed instead of the default      **
 **        handler (Pane_Hotspot).  It processes string hotspots as well   **
 **        as the usual HASH hotspots. All jumps are routed to the topic.  **
 ****************************************************************************/
BOOL CALLBACK MV_Hotspot(LPMV lpMV, LPMVHOTINFO lpHI)
	{
	VA va;
	HANDLE hTitle;
	HWND hWnd;
	extern LPTOPIC lpTopic;

	if (lpMV == 0)
		return(FALSE);
	
	switch (lpHI->nType)
		{
		case HOTSPOT_STRING:
			break;
		case HOTSPOT_HASH:
			/* a jump closes all open popups */
			hWnd = Pane_CloseAllPopups(hwndMVGetWindow(lpMV));

			/* this hWnd is the base Pane. Use its MV. */
			lpMV = GetPaneMV(hWnd);

			/* convert the hash data into a VA */
			hTitle = hMVGetTitle(lpMV, NULL);
			va = vaMVConvertHash(hTitle, (HASH)(lpHI->hash));
			/* in this model we handle the hotspot at the Topic level */
			MV_AddToHistory(Topic_ValidMV(lpTopic));
			if (!MV_ChangeAddress(0, lpTopic, va, 0))
				return(FALSE);
			break;
		case HOTSPOT_POPUPHASH:
			/* convert the hash data into a VA */
			hTitle = hMVGetTitle(lpMV, NULL);
			va = vaMVConvertHash(hTitle, (HASH)(lpHI->hash));

			if (!Pane_Popup(lpMV, &(lpHI->r), va))
				return(FALSE);
			break;

		case HOTSPOT_UNKNOWN:
		default:
			/* handle unknown hotspots here */
			return(FALSE);
		}
		
	return(TRUE);
	}


	
/****************************************************************************
 **     FUNCTION: MV_SetAddressInTitle                                     **
 **     PURPOSE: Set address in the MV watching out for an inter-file      **
 **        jump                                                            **
 **     COMMENTS:                                                          **
 **        Make sure that the address reflects a valid subTopic.           **
 ****************************************************************************/
LPMV MV_SetAddressInTitle(HTITLE hTitle, LPMV lpMV, VA va)
	{
	ERR err;
	
	/* check for invalid VA */
	if (va == vaNil)
		return(0);

	/* is this an interfile jump? */
	if (hTitle != hMVGetTitle(lpMV, 0))
		{
		if (!fMVSetTitle(lpMV, hTitle, 0, &err))
			return(0);
		}

	/* jump to the new topic ... be sure to get a valid subtopic */
	if (!fMVSetAddress(lpMV, va, NSR_PANE, 0, &err))
		if (!fMVSetAddress(lpMV, va, SR_PANE, 0, &err))
			return(0);

	return(lpMV);
	}

/****************************************************************************
 **     FUNCTION: MV_ChangeAddress                                         **
 **     PURPOSE: Update the display with new MV topics.                    **
 **     COMMENTS:                                                          **
 **       The model in this application is to only update the TOPIC window **
 **       (the TOC window stays constant).                                 **
 ****************************************************************************/
BOOL MV_ChangeAddress(HTITLE hTitle, LPTOPIC lpT, VA va, long scroll)
	{
	int statusP, statusN, statusC;
	LPMV lpMV = Topic_ValidMV(lpT);
	
	/* check for invalid VA */
	if (va == vaNil)
		return(0);

	/* are we crossing into another file? */
	if (hTitle && hTitle != hMVGetTitle(lpMV, NULL))
		{
		/* get the new title assigned to the Topic and TOC pane */
		lpT = UI_OpenFile(0, GetPaneInstance(lpMV), hTitle, 0);
		lpMV = Topic_ValidMV(lpT);
		}
 	Topic_SetTopic(lpT, va, scroll);

 	/*
 	 * If there is anything to do between setting the address and
	 * actually doing the layout, this is the spot.	In this
	 * application it is:
	 * 1. insert annotation markers for the new topic (scrolling region only).
	 * 2. process "macro" commands
	 * 3. Determine whether there are any visible search hits.
	 */


 	/* now update the display */
	Topic_Layout(lpT);

	/*
	 * determine whether there are any Next or Prev topics in the group
	 * and grey out the UI appropriately. If there is a contents button,
	 * indicate visible/grey status.
	 */
	statusP = (addrMVGetPrev(lpMV) != addrNil);
	statusN = (addrMVGetNext(lpMV) != addrNil);
	statusC = (vaMVGetContents(hMVGetTitle(lpMV, NULL)) != va);
	UI_UpdateEnable(statusP, statusN, statusC);

	return(TRUE);
	}
	
/****************************************************************************
 **     FUNCTION: MV_NextPrev                                              **
 **     PURPOSE: Jump to the next or previous topic in the browse group    **
 **     COMMENTS:                                                          **
 ****************************************************************************/
BOOL MV_NextPrev(LPTOPIC lpT, int type)
	{
	LPMV lpMV = Topic_ValidMV(lpT);
	ADDR addr;
	VA va;
	HTITLE hTitle = hMVGetTitle(lpMV, NULL);

	/* if there is a Next or Prev, jump to it */
	if (type == IDC_NEXT)
		{
		if ((addr = addrMVGetNext(lpMV)) == addrNil)
			return(FALSE);
		}
	else
		{
		if ((addr = addrMVGetPrev(lpMV)) == addrNil)
			return(FALSE);
		}

	MV_AddToHistory(Topic_ValidMV(lpT));

	/* we have an address, convert to a VA and set the topic */
	va = vaMVConvertAddr(hTitle, addr);
	MV_ChangeAddress(0, lpT, va, 0);
	return(TRUE);
	}

