/*
 * This file handles all of the user interface issues for
 * the MediaView example player.
 */
#include <stdlib.h>
#include <ctype.h>
#include <memory.h>	 
#include <windows.h>
#include <windowsx.h>
#include <commdlg.h>
#include <string.h>
#include <stdarg.h>
#include "ctl3d.h"

/* MediaView Include Files */
#include <medv14.h>

/* Application Include Files */
#include "topic.h"
#include "player.h"
#include "pane.h"
#include "annotate.h"
#include "resrc1.h"
#include "proto.h"

/*
 * This applicaton keeps around 1 TOPIC (window with scrolling and
 * non-scrolling regions), and a single PANE containing the contents
 * page of the current title.
 */
LPTOPIC lpTopic = {0};     
HWND hTOC = 0;

/*
 * Global Variables.
 */
int bShowHotspots = FALSE;	/* hotspots currently visible? */
int iKerningBoundary = 0;	/* kerning boundry for text */
HANDLE hAccTable = 0; 		/* handle to accelerator table */
HWND hMainWnd = 0;			/* handle to main window */
HWND hFP = 0;				/* face plate handle */
HWND hActiveTopic = 0;		/* Which Topic has the "focus" */
int numOpenTitles = 0;		/* number of open titles */
HTITLE *OpenTitles = 0;		/* table of open title handles */
char gErrBuff[512] = {0};	/* printer error buffer */
HWND hHist = 0;				/* handle of history list window */
HWND hSearch = 0;			/* handle of the search results window */
HANDLE hTopicList = 0;		/* search results topic list */
HANDLE hHighlights = 0;		/* search results highlights list */
int showHits = FALSE;		/* show the search highlights? */
HBRUSH hHitBrush = 0;		/* current brush for search hilights */
HANDLE hWordWheel = 0;		/* the current open word wheel */
int iIndex = 0;				/* the currently open index group number */
LPGROUPUI lpGroups = 0;		/* group list for the title */
LPGROUPUI lpKeyIndex = 0;	/* list of keyword index groups */
LPTOPIC lpSecondary = {0};	/* the secondary window */
HFILE hfAnnotation = 0;		/* Annotation file handle */
HINSTANCE ghInstance = 0;	/* global application instance */
int gStopSearch = FALSE;	/* Variable to indicate a search abort */
RECT rcSecondary = {160, 120, 480, 360};	/* Remember the secondary window position */

char szWindowTitle[] = "MediaView 1.4 Example Player";

LPSTR szInitialTopic = 0;

/* list of menu items that are grayed on entry, then enabled at file open */
int MenuItems[] =	{
			        ID_FILE_PRINTTOPIC,
        			ID_FILE_PRINTMARKED,
        			ID_FILE_PRINTALLTOPICS,
        			ID_EDIT_COPYTOPIC,
        			ID_EDIT_COPYSELECTION,
        			ID_EDIT_SEARCH,
        			ID_EDIT_CHARACTERSTYLES,
        			ID_BOOKMARK_DEFINE,
        			ID_TOPIC_INDEX,
	       			ID_TOPIC_MARK,
        			ID_VIEW_SEARCHRESULTS,
        			ID_VIEW_EMBEDDEDWINDOW,
					ID_VIEW_HOTSPOTS,
        			ID_OPTIONS_TEXTCOLOR,
        			ID_OPTIONS_HITCOLOR,
        			ID_OPTIONS_MAGNIFYTEXT,
					ID_OPTIONS_KERNING,
					ID_OPTIONS_SELECTCOLOR,
					};
/* variables for GetOpenFilename commmon dialog */
OPENFILENAME ofn = {0};
char szFilterSpec [128] =		/* file type filters */
             "MediaView 1.4 (*.M14)\0*.M14\0All Files (*.*)\0*.*\0";
char szFileName[_MAX_PATH] = {0};
char szFileTitle[_MAX_PATH] = {0};

/* variables for the Print common dialog */
PRINTDLG pd = {0};

/* variables for ChooseColor common dialog */
CHOOSECOLOR cc = {0};
COLORREF aclrCust[16] = {0};

/* cursors for different hotspot types */
HCURSOR curT = 0;
HCURSOR curB = 0;
HCURSOR curE = 0;
HCURSOR curU = 0;
HCURSOR curArrow = 0;
HCURSOR curIbeam = 0;
HCURSOR curWait = 0;

/* main Window Class */
#define MVAPPCLASS "MVAppClass"

/****************************************************************************
 **     FUNCTION: WinMain                                                  **
 **     PURPOSE: initialize and call do main processing loop               **
 **     COMMENTS:                                                          **
 ****************************************************************************/
int WINAPI WinMain(
	HINSTANCE hInstance,
	HINSTANCE hPrevInstance,
	LPSTR lpCmdLine,
	int nCmdShow)
	{
	MSG msg;  
	int ret = TRUE;
	int argc;
	LPSTR argv[10];
	LPSTR szFilename;

	ghInstance = hInstance;

	/* Enable the 3-D controls */
	Ctl3dRegister(hInstance);
	Ctl3dAutoSubclass(hInstance);
	argc = UI_ParseCmdLine(lpCmdLine, argv, DIMENSION(argv));
	/* Initialize the MV libraries */
	MVSetInstance(hInstance);

	if (!hPrevInstance)
		if (!InitApplication(hInstance))
			{
			ret = FALSE;
			goto exit;
			}
		
	if (!InitInstance(hInstance, nCmdShow))
			{
			ret = FALSE;
			goto exit;
			}

	/* do initial sizing of faceplate and source window */
	UI_ResizeFacePlate(hMainWnd, hFP, 0, 0);

	/* everything is set up, show the main window */
	ShowWindow(hMainWnd, SW_SHOW);

	/* was there a filename on the command line? */
	if (argc > 0)
		{
			szFilename = argv[0];

		if (UI_OpenFile(szFilename, hInstance, 0, szInitialTopic) != 0)
			{
			/* enable the button bar */
			UI_EnableControls(hFP, TRUE,
								IDC_INDEX,
								IDC_SEARCH,
								0);

			/* enable the portions of the menu that need an open file */
			UI_EnableMenuItems(hMainWnd, MenuItems, DIMENSION(MenuItems), MF_ENABLED|MF_BYCOMMAND);

			/*
			 * Test of Baggage file system APIs. This code does not
			 * do anything meaningful, it just demonstrates the use of
			 * these APIs.
			 */
			ret = UI_BaggageTest(OpenTitles[0]);
			}
		}

	/* the main message loop */
	while (GetMessage(&msg, 0, 0, 0)) 
		{ 
		/* modeless dialog boxes need custom dispatching */
		if (hSearch && IsDialogMessage(hSearch, &msg))
			continue;
		if (hHist && IsDialogMessage(hHist, &msg))
			continue;

		 /* otherwise, dispatch to the main message pump */
		else 
		if (!TranslateAccelerator(hMainWnd, hAccTable, &msg))
			{
			TranslateMessage(&msg);
			DispatchMessage(&msg); 
			}
		}
	ret = msg.wParam;
	exit:

	/* and terminate the MV libraries */
	MVTerminate(hInstance);

	/* clean up the 3 -D controls */
	Ctl3dUnregister(hInstance);
	return(ret);
	}


/****************************************************************************
 **     FUNCTION: InitApplication                                          **
 **     PURPOSE: initialize window data and registers window class         **
 **     COMMENTS:                                                          **
 ****************************************************************************/
BOOL InitApplication(HANDLE hInstance)
	{
	WNDCLASS  wc;

	wc.style = 0;
	wc.lpfnWndProc = MainWndProc;
	wc.cbClsExtra = 0;
	wc.cbWndExtra = 0;
	wc.hInstance = hInstance;
	wc.hIcon = LoadIcon (hInstance, "myicon");
	wc.hCursor = LoadCursor(0, IDC_ARROW);
	wc.hbrBackground = GetStockObject(LTGRAY_BRUSH); 
	wc.lpszMenuName =  "MVAppMenu";
	wc.lpszClassName = MVAPPCLASS;
	if (RegisterClass(&wc) == 0)
		return(FALSE);

	if (!Topic_Register(hInstance))
		return(FALSE);
		
	if (!Pane_Register(hInstance))
		return(FALSE);
	return(TRUE);
	}

/****************************************************************************
 **     FUNCTION: InitInstance                                             **
 **     PURPOSE: create the windows and create the button bar              **
 **     COMMENTS:                                                          **
 ****************************************************************************/
BOOL InitInstance(HANDLE hInstance, int nCmdShow)
	{
	
	hAccTable = LoadAccelerators(hInstance, "MVAppAccelerators");

	/* create the main application window */
	hMainWnd = CreateWindowEx(
#ifdef WIN32
			WS_EX_OVERLAPPEDWINDOW,
#else
			0,
#endif
			MVAPPCLASS,
			szWindowTitle,
    	    WS_OVERLAPPEDWINDOW,
			0,
			0,
			640,
			480,
			0,
			0,
			hInstance,
			0 );

	if (!hMainWnd)
		return (FALSE);      
		
	ShowWindow(hMainWnd, SW_HIDE);
	/* set the opeing topic of the right hand window */
	szInitialTopic = "opening";

	/* grey out the portions of the menu that need an open file */
	UI_EnableMenuItems(hMainWnd, MenuItems, DIMENSION(MenuItems), MF_GRAYED|MF_BYCOMMAND);
	UpdateWindow(hMainWnd);

	/* manifest the button bar */
	if ((hFP = CreateDialog(hInstance, "FacePlateDlg", hMainWnd, (DLGPROC)FacePlateDlgProc)) == 0)
		return(FALSE);

	/* fill in non-variant fields of OPENFILENAME struct. */
	ofn.lStructSize       = sizeof(OPENFILENAME);
	ofn.hwndOwner	  = hMainWnd;
	ofn.lpstrFilter	  = szFilterSpec;
	ofn.lpstrCustomFilter = 0;
	ofn.nMaxCustFilter	  = 0;
	ofn.nFilterIndex	  = 1;
	ofn.lpstrFile         = szFileName;
	ofn.nMaxFile	  		 = sizeof(szFileName);
	ofn.lpstrInitialDir   = 0;
	ofn.lpstrFileTitle    = szFileTitle;
	ofn.nMaxFileTitle     = sizeof(szFileTitle);
	ofn.lpstrTitle        = 0;
	ofn.lpstrDefExt       = "M14";
	ofn.Flags             = OFN_HIDEREADONLY;

	UI_InitCursors(hInstance);
	return (TRUE);
	}

/****************************************************************************
 **     FUNCTION: MainWndProc                                              **
 **     PURPOSE: Handle top level messages to the main window              **
 **     COMMENTS:                                                          **
 ****************************************************************************/
long WINAPI MainWndProc(HWND hWnd, UINT message, WPARAM wParam, LONG lParam)
	{
	UINT cmd;
	ERR err = 0;
	LPSTR szFilename;
#ifdef WIN32
	HINSTANCE hInst = (HINSTANCE)GetWindowLong(hWnd, GWL_HINSTANCE);
#else
	HINSTANCE hInst = (HINSTANCE)GetWindowWord(hWnd, GWW_HINSTANCE);
#endif	//WIN32

	switch (message) 
		{
#ifdef WIN32
		/* ALT-TAB to another application on NT loses the mouse
		 * capture, so re-capture it when re-activated.
		 */
		case WM_ACTIVATEAPP:
			{
			extern HWND hCaptureWnd;
			if (wParam == TRUE && hCaptureWnd)
				SetCapture(hCaptureWnd);
				break;
			}
#endif
		case WM_KEYDOWN:
			/* send keystrokes to the most recently clicked in region */
			if (hTOC && !hActiveTopic)
				
				SendMessage(hTOC, message, wParam, lParam);
			else
				if (hActiveTopic)
					SendMessage(GetTopicPtr(hActiveTopic)->hTopic, message, wParam, lParam);
				break;

		case WM_SIZE:
			UI_ResizeFacePlate(hWnd, hFP, (lpTopic?lpTopic->hTopic:0), hTOC);
			break;



		case WM_COMMAND:

			cmd = GET_WM_COMMAND_ID(wParam, lParam);
			switch (cmd) 
				{
				case ID_FILE_OPEN:
					/* call the common dialog to get a file name */
					if (!GetOpenFileName ((LPOPENFILENAME)&ofn))
						break;
					szFilename = ofn.lpstrFile;
					
					if (UI_OpenFile(szFilename, hInst, 0, szInitialTopic) != 0)
				 		{
						/* enable the button bar */
						UI_EnableControls(hFP, TRUE,
								IDC_INDEX,
								IDC_SEARCH,
								0);

						/* enable the portions of the menu that need an open file */
						UI_EnableMenuItems(hMainWnd, MenuItems, DIMENSION(MenuItems), MF_ENABLED|MF_BYCOMMAND);
						}
					else
						{
						/* disable buttons that need an open file */
						UI_EnableControls(hFP, FALSE,
								IDC_INDEX,
								IDC_SEARCH,
								0);

						/* disable the portions of the menu that need an open file */
						UI_EnableMenuItems(hMainWnd, MenuItems, DIMENSION(MenuItems), MF_GRAYED|MF_BYCOMMAND);
						}
					break;
				case ID_FILE_EXIT:
					DestroyWindow(hWnd);
					break;
				case ID_HELP_ABOUT:
					DialogBox(hInst, "AboutDlg", hWnd, AboutDlgProc);
					break;
				default:
					break;
				} 
			break;

		case WM_DESTROY:
			/* close down any open title */
			if (lpTopic && lpTopic->isOpen)
				UI_CloseFile();
			/* any other resources to free? */
			UI_CloseTitles();
			UI_DeleteCursors();  
			if ( hFP )
				DestroyWindow(hFP);
			if (hHitBrush)
				DeleteObject(hHitBrush);
			PostQuitMessage(0);            
			break;

		default:
			/* otherwise ... */
			return (DefWindowProc(hWnd, message, wParam, lParam));
		}
	return (0);
	}

/****************************************************************************
 **     FUNCTION: UI_ResizeFacePlate                                       **
 **     PURPOSE: Respond to size changes in the parent and resize the      **
 **       FacePlate.  This includes resizing the Topic display window.      **
 **     COMMENTS:                                                          **
 ****************************************************************************/
void UI_ResizeFacePlate(HWND hMain, HWND hFP, HWND hTopic, HWND hTOC)
	{
	RECT rect;
	int x, y;    
	int xT = 0;
	int xV = 0;
	
	/* size the whole face plate */
	GetClientRect(hMain, &rect);
	x = rect.right - rect.left;
	y = rect.bottom - rect.top;
	MoveWindow(hFP, 0, 0, x, y, TRUE);

	/* size the Topic display frame */
	#define FRAME 10
	#define BORDER 2
	GetClientRect(GetDlgItem(hFP, IDC_INDEX), &rect);
	UI_ConvertCoordinates(hFP, GetDlgItem(hFP, IDC_INDEX), &rect);
	/*******************************************************/
	/******** TUTORIAL: Insert Lesson 4 code here. *********/
	/*******************************************************/

	/* Topic */
	MoveWindow(GetDlgItem(hFP, IDC_STATIC), 2*FRAME+xT, rect.bottom + FRAME, xV, y - 2*FRAME - rect.bottom, TRUE);

	/* size the Topic with a 2 pixel border inside the right faceplate window */
	if (hTopic)
		UI_SizeTopicInWindow(hFP, IDC_STATIC, hTopic, BORDER);

	InvalidateRect(hFP, 0, TRUE);
	}

/****************************************************************************
 **     FUNCTION: UI_ConvertCoordinates                                    **
 **     PURPOSE: Convert the RECT coordiantes from one origin to another   **
 **     COMMENTS:                                                          **
 ****************************************************************************/
 void UI_ConvertCoordinates(HWND hTo, HWND hFrom, LPRECT lpR)
 	{
	POINT pt;

	pt.x = lpR->left;
	pt.y = lpR->top;
	ClientToScreen(hFrom, &pt);
	ScreenToClient(hTo, &pt);
	lpR->left = pt.x;
	lpR->top = pt.y;
	pt.x = lpR->right;
	pt.y = lpR->bottom;
	ClientToScreen(hFrom, &pt);
	ScreenToClient(hTo, &pt);
	lpR->right = pt.x;
	lpR->bottom = pt.y;
	}

/****************************************************************************
 **     FUNCTION: UI_SizeTopicInWindow                                     **
 **     PURPOSE: Size the Topic into a particular control in a window.     **
 **     COMMENTS:                                                          **
 **       The border allows the Topic to have a margin inside the control. **
 ****************************************************************************/
void UI_SizeTopicInWindow(HWND hWnd, int control, HWND hTopic, int border)
	{ 
	RECT rect;
	POINT pt;

	/* now size the Topic itself within the display frame */
	GetWindowRect(GetDlgItem(hWnd, control), &rect);
	pt.x = rect.left;
	pt.y = rect.top;
	ScreenToClient(hWnd, &pt);

	/* these force the layout and then the painting of the Topic */
	MoveWindow(hTopic, pt.x+border, pt.y+border, rect.right - rect.left-2*border, rect.bottom - rect.top-2*border, TRUE);
	InvalidateRect(hTopic, 0, TRUE);
	}

/****************************************************************************
 **     FUNCTION: FacePlateDlgProc                                         **
 **     PURPOSE: Dispatch the buttons from the button bar                  **
 **     COMMENTS:                                                          **
 ****************************************************************************/
BOOL WINAPI FacePlateDlgProc(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
	{   
	UINT	cmd;

	switch (message) 
		{
		case WM_INITDIALOG:
			UI_EnableControls(hDlg, FALSE,
											IDC_NEXTHIGHLIGHT,
											IDC_INDEX,
											IDC_BACK,
											IDC_HISTORY,
											IDC_NEXT,
											IDC_PREV,
											IDC_SEARCH,
											0);
			break;

		case WM_COMMAND:
			cmd = GET_WM_COMMAND_ID(wParam, lParam);
			switch (cmd)
				{
				}
			/*
			 * After a button is processed, put the focus back on the main
			 * window so that keystrokes will all be routed there.
			 */
			SetFocus(hMainWnd);
			break;
		}
	return (FALSE);
	}

/****************************************************************************
 **     FUNCTION: AboutDlgProc                                             **
 **     PURPOSE: About box handling                                        **
 **     COMMENTS:                                                          **
 ****************************************************************************/
BOOL WINAPI AboutDlgProc(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
	{
	UINT cmd;
	switch (message) 
		{
		case WM_INITDIALOG:
			SetFocus(hDlg);
			break;

		case WM_CLOSE:
			EndDialog(hDlg, FALSE);
			return(TRUE);
			break;

		case WM_COMMAND:
			cmd = GET_WM_COMMAND_ID(wParam, lParam);
			switch (cmd)
				{
				case IDOK:
					EndDialog(hDlg, TRUE);
					return(TRUE);
				}
			break;
		}
	return (FALSE);
	}
     

	         


/****************************************************************************
 **     FUNCTION: ErrorMsg                                                 **
 **     PURPOSE: show an error string dialog box                           **
 **     COMMENTS:                                                          **
 ****************************************************************************/
void ErrorMsg(LPSTR szText, LPSTR szCaption)
	{
	extern HWND hMainWnd;
	extern HWND hCaptureWnd;

	MessageBox (
					hMainWnd,
					szText,
					szCaption,
					MB_ICONASTERISK | MB_OK);

	/* MessageBox takes back the mouse. Reset the mouse capture. */
	if (hCaptureWnd)
		SetCapture(hCaptureWnd);
	}

/****************************************************************************
 **     FUNCTION: UI_EnableControls                                        **
 **     PURPOSE: Enable/Disable a set of controls                          **
 **     COMMENTS:                                                          **
 ****************************************************************************/
void UI_EnableControls(HWND hDlg, int status, ...)
	{
	va_list marker;
	int control;

	va_start(marker, status);
	while ((control = va_arg(marker, int)) != 0)
		EnableWindow(GetDlgItem(hDlg, control), status);

	va_end(marker);
	}

/****************************************************************************
 **     FUNCTION: UI_EnableMenuItems                                       **
 **     PURPOSE: Enable/Disable a set of menu items                        **
 **     COMMENTS:                                                          **
 ****************************************************************************/
void UI_EnableMenuItems(HWND hWnd, int * pItemList, int iCount, UINT uEnable)
	{
	HMENU hMenu = GetMenu(hWnd);
	while (iCount--)
		EnableMenuItem(hMenu, *pItemList++, uEnable);
	}

/****************************************************************************
 **     FUNCTION: UI_ToggleMenuItem                                        **
 **     PURPOSE: toggle the CHECKED/UNCHECKED state of the menu item       **
 **     COMMENTS:                                                          **
 ****************************************************************************/
int UI_ToggleMenuItem(HWND hWnd, int iMenuItem)
	{
	int state;

	state = GetMenuState(GetMenu(hWnd), iMenuItem, MF_BYCOMMAND);
	state = (state == MF_CHECKED ? MF_UNCHECKED : MF_CHECKED);
	CheckMenuItem(GetMenu(hWnd), iMenuItem, state);
	return(state);
	}

	
/****************************************************************************
 **     FUNCTION: UI_UpdateEnable                                          **
 **     PURPOSE: Enable/disable things in the UI according to the status   **
 **     COMMENTS:                                                          **
 ****************************************************************************/
void UI_UpdateEnable(int statusP, int statusN, int statusC)
	{

	HMENU hMenu = GetMenu(hMainWnd);



	}

/****************************************************************************
 **     FUNCTION: UI_InitCursors                                           **
 **     PURPOSE: Load up the cursors                                       **
 **     COMMENTS:                                                          **
 ****************************************************************************/
void UI_InitCursors(HINSTANCE hInst)
	{
	curT = LoadCursor(hInst, "HAND_T");
	curE = LoadCursor(hInst, "HAND_E");
	curB = LoadCursor(hInst, "HAND_B");
	curU = LoadCursor(hInst, "HAND_U");
	curArrow = LoadCursor(0, IDC_ARROW);
	curIbeam = LoadCursor(0, IDC_IBEAM);
	curWait = LoadCursor(0, IDC_WAIT);
	}

/****************************************************************************
 **     FUNCTION: UI_DeleteCursors                                         **
 **     PURPOSE: Free up the cursors                                       **
 **     COMMENTS:                                                          **
 ****************************************************************************/
void UI_DeleteCursors()
	{                  
	if ( curT )
		DestroyCursor(curT);
	if ( curE )
		DestroyCursor(curE);
	if ( curB )
		DestroyCursor(curB);
	if ( curU )
		DestroyCursor(curU);
	curT = curE = curB = curU = 0;
	
	/* no need to destroy the standard windows cursors. */
	}

/****************************************************************************
 **     FUNCTION: UI_CloseTitles                                           **
 **     PURPOSE: Close all open titles.                                    **
 **     COMMENTS:                                                          **
 ****************************************************************************/
void UI_CloseTitles()
	{
	int i;  
	if (OpenTitles)
		{
		for (i = 0; i < numOpenTitles; ++i)
			MVTitleClose(OpenTitles[i]);
		GlobalFreePtr(OpenTitles);
		OpenTitles = 0;
		}
	}

/****************************************************************************
 **     FUNCTION: UI_ParseCommandLine                                      **
 **     PURPOSE: Put the command line in the same form as a C program.     **
 **     COMMENTS:                                                          **
 ****************************************************************************/
int UI_ParseCmdLine(LPSTR lpC, LPSTR *argv, int maxArg)
	{
	int i;

	for (i = 0; i < maxArg; ++i)
		{
		if (*lpC == 0)
			break;
		while (*lpC && isspace(*lpC))
			++lpC;
		argv[i] = lpC;
		while (*lpC && !isspace(*lpC))
			++lpC;
		if (*lpC != 0)
			*lpC++ = 0;
		}
	return(i);
	}

LPTOPIC UI_OpenFile(LPSTR szFileName, HINSTANCE hInstance, HTITLE hTitle, LPSTR szInitialTopic)
	{          
	char buff[_MAX_PATH];
	HMENU hMenu = GetMenu(hMainWnd);
	
	/* destroy the secondary window if it exists */
	if (lpSecondary)
		SendMessage(lpSecondary->hTopic, WM_SYSCOMMAND, SC_CLOSE, 0);

	/* "close" any title that is already open */
	if (lpTopic && lpTopic->isOpen)
		UI_CloseFile();

	/* Open the title ... use WAIT cursor in case there is a delay */
	if (hTitle == 0)
		{
		SetCursor(curWait);
		hTitle = hMVTitleOpenEx(szFileName, (FARPROC)MV_SearchInterrupt, NULL);
		if (hTitle == 0)
			{
			ErrorMsg(szFileName, "MVTitleOpenEx fails");
			SetCursor(curArrow);
			return(0);
			}
		}

	/* save the Handle so we can later close it */
	if (numOpenTitles == 0)
		OpenTitles = (HTITLE *)GlobalAllocPtr(GHND, sizeof(HTITLE));
	else
		OpenTitles = (HTITLE *)GlobalReAllocPtr(OpenTitles,(numOpenTitles+1)*sizeof(HTITLE), GHND);
	OpenTitles[numOpenTitles++] = hTitle;

	/* create the right hand window TOPIC */
	lpTopic = Topic_Create(hInstance, hTitle, hFP, WS_CHILD, szInitialTopic, MV_Hotspot);
	if (lpTopic == 0)
		{
		SetCursor(curArrow);
		return(0);
		}
	hActiveTopic = lpTopic->hTopic;
		

	/*******************************************************/
	/******** TUTORIAL: Insert Lesson 4 code here. *********/
	/*******************************************************/

	/* position the whole thing correctly in the face plate */
	UI_ResizeFacePlate(hMainWnd, hFP, (lpTopic?lpTopic->hTopic:0), hTOC);
	/* set the title bar to the title */
	lMVTitleGetInfo(hTitle, TTLINF_TITLE, sizeof(buff)-1, (long)(LPSTR)buff);
	SetWindowText(hMainWnd, buff);

	SetCursor(curArrow);

	/* BUGBUG: map the DLL names for online or local DLLs */
	return(lpTopic);
	}

/****************************************************************************
 **     FUNCTION: UI_CreateTOC                                             **
 **     PURPOSE: Create the Table of Contents Pane.                        **
 **     COMMENTS:                                                          **
 **        The left side window will contain the Contents topic from the   **
 **        sample application. By setting its hotspot callback to          **
 **        MV_Hotspot all hotspot clicks in the TOC will only affect the   **
 **        right window.                                                   **
 ****************************************************************************/
 HWND UI_CreateTOC(HINSTANCE hInstance, HTITLE hTitle)
 	{
	HWND hWnd = 0;

	/*******************************************************/
	/******** TUTORIAL: Insert Lesson 4 code here. *********/
	/*******************************************************/
	return(hWnd);
	}

/****************************************************************************
 **     FUNCTION: UI_CloseFile                                             **
 **     PURPOSE: Close the open title.                                     **
 **     COMMENTS:                                                          **
 **       If the title is being "closed" in the middle of the application  **
 **       it is still in the history list, so only free the associated     **
 **       resources. 			   	                                       **
 ****************************************************************************/
 void UI_CloseFile()
 	{                   
	/* clear the display windows */
	if (lpTopic)
		Topic_Destroy(lpTopic);
	lpTopic = 0;
	if (hTOC)
		Pane_Destroy(hTOC);
	
	hTOC = 0;

	
	/* miscellaneous cleanup */
	if (hWordWheel)
		MVWordWheelClose(hWordWheel);
	hWordWheel = 0;
	iIndex = 0;	
	if (hTopicList)
		MVTopicListDestroy(hTopicList);
	hTopicList = 0;
	}


/****************************************************************************
 **     FUNCTION: UI_NextHighlight                                         **
 **     PURPOSE: Turn on the Next Highlight menu/button?                   **
 **     COMMENTS:                                                          **
 ****************************************************************************/
void UI_NextHighlight(BOOL state)
	{
	HMENU hMenu = GetMenu(hMainWnd);

	UI_EnableControls(hFP, state, IDC_NEXTHIGHLIGHT, 0);
	EnableMenuItem(hMenu, ID_VIEW_NEXTHIGHLIGHT, MF_BYCOMMAND | (state ? MF_ENABLED: MF_GRAYED));
	}

/*****************************************************************************
 **     FUNCTION: UI_BaggageTest											**
 **     PURPOSE: Test the Baggage APIs										**
 **     COMMENTS:															**
 **			The file "TEST.XYZ" was placed in baggage of the MYTHS.M14		**
 **			test title. This routine demonstrates the use of the Baggage	**
 **			APIs.															**
 *****************************************************************************/
BOOL UI_BaggageTest(HTITLE hTitle)
	{
	HMVFILE hf;
	HANDLE hMem;
	LPBYTE lpB;
	DWORD dw;
	char c;
	ERR err;

	/* 100th char in file is 99 */
	hMem = hMVBaggageGetFile(hTitle, "TEST.XYZ", TRUE, 0, 100, &dw, &err);
	if (hMem == 0)
		return(FALSE);
	lpB = GlobalLock(hMem);
	if (lpB[99] != 99)
		return(FALSE);
	GlobalFree(hMem);
	hMem = 0;
	
  	/* open, check that size is 100 */
  	hf = hMVBaggageOpen(hTitle, "test.xyz", TRUE, &err);
	if (hf == 0)
		return(FALSE);
 	if (lMVBaggageSize(hf, &err) != 100)
		return(FALSE);

	/* read value, seek to it */
 	lMVBaggageRead(hf, (LPBYTE)&dw, sizeof(DWORD), &err);
 	lMVBaggageSeek(hf, dw, 0, &err);

	/* read value, seek and read */
 	lMVBaggageRead(hf, (LPBYTE)&dw, sizeof(DWORD), &err);
	lMVBaggageSeekRead(hf, dw, 0, &c, 1, &err);
	if (c != 99)
		return(FALSE);

	/* close the Baggage file */
	MVBaggageClose(hf, &err);

	return(TRUE);
	}       
	


