/*
 * This file contains the MVPANECLASS window class code for the example player.
 */

#include <stdlib.h>
#include <windows.h>
#include <windowsx.h>
#include <commdlg.h>
#include <string.h>
#include <stdarg.h>

/* MediaView Include Files */
#include <medv14.h>

/* Application Include Files */
#include "topic.h"
#include "player.h"
#include "pane.h"
#include "resrc1.h"
#include "proto.h"

/* Global Variables for Panes */
HWND hCaptureWnd = 0;                           /* initially un-captured */

/* are we currently selecting? */
int selecting = FALSE;
HWND hSelectCapture = 0;

/*****************************************************************************
 **	FUNCTION: Pane_Register													**
 **	PURPOSE: Register the window class for the Pane							**
 **	COMMENTS:																**
 **		The MVPaneClass contains extra bytes for holding Pane data.			**
 **		See pane.h for details.												**
 *****************************************************************************/
BOOL Pane_Register(HINSTANCE hInstance)
	{
	WNDCLASS wc;

	/* create the Pane class */
	wc.style = CS_DBLCLKS;
	wc.lpfnWndProc = PaneWndProc;
	wc.cbClsExtra = 0;
	wc.cbWndExtra = EXTRA_PANE_SIZE;
	wc.hInstance = hInstance;
  	wc.hIcon = LoadIcon (hInstance, "myicon");
	wc.hCursor = 0;
	wc.hbrBackground = GetStockObject(WHITE_BRUSH); 
	wc.lpszMenuName =  0;
	wc.lpszClassName = MVPANECLASS;
	if (RegisterClass(&wc) == 0)
		return(FALSE);

	return(TRUE);
	}

/*****************************************************************************
 **	FUNCTION: Pane_Create													**
 **	PURPOSE: Create a new Pane.												**
 **	COMMENTS:																**
 **		This call creates a MediaView (MV) and a window for displaying		**
 **		it.																	**
 *****************************************************************************/
HWND Pane_Create(HWND hParentWnd, HINSTANCE hInstance, HTITLE hTitle, int type, int paneFlags, PANEHOTPROC fpHotProc)
	{
	HWND hWnd;
	LPMV lpMV;
	ERR err;
	LPPANE lpP;

	/* Create the PANE Window */
	hWnd = CreateWindowEx(
#if defined(WIN32) && !defined(NT350)
			WS_EX_CLIENTEDGE,
#else
			0,
#endif
			MVPANECLASS,
			"Pane",
			WS_CHILD|WS_VISIBLE|WS_HSCROLL|WS_VSCROLL,
			CW_USEDEFAULT,
			CW_USEDEFAULT,
			CW_USEDEFAULT,
			CW_USEDEFAULT,
			hParentWnd,
			0,
			hInstance,
			0 );
	if (hWnd == 0)
		return(0);

	/* make it initially invisible without scroll bars */
	ShowWindow(hWnd, SW_HIDE);
	ShowScrollBar(hWnd, SB_BOTH, FALSE);

	/* now Create the MediaView (MV) */
	lpMV = lpMVNew(hTitle, &err);
	if (lpMV == 0)
		return(0);

	/* bind MV to the Window handle */
	if (!fMVSetWindow(lpMV, hWnd, &err))
		return(0);

	/* allocate a structure to store the PANE data */
	if ((lpP = (LPPANE)GlobalAllocPtr(GHND, sizeof(PANE))) == 0)
		{
		MVDelete(lpMV);
		DestroyWindow(hWnd);
		return(0);
		}

	/* store away the MV and the type */
	SetPaneMV(hWnd, lpMV);
	SetPaneData(hWnd, lpP);

	/* initialize PANE data */
	lpP->iType =  type;
	lpP->flags = paneFlags;
	
	/* if there is no Hotspot callback pointer, use the default */
	if (fpHotProc == 0)
		fpHotProc = Pane_Hotspot;
	lpP->fpHotProc = fpHotProc;

	/*
	 * The Windows GetParent function will make the main window the parent
	 * of the first Popup.  We need the parent to be the base Pane,
	 * so we track all of this outside Windows.
	 */
	lpP->hWndParent =  hParentWnd;

	/* allow selection of embedded windows */
	fMVAllowEWSelection(lpMV, TRUE);
#ifdef DOCPLAYER
	MVSetKerningBoundary(lpMV, 100);
#endif

	return(hWnd);
	}

/*****************************************************************************
 **	FUNCTION: Pane_Duplicate												**
 **	PURPOSE: Create a new Pane that has all the properties of another.		**
 **	COMMENTS:																**
 **		This allows creation of a pane with all the attributes of an		**
 **		existing one, e.g., search hits list, text color, etc.				**
 *****************************************************************************/
HWND Pane_Duplicate(LPMV lpParentMV, int type)
	{
	HWND hWnd, hWndParent;
	LPMV lpMV;
	ERR err;
	long style = 0;
	LPPANE lpP, lpOldP;

	hWndParent = hwndMVGetWindow(lpParentMV);
	lpOldP = GetPaneData(hWndParent);

	/* Create the PANE Window */
	switch(type)
		{
		case POPUP_PANE:
			style = WS_BORDER|WS_POPUP;
			break;
		case SR_PANE:
			style = WS_CHILD|WS_VSCROLL|WS_HSCROLL;
			break;
		case NSR_PANE:
			style = WS_CHILD;
			break;
		}

	hWnd = CreateWindow(
			MVPANECLASS,
			"Pane",
			style,
			CW_USEDEFAULT,
			CW_USEDEFAULT,
			CW_USEDEFAULT,
			CW_USEDEFAULT,
			hwndMVGetWindow(lpParentMV),
			0,
			GetPaneInstance(lpParentMV),
			0 );
	if (hWnd == 0)
		return(0);

	/* make it initially invisible without scroll bars */
	ShowWindow(hWnd, SW_HIDE);
	ShowScrollBar(hWnd, SB_BOTH, FALSE);

	/* now Create the MediaView (MV) */
	lpMV = lpMVDuplicate(lpParentMV, &err);
	if (lpMV == 0)
		return(0);

	/* bind MV to the Window handle */
	if (!fMVSetWindow(lpMV, hWnd, &err))
		return(0);

	/* allocate a structure to store the PANE data */
	if ((lpP = (LPPANE)GlobalAllocPtr(GHND, sizeof(PANE))) == 0)
		{
		MVDelete(lpMV);
		DestroyWindow(hWnd);
		return(0);
		}

	/* save information with the handle */
	SetPaneMV(hWnd,lpMV);
	SetPaneData(hWnd, lpP);

	/* initialize PANE data */
	*lpP = *lpOldP;

	return(hWnd);
	}

/*****************************************************************************
 **	FUNCTION: Pane_Destroy													**
 **	PURPOSE: Destroy the Pane.												**
 **	COMMENTS:																**
 *****************************************************************************/
void Pane_Destroy(HWND hWnd)
	{
	LPMV lpMV = GetPaneMV(hWnd);
	LPPANE lpP = GetPaneData(hWnd);
	HTHLITE hHits;
	
	if (lpMV)
		{
		hHits = hMVGetHighlights(lpMV);

		/* undo any highlights */
		if (hHits)
			{
			GlobalFree(hHits);
			hHits = 0;
			}
		}

	if (lpP)
		GlobalFreePtr(lpP);

	/* destroy the MV */
	if (lpMV)
		MVDelete(lpMV);

	/* and destroy the window */
	DestroyWindow(hWnd);

	return;
	}

/*****************************************************************************
 **	FUNCTION: PaneWndProc													**
 **	PURPOSE: Process messages sent to the Pane.								**
 **	COMMENTS:																**
 *****************************************************************************/
long WINAPI PaneWndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
	{
	PAINTSTRUCT ps;
	HDC hDC;
	LPPANE lpP = GetPaneData(hWnd);

	switch (message) 
		{
#ifdef DOCPLAYER
		case WM_ERASEBKGND:
			if (lpP && lpP->iType != SR_PANE)
				{
				HBRUSH hBr = CreateSolidBrush(RGB(255,255,196));
				RECT rc;
				GetClientRect(hWnd, &rc);
				MVSetBkColor(GetPaneMV(hWnd), RGB(255,255,196));
				FillRect((HDC)wParam, &rc, hBr);
				DeleteObject(hBr);
				return(1);
				}
			else
				return (DefWindowProc(hWnd, message, wParam, lParam));
			break;
#endif

		case WM_KEYDOWN:
			Pane_KeyDown(hWnd, wParam, lParam);
			break;

		case WM_SIZE:
			/* if sizing is not suppressed, do the layout */
			if (lpP && !(lpP->flags & P_NOSIZE))
				Pane_Layout(hWnd, 0);
			break;

		case WM_PAINT:
			/* draw the Pane on the display */
			hDC = BeginPaint (hWnd, &ps);
        	Pane_Draw(hWnd, hDC, &(ps.rcPaint));
			EndPaint (hWnd, &ps);
			break;

		case WM_MOUSEMOVE:
			Pane_MouseMove(hWnd, wParam, lParam);
			break;

		case WM_LBUTTONUP:
			Pane_MouseLButtonUp(hWnd, wParam, lParam);
			break;

		case WM_HSCROLL:
			Pane_ScrollHorz(hWnd, wParam, lParam);
			break;

		case WM_VSCROLL:
			Pane_ScrollVert(hWnd, wParam, lParam);
			break;

		case WM_LBUTTONDOWN:
			Pane_MouseLButtonDown(hWnd, wParam, lParam);
			break;

		case WM_LBUTTONDBLCLK:
			Pane_MouseDoubleClick(hWnd, wParam, lParam);
			break;
			
		default:
			return (DefWindowProc(hWnd, message, wParam, lParam));
		}
	return (0);
	}

/*****************************************************************************
 **	FUNCTION: Pane_Draw														**
 **	PURPOSE: Repaint the pane.												**
 **	COMMENTS:																**
 *****************************************************************************/
int Pane_Draw(HWND hWnd, HDC hDC, LPRECT lpR)
	{
	ERR err;
	RECT rect;
	LPMV lpMV = GetPaneMV(hWnd);
	LPPANE lpP = GetPaneData(hWnd);

	if (lpMV == 0)
		return(ERR_FAILED);

	if (lpR == NULL)
		{
		GetClientRect (hWnd, &rect);
		lpR = &rect;
		}
	if (!fMVApplyToDC (lpMV, hDC, lpR, &err))
		return(err);

	/* the non-scrolling region needs a border at the bottom */
	if (lpP->iType == NSR_PANE && fMVHasSR(lpMV))
		{
		/* be sure to use the whole window rect, not just the update rect */
		GetClientRect (hWnd, &rect);
		MoveToEx(hDC, rect.left, rect.bottom-1, NULL);
		LineTo(hDC, rect.right, rect.bottom-1);
		}
	return(0);
	}

/*****************************************************************************
 **	FUNCTION: Pane_Layout													**
 **	PURPOSE: Realize the Pane ... this does not actually paint, that		**
 **		is done handling the WM_PAINT message.								**
 **	COMMENTS:																**
 *****************************************************************************/
void Pane_Layout(HWND hWnd, LPRECT lpR)
	{
	LPMV lpMV = GetPaneMV(hWnd);
	LPPANE lpP = GetPaneData(hWnd);
	ERR err;

	if (lpMV == 0)
		return;

	/* if it is a scrolling Pane, Realize and set the scroll bar ranges too */
	if (lpP->iType == SR_PANE)
		Pane_ScrollRanges(lpMV);
	else
		{
		/* this does the MediaView layout of the subTopic */
		fMVRealize(lpMV, lpR, (LPERR)&err);
		}

	return;
	}

/*****************************************************************************
 **	FUNCTION: Pane_CheckHorz												**
 **	PURPOSE: Check whether horizontal scrollbar is needed.					**
 **	COMMENTS:																**
 **		MediaView only does a layout on what is visible in the window, so	**
 **		as we scroll vertically we might uncover the need for a horizontal	**
 **		scrollbar later in the topic. For example, a wide table in the		**
 **		middle of text is only recognized to need scrollbars when it gets a	**
 **		layout. This code checks for just such a condition and adds a HORZ	**
 **		scroll bar if needed.												**
 *****************************************************************************/
void Pane_CheckHorz(HWND hWnd, LPMV lpMV)
	{
	int x, minx;
	POINT pt;
	ERR err;

	GetScrollRange(hWnd, SB_HORZ, &minx, &x);
	pt = ptMVGetScrollSizes(lpMV);

	/* the Scroll Range uses the default (100) if it is turned off */
	if (pt.x && x == 100)
		{
		/* 
		 * Change of state ... HORZ scrollbar appearing. Windows will not allow
		 * SetScrollRange to be called when processing a scrollbar notification
		 * message (which is being done here), so we cannot turn the scrollbar
		 * off until there is a size of topic change.
		 */
		SetScrollRange(hWnd, SB_HORZ, 0, pt.x, FALSE);
		Pane_ProportionScroll(hWnd, SB_HORZ);

 		/* this requires the Pane to be realized again */
		fMVRealize(lpMV, 0, &err);
		}
	}

	 
/*****************************************************************************
 **	FUNCTION: Pane_ScrollVert												**
 **	PURPOSE: handle vertical scrolling.										**
 **	COMMENTS:																**
 **		these messages only come to a scrolling Pane						**
 *****************************************************************************/
void Pane_ScrollVert(HWND hWnd, WPARAM wParam, LPARAM lParam)
	{
	LPMV lpMV = GetPaneMV(hWnd);
	LPPANE lpP = GetPaneData(hWnd);
	POINT pt;
	ERR err;
	int y;
	int yTop = 0, yBottom = 0;
	RECT rc;
	WORD pos, code;
	
	/* make sure it is a scrolling Pane */
	if (lpMV == 0 ||lpP->iType != SR_PANE)
		return;
	
	code = GET_WM_VSCROLL_CODE(wParam, lParam);
	pos = GET_WM_VSCROLL_POS(wParam, lParam);

	switch (code)
		{
		case SB_TOP:
		case SB_BOTTOM:
		case SB_LINEUP:
		case SB_LINEDOWN:
		case SB_PAGEUP:
		case SB_PAGEDOWN:
			/*
			 * MediaView returns the scroll pixels in pt.
			 * Just set the scroll position and pass them to Windows.
			 */
			Pane_CheckHorz(hWnd, lpMV);
			y = yMVScrollY(lpMV, &pt, &yTop, &yBottom, code, 0, &err);
			SetScrollPos (hWnd, SB_VERT, y, TRUE);
			ScrollWindow(hWnd, pt.x, pt.y, 0, 0);

			/* 
			 * When partial lines are hidden, there might be remnants
			 * that need cleaning up. yTop and yBottom will be 0 if 
			 * partial lines are allowed.
			 */
			if( (code != SB_TOP) && (code != SB_BOTTOM) )
				{
				GetClientRect(hWnd, &rc);
				rc.bottom = rc.top + yTop;
				InvalidateRect(hWnd, &rc, TRUE);
				UpdateWindow(hWnd);
				GetClientRect(hWnd, &rc);
				rc.top = rc.bottom - yBottom;
				InvalidateRect(hWnd, &rc, TRUE);
				UpdateWindow(hWnd);
				}
			break;

		case SB_THUMBPOSITION:
			/* Vertical scrolling might create the need for a horizontal scrollbar.*/
			Pane_CheckHorz(hWnd, lpMV);

			/* set the vertical scroll position in MediaView */
			yMVScrollY(lpMV, &pt, &yTop, &yBottom, code, pos, (LPERR)&err);
			RedrawWindow(hWnd, 0, 0, RDW_INVALIDATE | RDW_UPDATENOW | RDW_ERASE);
			break;

		case SB_THUMBTRACK:
			/* track thumb, but don't update the display until SB_THUMBPOSITION */
			SetScrollPos (hWnd, SB_VERT, pos, TRUE);
			break;
		}

	}                
	
/*****************************************************************************
 **	FUNCTION: Pane_ScrollHorz												**
 **	PURPOSE: handle horizontal scrolling (these messages only come			**
 **	to the scrolling region.												**	
 **	COMMENTS:																**
 *****************************************************************************/
void Pane_ScrollHorz(HWND hWnd, WPARAM wParam, LPARAM lParam)
	{
	LPMV lpMV = GetPaneMV(hWnd);
	LPPANE lpP = GetPaneData(hWnd);
	POINT pt;
	int err;
	int x;
	WORD pos, code;
	
	/* make sure it is a scrolling Pane */
	if (lpMV == 0 || lpP->iType != SR_PANE)
		return;
	
	code = GET_WM_HSCROLL_CODE(wParam, lParam);
	pos = GET_WM_HSCROLL_POS(wParam, lParam);

	switch (code)
		{
		case SB_TOP:
		case SB_BOTTOM:
		case SB_LINEUP:
		case SB_LINEDOWN:
		case SB_PAGEUP:
		case SB_PAGEDOWN:
			/*
			 * MediaView returns the scroll pixels in pt.
			 * Just set the scroll position and pass them to Windows.
			 */
			x = xMVScrollX(lpMV, &pt, code, 0, (LPERR)&err);
			SetScrollPos (hWnd, SB_HORZ, x, TRUE);
			ScrollWindow(hWnd, pt.x, pt.y, 0, 0);
			break;

		case SB_THUMBPOSITION:
			/* set the horizontal scroll position in MediaView */
			xMVScrollX(lpMV, &pt, code, pos, (LPERR)&err);
			RedrawWindow(hWnd, 0, 0, RDW_INVALIDATE | RDW_UPDATENOW | RDW_ERASE);
			break;

		case SB_THUMBTRACK:
			/* track thumb, but don't update the display until SB_THUMBPOSITION */
			SetScrollPos(hWnd, SB_HORZ, pos, TRUE);
			break;
		}
	}

/*****************************************************************************
 **	FUNCTION: Pane_ScrollRanges												**
 **	PURPOSE: Calculate the new scroll ranges								**
 **	COMMENTS:																**
 **		Adding a vertical scrollbar uses up horizontal space, and vice		**
 **		versa. When a scroll bar is added, check whether this creates a		**
 **		need for one in the other direction.								**
 *****************************************************************************/
void Pane_ScrollRanges(LPMV lpMV)
	{
	POINT pt;
	int x, y;
	HWND hWnd = hwndMVGetWindow(lpMV);
	int flags;
	int didVert = FALSE;
	ERR err;
	LPPANE lpP = GetPaneData(hWnd);

	/* turn off WM_SIZE processing to avoid infinite recursion */
	flags = lpP->flags;
	lpP->flags |= P_NOSIZE;
	
	/* start with no scrollbars on the window */
 	SetScrollRange(hWnd, SB_VERT, 0, 0, FALSE);
	SetScrollRange(hWnd, SB_HORZ, 0, 0, FALSE);
	fMVRealize(lpMV, 0, (LPERR)&err);
	
	/* have MediaView tell us the vertical scroll range, and pass it to Windows */
	pt = ptMVGetScrollSizes(lpMV);
	if (pt.y)
		{
		SetScrollRange(hWnd, SB_VERT, 0, pt.y, FALSE);
		Pane_ProportionScroll(hWnd, SB_VERT);

		/* this requires the Pane to be realized again */
		fMVRealize(lpMV, 0, &err);
		didVert = TRUE;
		}
	
	/* now do the same for the horizontal */
	pt = ptMVGetScrollSizes(lpMV);
	if (pt.x)
		{
		SetScrollRange(hWnd, SB_HORZ, 0, pt.x, FALSE);
 		Pane_ProportionScroll(hWnd, SB_HORZ);

 		/* this requires the Pane to be realized again */
		fMVRealize(lpMV, 0, &err);
		
		if (!didVert)
			{
			/* Adding HORZ without VERT could now make us need VERT */
			pt = ptMVGetScrollSizes(lpMV);
			if (pt.y)
				{
				SetScrollRange(hWnd, SB_VERT, 0, pt.y, FALSE);
				Pane_ProportionScroll(hWnd, SB_VERT);
		
				/* this requires the Pane to be realized again */
				fMVRealize(lpMV, 0, &err);
				}
			}
		}
 
	/* Set the thumb position */
	x = xMVGetXScrollPos(lpMV);
	y = yMVGetYScrollPos(lpMV);
	SetScrollPos(hWnd, SB_VERT, y, TRUE);
	SetScrollPos(hWnd, SB_HORZ, x, TRUE);
	
	/* set the flags back to the original value */
	lpP->flags = flags;
	}

/*****************************************************************************
 **	FUNCTION: Pane_MouseMove												**
 **	PURPOSE: handle the changing of the cursor over hotspots				**
 **	COMMENTS:																**
 *****************************************************************************/
void Pane_MouseMove(HWND hWnd, WPARAM wParam, LPARAM lParam)
	{
	POINT pt;
	MVPOINTINFO mvpi;
	LPMV lpMV = GetPaneMV(hWnd);
	LPPANE lpP = GetPaneData(hWnd);
	HINSTANCE hInst = GetPaneInstance(lpMV);
	ERR err;
	RECT rect;
	extern HCURSOR curT, curB, curE, curU, curArrow, curIbeam;
	extern HCURSOR curH;

	if (lpMV == 0)
		return;

#ifdef WIN32
	pt.x = MAKEPOINTS(lParam).x;
	pt.y = MAKEPOINTS(lParam).y;
#else
	pt.x = LOWORD(lParam);
	pt.y = HIWORD(lParam);
#endif

	/* if selecting, extend the selection. If we are not already in a
	 * hotspot, don't allow the cursor to change if we cross into one.
	 */
	if (selecting)
		{
		/* if this is a SR and the mouse moves outside the window, scroll */
		if (lpP->iType == SR_PANE)
			{
			GetClientRect(hWnd, &rect);
			if (pt.y < rect.top)
				Pane_ScrollVert(hWnd, SB_LINEUP, 0);
			else if (pt.y > rect.bottom)
				Pane_ScrollVert(hWnd, SB_LINEDOWN, 0);
			if (pt.x < rect.left)
				Pane_ScrollHorz(hWnd, SB_LINEUP, 0);
			else if (pt.x > rect.right)
				Pane_ScrollHorz(hWnd, SB_LINEDOWN, 0);
			}

		/* select characters up to this point */
		MVSelectPoint(lpMV, MVSEL_CHAR, pt, TRUE, (LPERR)&err);
		return;
		}

	/*
	 * If we are positioned over a hotspot, set the cursor
	 * according to the hotspot type.  Otherwise, use the IBEAM over
	 * text, or the ARROW over everything else.
	 */
	mvpi.dwSize = sizeof(mvpi);
	if (fMVGetPointInfo (lpMV, pt, &mvpi, NULL, &err))
		{
#ifdef DOCPLAYER
		SetCursor(curH);
#else
		switch (mvpi.ct)
			{
			case CONTENT_TEXT:
				SetCursor(curT);
				break;
			case CONTENT_BITMAP:
				SetCursor(curB);
				break;
			case CONTENT_WINDOW:
				SetCursor(curE);
				break;
			default:
				/* includes CONTENT_UNKNOWN */
				SetCursor(curU);
				break;
			}
#endif
		}
	else
		{
		/* not a hot spot ... is it over text? */
		if (mvpi.ct == CONTENT_TEXT)
			SetCursor(curIbeam);
		else
			SetCursor(curArrow);
		}
	}

/*****************************************************************************
 **	FUNCTION: Pane_MouseLButtonDown											**
 **	PURPOSE: handle mouse clicks											**
 **	COMMENTS:																**
 **		This is the routine that anchors a selection. It also handles		**
 **		clicks outside an active popup (who will have the mouse				**
 **		captured).															**
 *****************************************************************************/
void Pane_MouseLButtonDown(HWND hWnd, WPARAM wParam, LPARAM lParam)
	{
	ERR err;
	LPMV lpMV = GetPaneMV(hWnd);
	LPPANE lpP;
	HWND hOldWnd = hWnd;
	MVHOTINFO mvhi;
	POINT pt;

#ifdef WIN32
	pt.x = MAKEPOINTS(lParam).x;
	pt.y = MAKEPOINTS(lParam).y;
#else
	pt.x = LOWORD(lParam);
	pt.y = HIWORD(lParam);
#endif

	/* if there is a popup active, handle the click here */
	if (hCaptureWnd != 0)
		{
		/*
		 * returns an hWnd if processing continues, and may pass an
		 * ancestor hWnd. Note that pt will have been adjusted for the new
		 * Client coordinates.
		 */
		hWnd = Pane_PopupButtonDown(hWnd, wParam, &pt);
		}

	/* The popup window may have disappeared ... don't start selecting */
	if (hWnd == 0)
		{
		selecting = FALSE;
		hSelectCapture = 0;
		return;
		}

	/* hWnd might have changed */
	lpMV = GetPaneMV(hWnd);
 	lpP = GetPaneData(hWnd);

	/*
	 * In order to identify the Pane with the "focus" we send a message
	 * to the parent. Higher levels can then track where to send key strokes,
	 * which pane to clear the selection from, etc.
	 */
	 SendMessage(GetParent(hWnd), UWM_PANE_FOCUS, 0, (long)hWnd);

	/*
	 * If we are over a hotspot, disable selection and remember the hotspot so we can recognize
	 * a LBUTTONDOWN/LBUTTONUP click pair.
	 */
	/* do not begin selecting if we are over a hotspot */
	mvhi.dwSize = sizeof(mvhi);
	mvhi.lpBuffer = 0;
	mvhi.dwBufSize = 0;
	if (fMVGetPointInfo (lpMV, pt, NULL, &mvhi, &err))
		{
		SetCapture(hWnd);
		lpP->rcHotspot = mvhi.r;
		lpP->bClickPending = TRUE;
		return;
		}
	else
		{
		/* prepare for selection */
		lpP->bClickPending = FALSE;
		}

	/* are selections supressed for this pane? */
	if (lpP->flags & P_NOSELECT)
		return;

	/* anchor the selection (arg 4 == FALSE means start) */
	MVSelectPoint(lpMV, MVSEL_CHAR, pt, FALSE, (LPERR)&err);

	/*
	 * Capture the mouse so that we can detect selections outside the Pane
	 * and force scrolling. 
	 */
	hSelectCapture = SetCapture(hWnd);
	selecting = TRUE;

	}
/*****************************************************************************
 **	FUNCTION: Pane_MouseLButtonUp											**
 **	PURPOSE: handle mouse up												**
 **	COMMENTS:																**
 **		This is a completed "click", so tell MediaView.It also				**
 **		completes a selection, so process that too.							**
 *****************************************************************************/
void Pane_MouseLButtonUp(HWND hWnd, WPARAM wParam, LPARAM lParam)
	{
	POINT pt;
	LPMV lpMV = GetPaneMV(hWnd);
	LPPANE lpP = GetPaneData(hWnd);
	ERR err;
	char buff[256];		/* NOTE: 256 is the longest allowed hotspot string. */
	MVHOTINFO mvhi;
	MVPOINTINFO mvpi;

	if (lpMV == 0)
		return;

#ifdef WIN32
	pt.x = MAKEPOINTS(lParam).x;
	pt.y = MAKEPOINTS(lParam).y;
#else
	pt.x = LOWORD(lParam);
	pt.y = HIWORD(lParam);
#endif

	/* mark the end of the selection */
	if (selecting)
		{
		/* restore captured mouse and turn off the selection */
		selecting = FALSE;
		ReleaseCapture();
		if (hSelectCapture)
			SetCapture(hSelectCapture);
		MVSelectPoint(lpMV, MVSEL_CHAR, pt, TRUE, (LPERR)&err);
		return;
		}
 
	/*
	 * If there was a BUTTONDOWN on the hotspot, the mouse is captured.
	 * Make sure the hotspot number matches (i.e., the DOWN and UP are 
	 * on the same hotspot.
	 */
	if (lpP->bClickPending)
		{
		ReleaseCapture();
		lpP->bClickPending = FALSE;

		/* is it the same hotspot? */
		if (!PtInRect(&lpP->rcHotspot, pt))
			return;
		/* get hotspot data if it exists */
		mvpi.dwSize = sizeof(mvpi);
		mvhi.dwSize = sizeof(mvhi);
		mvhi.lpBuffer = buff;
	  	mvhi.dwBufSize = sizeof(buff);
		fMVGetPointInfo (lpMV, pt, &mvpi, &mvhi, &err);

		/* if we are hot and there was enough space to copy the hotspot data */
		if (mvhi.dwBufSize >= mvhi.dwSizeNeeded)
			{
			/* process the hotspot data in the installable hotspot handler */
			(lpP->fpHotProc)(lpMV, (LPMVHOTINFO)&mvhi);
			}
		}
	}

/*****************************************************************************
 **	FUNCTION: Pane_MouseDoubleClick											**
 **	PURPOSE: handle the double click (select a word)						**
 **	COMMENTS:																**
 *****************************************************************************/
void Pane_MouseDoubleClick(HWND hWnd, WPARAM wParam, LPARAM lParam)
	{
	POINT pt;
	LPMV lpMV = GetPaneMV(hWnd);
	ERR err;
	extern HCURSOR curIbeam;

#ifdef WIN32
	pt.x = MAKEPOINTS(lParam).x;
	pt.y = MAKEPOINTS(lParam).y;
#else
	pt.x = LOWORD(lParam);
	pt.y = HIWORD(lParam);
#endif

	/* if we are over text, select the next word */
	if (GetCursor() == curIbeam)
		{
		MVSelectPoint(lpMV, MVSEL_WORD, pt, FALSE, &err);
		MVSelectPoint(lpMV, MVSEL_WORD, pt, TRUE, &err);
		}
	}


/*****************************************************************************
 **	FUNCTION: Pane_Hotspot													**
 **	PURPOSE: This is the default PANE hotspot handler hotspot handler		**
 **	COMMENTS:																**
 **		This is what is called if the Pane_Open was passed a NULL			**
 **		pointer for the fpHotspotProc argument.								**
 *****************************************************************************/
BOOL CALLBACK Pane_Hotspot(LPMV lpMV, LPMVHOTINFO lpHI)
	{
	VA va;
	HTITLE hTitle;
	HWND hWnd = hwndMVGetWindow(lpMV);

	if (lpMV == 0)
		return(FALSE);
	
	switch (lpHI->nType)
		{

		case HOTSPOT_HASH:
			/* a jump closes all open popups */
			hWnd = Pane_CloseAllPopups(hwndMVGetWindow(lpMV));

			/* this hWnd is the base Pane. Use its MV. */
			lpMV = GetPaneMV(hWnd);
			/* convert the hash data into a VA */
			hTitle = hMVGetTitle(lpMV, NULL);
			va = vaMVConvertHash(hTitle, (HASH)(lpHI->hash));

			/* notify MediaView that we are using the online connection */
			MVTitleNotifyLayout(hTitle, 0, TRUE);

			Pane_SetAddress(hWnd, va, 0);
			Pane_Layout(hWnd, 0);

			/* notify MediaView that we are done using the online connection */
			MVTitleNotifyLayout(hTitle, 0, FALSE);

			InvalidateRect(hWnd, 0, TRUE);
			break;

		case HOTSPOT_POPUPHASH:
			/* convert the hash data into a VA */
			hTitle = hMVGetTitle(lpMV, NULL);
			va = vaMVConvertHash(hTitle, (HASH)(lpHI->hash));

			if (!Pane_Popup(lpMV, &(lpHI->r), va))
				return(FALSE);
			break;

		case HOTSPOT_STRING:
			/* put string hotspot handling here */
			break;
			
		case HOTSPOT_UNKNOWN:
		default:
			/* handle unknown hotspots here */
			return(FALSE);
			break;
		}
	return(TRUE);
	}
 
/*****************************************************************************
 **	FUNCTION: Pane_SetAddress												**
 **	PURPOSE: Set the MediaView address.										**
 **	COMMENTS:																**
 *****************************************************************************/
int Pane_SetAddress(HWND hWnd, VA va, long scroll)
	{
	ERR err;
	LPMV lpMV = GetPaneMV(hWnd);
	LPPANE lpP = GetPaneData(hWnd);
	int iSubTopic;

	/* Set the address. We retrieve the subTopic from the hWnd */
	iSubTopic = lpP->iType;
	/* for a POPUP, try the scrolling region first */
	if (iSubTopic == POPUP_PANE)
		iSubTopic = SR_PANE;
	if (!fMVSetAddress(lpMV, va, iSubTopic, scroll, (LPERR)&err))
		{
		/* the failure might have been because this is a popup that only
		 * has a non-scrolling region. Try it again.
		 */
		if (lpP->iType == POPUP_PANE)
			{
			if (!fMVSetAddress(lpMV, va, NSR_PANE, scroll, (LPERR)&err))
				return(err);
			}
		else
			return(err);
		}

	/* the display update is done from the calling routine */
	return(0);
	}
 
/*****************************************************************************
 **	FUNCTION: Pane_Popup													**
 **	PURPOSE: manifest a topic in a POPUP window								**
 **	COMMENTS:																**
 **		Popups will cascade. If the mouse is clicked outside an active		**
 **		popup, this code will clean them up back to the window with the		**
 **		click.																**
 *****************************************************************************/
int Pane_Popup(LPMV lpParentMV, LPRECT lpR, VA va)
	{
	ERR err;
	HWND hParentWnd = hwndMVGetWindow(lpParentMV);
	LPMV lpMV;
	HWND hWnd;
	POINT ptxy;
	LPPANE lpP;

	/* create the popup Pane window (a copy of the parentMV) */
	hWnd = Pane_Duplicate(lpParentMV, POPUP_PANE);
	lpMV = GetPaneMV(hWnd);
	lpP = GetPaneData(hWnd);

	/* set the PANE attributes */
	lpP->iType = POPUP_PANE;
	lpP->hWndParent = hParentWnd;


	/* supress re-layout on size messages */
	lpP->flags |= P_NOSIZE;
		
	/* make an initial guess at the popup size */
	ptxy = Pane_PopupInitSize(hWnd, hParentWnd, lpR);
	
	/* duplicate the parent MV and set the subTopic address */
	Pane_SetAddress(hWnd, va, 0);

	/* do the layout */
	Pane_Layout(hWnd, 0);

	/* The Pane is Realized, so size and position it */
	Pane_PopupPosition(lpMV, hWnd, ptxy.x, ptxy.y);

	/* popup windows need to process mouseClicks to see when to disappear */
	SetCapture(hWnd);
	hCaptureWnd = hWnd;

	/* clear any selection in the parent MV */
	fMVClearSelection(lpParentMV, &err);
	return(TRUE);
	}

/*****************************************************************************
 **	FUNCTION: Pane_CloseAllPopups											**
 **	PURPOSE: If any Popups are in effect, close them.						**
 **	COMMENTS:																**
 *****************************************************************************/
HWND Pane_CloseAllPopups(HWND hWnd)
	{
	while (hCaptureWnd)
		hWnd = Pane_UnPopup(hCaptureWnd);
	return(hWnd);
	}

/*****************************************************************************
 **	FUNCTION: Pane_UnPopup													**
 **	PURPOSE: Clean up a popup window.										**
 **	COMMENTS:																**
 **	If the parent pane is also a popup, pass the mouse capture to it.		**
 *****************************************************************************/
HWND Pane_UnPopup(HWND hWnd)
	{
	LPMV lpMV = GetPaneMV(hWnd);
	LPPANE lpP = GetPaneData(hWnd);
	HWND hParentWnd = lpP->hWndParent;
	LPPANE lpPParent = GetPaneData(hParentWnd);

	if (lpPParent->iType == POPUP_PANE)
		{
		SetCapture(hParentWnd);
		hCaptureWnd = hParentWnd;
		}
	else
		{
		ReleaseCapture();
		hCaptureWnd = 0;
		}

	/* cean up the Pane */
	Pane_Destroy(hWnd);
	return(hParentWnd);
	}
 
/*****************************************************************************
 **	FUNCTION: Pane_PopupButtonDown											**
 **	PURPOSE: Handle ButtonDown in a popup window.							**
 **	COMMENTS:																**
 **		A click outside the popup closes the popup.There may be a			**
 **		chain of popups that need cleaning up down to the one getting		**
 **		the click.When called, the top Popup window has the					**
 **		mouse captured.														**
 *****************************************************************************/
HWND Pane_PopupButtonDown(HWND hWnd, WPARAM wParam, LPPOINT lpt)
	{
	HWND hParentWnd;
	LPPANE lpPParent;
	HWND hTargetWnd;
	RECT rect;

	/* was the click inside this popup? */
	ClientToScreen(hWnd, lpt);
	GetWindowRect(hCaptureWnd, &rect);
	hTargetWnd = WindowFromPoint(*lpt);
	if (hTargetWnd == hWnd)
		{
		ScreenToClient(hWnd, lpt);
		return(hWnd);
		}

	/* or was it inside an embedded window in this popup? */
	else if (PtInRect(&rect, *lpt))
		goto sendClick;

	/* otherwise it was outside the popup */
	else
		{
		/* No, pass the torch to the parent */
		hParentWnd = Pane_UnPopup(hWnd);

		/* if Parent is a POPUP, have it process the click */
		lpPParent = GetPaneData(hParentWnd);
		if (lpPParent->iType == POPUP_PANE)
			{
			/* Convert the point from screen to the parent's coord system */
			ScreenToClient(hParentWnd, lpt);
			return(Pane_PopupButtonDown(hParentWnd, wParam, lpt));
			}
		}

	/* the click is outside the popup chain */
	ReleaseCapture();
	hCaptureWnd = 0;

	/*      The click was outside the popup chain. Was it inside the base Pane? */
	if (hTargetWnd == hParentWnd)
		{
		ScreenToClient(hParentWnd, lpt);
		return(hParentWnd);
		}

	/* otherwise it is outside the Pane or in another application */
	sendClick:
	ScreenToClient(hTargetWnd, lpt);

	/* hTargetWnd might be 0 if the click was on the desktop */
	if (hTargetWnd)
		PostMessage(hTargetWnd, WM_LBUTTONDOWN, wParam, MAKELONG(lpt->x, lpt->y));
	SetFocus(hTargetWnd);
	return(0);
	}

/*****************************************************************************
 **	FUNCTION: Pane_PopupPosition											**
 **	PURPOSE: Calculate the best position for a popup window					**
 **	COMMENTS:																**
 **		Anchor it to the input x,y (the middle of the hotspot), but			**
 **		choose an orientation that keeps most of it on the screen.			**
 **		Note that x and y are already screen coordinates.					**
 *****************************************************************************/
void Pane_PopupPosition(LPMV lpMV, HWND hWnd, int x, int y)
	{
	POINT pt;
	int screenX = GetSystemMetrics(SM_CXSCREEN);
	int screenY = GetSystemMetrics(SM_CYSCREEN);
	int width,length;
	int Rslop, Bslop;

	/* what size do we really need? */
	pt = ptMVGetSize(lpMV);
	length = pt.y;
	width = pt.x;
#ifdef DOCPLAYER
	/* keep MediaView from doing layout right up to right/bottom borders */
	length += 4;
	width += 4;
#endif

	/* how much goes off the screen to the right? */
	Rslop = x + width - screenX;
	if (Rslop > 0 )
		x -= width;

	/* if it still doesn't fit, center it on the screen */
	if (x < 0)
		x = (screenX - width) / 2;

	/* how much goes off the screen to the bottom? */
	Bslop = y + length - screenY;
	if (Bslop > 0 )
		y -= length;

	/* if it still doesn't fit, center it on the screen */
	if (y < 0)
		y = (screenY - length) / 2;
	
	MoveWindow(hWnd, x, y, width, length, TRUE);
	ShowWindow(hWnd, SW_SHOW);
	}

/*****************************************************************************
 **	FUNCTION: RectClientToScreen											**
 **	PURPOSE: ClientToScreen for both POINTs in the RECT						**
 **	COMMENTS:																**
 *****************************************************************************/
void RectClientToScreen(HWND hWnd, LPRECT lpR)
	{
	POINT pt;

	pt.x = lpR->left;
	pt.y = lpR->top;
	ClientToScreen(hWnd, &pt);
	lpR->left = pt.x;
	lpR->top = pt.y;
	pt.x = lpR->right;
	pt.y = lpR->bottom;
	ClientToScreen(hWnd, &pt);
	lpR->right = pt.x;
	lpR->bottom = pt.y;
	}

/*****************************************************************************
 **	FUNCTION: Pane_CopySelection											**
 **	PURPOSE: If there is a selection, copy it to the clipboard.				**
 **	COMMENTS:																**
 **		This will select all possible clipboard formats and post them		**
 **		to the clipboard.													**
 *****************************************************************************/
BOOL Pane_CopySelection(LPMV lpMV)
	{
	ERR err;
	HANDLE hMem;
	LPMVCOPYDATA lpCD;
	int i;

	/* is there an active selection? */
	if (!fMVIsSelected(lpMV))
		return(FALSE);
	
	/* MediaView allocates memory for each COPYPAIR, but the clipboard will free it */
	if ((hMem = hMVCopyAdvanced(lpMV, TRUE, CF_ALL, &err)) != 0)
		{
		lpCD = (LPMVCOPYDATA)GlobalLock(hMem);
   		OpenClipboard(hwndMVGetWindow(lpMV));
 		EmptyClipboard();
		
		/* for each clipboard format */
		for (i = 0; i < lpCD->nNumHandles; ++i) 
			{
			/* set the data on the clipboard ... clipboard now owns the handle */
			SetClipboardData(lpCD->aCopyPairs[i].cfType,
  													lpCD->aCopyPairs[i].hShared);
			}

		CloseClipboard();
		GlobalFree(hMem);
		return(TRUE);
		}
	return(FALSE);
	}

/*****************************************************************************
 **	FUNCTION: Pane_KeyDown													**
 **	PURPOSE: Process the keydown message.									**
 **	COMMENTS:																**
 *****************************************************************************/
void Pane_KeyDown(HWND hWnd, WPARAM wParam, LPARAM lParam)
	{
	ERR err;
	LPMV lpMV = GetPaneMV(hWnd);
	LPPANE lpP = GetPaneData(hWnd);
	
	switch (wParam)
		{
		case VK_TAB:
			/* Tabbing through hot spots */
			fMVMoveFocus(lpMV, !(GetKeyState(VK_SHIFT) & 0x8000), &err);
			break;

		case VK_RETURN:
			{
			/* simulate "clicking" on a hot spot */
			BYTE  buff[256];			/* 256 is longest hotspot string */	
			MVHOTINFO mvhi;

			mvhi.dwSize = sizeof(mvhi);
			mvhi.lpBuffer = buff;
		  	mvhi.dwBufSize = sizeof(buff);

			/* if we are hot and there was enough space to copy the hotspot data */
			if (fMVFocusInfo (lpMV, &mvhi, &err) && (mvhi.dwBufSize >= mvhi.dwSizeNeeded))
				{
				(lpP->fpHotProc)(lpMV, (LPMVHOTINFO)&mvhi);
				}
			break;
			}

		/* text selection using keys */
		case VK_UP:
			MVSelectKey(lpMV, MVKEY_UP, (GetKeyState(VK_SHIFT) & 0x8000), &err);
			break; 
		case VK_DOWN:
			MVSelectKey(lpMV, MVKEY_DOWN, (GetKeyState(VK_SHIFT) & 0x8000), &err);
			break;
		case VK_LEFT:
			MVSelectKey(lpMV, MVKEY_LEFT, (GetKeyState(VK_SHIFT) & 0x8000), &err);
			break;
		case VK_RIGHT:
			MVSelectKey(lpMV, MVKEY_RIGHT, (GetKeyState(VK_SHIFT) & 0x8000),&err);
			break;
		case VK_HOME:
			MVSelectKey(lpMV, MVKEY_START, (GetKeyState(VK_SHIFT) & 0x8000),&err);
			break;
		case VK_END:
			MVSelectKey(lpMV, MVKEY_END, (GetKeyState(VK_SHIFT) & 0x8000), &err);
			break;
		
		/* ctrl-C is the common short cut for Copy Selection */
		case 'C':
			if (GetKeyState(VK_CONTROL) & 0x8000 )
				Pane_CopySelection(lpMV);
			break;

		/* scrolling using keys */
		case VK_NEXT:
			Pane_ScrollVert(hWnd, SB_PAGEDOWN, 0);
			break;
		case VK_PRIOR:
			Pane_ScrollVert(hWnd, SB_PAGEUP, 0);
			break;

		/* if any popups are in effect, wind them back */
		case VK_ESCAPE:
			Pane_CloseAllPopups(0);
			break;
		}
	}

/*****************************************************************************
 **	FUNCTION: Pane_PopupInitSize											**
 **	PURPOSE: Guess at the inital size of the popup.							**
 **	COMMENTS:																**
 **		The size of a popup can't be known until it is layed out. But		**
 **		we can't lay it out without having a window to lay it out in.		**
 **		The strategy we choose is to guess at an aesthetically pleasing 	**
 **		size, lay out the popup, then later shrink the window if it			**
 **		is too large.														**
 *****************************************************************************/
POINT Pane_PopupInitSize(HWND hWnd, HWND hParentWnd, LPRECT lpR)
	{
	int xLen, yLen;
	POINT pt;

	/*
	 * The coordinates are relative to the parent window.
	 * A popup needs to be positioned with Screen coordinates.
	 */
	RectClientToScreen(hParentWnd, lpR);

	/* guess at an initial size and placement so we can Realize the MV */
	xLen = yLen = 400;
	pt.x = lpR->left + (lpR->right - lpR->left) / 2;
	pt.y = lpR->top + (lpR->bottom - lpR->top) / 2;
	ShowWindow(hWnd, SW_HIDE);
	MoveWindow(hWnd, pt.x, pt.y, xLen, yLen, FALSE);
	return(pt);
	}

/*****************************************************************************
 **	FUNCTION: Pane_ProportionScroll											**
 **	PURPOSE: Create Windows95 proportional thumb on the scrollbar.			**
 **	COMMENTS:																**
 *****************************************************************************/
void Pane_ProportionScroll(HWND hWnd, int iType)
	{

/* if you are using NT 3.50 or earlier, be sure to define NT350 for the preprocessor */
#if defined(WIN32) && !defined(NT350)
	RECT rc;
	SCROLLINFO si;
	int iTopic,iRange, iPage;
	POINT ptRange, ptTopic;
	LPMV lpMV = GetPaneMV(hWnd);
	
	/* get the scroll range, the topic size, and the page size (window dimension) */
	GetClientRect(hWnd, &rc);
	ptRange = ptMVGetScrollSizes(lpMV);
	ptTopic = ptMVEstimateSize(lpMV);
	if (iType == SB_HORZ)
		{
		iRange = ptRange.x;
		iPage = rc.right;
		iTopic = ptTopic.x;
		}
	else
		{
		iRange = ptRange.y;
		iPage =	rc.bottom;
		iTopic = ptTopic.y;
		}

	/* calculate the page size as a proportion of the range */
	si.nPage = (int)(((long)iPage * iRange + iTopic-1) / iTopic);

	/* now set the scrollbar proportion */
	si.fMask = SIF_PAGE | SIF_RANGE;
	si.cbSize = sizeof(SCROLLINFO);
	si.nMin = 0;
	si.nMax = iRange + si.nPage - 1;
	SetScrollInfo(hWnd, iType, &si, FALSE);
#endif
	}
