// SwitchTrack.java

/*
 * Copyright (c) 1996 by Corey Trager.
 *
 * Permission to use, copy, and distribute this software for
 * any purpose and without fee is hereby granted provided
 * that this copyright notice appears in all copies.
 *
 */

// graphically, a hybrid of StraightTrack and CurvedTrack.
// More logic, however, to support the automatic toggling
// of the switch.


import java.awt.Graphics;
import java.awt.Color;

class SwitchTrack extends Track {

	// Main lines
	Arc arc;
	Line ln;
	static final int LEFT = 0;
	static final int RIGHT = 1;
	int side;
	boolean toggle = true;
	
	public SwitchTrack (int x, int y, int side)
	{
		tieCnt = 5;
		connectionCnt = 3;
		this.side = side;
		
		instantiateObjects();
				
		ox = (double) x;
		oy = (double) y;

		restore ();		
	}


	void setMainLines () {
		if (side == RIGHT)
			arc = new Arc(
				new DblPoint(x,y+40),
				40,
				Geometry.FORTY_FIVE_DEGREES,
				Geometry.NINETY_DEGREES);
		else
			arc = new Arc(
				new DblPoint(x+40,y+40),
				40,
				Geometry.NINETY_DEGREES,
				Geometry.degreesToRadians (135.0));
			
		ln = new Line(new DblPoint(x,y),new DblPoint(x+40,y));
	}
	
	void setTies () {

		double up = ln.c.y + TIE_SIZE;
		double down = ln.c.y - TIE_SIZE;
		double inc = 10.0;
		
		if (side == RIGHT) {
			for (int i = 0; i < 3; i++) {
				ties[i].v1.x = ties[i].v2.x
					= ln.v1.x + inc + (i * inc);
				ties[i].v1.y = up;
				ties[i].v2.y = down;
			}
		}
		else {
			for (int i = 0; i < 3; i++) {
				ties[i].v1.x = ties[i].v2.x
					= ln.v2.x - inc - (i * inc);
				ties[i].v1.y = up;
				ties[i].v2.y = down;
			}
		}


		double centerX = arc.c.x;
		double centerY = arc.c.y;
		
		double theta;
		DblPoint vector = DblPoint.vector (Geometry.ORIGIN, arc.c);
		DblPoint pt = new DblPoint (0,0);
		
		for (int i = 3; i < tieCnt; i++) {
			ties[i].v1.x = centerX;
			ties[i].v1.y = centerY + 3;
			ties[i].v2.x = centerX;
			ties[i].v2.y = centerY - 3;
			if (side == RIGHT) {
				theta = Geometry.degreesToRadians (-15.0 * (i - 2));
				ties[i].rotate (vector, theta);
			}
			else {
				theta = Geometry.degreesToRadians (15.0 * (i - 2));
				ties[i].rotate (vector, theta);
			}
			pt.x = Math.sin (-theta) * arc.radius;
			pt.y = - Math.cos (theta) * arc.radius;
			ties[i].translate (pt);
		}
	}

	void setConnectionPoints () {
		if (side == RIGHT) {
			connectionPoints[0].x = ln.v1.x;
			connectionPoints[0].y = ln.v1.y;
			connectionPoints[1].x = ln.v2.x;
			connectionPoints[1].y = ln.v2.y;
			connectionPoints[2].x = arc.v2.x;
			connectionPoints[2].y = arc.v2.y;
		}
		else {
			connectionPoints[0].x = ln.v2.x;
			connectionPoints[0].y = ln.v2.y;
			connectionPoints[1].x = ln.v1.x;
			connectionPoints[1].y = ln.v1.y;
			connectionPoints[2].x = arc.v1.x;
			connectionPoints[2].y = arc.v1.y;
		}
		
		cx = ln.c.x;
		cy = ln.c.y;
	}

    public void restore() {

		x = ox;
		y = oy;
		super.restore();
		setMainLines();
		setTies();
		setConnectionPoints();
    }

	public void move(int argX, int argY) {

		DblPoint vector = ln.moveTo (
			(double) argX, (double) argY);

		arc.translate(vector);
		
		for (int i = 0; i < tieCnt; i++)
			ties[i].translate(vector);

		setConnectionPoints();

    }

    public void rotate(double theta) {
	
		DblPoint vector;

		if (transformMode == ROTATE_MODE_XY) {
			if (side == RIGHT)
				vector = DblPoint.vector (Geometry.ORIGIN, ln.v1);
			else
				vector = DblPoint.vector (Geometry.ORIGIN, ln.v2);
		}
		else if (transformMode == ROTATE_MODE_PQ) {
			if (side == RIGHT)
				vector = DblPoint.vector (Geometry.ORIGIN, ln.v2);
			else
				vector = DblPoint.vector (Geometry.ORIGIN, ln.v1);
		}
		else
			vector = DblPoint.vector (Geometry.ORIGIN, ln.c);

		ln.rotate(vector, theta);
		arc.rotate(vector, theta);
		for (int i = 0; i < tieCnt; i++)
			ties[i].rotate(vector, theta);
		
		setConnectionPoints();
    }

	public void draw (Graphics g) {

		if (transformMode == NOT_GRIPPED)
			g.setColor(Color.black);
		else
			g.setColor(Color.blue);
		
		ln.draw (g);
		arc.draw (g);
		
		super.draw(g);
	}

	// Get track connected to this track which is
	// NOT the same as the incoming track
	int getNext (int id) {

		// check the connection at the
		// single end;
		
		if (connections[0] == id) {
			if (toggle) {
				if (connections[1] != EMPTY) {
					toggle = false;
					return connections[1];
				}
				else {
					return -1;
				}
			}
			else {	
				if (connections[2] != EMPTY) {
					toggle = true;
					return connections[2];
				}
				else {
					return -1;
				}	
			}
		}
		else {
			if (connections [0] != EMPTY)
				return connections[0];
			else
				return -1;
		}
	}

	int appendRunPoints (DblPoint pts[], int index, int id) {
		// coming in from the single side - toggle
		// the way we leave
		if (connections[0] == id) {
			if (toggle) {
				pts[index] = new DblPoint (ties[0].c);
				index++;
				pts[index] = new DblPoint (ties[1].c);
				index++;
				pts[index] = new DblPoint (ties[2].c);
				index++;
				pts[index] = new DblPoint (connectionPoints[1]);
				index++;
			}
			else {
				pts[index] = new DblPoint (ties[3].c);
				index++;
				pts[index] = new DblPoint (ties[4].c);
				index++;
				pts[index] = new DblPoint (connectionPoints[2]);
				index++;
			}
		}
		else if (connections[1] == id) {
			// coming in via the straight track
			pts[index] = new DblPoint (ties[2].c);
			index++;
			pts[index] = new DblPoint (ties[1].c);
			index++;
			pts[index] = new DblPoint (ties[0].c);
			index++;
			pts[index] = new DblPoint (connectionPoints[0]);
			index++;
		} else {
			// coming in via the curved track
			pts[index] = new DblPoint (ties[4].c);
			index++;
			pts[index] = new DblPoint (ties[3].c);
			index++;
			pts[index] = new DblPoint (connectionPoints[0]);
			index++;
		}

		return index;
	}

	void prepareForCircuitInquiry ()
	{
		toggle = true;
	}
}
