/*
 * @(#)StockStreamParser.java	1.10 95/12/09 Jim Graham
 * 
 * Copyright (c) 1994-1995 Sun Microsystems, Inc. All Rights Reserved.
 *
 * Permission to use, copy, modify, and distribute this software
 * and its documentation for NON-COMMERCIAL or COMMERCIAL purposes and
 * without fee is hereby granted. 
 * Please refer to the file http://java.sun.com/copy_trademarks.html
 * for further important copyright and trademark information and to
 * http://java.sun.com/licensing.html for further important licensing
 * information for the Java (tm) Technology.
 * 
 * SUN MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. SUN SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 * 
 * THIS SOFTWARE IS NOT DESIGNED OR INTENDED FOR USE OR RESALE AS ON-LINE
 * CONTROL EQUIPMENT IN HAZARDOUS ENVIRONMENTS REQUIRING FAIL-SAFE
 * PERFORMANCE, SUCH AS IN THE OPERATION OF NUCLEAR FACILITIES, AIRCRAFT
 * NAVIGATION OR COMMUNICATION SYSTEMS, AIR TRAFFIC CONTROL, DIRECT LIFE
 * SUPPORT MACHINES, OR WEAPONS SYSTEMS, IN WHICH THE FAILURE OF THE
 * SOFTWARE COULD LEAD DIRECTLY TO DEATH, PERSONAL INJURY, OR SEVERE
 * PHYSICAL OR ENVIRONMENTAL DAMAGE ("HIGH RISK ACTIVITIES").  SUN
 * SPECIFICALLY DISCLAIMS ANY EXPRESS OR IMPLIED WARRANTY OF FITNESS FOR
 * HIGH RISK ACTIVITIES.
 */

import java.io.InputStream;
import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.StreamTokenizer;
import java.util.Vector;
import java.util.StringTokenizer;
import java.net.URL;
import java.awt.Color;

/**
 * A simple class to read a stock quote data stream and execute callbacks
 * on the (StockWatcher) parent with the parsed data.
 *
 * @author	Jim Graham
 * @version 	1.10, 12/09/95
 */
public class StockStreamParser implements Runnable {
    int numURLs;
    String dataURLs[];
    boolean isStatic[];

    String stocks;

    StockWatcher watcher;
    URL documentURL;

    boolean canFudge;
    int delay;
    int history;

    URL dataURL;
    InputStream dataStream;
    Thread kicker;

    public StockStreamParser(StockWatcher parent, URL docURL,
			     boolean fudge, int d, int h) {
	watcher = parent;
	documentURL = docURL;
	canFudge = fudge;
	delay = d;
	history = h;

	Vector URLlist = new Vector();
	String URLs = parent.getParameter("livedata");
	int numLive = 0;
	if (URLs != null) {
	    StringTokenizer st = new StringTokenizer(URLs, "|");
	    while (st.hasMoreTokens()) {
		String nextURL = (String) st.nextToken();
		URLlist.addElement(nextURL);
		numLive++;
	    }
	}
	URLs = parent.getParameter("staticdata");
	int numStatic = 0;
	if (URLs != null) {
	    StringTokenizer st = new StringTokenizer(URLs, "|");
	    while (st.hasMoreTokens()) {
		String nextURL = (String) st.nextToken();
		URLlist.addElement(nextURL);
		numStatic++;
	    }
	}
	if (numLive == 0 && numStatic == 0) {
	    URLs = parent.getParameter("data");
	    if (URLs == null) {
		URLs = "http://benden:8888/stock.dat";
	    }
	    URLlist.addElement(URLs);
	    String s = parent.getParameter("static");
	    if (s != null && s.equals("true")) {
		numStatic = 1;
	    } else {
		numLive = 1;
	    }
	}
	numURLs = numLive + numStatic;
	isStatic = new boolean[numURLs];
	for (int i = 0; i < numURLs; i++) {
	    isStatic[i] = (i >= numLive);
	}
	dataURLs = new String[numURLs];
	URLlist.copyInto(dataURLs);
    }

    public void setStock(String stock) {
	stocks = "/" + stock;
    }

    public void setStock(String stock[]) {
	stocks = "";
	for (int i = 0; i < stock.length; i++) {
	    stocks += "/" + stock[i];
	}
    }

    public void run() {
	int index = 0;
	boolean done = false;
	while (kicker != null && !done) {
	    try {
		String sendURL;
		if (isStatic[index]) {
		    sendURL = dataURLs[index];
		    int percent = sendURL.indexOf("%");
		    if (percent >= 0) {
			sendURL = sendURL.substring(0, percent) + stocks;
		    }
		} else {
		    sendURL = dataURLs[index]
			+ stocks
			+ (canFudge ? "?f" : "?")
			+ ((history == 0)
			   ? ("dle," + delay)
			   : ("dhle," + delay + "," + history + "s"));
		}
		dataURL = new URL(documentURL, sendURL);
		dataStream = new BufferedInputStream(dataURL.openStream());
		done = true;
		readStream(dataStream);
	    } catch(ThreadDeath td) {
		done = true;
	    } catch(IOException e) {
	    //} catch(Exception e) {
		//e.printStackTrace();
		//close();
		//throw e;
	    }
	    close();
	    if (!done) {
		index++;
		if (index == numURLs) {
		    index = 0;
		    try {
			Thread.sleep(5000);
		    } catch (InterruptedException e) {
			done = true;
		    }
		}
	    }
	}
	watcher.feedEOF(this);
	System.out.println("Exitting");
    }

    public void start() {
	kicker = new Thread(this);
	kicker.setPriority(Thread.MIN_PRIORITY);
	kicker.start();
    }

    public void stop() {
	Thread t = kicker;
	kicker = null;
	t.stop();
	close();
    }

    public synchronized void close() {
	try {
	    dataStream.close();
	} catch (Exception e) {
	}
	dataStream = null;
    }

    private static final int IGNORE = 0;
    private static final int QUOTE = 1;
    private static final int HIGH = 2;
    private static final int LOW = 3;
    private static final int CLOSE = 4;
    private static final int HIST_TIME = 5;
    private static final int HIST_VAL = 6;
    private static final int ERROR = 7;

    public void readStream(InputStream dataStream) throws IOException {
	StreamTokenizer st = new StreamTokenizer(dataStream);
	st.eolIsSignificant(false);
	int nextval = IGNORE;
	int histTime = 0;
	String symbol = null;
	while (kicker != null) {
	    if (dataStream.available() < 1) {
		watcher.flushQuotes();
	    }
	    int tok = st.nextToken();
	    if (kicker == null) {
		break;
	    }
	    switch (tok) {
	      case st.TT_EOF:
		System.out.println("EOF "+stocks);
		return;
	      case st.TT_NUMBER:
		switch (nextval) {
		case HIGH:
		    watcher.newHigh(symbol, st.nval);
		    break;
		case LOW:
		    watcher.newLow(symbol, st.nval);
		    break;
		case CLOSE:
		    watcher.newClose(symbol, st.nval);
		    break;
		case HIST_TIME:
		    histTime = (int) st.nval;
		    nextval = HIST_VAL;
		    continue;
		case HIST_VAL:
		    watcher.newQuote(symbol, st.nval, histTime);
		    break;
		case QUOTE:
		    watcher.newQuote(symbol, st.nval);
		    break;
		default:
		    System.out.println("Ignoring "+st.nval);
		    break;
		}
		nextval = IGNORE;
		break;
	      case st.TT_WORD:
		symbol = st.sval;
		if (nextval == ERROR) {
		    watcher.setFeedError(symbol, true);
		    nextval = IGNORE;
		} else {
		    if (nextval == IGNORE) {
			nextval = QUOTE;
		    }
		    watcher.setFeedError(symbol, false);
		}
		break;
	      default:
		int prefix = st.ttype;
		switch (prefix) {
		case '^':
		    nextval = HIGH;
		    break;
		case '_':
		    nextval = LOW;
		    break;
		case '<':
		    nextval = CLOSE;
		    break;
		case '#':
		    nextval = ERROR;
		    break;
		case '@':
		    nextval = HIST_TIME;
		    break;
		case '*':
		    // Server side fudge indicator - ignore it...
		    break;
		default:
		    System.out.println("Unknown quote token: "+st);
		    break;
		}
		break;
	    }
	}
    }
}
