// NTOLE.cpp

// (c) 1995 Microsoft Corporation. All rights reserved. 
// 		Developed by hip communications inc., http://info.hip.com/info/

extern "C" {
#include <windows.h>
#define WIN32_LEAN_AND_MEAN

#pragma  warning(disable: 4091)

#include "EXTERN.h"
#include "perl.h"

#include "XSub.h"
}

#include <winnt.h>

#ifdef _DEBUG
inline void ODS(LPSTR x)
{
	OutputDebugString(x);
	OutputDebugString("\n");
}

#if defined(UNICODE)
#define ODSOLE(x) ODS(x)
#else
inline void ODSOLE(LPCWSTR x)
{
	char bufA[256];
	WideCharToMultiByte(CP_ACP, NULL, x, -1, bufA, 256, NULL, NULL);
	ODS(bufA);
}
#endif

#else
#define ODS(x)
#define ODSOLE(x)
#endif

#define CROAK croak
#define SUCCESSRETURNED(x)	(x == ERROR_SUCCESS)
#define SETIV(index,value) sv_setiv(ST(index), value)
#define SETNV(index,value) sv_setnv(ST(index), value)
#define SETPV(index,string) sv_setpv(ST(index), string)
#define SETPVN(index, buffer, length) sv_setpvn(ST(index), (char*)buffer, length)
#define RETURNRESULT if(bSuccess){ XST_mYES(0); }\
					 else {	XST_mNO(0); }\
					 XSRETURN(1)



static char thisFile[] = __FILE__;
const DWORD NTOLE_MAGIC = 0x12344321;
const int bufferSize = 1024;

const LCID lcidDefault = (0x02 << 10) /* LOCALE_SYSTEM_DEFAULT */;

typedef struct _tagNTOLEObject *LPNTOLEOBJECT;
typedef struct _tagNTOLEObject
{
	long NTOLEMagic;
	LPNTOLEOBJECT pNext;
	LPNTOLEOBJECT pPrevious;
	IDispatch*	pDispatch;
	ITypeInfo*	pTypeInfo;
	ITypeLib*	pTypeLib;
} NTOLEOBJECT; 

static LPNTOLEOBJECT g_lpObj = NULL;

void ReleaseObjects(LPNTOLEOBJECT lpObj)
{
	if(lpObj->pTypeLib != NULL)
		lpObj->pTypeLib->Release();

	if(lpObj->pTypeInfo != NULL)
		lpObj->pTypeInfo->Release();

	if(lpObj->pDispatch != NULL)
		lpObj->pDispatch->Release();
}

LPNTOLEOBJECT NewOleObject(void)
{
	LPNTOLEOBJECT lpObj;
	New(2101, lpObj, 1, NTOLEOBJECT);
	lpObj->NTOLEMagic = NTOLE_MAGIC;
	lpObj->pNext = g_lpObj;
	lpObj->pPrevious = NULL;
	lpObj->pDispatch = NULL;
	lpObj->pTypeInfo = NULL;
	lpObj->pTypeLib = NULL;
	g_lpObj = lpObj;
	return lpObj;
}

LPNTOLEOBJECT NewDispatch(IDispatch* pDisp)
{
	LPNTOLEOBJECT lpObj = NewOleObject();
	lpObj->pDispatch = pDisp;
	return lpObj;
}

LPNTOLEOBJECT NewTypeLib(ITypeLib* ptLib)
{
	LPNTOLEOBJECT lpObj = NewOleObject();
	lpObj->pTypeLib = ptLib;
	return lpObj;
}

void __cdecl NTOLECleanUp(void)
{	// release all unreleased OLE Objects
	LPNTOLEOBJECT lpObj;
	for(lpObj = g_lpObj; lpObj != NULL; lpObj = lpObj->pNext)
		ReleaseObjects(lpObj);

	OleUninitialize();
}


XS(NTOLETypeInfoVar)
{
	dXSARGS;
	unsigned int length;
	LPNTOLEOBJECT lpObj;
	BSTR bstrName, bstrDocString, bstrHelpFile;
	VARDESC* pvardesc;
	BOOL bSuccess = FALSE;

	if(items != 8)
	{
		CROAK("usage: NTOLETypeInfoVar($object, $index, $varName, $varMEMBERID, $varType, $docStr, $helpID, $helpFile);\n");
	}
	lpObj = (LPNTOLEOBJECT)SvIV(ST(0));
	if(lpObj != NULL && lpObj->NTOLEMagic == NTOLE_MAGIC && lpObj->pTypeInfo != NULL)
	{
		if(lpObj->pTypeInfo->GetVarDesc(SvIV(ST(1)), &pvardesc) == S_OK)
		{
			DWORD dwHelpContext;
			if(lpObj->pTypeInfo->GetDocumentation(pvardesc->memid, &bstrName,
									&bstrDocString, &dwHelpContext, &bstrHelpFile) == S_OK)
			{
				char bufA[bufferSize];
				WideCharToMultiByte(CP_ACP, NULL, bstrName, -1, bufA, sizeof(bufA), NULL, NULL);
				SysFreeString(bstrName);
				SETPV(2, bufA);
				SETIV(3, (long)pvardesc->memid);
				SETIV(4, (long)(DWORD)pvardesc->elemdescVar.tdesc.vt);

				WideCharToMultiByte(CP_ACP, NULL, bstrDocString, -1, bufA, sizeof(bufA), NULL, NULL);
				SysFreeString(bstrDocString);
				SETPV(5, bufA);
				SETIV(6, (long)dwHelpContext);

				WideCharToMultiByte(CP_ACP, NULL, bstrHelpFile, -1, bufA, sizeof(bufA), NULL, NULL);
				SysFreeString(bstrHelpFile);
				SETPV(7, bufA);
				bSuccess = TRUE;
			}
			lpObj->pTypeInfo->ReleaseVarDesc(pvardesc);
		}
	}

	RETURNRESULT;
}

XS(NTOLETypeInfoFunc)
{
	dXSARGS;
	unsigned int length;
	LPNTOLEOBJECT lpObj;
	BSTR bstrName, bstrDocString, bstrHelpFile;
	FUNCDESC* pfuncdesc;
	BOOL bSuccess = FALSE;

	if(items != 10)
	{
		CROAK("usage: NTOLETypeInfoFunc($object, $funcNum, $funcName, $funcDISPID, $funcReturnType, $paramCount, $funcMagic, $docStr, $helpID, $helpFile);\n");
	}
	lpObj = (LPNTOLEOBJECT)SvIV(ST(0));
	if(lpObj != NULL && lpObj->NTOLEMagic == NTOLE_MAGIC && lpObj->pTypeInfo != NULL)
	{
		if(lpObj->pTypeInfo->GetFuncDesc(SvIV(ST(1)), &pfuncdesc) == S_OK)
		{
			DWORD dwHelpContext;
			if(lpObj->pTypeInfo->GetDocumentation(pfuncdesc->memid, &bstrName,
									&bstrDocString, &dwHelpContext, &bstrHelpFile) == S_OK)
			{
				char bufA[bufferSize];
				WideCharToMultiByte(CP_ACP, NULL, bstrName, -1, bufA, sizeof(bufA), NULL, NULL);
				SysFreeString(bstrName);
				SETPV(2, bufA);
				SETIV(3, (long)pfuncdesc->memid);
				SETIV(4, (long)(DWORD)pfuncdesc->elemdescFunc.tdesc.vt);
				SETIV(5, (long)(DWORD)pfuncdesc->cParams);

				// make function magic
				bufA[0] = '\0';
				for(length = 0; length < pfuncdesc->cParams; ++length)
					sprintf(&bufA[length*4], "%04x", pfuncdesc->lprgelemdescParam[length].tdesc.vt);

				SETPV(6, bufA);

				WideCharToMultiByte(CP_ACP, NULL, bstrDocString, -1, bufA, sizeof(bufA), NULL, NULL);
				SysFreeString(bstrDocString);
				SETPV(7, bufA);
				SETIV(8, (long)dwHelpContext);

				WideCharToMultiByte(CP_ACP, NULL, bstrHelpFile, -1, bufA, sizeof(bufA), NULL, NULL);
				SysFreeString(bstrHelpFile);
				SETPV(9, bufA);
				bSuccess = TRUE;
			}
			lpObj->pTypeInfo->ReleaseFuncDesc(pfuncdesc);
		}
	}

	RETURNRESULT;
}

XS(NTOLETypeInfoFuncInfo)
{
	dXSARGS;
	unsigned int length;
	LPNTOLEOBJECT lpObj;
	BSTR *pBStrNames;
	FUNCDESC* pfuncdesc;
	BOOL bSuccess = FALSE;

	if(items != 5)
	{
		CROAK("usage: NTOLETypeInfoFuncInfo($object, $index, $paraIndex, $paraName, $paraType);\n");
	}
	lpObj = (LPNTOLEOBJECT)SvIV(ST(0));
	if(lpObj != NULL && lpObj->NTOLEMagic == NTOLE_MAGIC && lpObj->pTypeInfo != NULL)
	{
		if(lpObj->pTypeInfo->GetFuncDesc(SvIV(ST(1)), &pfuncdesc) == S_OK)
		{
			New(2101, pBStrNames, pfuncdesc->cParams, BSTR);
			if(lpObj->pTypeInfo->GetNames(pfuncdesc->memid, pBStrNames, pfuncdesc->cParams, &length) == S_OK)
			{
				char bufA[bufferSize];
				int paramNum = SvIV(ST(2));
				if(length > paramNum)
				{
					WideCharToMultiByte(CP_ACP, NULL, pBStrNames[paramNum], -1, bufA, sizeof(bufA), NULL, NULL);
					SETPV(3, bufA);
					SETIV(4, (long)(DWORD)pfuncdesc->lprgelemdescParam[paramNum].tdesc.vt);
					bSuccess = TRUE;
				}
				else
				{
					SETPV(3, "<not available>");
				}

				for(int index = 0; index < length; ++index)
					SysFreeString(pBStrNames[index]);

			}
			lpObj->pTypeInfo->ReleaseFuncDesc(pfuncdesc);
			Safefree(pBStrNames);
		}
	}

	RETURNRESULT;
}


XS(NTOLETypeInfo)
{
	dXSARGS;
	unsigned int length;
	LPNTOLEOBJECT lpObj;
	BOOL bSuccess = FALSE;

	if(items != 3)
	{
		CROAK("usage: NTOLETypeInfo($object, $funcCount, $varCount);\n");
	}

	lpObj = (LPNTOLEOBJECT)SvIV(ST(0));
	if(lpObj != NULL && lpObj->NTOLEMagic == NTOLE_MAGIC && lpObj->pDispatch != NULL)
	{
		if(lpObj->pTypeInfo != NULL)
		{
			 lpObj->pTypeInfo->Release();
			 lpObj->pTypeInfo = NULL;
		}

		ITypeInfo *ptInfo;
		if(lpObj->pDispatch->GetTypeInfo(0, lcidDefault, &ptInfo) == S_OK)
		{	// we have a type info
			TYPEATTR* ptAttr;
			if(ptInfo->GetTypeAttr(&ptAttr) == S_OK)
			{
				SETIV(1,(long)ptAttr->cFuncs);
				SETIV(2,(long)ptAttr->cVars);
				ptInfo->ReleaseTypeAttr(ptAttr);

				lpObj->pTypeInfo = ptInfo;
				bSuccess = TRUE;
			}
			else
				ptInfo->Release();
		}
	}

	RETURNRESULT;
}

XS(NTOLECreateObject)
{
	dXSARGS;
	CLSID CLSIDObj;
	OLECHAR bBuffer[bufferSize];
	unsigned int length;
	char *buffer;
	HKEY handle;
	IDispatch *pDisp;
	BOOL bSuccess = FALSE;

	if(items != 2)
	{
		CROAK("usage: NTOLECreateObject($class, $object);\n");
	}

	buffer = SvPV(ST(0), length);
	MultiByteToWideChar(CP_ACP, NULL, buffer, -1, bBuffer, sizeof(bBuffer));

	if(CLSIDFromProgID(bBuffer, &CLSIDObj) == S_OK)
	{
		if(CoCreateInstance(CLSIDObj, NULL, CLSCTX_LOCAL_SERVER, IID_IDispatch, (void**)&pDisp) == S_OK)
		{
			SETIV(1,(long)NewDispatch(pDisp));
			bSuccess = TRUE;
		}
	}

	RETURNRESULT;
}

XS(NTOLEDestroyObject)
{
	dXSARGS;
	LPNTOLEOBJECT lpObj;
	BOOL bSuccess = FALSE;

	if(items != 1)
	{
		CROAK("usage: NTOLEDestroyObject($object);\n");
	}

	lpObj = (LPNTOLEOBJECT)SvIV(ST(0));
	if(lpObj != NULL && lpObj->NTOLEMagic == NTOLE_MAGIC)
	{
		ReleaseObjects(lpObj);
			
		// unlink from list
		if(lpObj->pPrevious == NULL)
			g_lpObj = lpObj->pNext;
		else if(lpObj->pNext == NULL)
			lpObj->pPrevious->pNext = NULL;
		else
		{
			lpObj->pPrevious->pNext = lpObj->pNext;
			lpObj->pNext->pPrevious = lpObj->pPrevious;
		}

		Safefree(lpObj);
		SETIV(0,0);
		bSuccess = TRUE;
	}

	RETURNRESULT;
}

XS(NTOLEGetIDofName)
{
	dXSARGS;
	char *buffer;
	unsigned int length;
	LPNTOLEOBJECT lpObj;
	BOOL bSuccess = FALSE;

	if(items != 3)
	{
		CROAK("usage: NTOLEGetIDofName($object, $Name, $NameID);\n");
	}

	buffer = (char *) SvPV(ST(1), length);
	lpObj = (LPNTOLEOBJECT)SvIV(ST(0));
	if(lpObj != NULL && lpObj->NTOLEMagic == NTOLE_MAGIC && lpObj->pDispatch != NULL)
	{
		DISPID id;
		OLECHAR bBuffer[bufferSize], *bBufPtr = bBuffer;
		MultiByteToWideChar(CP_ACP, NULL, buffer, -1, bBuffer, sizeof(bBuffer));
		if(lpObj->pDispatch->GetIDsOfNames(IID_NULL, &bBufPtr, 1, lcidDefault, &id) == S_OK)
		{
			SETIV(2, (long)id);
			bSuccess = TRUE;
		}  
	}

	RETURNRESULT;
}

static void TranslateToVarType(char *pChr, VARTYPE &vt)
{	// assumes 4 characters
	int index, hex;

	for(index = hex = 0; index < 4; ++index, ++pChr)
	{
		hex *= 16;
		switch(*pChr)
		{		  
			case '0':
			case '1':
			case '2':
			case '3':
			case '4':
			case '5':
			case '6':
			case '7':
			case '8':
			case '9':
				hex += *pChr-'0';
				break;

			case 'a':
			case 'b':
			case 'c':
			case 'd':
			case 'e':
			case 'f':
				hex += *pChr-'a'+10;
				break;
			case 'A':
			case 'B':
			case 'C':
			case 'D':
			case 'E':
			case 'F':
				hex += *pChr-'A'+10;
				break;
		}
	}
	vt = (VARTYPE)hex;
}


#define SETiVRETURN(x,f)\
					if(x.vt&VT_BYREF) {\
						SETIV(3, (long)*x.p##f);\
					} else {\
						SETIV(3, (long)x.f);\
					}

#define SETnVRETURN(x,f)\
					if(x.vt&VT_BYREF) {\
						SETNV(3, (double)*x.p##f);\
					} else {\
						SETNV(3, (double)x.f);\
					}
#undef bool

XS(NTOLEMethod)
{
	dXSARGS;
	char *buffer;
	char *ptr;
	unsigned int length;
	int index;
	LPNTOLEOBJECT lpObj;
	BOOL bSuccess = FALSE;

	if(items < 4)
	{
		CROAK("usage: NTOLEMethod($object, $funcMagic, $funcDISPID, $funcReturn, ...);\n");
	}

	buffer = (char *) SvPV(ST(1), length);
	if(items != (length/4)+4)
	{
		char errorBuffer[256];
		sprintf(errorBuffer, "usage: NTOLEMethod($object, $funcMagic, $funcDISPID, $funcReturn, ...);\n\tHas %x parameters but requires %x\n", items, (length/4)+4);
		CROAK(errorBuffer);
	}

	lpObj = (LPNTOLEOBJECT)SvIV(ST(0));
	if(lpObj != NULL && lpObj->NTOLEMagic == NTOLE_MAGIC && lpObj->pDispatch != NULL)
	{
		VARIANT result;
		DISPPARAMS dispParams;
		dispParams.rgvarg = NULL;
		dispParams.rgdispidNamedArgs = NULL;
		dispParams.cNamedArgs = 0;
		dispParams.cArgs = (length/4);

		VariantInit(&result);
		if(dispParams.cArgs > 0)
		{
			New(2101, dispParams.rgvarg, dispParams.cArgs, VARIANTARG);
			for(index = 0; index < dispParams.cArgs; ++index)
			{
				VARTYPE param;
				VariantInit(&dispParams.rgvarg[index]);
				TranslateToVarType(&buffer[index*4], param);
				switch(param)
				{

					case VT_UI1:
						dispParams.rgvarg[index].vt = VT_UI1; 
						dispParams.rgvarg[index].bVal = (unsigned char)SvIV(ST(index+4));
						break;

					case VT_I2:
						dispParams.rgvarg[index].vt = VT_I2; 
						dispParams.rgvarg[index].iVal = (short)SvIV(ST(index+4));
						break;

					case VT_USERDEFINED:
					case VT_I4:
						dispParams.rgvarg[index].vt = VT_I4; 
						dispParams.rgvarg[index].lVal = SvIV(ST(index+4));
						break;

					case VT_BOOL:
						dispParams.rgvarg[index].vt = VT_BOOL; 
						dispParams.rgvarg[index].bool = SvIV(ST(index+4));
						break;

					case VT_R4:
						dispParams.rgvarg[index].vt = VT_R4; 
						dispParams.rgvarg[index].fltVal = (float)SvNV(ST(index+4));
						break;

					case VT_R8:
						dispParams.rgvarg[index].vt = VT_R8; 
						dispParams.rgvarg[index].dblVal = SvNV(ST(index+4));
						break;

					case VT_DATE:
						dispParams.rgvarg[index].vt = VT_DATE; 
						dispParams.rgvarg[index].date = SvNV(ST(index+4));
						break;

					case VT_SAFEARRAY:
					case VT_BSTR:
						dispParams.rgvarg[index].vt = VT_BSTR; 
						ptr = SvPV(ST(index+4), length);
						length += 2;
						dispParams.rgvarg[index].bstrVal = SysAllocStringByteLen(NULL, length*2);
						if(dispParams.rgvarg[index].bstrVal != NULL)
							MultiByteToWideChar(CP_ACP, NULL, ptr, -1, dispParams.rgvarg[index].bstrVal, length);
						break;

					case VT_CY:
					default:
						CROAK("NTOLEMethod: unsupported parameter type");
						break;

				}
			}
				
		}
		DISPID dispID = (DISPID)SvIV(ST(2));
		HRESULT hResult = lpObj->pDispatch->Invoke(dispID, IID_NULL, lcidDefault,
							DISPATCH_METHOD, &dispParams, &result, NULL, NULL);
		if(hResult != S_OK)
		{	// mega kludge
			// if a method in WORD is called and we ask for a result then hResult == DISP_E_EXCEPTION
			// this only happens on functions whose DISPID > 0x8000
			if(hResult == DISP_E_EXCEPTION && dispID > 0x8000)
			{
				VariantClear(&result);
				hResult = lpObj->pDispatch->Invoke(dispID, IID_NULL, lcidDefault,
							DISPATCH_METHOD, &dispParams, NULL, NULL, NULL);
				if(hResult == S_OK)
					bSuccess = TRUE;
			}
		}
		else
			bSuccess = TRUE;


		if(bSuccess)
		{	// handle result
			switch(result.vt&~VT_BYREF)
			{
				case VT_EMPTY:
				case VT_NULL:
					break;

				case VT_UI1:
					SETiVRETURN(result,bVal)
					break;

				case VT_I2:
					SETiVRETURN(result,iVal)
					break;

				case VT_I4:
					SETiVRETURN(result,lVal)
					break;

				case VT_R4:
					SETnVRETURN(result,fltVal)
					break;

				case VT_R8:
					SETnVRETURN(result,dblVal)
					break;

				case VT_BSTR:
					{
						char *pStr;
						if(result.vt&VT_BYREF)
							length = SysStringLen(*result.pbstrVal)+2;
						else
							length = SysStringLen(result.bstrVal)+2;

						New(1110, pStr, length, char);

						if(result.vt&VT_BYREF)
							WideCharToMultiByte(CP_ACP, NULL, *result.pbstrVal, -1, pStr, length, NULL, NULL);
						else
							WideCharToMultiByte(CP_ACP, NULL, result.bstrVal, -1, pStr, length, NULL, NULL);

						SETPV(3, pStr);
						Safefree(pStr);
					}
					break;

				case VT_ERROR:
					SETiVRETURN(result,scode)
					break;

				case VT_BOOL:
					SETiVRETURN(result,bool)
					break;

				case VT_DATE:
					SETnVRETURN(result,date)
					break;

				case VT_DISPATCH:
					if(result.vt&VT_BYREF) 
					{
						SETIV(3,(long)NewDispatch((IDispatch*)*result.ppunkVal));
					}
					else
					{
						SETIV(3,(long)NewDispatch((IDispatch*)result.punkVal));
					}
					break;

				case VT_CY:
				case VT_VARIANT:
				case VT_UNKNOWN:
				default:
					CROAK("NTOLEMethod: unsupported return type");
					break;

			}

		}

		VariantClear(&result);
		if(dispParams.cArgs != 0)
		{
			for(index = 0; index < dispParams.cArgs; ++index)
			{
				if(dispParams.rgvarg[index].vt == VT_BSTR)
					SysFreeString(dispParams.rgvarg[index].bstrVal);
			}
			Safefree(dispParams.rgvarg);
		}
	}

	RETURNRESULT;
}

XS(NTOLEPropertyGet)
{
	dXSARGS;
	unsigned int length;
	LPNTOLEOBJECT lpObj;
	BOOL bSuccess = FALSE;

	if(items != 4)
	{
		CROAK("usage: NTOLEPropertyGet($object, $varMEMBERID, $varType, $varReturn);\n");
	}

	lpObj = (LPNTOLEOBJECT)SvIV(ST(0));
	if(lpObj != NULL && lpObj->NTOLEMagic == NTOLE_MAGIC && lpObj->pDispatch != NULL)
	{
		VARIANT result;

		VariantInit(&result);
		bSuccess = (lpObj->pDispatch->Invoke((DISPID)SvIV(ST(1)), IID_NULL, lcidDefault,
							DISPATCH_PROPERTYGET, NULL, &result, NULL, NULL) == S_OK);

		if(bSuccess)
		{	// handle result
			switch(result.vt&~VT_BYREF)
			{
				case VT_EMPTY:
				case VT_NULL:
					break;

				case VT_UI1:
					SETiVRETURN(result,bVal)
					break;

				case VT_I2:
					SETiVRETURN(result,iVal)
					break;

				case VT_I4:
					SETiVRETURN(result,lVal)
					break;

				case VT_R4:
					SETnVRETURN(result,fltVal)
					break;

				case VT_R8:
					SETnVRETURN(result,dblVal)
					break;

				case VT_BSTR:
					{
						char *pStr;
						if(result.vt&VT_BYREF)
							length = SysStringLen(*result.pbstrVal)+2;
						else
							length = SysStringLen(result.bstrVal)+2;

						New(1110, pStr, length, char);

						if(result.vt&VT_BYREF)
							WideCharToMultiByte(CP_ACP, NULL, *result.pbstrVal, -1, pStr, length, NULL, NULL);
						else
							WideCharToMultiByte(CP_ACP, NULL, result.bstrVal, -1, pStr, length, NULL, NULL);

						SETPV(3, pStr);
						Safefree(pStr);
					}
					break;

				case VT_ERROR:
					SETiVRETURN(result,scode)
					break;

				case VT_BOOL:
					SETiVRETURN(result,bool)
					break;

				case VT_DATE:
					SETnVRETURN(result,date)
					break;

				case VT_DISPATCH:
					if(result.vt&VT_BYREF) 
					{
						SETIV(3,(long)NewDispatch((IDispatch*)*result.ppunkVal));
					}
					else
					{
						SETIV(3,(long)NewDispatch((IDispatch*)result.punkVal));
					}
					break;

				case VT_CY:
				case VT_VARIANT:
				case VT_UNKNOWN:
				default:
					CROAK("NTOLEMethod: unsupported return type");
					break;

			}
			VariantClear(&result);
		}
	}

	RETURNRESULT;
}

XS(NTOLEPropertyPut)
{
	dXSARGS;
	unsigned int length;
	char *buffer;
	LPNTOLEOBJECT lpObj;
	BOOL bSuccess = FALSE;

	if(items != 4)
	{
		CROAK("usage: NTOLEPropertyPut($object, $varMEMBERID, $varType, $varValue);\n");
	}

	lpObj = (LPNTOLEOBJECT)SvIV(ST(0));
	if(lpObj != NULL && lpObj->NTOLEMagic == NTOLE_MAGIC && lpObj->pDispatch != NULL)
	{
		VARIANTARG vArg;
		DISPPARAMS dispParams;
		dispParams.rgvarg = &vArg;
		dispParams.rgdispidNamedArgs = NULL;
		dispParams.cNamedArgs = 0;
		dispParams.cArgs = 1;

		VariantInit(&vArg);
		switch((short)SvIV(ST(2)))
		{

			case VT_UI1:
				vArg.vt = VT_UI1; 
				vArg.bVal = (unsigned char)SvIV(ST(3));
				break;

			case VT_I2:
				vArg.vt = VT_I2; 
				vArg.iVal = (short)SvIV(ST(3));
				break;

			case VT_USERDEFINED:
			case VT_I4:
				vArg.vt = VT_I4; 
				vArg.lVal = SvIV(ST(3));
				break;

			case VT_BOOL:
				vArg.vt = VT_BOOL; 
				vArg.bool = SvIV(ST(3));
				break;

			case VT_R4:
				vArg.vt = VT_R4; 
				vArg.fltVal = (float)SvNV(ST(3));
				break;

			case VT_R8:
				vArg.vt = VT_R8; 
				vArg.dblVal = SvNV(ST(3));
				break;

			case VT_DATE:
				vArg.vt = VT_DATE; 
				vArg.date = SvNV(ST(3));
				break;

			case VT_SAFEARRAY:
			case VT_BSTR:
				vArg.vt = VT_BSTR; 
				buffer = SvPV(ST(3), length);
				length += 2;
				vArg.bstrVal = SysAllocStringByteLen(NULL, length*2);
				if(vArg.bstrVal != NULL)
					MultiByteToWideChar(CP_ACP, NULL, buffer, -1, vArg.bstrVal, length);
				break;

			case VT_CY:
			default:
				CROAK("NTOLEMethod: unsupported parameter type");
				break;

		}
		bSuccess = (lpObj->pDispatch->Invoke((DISPID)SvIV(ST(1)), IID_NULL, lcidDefault,
							DISPATCH_PROPERTYPUT, &dispParams, NULL, NULL, NULL) == S_OK);


		if(vArg.vt == VT_BSTR)
			SysFreeString(vArg.bstrVal);

	}

	RETURNRESULT;
}

XS(NTOLECreateTypeLib)
{
	dXSARGS;
	unsigned int length;
	char *buffer;
	OLECHAR sztlib[bufferSize];
	BOOL bSuccess = FALSE;
	ITypeLib *ptLib;

	if(items != 3)
	{
		CROAK("usage: NTOLECreateTypeLib($fileName, $object, $count);\n");
	}
	
	buffer = (char *) SvPV(ST(0), length);
	// include NULL byte
	MultiByteToWideChar(CP_ACP, NULL, buffer, -1, sztlib, sizeof(sztlib));
	// load the type library
	if(LoadTypeLib(sztlib, &ptLib) == S_OK)
	{
		SETIV(1,(long)NewTypeLib(ptLib));
	    // get the number of type infos
		SETIV(2,(long)ptLib->GetTypeInfoCount());
		bSuccess = TRUE;
	}

	RETURNRESULT;
}

XS(NTOLETypeLibTypeKind)
{
	dXSARGS;
	unsigned int length;
	LPNTOLEOBJECT lpObj;
	BOOL bSuccess = FALSE;

	if(items != 3)
	{
		CROAK("usage: NTOLETypeLibTypeKind($object, $index, $typeKind);\n");
	}

	lpObj = (LPNTOLEOBJECT)SvIV(ST(0));
	if(lpObj != NULL && lpObj->NTOLEMagic == NTOLE_MAGIC && lpObj->pTypeLib != NULL)
	{
		TYPEKIND tkind;
		if((lpObj->pTypeLib->GetTypeInfoType(SvIV(ST(1)), &tkind) == S_OK))
		{
			SETIV(2,(long)tkind);
			bSuccess = TRUE;
		}
	}

	RETURNRESULT;
}

XS(NTOLETypeLibGetTypeInfo)
{
	dXSARGS;
	unsigned int length, dwIndex;
	LPNTOLEOBJECT lpObj;
	BOOL bSuccess = FALSE;

	if(items != 4)
	{
		CROAK("usage: NTOLETypeLibGetTypeInfo($object, $index, $funcCount, $varCount);\n");
	}

	lpObj = (LPNTOLEOBJECT)SvIV(ST(0));
	if(lpObj != NULL && lpObj->NTOLEMagic == NTOLE_MAGIC && lpObj->pTypeLib != NULL)
	{
		TYPEKIND tkind;
		dwIndex = SvIV(ST(1));

		if(lpObj->pTypeInfo != NULL)
		{
			 lpObj->pTypeInfo->Release();
			 lpObj->pTypeInfo = NULL;
		}

		if((lpObj->pTypeLib->GetTypeInfoType(dwIndex, &tkind) == S_OK) && (tkind == TKIND_DISPATCH))
		{
			ITypeInfo *ptInfo;
			if(lpObj->pTypeLib->GetTypeInfo(dwIndex, &ptInfo) == S_OK)
			{	// we have a type info
				TYPEATTR* ptAttr;
				if(ptInfo->GetTypeAttr(&ptAttr) == S_OK)
				{
					SETIV(2,(long)ptAttr->cFuncs);
					SETIV(3,(long)ptAttr->cVars);
					ptInfo->ReleaseTypeAttr(ptAttr);

					lpObj->pTypeInfo = ptInfo;
					bSuccess = TRUE;
				}
				else
					ptInfo->Release();
			}
		}
	}

	RETURNRESULT;
}

XS(NTOLETypeLibGetDispatchCount)
{
	dXSARGS;
	LPNTOLEOBJECT lpObj;
	BOOL bSuccess = FALSE;

	if(items != 2)
	{
		CROAK("usage: NTOLETypeLibGetDispatchCount($object, $infoCount);\n");
	}

	lpObj = (LPNTOLEOBJECT)SvIV(ST(0));
	if(lpObj != NULL && lpObj->NTOLEMagic == NTOLE_MAGIC && lpObj->pTypeLib != NULL)
	{
		SETIV(1,(long)lpObj->pTypeLib->GetTypeInfoCount());
		bSuccess = TRUE;
	}

	RETURNRESULT;
}

static BOOL GetProgInfo(ITypeInfo *ptInfo, char* buffer, int bufferSize, long& nFuncs, long& nVars)
{
	TYPEATTR* ptAttr;
	BOOL bReturn = FALSE;
	if(ptInfo->GetTypeAttr(&ptAttr) == S_OK)
	{
		if(buffer != NULL)
		{
			LPOLESTR psz;
			if(ProgIDFromCLSID(ptAttr->guid, &psz) == S_OK)
			{
				IMalloc FAR* pMalloc;
				if(WideCharToMultiByte(CP_ACP, NULL, psz, -1, buffer, bufferSize, NULL, NULL) != 0)
				{
					nFuncs =(long)ptAttr->cFuncs;
					nVars = (long)ptAttr->cVars;
					bReturn = TRUE;
				}

				if(CoGetMalloc(MEMCTX_TASK, &pMalloc) == S_OK)
				{
					pMalloc->Free(psz);
					pMalloc->Release();
				}
			}
		}
		else
		{	// only want counts
			nFuncs =(long)ptAttr->cFuncs;
			nVars = (long)ptAttr->cVars;
			bReturn = TRUE;
		}
		ptInfo->ReleaseTypeAttr(ptAttr);
	}
	return bReturn;
}

XS(NTOLETypeLibGetDispatchIndex)
{
	dXSARGS;
	LPNTOLEOBJECT lpObj;
	BOOL bSuccess = FALSE;

	if(items != 5)
	{
		CROAK("usage: NTOLETypeLibGetDispatchIndex($object, $index, $dispString, $funcCount, $varCount);\n");
	}

	lpObj = (LPNTOLEOBJECT)SvIV(ST(0));
	if(lpObj != NULL && lpObj->NTOLEMagic == NTOLE_MAGIC && lpObj->pTypeLib != NULL)
	{
		TYPEKIND tkind;
		ITypeInfo *ptInfo;
		long nFuncs, nVars;
		char buffer[bufferSize];
		int index = SvIV(ST(1));

		if(lpObj->pTypeInfo != NULL)
		{
			 lpObj->pTypeInfo->Release();
			 lpObj->pTypeInfo = NULL;
		}

		if(lpObj->pTypeLib->GetTypeInfoType(index, &tkind) == S_OK)
		{
			if((tkind == TKIND_DISPATCH) || (tkind == TKIND_COCLASS))
			{
				if(lpObj->pTypeLib->GetTypeInfo(index, &ptInfo) == S_OK)
				{	// we have a type info
					if(GetProgInfo(ptInfo, buffer, bufferSize, nFuncs, nVars))
					{
						if(tkind == TKIND_COCLASS && nFuncs == 0 && nVars == 0)
						{	// we have a coclass the has nothing fall back to previous index
							ITypeInfo *ptInfo1;
							if(index > 0)
							{
								if(lpObj->pTypeLib->GetTypeInfoType(index-1, &tkind) == S_OK)
								{
									if(tkind == TKIND_DISPATCH)
									{	// only interesting if dispatch
										if(lpObj->pTypeLib->GetTypeInfo(index-1, &ptInfo1) == S_OK)
										{	// we have a type info
											if(GetProgInfo(ptInfo1, NULL, 0, nFuncs, nVars))
											{
												ptInfo->Release();
												ptInfo = ptInfo1;
												bSuccess = TRUE;
											}
										}
									}
								}
							}
						}
						else
							bSuccess = TRUE;

						if(bSuccess)
						{
							SETPV(2, buffer);
							SETIV(3, nFuncs);
							SETIV(4, nVars);
							lpObj->pTypeInfo = ptInfo;
							ptInfo->AddRef();		// so we keep it
						}
					}
					ptInfo->Release();
				}
			}
		}
	}

	RETURNRESULT;
}

XS(NTOLEIsDispatch)
{
	dXSARGS;
	LPNTOLEOBJECT lpObj;
	BOOL bSuccess = FALSE;

	if(items != 1)
	{
		CROAK("usage: NTOLEIsDispatch($object);\n");
	}

	lpObj = (LPNTOLEOBJECT)SvIV(ST(0));
	if(lpObj != NULL && lpObj->NTOLEMagic == NTOLE_MAGIC && lpObj->pDispatch != NULL)
		bSuccess = TRUE;

	RETURNRESULT;
}

XS(NTOLECreateMagicString)
{
	dXSARGS;
	LPNTOLEOBJECT lpObj;
	char *buffer;
	int index;
	BOOL bSuccess = FALSE;

	if(items > 1)
	{
		CROAK("usage: NTOLECreateMagicString($string, ....);\n");
	}

	New(2101, buffer, items*4, char);
	for(index = 0; index < items-1; ++index)
		sprintf(&buffer[index*4], "%04x", SvIV(ST(index+1)));

	SETPV(0,buffer);
	Safefree(buffer);

	bSuccess = TRUE;

	RETURNRESULT;
}

void NTOLEInit(void)
{
	OleInitialize(NULL);

	newXS("NTOLETypeInfoVar", NTOLETypeInfoVar, thisFile);
	newXS("NTOLETypeInfoFunc", NTOLETypeInfoFunc, thisFile);
	newXS("NTOLETypeInfoFuncInfo", NTOLETypeInfoFuncInfo, thisFile);
	newXS("NTOLETypeInfo", NTOLETypeInfo, thisFile);
	newXS("NTOLECreateObject", NTOLECreateObject, thisFile);
	newXS("NTOLEDestroyObject", NTOLEDestroyObject, thisFile);
	newXS("NTOLEGetIDofName", NTOLEGetIDofName, thisFile);
	newXS("NTOLEMethod", NTOLEMethod, thisFile);
	newXS("NTOLEPropertyGet", NTOLEPropertyGet, thisFile);
	newXS("NTOLEPropertyPut", NTOLEPropertyPut, thisFile);
	newXS("NTOLECreateTypeLib", NTOLECreateTypeLib, thisFile);
	newXS("NTOLETypeLibTypeKind", NTOLETypeLibTypeKind, thisFile);
	newXS("NTOLETypeLibGetTypeInfo", NTOLETypeLibGetTypeInfo, thisFile);
	newXS("NTOLETypeLibGetDispatchCount", NTOLETypeLibGetDispatchCount, thisFile);
	newXS("NTOLETypeLibGetDispatchIndex", NTOLETypeLibGetDispatchIndex, thisFile);
	newXS("NTOLEIsDispatch", NTOLEIsDispatch, thisFile);
	newXS("NTOLECreateMagicString", NTOLECreateMagicString, thisFile);
	atexit(NTOLECleanUp);
}
